/**
   <b>UnsatVirusAttenuator</b> class.
<P>
Computes the unsaturated gravity flow water flux and attenuation
factor (<i>A=Mf/cmax</i>) for the given hydrogeologic layer.
Fairly single-minded purpose for existing.
<hr>
<b>References</b>
<p>
Durner,W. Priesack, E., Vogel, H., Zurmuhl, T. 1997,
Determination of Parameters for Flexible
Hydraulic Functions by Inverse Modeling (In) <i>Characterization
and Measurement of the Hydraulic Properties of
Unsaturated Porous Media</i>. pp. 817-829:
<p>
Rose, W., Bruce, W.A. 1949. Evaluation of capillary character in
petroleum reservoir rock. <i>Transactions of the American Institute
of Mining and Metallurgical Engineers 186</i>:127-142.
<p>
Schaefer, C.E., Arands, R.R., van der Sloot, H.A. Kosson, D.S.
Prediction and experimental validation of liquid-phase diffusion
resistance in unsaturated soils. <i>Journal of Contaminant Hydrology
20</i>:145-166.
<p>
Skopp, J., 1985. Oxygen uptake and transport in soils: analysis of
the airwater interfacial area.
<i>Soil Sci. Soc. Am. J. 49</i>(6):1327-1331.

@author Barton R. Faulkner<br>
	U.S. EPA Office of Research and Development<br>
	National Risk Management Research Laboratory<br>
	Ada, Oklahoma, USA<br>
@version 12 February 2002
@see Medium
@see Operandum
*/
public class UnsatVirusAttenuator implements Attenuator {

/* ------------------------
   Constructor
 * ------------------------ */

        /** Construct one.
        */
	public UnsatVirusAttenuator(Medium soildata, Operandum virusdata) {
		sd=soildata;
		vd=virusdata;
		random=new Random(300.,1000);
	}
/* ------------------------
   Public Methods
 * ------------------------ */

	/** Compute the gravity flow water flux.
	@return	qf	The flux rate.
	*/
	public double getAdvection() {
		qf=getKh(sd.theta_m.mean)*(0.1-geth(sd.theta_m.mean))/sd.z.mean+
			getKh(sd.theta_m.mean)*sd.z.mean;
		return qf;
	}
	public double getKh(double wc) {
		Se=(wc-sd.theta_r.mean)/
			(sd.theta_s.mean-sd.theta_r.mean);
		innermost=Math.pow( Se , alog_n/(alog_n-1) );
		double kh=alog_K0*Math.sqrt(Se)*
                   Math.pow(1- Math.pow(1-innermost, 1-1/alog_n) ,2);
		return kh;
	}
	public double geth(double wc) {
		Se=(wc-sd.theta_r.mean)/
                        (sd.theta_s.mean-sd.theta_r.mean);
		innermost=Math.pow(Se, alog_n/(1-alog_n));
		return -(1/ALPHA)*Math.pow(innermost-1,1/alog_n);
	}
	/** Compute the attenuation factor.
	@param q	The flux rate.
	@return Mf/Cmax	The attenuation factor.
	*/
	public double getAttenuation() {
		// Adjust the variables stored as logarithms:
		alog_a=Math.pow(10,sd.a.mean);
		alog_n=Math.pow(10,sd.n.mean);
		alog_K0=Math.pow(10,sd.K0.mean);
		alog_lambda0=Math.pow(10,vd.lambda0.mean);
		alog_lambda1=Math.pow(10,vd.lambda1.mean);
		T=sd.t.mean+273.16;
		q=getAdvection();

		at=3.*(1-sd.theta_s.mean)/sd.rp.mean;
		k=vd.kappa0.mean*at;

		arg=Math.pow(Se , alog_n/(1.-alog_n));

//  This is Rose-Bruce (1949):
		atdia=RHOW*GRAVITY*sd.theta_m.mean*Math.pow(
			arg-1,1/alog_n)/(SIGMA*alog_a);
/*
 *  This is Skopp (1985):
 * 		atdia=1.2*RHOW*GRAVITY*sd.theta_m.mean*Math.pow(
 *			arg-1,1/alog_n)/(SIGMA*alog_a);
 */

/*
 *  This is Cary:
 *		r0=2*SIGMA/(RHOW*GRAVITY*sd.h0.mean);
 *		arg1=-sd.b.mean/(Math.pow(sd.theta_s.mean,-sd.b.mean)-
 *			Math.pow(LESSTINY,-sd.b.mean));
 *		arg2=(Math.pow(sd.theta_s.mean,-sd.b.mean+1)-Math.pow(
 *			LESSTINY,-sd.b.mean+1))/
 *				(-sd.b.mean+1);
 *		eta=at*r0/(2*LESSTINY*Math.pow(sd.theta_s.mean,sd.b.mean))*
 *			arg1-arg2*arg1/LESSTINY;
 * 		atdia=2*Math.pow(sd.theta_s.mean,sd.b.mean)/r0*(
 *					eta*sd.theta_r.mean/arg1+arg2);
 */
		kdia=vd.kappa1.mean*atdia;
		gamma=alog_lambda0
				+alog_lambda1*sd.rho.mean*k
				/(k*sd.theta_m.mean/vd.Kd.mean+alog_lambda1*sd.rho.mean)
				+kdia;
		V=q/sd.theta_m.mean;
		// Use Schaefer et al (1995):
		if (sd.theta_m.mean<=0.2) {
			tau=Math.pow(sd.theta_s.mean,2)/
				Math.pow(sd.theta_m.mean,11/5);
		} else {
			tau=Math.pow(sd.theta_s.mean,2)/
				Math.pow(sd.theta_m.mean,7/3);
		}
		D=BOLTZMANN*T/(6*Math.PI*VISCOSITY*vd.rv.mean);
		De=D*tau;
		Dz=sd.alphaz.mean*V+De;
		Gamma=(V-Math.sqrt(V*V+4*Dz*gamma))/(2*Dz);
		A=Math.exp(Gamma*sd.z.mean);
 		// Cannot create viruses and cannot have less than zero, hence
		if (A>1.) A=1.;
		if (A<=INFINITESIMAL) {
			pA=-random.nextDouble();
		} else {
			pA=HspBasicMath.log10(A);
		}
		return pA;
	}
	public Medium getMedium() {
		return sd;
	}
	public Operandum getOperandum() {
		return vd;
	}
	public void replace(Medium sdat, Operandum vdat) {
		sd=sdat;
		vd=vdat;

		/*
		 *  Perform some checks on the physical reasonableness
		 *  of the perturbed parameters, since they were
		 *  generated randomly, and condition them as necessary:
		 *
		 *  First, all normally distributed parameters
		 *  must be > zero:
		 */

		if (sd.theta_s.mean<=0.) sd.theta_s.mean=INFINITESIMAL;
		if (sd.theta_r.mean<=0.) sd.theta_r.mean=INFINITESIMAL;
		if (sd.theta_m.mean<=0.) sd.theta_m.mean=INFINITESIMAL;
		if (sd.z.mean<=0.) sd.z.mean=INFINITESIMAL;

		/*
		 *  We use n<10 (or log10(n)<1) (Durner et al. 1997)
		 */

		if (sd.n.mean>1.) sd.n.mean=1.-INFINITESIMAL;
		if (sd.rho.mean<=0.) sd.rho.mean=INFINITESIMAL;
		if (sd.rp.mean<=0.) sd.rp.mean=INFINITESIMAL;
		if (vd.kappa0.mean<0.) vd.kappa0.mean=INFINITESIMAL;
		if (vd.kappa1.mean<=0.) vd.kappa1.mean=INFINITESIMAL;
		if (vd.Kd.mean<=0.) vd.Kd.mean=INFINITESIMAL;

		/*
		 *  Next, all water contents must also be less than 1:
		 */

		if (sd.theta_s.mean>=1) sd.theta_s.mean=1-INFINITESIMAL;
		if (sd.theta_r.mean>=1) sd.theta_r.mean=1-INFINITESIMAL;
		if (sd.theta_m.mean>=1) sd.theta_m.mean=1-INFINITESIMAL;

		/*
		 *  Next, the residual moisture must be less than
		 *  the moisture, and we place more trust in the
		 *  in the moisture value:
		 */

		if (sd.theta_r.mean>=sd.theta_m.mean) {
			sd.theta_r.mean=sd.theta_m.mean-LESSTINY;
		}

		/*
		 *  Next, the saturated moisture must be greater
		 *  than the soil moisture content:
		 */

		if (sd.theta_s.mean<=sd.theta_m.mean) {
			sd.theta_s.mean=sd.theta_m.mean+LESSTINY;
		}
	}
	private Random random;
	private Medium sd;
	private Operandum vd;
	private double alog_a, alog_n, alog_K0, alog_lambda0, alog_lambda1;
	private double q, innermost, qf, T;
	private double at, k, atdia, arg;
	private double kdia, gamma, V, tau, D, De;
	private double Se, Dz, Gamma, A, pA;
	private static final double ALPHA=1.0;
	private static final double SIGMA=962020800;		// g*m/hr^2
	private static final double RHOW=1.e6;			// g/m^3
	private static final double GRAVITY=127008000.;		// m/hr^2
	private static final double BOLTZMANN=1.789332768e-13;	//(g*m)/hr^2 (CRC)
	private static final double VISCOSITY=4.705200006e15;	// g/(m*hr) (CRC)
	private static final double INFINITESIMAL=1e-300;
	private static final double TINY=1e-15;			// A small number,
								// but not the smallest.
	private static final double LESSTINY=1e-6;		// Another small number.
}
