classdef class_REVS_vehicle
    %class_REVS_vehicle
    %	Vehicle description class for REVS model for single axle drive
    
    properties
        
        name;							% vehicle Description
        class;							% vehicle Type Specifier
        fuel;							% vehicle Fuel Specifier - Specified via class_REVS_fuel
        
        mass_static_kg;                 % Vehicle static mass (curb weight)
        mass_dynamic_kg;                % vehicle equivalent inertia mass, e.g. static mass * 1.03
        
        delta_static_mass_kg;           % dynamic lookup for altering static mass during a drive cycle
        delta_dynamic_mass_kg;          % dynanic lookup for altering dynamic mass during a drive cycle
        
        use_abc_roadload@logical;		% Use ABC roadload vs Cd & Crr
        
        coastdown_A_N;                  % coastdown target "A" term, SI units
        coastdown_B_Npms;               % coastdown target "B" term, SI units
        coastdown_C_Npms2;              % coastdown target "C" term, SI units
        
        frontal_area_m2;                % vehicle frontal area, square meters
        aerodynamic_drag_coeff;         % vehicle aerodynamic drag coefficient
        
        speed_limit_mps = inf;          % vehicle speed limiter, truncates drive trace (?)
        
        drive_axle1 = class_REVS_drive_axle;	% Drive Axle Definition Class
        
        max_brake_force_N;			% Total combined braking torque of all axles
        
    end
    
    methods
        
        function obj = class_REVS_vehicle( )
            obj.delta_static_mass_kg.axis_1.signal       = 'cycle_phase_num';
            obj.delta_static_mass_kg.axis_1.breakpoints  = [0 1];
            obj.delta_static_mass_kg.table               = [0 0];
            
            obj.delta_dynamic_mass_kg.axis_1.signal      = 'cycle_phase_num';
            obj.delta_dynamic_mass_kg.axis_1.breakpoints = [0 1];
            obj.delta_dynamic_mass_kg.table              = [0 0];
        end
        
        function val = get.max_brake_force_N(obj)
            val = REVS_class_default( obj.max_brake_force_N , obj.mass_static_kg * 9.8 * 1.75);  % changed from 2.0 to 1.4 for REVS1 compatible brake gain and alternator regen...
        end
        
        
        function val = get.drive_axle1( obj )
            
            %Set axle parameters that require vehicle level inputs
            val = obj.drive_axle1;
            
            val.brake.max_torque_Nm = REVS_class_default( val.brake.max_torque_Nm, obj.max_brake_force_N * val.brake.proportion_norm * val.tire.radius_m);
            
            val.tire.wheel_slip_torque_Nm = REVS_class_default( val.tire.wheel_slip_torque_Nm, obj.mass_static_kg * val.tire.vehicle_weight_norm * 9.81 * val.tire.radius_m);
            
        end
        
        
        function val = get.coastdown_A_N(obj)
            if obj.use_abc_roadload
                val = obj.coastdown_A_N;
            else
                val = REVS_class_default(obj.coastdown_A_N,0);
            end
        end
        
        function val = get.coastdown_B_Npms(obj)
            if obj.use_abc_roadload
                val = obj.coastdown_B_Npms;
            else
                val = REVS_class_default(obj.coastdown_B_Npms,0);
            end
        end
        
        function val = get.coastdown_C_Npms2(obj)
            if obj.use_abc_roadload
                val = obj.coastdown_C_Npms2;
            else
                val = REVS_class_default(obj.coastdown_C_Npms2,0);
            end
        end
        
        function val = get.aerodynamic_drag_coeff(obj)
            if obj.use_abc_roadload
                val = REVS_class_default(obj.aerodynamic_drag_coeff,0);
            else
                val = obj.aerodynamic_drag_coeff;
            end
        end
        
        function val = get.frontal_area_m2(obj)
            if obj.use_abc_roadload
                val = REVS_class_default(obj.frontal_area_m2,0);
            else
                val = obj.frontal_area_m2;
            end
        end
        
        
    end
    
end

