function  exe_info = REVS_build_exe( mdl, varargin )
%UNTITLED Summary of this function goes here
%   Detailed explanation goes here

mdl_exe = parse_varargs(varargin,'model_exe_name',[mdl, '_exe'],'char');

%params = parse_varargs(varargin,'tunable_parameters',{'REVS','engine','transmission','electric','vehicle','driver','ambient','controls'});
params = parse_varargs(varargin,'tunable_parameters',{});

lookups = parse_varargs(varargin,'dynamic_size_lookups',{});

variable_drive_cycle = parse_varargs(varargin,'variable_drive_cycle',false,'toggle');

output_generate_mat = parse_varargs( varargin,'disable_mat_output',true,'toggle');
output_remove_verbose_logging =  parse_varargs( varargin,'remove_verbose_logging',inf,'numeric');
output_buffer_size = parse_varargs(varargin,'output_buffer_size',-1,'numeric');

custom_source = parse_varargs( varargin, 'custom_source',{});
custom_include = parse_varargs( varargin, 'custom_include',{});
custom_library = parse_varargs( varargin, 'custom_library',{});

close_model = parse_varargs( varargin, 'leave_model_open', true, 'toggle');
cleanup_files = parse_varargs( varargin, 'cleanup_files', false, 'toggle');


exe_info.base_parameters = params;
exe_info.full_parameters = REVS_full_param_list(params);
% exe_info.lookup_info = {};
exe_info.dyn_expr_info = {};
exe_info.dyn_lookup_info = {};
exe_info.nD_lookup_info = {};

% Disable Warning about modifying parameter in a link
warning('off','Simulink:Commands:SetParamLinkChangeWarn');
% Disable Warning about bus signal names remapping - when variant changes
warning('off','Simulink:blocks:BusSelectorOutSignalNameUpdate');
% Disable Warning about tunable parameters in non-tunable field
warning('off','Simulink:Parameters:TunablePrmInNonTunableField');
% Disable Warnig About Saving with parameterized links
warning('off','Simulink:Engine:SaveWithParameterizedLinks_Warning');
%
warning('off','Simulink:blocks:RefBlockUnknownParameter');


% Make a copy to work on
open(mdl);					% Open model (if not already open)
bdclose(mdl_exe);			% Close old exe model (if open)
save_system(mdl,mdl_exe);	% Save copy for executable creation

% Setup EXE Options
set_param(mdl_exe,'SystemTargetFile','grt.tlc')
%set_param(mdl_exe,'TemplateMakefile','REVS_vcx64.tmf');
%set_param(mdl_exe,'MakeCommand','make_rtw OPTS="-DDEFAULT_BUFFER_SIZE=1048576"')

set_param(mdl_exe,'TargetHWDeviceType','Generic->32-bit x86 compatible');

set_param(mdl_exe,'UnderspecifiedInitializationDetection','Simplified');
set_param(mdl_exe,'SupportNonFinite','on');


set_param(mdl_exe,'ProdHWDeviceType','Intel->x86-64')
set_param(mdl_exe,'BuildConfiguration','Faster Runs')


% Setup Lookup Tables
% if ~isempty(lookups)
% 	fprintf('\nConfiguring Dynamic Sized Tables\n');
% 	exe_info.lookup_info = REVS_setup_dynamic_table_size(mdl_exe, lookups );
% end

exe_info.dyn_expr_info = REVS_collect_dynamic_expr_info( mdl_exe );
exe_info.dyn_lookup_info = REVS_setup_dynamic_lookups( mdl_exe, params );
exe_info.nD_lookup_info = REVS_setup_nD_lookups( mdl_exe, params );


if variable_drive_cycle
	REVS_variable_drive_cycle( mdl_exe , 'drive_cycle.cycle_duration_secs' );
end



% Set Up Model Parametization
set_param(mdl_exe,'InlineParams','on')

% Handle Model Parameter Inputs
if ~isempty( params )
	
	for i = 1:length(params)
		param = evalin('base',params{i});
		param =  REVS_cleanup_exe_structure(param,params{i});
		assignin('base',params{i},param);
	end
	
	
	
	tunable_var_str = [params{1}, sprintf(', %s',params{2:end})];
	tunable_class_str = regexprep(tunable_var_str, '[^,]+','Auto');
	tunable_type_str = regexprep(tunable_var_str, '[^,]+','');
	
	set_param(mdl_exe,'TunableVars',tunable_var_str)
	set_param(mdl_exe,'TunableVarsStorageClass',tunable_class_str)
	set_param(mdl_exe,'TunableVarsTypeQualifier',tunable_type_str)
	
	set_param(mdl_exe,'RTWCAPIParams','on')
	
	% Source for custom input parser
	custom_source{end+1} = 'param_input.c';
end


% Change Solver if Not Fixed Step
if ~strcmp(get_param(mdl_exe,'SolverType') ,'Fixed-step')
	fprintf('\nChanging Solver to Fixed Step ODE 3!\n');
	set_param(mdl_exe,'solver','ode3');
end

% Only generate MAT file output if requested (default)
if ~output_generate_mat
	set_param(mdl_exe,'MatFileLogging','off')
else
	set_param(mdl_exe,'MatFileLogging','on')
	set_param(mdl_exe,'LogVarNameModifier','none')
	
	if(  output_buffer_size > 0 )
		REVS_update_to_workspace( mdl_exe, output_buffer_size );
	end
	
	if( output_remove_verbose_logging < Inf )
		REVS_remove_verbose_logging(mdl_exe, output_remove_verbose_logging)
	end
	
end

% Source for custom input parser
set_param(mdl_exe,'CustomSource', strjoin(custom_source, ' '));
set_param(mdl_exe,'CustomInclude', strjoin(custom_include, ' '));
set_param(mdl_exe,'CustomLibrary', strjoin(custom_library, ' '));

% Save Model Before Build
fprintf('\nSaving model before build\n');
save_system(mdl_exe,mdl_exe);


% Build Simulink Executable
fprintf('\nBeginning code generation & compilation!\n');
slbuild(mdl_exe);


if close_model
	bdclose(mdl_exe);
end

if cleanup_files
	delete([mdl_exe,'.slx']);
	delete([mdl_exe,'_sfun.mexw32']);
	delete([mdl_exe,'_sfun.mexw64']);
	rmdir( [mdl_exe,'_grt_rtw'],'s');
end



% Restore Warnings
warning('on','all')

end

function c = REVS_clean_block_path(str)

bad = ismember(str, char([9 10 11 12 13]));

c = str;
c(bad) = [];

end


function list = REVS_full_param_list(list_in)

list = list_in;

idx = 1;
while idx <= length(list)
	
	if( evalin('base',['isstruct( ',list{idx} ,');']) )
		base_name = [list{idx},'.'];
		add_fields = strcat(base_name , evalin('base',['sort( fieldnames( ',list{idx}, '));'] ));
		list = {list{:}, add_fields{:}};
		list(idx) = [];
	else
		idx = idx+1;
	end
	
end

end


function REVS_remove_verbose_logging(mdl, removoval_level)

fprintf('\nRemoving datalog blocks with verbosity >= %d.\n', removoval_level);

blks = find_system(mdl,'lookundermasks','all','followlinks','on','regexp','on','ReferenceBlock','^.*to datalog[^/]*$','BlockType','SubSystem');
for i = 1:length(blks)
	if( str2double(get_param(blks{i},'VerboseLevel')) >= removoval_level )
		fprintf(' - Commenting out block %s\n', blks{i})
		%set_param(blks{i},'LinkStatus','inactive');
		set_param(blks{i},'commented','on');
	end
end


if removoval_level <= 1
	
	fprintf('\nRemoving audit blocks.\n');
	blks = find_system(mdl,'lookundermasks','all','followlinks','on','regexp','on','ReferenceBlock','^audit_lib.*$','BlockType','SubSystem');
	for i = 1:length(blks)
		fprintf(' - Commenting out block %s\n', blks{i})
		%set_param(blks{i},'LinkStatus','inactive');
		set_param(blks{i},'commented','on');
	end
	
end
end


function  REVS_update_to_workspace( mdl, pts )
%REVS_update_to_workspace(mdl)
%   Change settings of to workspace blocks for executable generations

fprintf('\nUpdating to workspace bolcks to set maximum buffer size. \n');
blks = find_system(mdl,'lookundermasks','all','followlinks','on','regexp','on','blocktype','ToWorkspace');

for i = 1:length(blks)
	
	mdl_pts = eval(get_param(blks{i},'MaxDataPoints' ));
	
	if mdl_pts > pts
		fprintf(' - Updating Size of block %s\n', blks{i})
		set_param(blks{i},'MaxDataPoints', num2str(pts) );
	end
end

end


function dyn_expr_info = REVS_collect_dynamic_expr_info( mdl )

blks = find_system(mdl,'lookundermasks','all','followlinks','on','regexp','on','blocktype','SubSystem','ReferenceBlock','general_lib/dynamic equation' );
dyn_expr_info = cell(1,length(blks));

for i = 1:length(blks)
	
	dyn_expr_info{i}.block = blks{i};
	dyn_expr_info{i}.expression = get_param( blks{i}, 'DynamicExpressionEval');
	expr = get_param( blks{i}, 'DynamicExpression');
	
	if regexp(expr,'^''.*''$')
		% String definition '????'
		dyn_expr_info{i}.expression_param = expr;
	else
		% Variable Reference
		dyn_expr_info{i}.expression_param  = REVS_get_complete_param(blks{i}, expr);
	end
end

end

function param_str = REVS_get_complete_param( blk, expr )

blk = strrep( blk, '//','<-|@@|->');

blk_path = strsplit(blk,'/','CollapseDelimiters',false);

blk_path = strrep( blk_path,'<-|@@|->','//');

[param_base, param_remain] = strtok( expr ,'.');
param_str = expr;

for b = (length(blk_path)-1):-1:2
	
	curr_blk = strjoin(blk_path(1:b),'/');
	block_params = fieldnames(get_param(curr_blk,'DialogParameters'));
	if ismember( param_base, block_params )
		
		base_expr = get_param(curr_blk,param_base);
		param_str = strcat(base_expr,param_remain);
		
		[param_base, param_remain] = strtok( param_str ,'.');
	end
	
end


end


function dyn_lookup_info = REVS_setup_dynamic_lookups( mdl, params )

blks = find_system(mdl,'lookundermasks','all','followlinks','on','regexp','on','blocktype','SubSystem','ReferenceBlock','general_lib/dynamic lookup' );
dyn_lookup_info = cell(1,length(blks));

fprintf('Updating Dynamic Lookup blocks for dynamic sizing...\n');
l = 1;
for i = 1:length(blks)
	
	fprintf(' -Updating Dynamic Lookup Block: %s \n',REVS_clean_block_path(blks{l}));
	
	
	in_var = REVS_get_complete_param(blks{i}, get_param( blks{i}, 'LookupStructure') );
	in_var_base = strtok(in_var,'.');
	
	if ~ismember( in_var_base, params)
		continue;	% Table isn't a tunable param
	end
	
	
	dyn_lookup_info{l}.block = blks{l};
	dyn_lookup_info{l}.block_param = get_param( blks{l}, 'LookupStructure');
	
	dyn_lookup_info{l}.base_param = in_var;
	%     dyn_lookup_info{i}.definition = evalin('base',in_var);
	
	var_size = evalin('base',['size( ' in_var '.table) ;' ]) ;
	var_size = var_size( var_size > 1 );
	dyn_lookup_info{l}.max_dims = var_size;
	
	for a = 1:length(var_size)
		dyn_lookup_info{l}.bus_signals{a} = evalin('base',[in_var,'.axis_',int2str(a),'.signal']);
	end
	
	size_var = [in_var,'.table_size'];
	fprintf(' --Adding Size Variable to Workspace: %s \n',size_var);
	evalin('base',[size_var,' = uint32( [' num2str(var_size - 1) '] );' ]);
	set_param(blks{l},'SupportTunableTableSize','on');
	
	l = l+ 1;
	
end


end

function nD_lookup_info = REVS_setup_nD_lookups( mdl , params)

blks = find_system(mdl,'lookundermasks','all','followlinks','on','regexp','on','blocktype','Lookup_n-D' );
dyn_blks = find_system(mdl,'lookundermasks','all','followlinks','on','regexp','on','blocktype','SubSystem','ReferenceBlock','general_lib/dynamic lookup' );

nD_lookup_info = {}; %cell(1,length(blks));

fprintf('Updating nD Lookup blocks for dynamic sizing...\n');


l = 1;
for i = 1:length(blks)
	
	
	% If nD Lookup is part of a dynamic lookup skip it
	j = 1;
	
	while j <= length(dyn_blks) && ~strncmp( dyn_blks{j} , blks(i), length(dyn_blks{j}) )
		j = j + 1;
	end
	
	if j <= length(dyn_blks)
		continue;
	end
	
	% Get Base variable
	table_param = get_param( blks{i}, 'Table');
	
	if  any(ismember( '[:]',table_param))
		continue;
	end
	
	in_var = REVS_get_complete_param(blks{i}, table_param );
	in_var_base = strtok(in_var,'.');
	
	if ~ismember( in_var_base, params)
		continue;	% Table isn't a tunable param
	end
	
	
	fprintf(' -Updating nD Lookup Block: %s \n',REVS_clean_block_path(blks{i}));
	
	nD_lookup_info{l}.block = blks{i};
	nD_lookup_info{l}.block_param = table_param;
	nD_lookup_info{l}.base_param = in_var;
	%     nD_lookup_info{l}.lookup_definition = evalin('base',in_var);
	
	var_size = evalin('base',['size( ' in_var ') ;' ]) ;
	var_size = var_size( var_size > 1 );
	nD_lookup_info{l}.max_dims = var_size;
	
	size_var = [in_var,'_size'];
	fprintf(' --Adding Size Variable to Workspace: %s \n',size_var);
	evalin('base',[size_var,' = uint32( [' num2str(var_size - 1) '] );' ]);
	blk_size_var = [table_param,'_size'];
	set_param(blks{i},'SupportTunableTableSize','on');
	set_param(blks{i},'MaximumIndicesForEachDimension', blk_size_var);
	
	l = l + 1;
	
	
end

end



function REVS_variable_drive_cycle( mdl , cycle_length_var )

% Change end of cycle time
set_param([mdl,'/driver/drive cycle lookups/end of cycle'],'const',cycle_length_var);

end