/*
 * Copyright 2004-2008 The MathWorks, Inc.
 *
 * File: rtw_capi_examples.c     $Revision: 1.1.4.3 $
 *
 * Abstract:
 *   Provides utility functions to traverse and access RTW generated C-API
 *   structures. The C-API structure types are provided in following 2 files.
 *
 *       matlabroot/rtw/c/src/rtw_capi.h & rtw_modelmap.h
 *
 *   The actual data structures arrays are generated by code generator in 
 *   <MODEL>_capi.c, where <MODEL> is the name of the model.
 * 
 *   The functions provided in this file are
 *       capi_PrintModelParameter - Prints a model parameter value to STDOUT
 */


#include <stdio.h>
#include <assert.h>
#include <math.h>
#include "rtwtypes.h"
#include "rtw_modelmap.h"
#include "rtw_capi.h"
#include <stddef.h>
#include <string.h>
#include <malloc.h>
//#include <ctype.h>


/*
 * Function to read the param description file.
 */
#define   MAX_LINE_SZ       4096
//#define   PARAM_NAME_SIZE   4096

#define		DEBUG_LEVEL		0
//sscanf( val_str, "%a", &val );\

#define		HANDLE_DATA_LINE( TYPE, LINE ) \
{\
\
	TYPE		*param_ptr = model_line_start_addr;\
	char_T		*val_str = strtok(LINE, " \t\n");\
	real64_T	val;\
	real64_T	prev_val;\
	while( val_str != NULL && value_array_idx < model_param_size)\
	{\
		prev_val = (real64_T) param_ptr[value_array_idx];\
		if( strncmp( val_str, "Inf", 3 ) == 0)             val = 0x7FF0000000000000;\
        else if( strncmp( val_str, "-Inf", 4 ) == 0)       val = 0xFFF0000000000000;\
        else if( strncmp( val_str, "NaN", 3 ) == 0 )       val = 0x7FF8000000000000;\
        else                                               val = strtod( val_str, NULL );\
		if( DEBUG_LEVEL >= 10 ) 		printf("Change %s @%x to from %lf -> %lf \n", file_param_name, ((char*) model_line_start_addr) + value_array_idx* model_type_size, prev_val, val); \
		param_ptr[value_array_idx++] = ( TYPE ) val;\
		val_str = strtok(NULL, " \t\n");\
	}\
	if( val_str != NULL && value_array_idx >= model_param_size)\
		printf("Parameter '%s' in file exceeds size %d in model\n", file_param_name, model_param_size);\
}\


int load_parameters( rtwCAPI_ModelMappingInfo* capiMap, char_T* in_file)
{
   FILE    *fd;
	char_T	line[MAX_LINE_SZ];
	int32_T	line_num = 0;
	int32_T	i;

	char_T			*file_param_name;
    char_T			*file_param_indexing;
	char_T			*file_param_name_base;
	char_T			*file_param_element;
	
    char_T          *index_str;
    uint32_T        index_val;
    uint32_T        index_dim;
	
	int32_T			model_num_params;

	rtwCAPI_ModelParameters			*model_params;
	int32_T							model_param_index = -1;
	void							**model_addr_map;
	void							*model_param_addr;
    void							*model_line_start_addr;
 //   void							*model_current_addr;

	rtwCAPI_DataTypeMap				*model_type_map;
	int32_T							model_type_index;
	uint8_T							model_type;
	uint16_T						model_type_size;

	rtwCAPI_ElementMap				*model_element_map;
	int32_T							model_element_index;
	int32_T							model_element_index_end;
	
	rtwCAPI_DimensionMap			*model_dim_map;
	uint_T							*model_dim_array;
    uint32_T						model_dim_start;
	uint32_T						model_dim_index;
	uint32_T						model_param_size;
	uint32_T                        model_block_size;
    uint32_T                        model_sub_dim_block_size;
    uint32_T                        model_line_step_size;    
    
	uint32_T						value_array_idx;
	
	char_T							*model_param_name;

	model_num_params = rtwCAPI_GetNumModelParameters(capiMap);
    

	// Check of Model Contains Parameters
	if( model_num_params <= 0 )
	{
		printf("No Tunable Model Parameters in the model \n");
        fflush(stdout);
		return 0;
	}
	
	// Get Components of Model CAPI Map
	model_params = rtwCAPI_GetModelParameters(capiMap);
	model_addr_map = rtwCAPI_GetDataAddressMap(capiMap);
	model_type_map = rtwCAPI_GetDataTypeMap(capiMap);
	model_element_map = rtwCAPI_GetElementMap(capiMap);
	model_dim_map = rtwCAPI_GetDimensionMap(capiMap);
	model_dim_array = rtwCAPI_GetDimensionArray(capiMap);
	
	// Open the parameter description file 
	fd = fopen(in_file, "r");
	if (fd == NULL) 
	{
		printf("Unable to open parameter descriptor file\n");
        fflush(stdout);
		return  1;
	}

	printf("** loading model parameters **\n");
    
	// Loop to End of File
	while ( !feof(fd))
	{
        
		// Read a Line
		line_num += 1;
		fgets(line, MAX_LINE_SZ, fd);
		if( DEBUG_LEVEL >= 7)               printf("\nLine %d: %s", line_num, line);
        fflush(stdout);
		
		i = 0;
		while( line[i] != '\0' && isspace(line[i]) ) 		i++;
		
 		if( line[i] == '\0' )			continue;		// Nothing on This Line
		
		if( strncmp(&(line[i]), "Parameter: ", 10 ) == 0 )	// New Parameter Section	
		{

			// Skip Leading White Space 
			file_param_name = strdup(strtok(&(line[i+10])," \t\n"));
            
            file_param_name = strtok( file_param_name,"(");
            file_param_indexing= strtok( NULL,  ")");
                        
			if( DEBUG_LEVEL >= 1)		printf("\nVariable: '%s' Indexing: '%s' \n", file_param_name, file_param_indexing);
			fflush(stdout);
            
			file_param_name_base = strdup(file_param_name);
			file_param_name_base = strtok( file_param_name_base, ".\n");
			
			if( DEBUG_LEVEL >= 2 )		printf("Base Parameter: '%s' \n", file_param_name_base);
			fflush(stdout);
			
			model_param_index = model_num_params - 1;
			while( model_param_index >= 0  )
			{
				model_param_name   = rtwCAPI_GetModelParameterName(model_params, model_param_index);
				if( strcmp(file_param_name_base, model_param_name) == 0 )		break;
				model_param_index--;
			}
			
			// model_param_index < 0  --> parameter not in model
			if( model_param_index < 0)				
			{
				printf ("Parameter '%s' not found in model!\n", file_param_name);
				continue;
			}

			// Get Parameter details
			model_param_addr = (void *) rtwCAPI_GetDataAddress(model_addr_map, rtwCAPI_GetModelParameterAddrIdx(model_params, model_param_index));
			model_type_index = rtwCAPI_GetModelParameterDataTypeIdx(model_params, model_param_index);			
			model_dim_index = rtwCAPI_GetModelParameterDimensionIdx(model_params, model_param_index);

			// Look Up Model Parameter Type
			model_type = rtwCAPI_GetDataTypeSLId(model_type_map, model_type_index);

			
			file_param_element = strtok(NULL,". \t\n");
			
			if( DEBUG_LEVEL >= 2 )		printf("Class: %d MW Name: %s @ Address: %x \n", model_type,  rtwCAPI_GetDataTypeMWName(model_type_map, model_type_index ), model_param_addr);
			fflush(stdout);
            
            while( file_param_element != NULL && model_type == SS_STRUCT  ) 
			//while( model_param_index >= 0 && file_param_element != NULL && model_type == SS_STRUCT  ) 
			{
                if( DEBUG_LEVEL >= 2 )		printf("Parameter Element: '%s' \n",file_param_element);

				model_element_index = rtwCAPI_GetDataTypeElemMapIndex(model_type_map,model_type_index);
				model_element_index_end = model_element_index + rtwCAPI_GetDataTypeNumElements(model_type_map, model_type_index) -1;
			 
                
				if( DEBUG_LEVEL >= 3 )		printf("Check Elements: %d - %d \n", model_element_index, model_element_index_end);
                fflush(stdout);
				            
                // Find Matching Element
				while( model_element_index <= model_element_index_end && strcmp( file_param_element, rtwCAPI_GetElementName(model_element_map, model_element_index)) != 0 )
                {
                    if( DEBUG_LEVEL >= 5 )		printf("Checking Element Index %d: Name: '%s' \n", model_element_index, rtwCAPI_GetElementName(model_element_map, model_element_index) );
                    fflush(stdout);                     
                    model_element_index++;
                }
				
                // Check if end reached without finding match
				if( model_element_index > model_element_index_end)
				{
					//Sub field is not a member of the model structure - set as if parameter does not exist
					model_param_index = -1;
					break;
				}
			
                if( DEBUG_LEVEL >= 3 )		printf("Found Element Index %d: Name: '%s' Offset: %d \n", model_element_index, rtwCAPI_GetElementName(model_element_map, model_element_index), rtwCAPI_GetElementOffset(model_element_map, model_element_index) );              
                fflush(stdout);
                
				model_param_addr = ((char *) model_param_addr)  + rtwCAPI_GetElementOffset(model_element_map, model_element_index);
				model_type_index = rtwCAPI_GetElementDataTypeIdx(model_element_map, model_element_index);
				model_dim_index = rtwCAPI_GetElementDimensionIdx(model_element_map, model_element_index);

				model_type = rtwCAPI_GetDataTypeSLId(model_type_map, model_type_index);
				model_type_size = rtwCAPI_GetDataTypeSize(model_type_map, model_type_index);
				
				if( DEBUG_LEVEL >= 2 )		printf("Class: %d MW Name: %s @ Address: %x \n", model_type,  rtwCAPI_GetDataTypeMWName(model_type_map, model_type_index ), model_param_addr);
                fflush(stdout);
            
				file_param_element = strtok(NULL,". \t\n");
				
			 }
			
			if( model_param_index < 0 || file_param_element != NULL || model_type == SS_STRUCT )
			{
				model_param_index = -1;
				printf ("Parameter '%s' subfield '%s' not found in model!\n", file_param_name, file_param_element);
				continue;
			}
				
            // Find dimension information in dimension table
			model_dim_start = rtwCAPI_GetDimArrayIndex(model_dim_map, model_dim_index);
			
            // Compute total parameter size - soon to be unused
            model_param_size = 1;
            for( i = 0 ; i < rtwCAPI_GetNumDims(model_dim_map, model_dim_index) ; i++)
				model_param_size = model_param_size * model_dim_array[ model_dim_start + i];
			
            // Compute 2d size
            model_block_size = 1;	
            if (rtwCAPI_GetNumDims(model_dim_map, model_dim_index) > 1)     model_block_size *= model_dim_array[ model_dim_start];
            if (rtwCAPI_GetNumDims(model_dim_map, model_dim_index) >= 2)     model_block_size *= model_dim_array[ model_dim_start+1];
            
            // Deconstruct higher order indexing
            if(  file_param_indexing != NULL )
            {   
                model_line_step_size = model_dim_array[ model_dim_start];       // Step by 1st dimension at each line
                index_dim = 2;
                index_str = strtok(file_param_indexing,", :");
                model_sub_dim_block_size = model_block_size;
                
                 while( index_str != NULL )
                {
                    index_val = atoi(index_str);
                    model_param_addr = ((char *) model_param_addr) +  model_sub_dim_block_size * (index_val - 1) * model_type_size;
                                        
                    if( DEBUG_LEVEL >= 3 )		printf("Subscript Dimension %d: Subscript Value: %d Address: %x SubBlock Size: %d \n", index_dim, index_val, model_param_addr, model_sub_dim_block_size );              
                    fflush(stdout);
                    
                    model_sub_dim_block_size *= model_dim_array[ model_dim_start+index_dim];
                    index_dim++;
                    
                    index_str = strtok(NULL,", :");
                }
                  
            }
            else
            {
                model_line_step_size = 0;   // 1D & 2D tables are "packed"
            }
                
            
            model_line_start_addr = model_param_addr;
            
			if( DEBUG_LEVEL >= 1 )		printf("Class: %d MW Name: %s @ Address: %x - Size: %lu x %d bytes\n", model_type, rtwCAPI_GetDataTypeMWName(model_type_map, model_type_index ), model_param_addr,  model_param_size, model_type_size);
			fflush(stdout);
            
			// Reset to Beginning of Array
			//value_array_idx = 0;
			
		}
		else if( model_param_index < 0 )
		{
			// Not A Valid Param Skipping Data
		}
		else 
        {
            
            value_array_idx = 0;
            
            if( model_type == SS_DOUBLE )
            {
                HANDLE_DATA_LINE( real64_T, line );
            }
            else if( model_type == SS_SINGLE )
            {
                HANDLE_DATA_LINE( real32_T, line );
            }
            else if( model_type == SS_UINT8 )
            {
                HANDLE_DATA_LINE( uint8_T, line );
            }
            else if( model_type == SS_INT8 )
            {
                HANDLE_DATA_LINE( int8_T, line );
            }
            else if( model_type == SS_UINT16 )
            {
                HANDLE_DATA_LINE( uint16_T, line );
            }
            else if( model_type == SS_INT16 )
            {
                HANDLE_DATA_LINE( int16_T, line );
            }
            else if( model_type == SS_UINT32 )
            {
                HANDLE_DATA_LINE( uint32_T, line );
            }
            else if( model_type == SS_INT32 )
            {
                HANDLE_DATA_LINE( int32_T, line );
            }
            else if( model_type == SS_BOOLEAN )
            {
                HANDLE_DATA_LINE( uint8_T, line );
            }
            else if( model_type == SS_ENUM_TYPE && model_type_size == 1 )
            {
                HANDLE_DATA_LINE( int8_T, line );
            }
            else if( model_type == SS_ENUM_TYPE && model_type_size == 2 )
            {
                HANDLE_DATA_LINE( int16_T, line );
            }
            else if( model_type == SS_ENUM_TYPE && model_type_size == 4 )
            {
                HANDLE_DATA_LINE( int32_T, line );
            }
            else 
            {
                printf("Parameter '%s' has unknown data type code %d unable to assign parameter!",file_param_name, model_type);
            }
            
            if(model_line_step_size > 0)
            {
                model_line_start_addr = ((char *) model_line_start_addr) +  model_line_step_size * model_type_size;
            
            }
            else
            {
                model_line_start_addr = ((char *) model_line_start_addr) + value_array_idx * model_type_size;
                
                
            }
        }

        
	}

	fflush(stdout);
    
    return 0;
    
}
   
/* EOF - rtw_capi_examples.c */
