function REVS_plot_engine_diff(e1, e2, varargin)
%function REVS_plot_engine_efficiency(e, varargin)
%
% no_figure                   = parse_varargs(varargin,'no_figure',false,'toggle');
% contour_levels              = parse_varargs(varargin,'contours',[0,10, 15, 20, 23, 25:50],'numeric',{'vector'});
% contour_filled              = parse_varargs(varargin,'filled',false,'toggle');
% contours_labels             = parse_varargs(varargin,'no_contour_labels',true,'toggle');
% contour_width               = parse_varargs(varargin,'line_width',1,'numeric');
% contour_color               = parse_varargs(varargin,'line_color','auto');
% power_lines                 = parse_varargs(varargin,'no_power_lines',true,'toggle');
% cmap                        = parse_varargs(varargin,'colormap','jet');
% highlight_patch             = parse_varargs(varargin,'highlight_contour',[],'numeric');
% highlight_color             = parse_varargs(varargin,'highlight_color',[0,.6,0],'numeric');
% highlight_alpha             = parse_varargs(varargin,'highlight_alpha',[],'numeric',{'scalar'});
% full_throttle_width         = parse_varargs(varargin,'line_width',3,'numeric');
% fill_above_full_throttle    = parse_varargs(varargin,'showabovefullthrottle',true,'toggle');
% plot_audit                  = parse_varargs(varargin,'plot_audit',false,'');


no_figure                   = parse_varargs(varargin,'no_figure',false,'toggle');
contour_levels              = parse_varargs(varargin,'contours',[],'numeric',{'vector'});
contour_filled              = parse_varargs(varargin,'filled',false,'toggle');
contours_labels             = parse_varargs(varargin,'no_contour_labels',true,'toggle');
contour_width               = parse_varargs(varargin,'line_width',1,'numeric');
contour_color               = parse_varargs(varargin,'line_color','auto');
contour_cmap                = parse_varargs(varargin,'colormap','jet');

plot_power_lines            = parse_varargs(varargin,'no_power_lines',true,'toggle');
plot_min_bsfc               = parse_varargs(varargin,'no_min_bsfc',true,'toggle');
plot_na_curve               = parse_varargs(varargin,'no_na_curve',true,'toggle');

highlight_patch             = parse_varargs(varargin,'highlight_contour',[],'numeric');
highlight_color             = parse_varargs(varargin,'highlight_color',[0,.6,0],'numeric');
highlight_alpha             = parse_varargs(varargin,'highlight_alpha',[],'numeric',{'scalar'});

full_throttle_width         = parse_varargs(varargin,'line_width',3,'numeric');

fill_above_full_throttle    = parse_varargs(varargin,'showabovefullthrottle',true,'toggle');

compare_mode				= parse_varargs(varargin,'compare_mode','efficiency','char',{'efficiency','bsfc','bsfc_pct','gps','gps_pct','fuel','fuel_pct' });
correct_fuel				= parse_varargs(varargin,'no_fuel_adjust',true,'toggle');


scaling_mode				= parse_varargs(varargin,'scaling','avg_power','char',{'none','max_power','max_torque','avg_torque','avg_power'});
scaling_window				= parse_varargs( varargin,'scaling_window',linspace(1800, 5500, 25)* convert.rpm2radps, 'numeric' );


% Get statistics on engine 1
e1_torque = max(0,interp1( e1.full_throttle_speed_radps, e1.full_throttle_torque_Nm, scaling_window,'linear','extrap' ));
e1_average_torque = sum(e1_torque )./ length(scaling_window);
e1_average_power = sum(e1_torque .* scaling_window )./ length(scaling_window);


if correct_fuel
	e2.fuel_map_gps = e2.fuel_map_gps * e2.fuel.energy_density_MJpkg / e1.fuel.energy_density_MJpkg;
end



% Scale 2nd engine if needed
switch lower(scaling_mode)
	
	case 'max_power'
		e2 = REVS_modify_engine( e2, 'scale_max_power', e1.max_power_kW);
	case 'max_torque'
		e2 = REVS_modify_engine( e2, 'scale_max_torque', e1.max_torque_Nm);
	case 'avg_power'
		e2 = REVS_modify_engine( e2, 'scale_avg_power', e1_average_power, 'avg_window', scaling_window );
	case 'avg_torque'
		e2 = REVS_modify_engine( e2, 'scale_avg_torque', e1_average_torque, 'avg_window', scaling_window);	
		
end

speeds_rpm = linspace(min(min(e1.fuel_map_speed_radps),min(e2.fuel_map_speed_radps)) .* convert.radps2rpm, max(max(e1.fuel_map_speed_radps), max(e2.fuel_map_speed_radps)) .* convert.radps2rpm, 200 );
torques_Nm = linspace(min(min(e1.fuel_map_torque_Nm), min(e2.fuel_map_torque_Nm)),max(max(e1.fuel_map_torque_Nm), max(e2.fuel_map_torque_Nm)),250);

[mesh_speed_rpm, mesh_torque_Nm] = meshgrid( speeds_rpm, torques_Nm);


%Compute Gridded Maps for comparison
[e1_gps, e1_eff, e1_bsfc] = interp_engine( e1, mesh_speed_rpm, mesh_torque_Nm );
[e2_gps, e2_eff, e2_bsfc] = interp_engine( e2, mesh_speed_rpm, mesh_torque_Nm );


switch lower(compare_mode)
	case 'efficiency'
		plot_data = e1_eff - e2_eff;
		contour_scale = 1.0;
	case 'bsfc'
		plot_data = e1_bsfc - e2_bsfc;
		contour_scale = 10.0;
	case {'gps','fuel'}
		plot_data =  e1_gps-  e2_gps;
		contour_scale = 0.1;
	case {'gps_pct','bsfc_pct','fuel_pct'}
		plot_data = 100 * (e1_gps-  e2_gps) ./ e1_gps;
		contour_scale = 1.0;
	otherwise
		error( 'Unknown comparison mode');
		
end

if isempty(contour_levels)
	contour_levels = [-15, -10:-1, -0.5, -0.2, 0, 0.2, 0.5, 1:10, 15] * contour_scale;
end

if no_figure
	ax = gca;
else
	figure;
	ax = axes;
end



if ~isempty(highlight_patch)
	
	c = contourc(speeds_rpm, torques_Nm, plot_data , contour_levels);
	
	idx = 1;
	contour_size = size(c,2);
	while idx < contour_size
		if c(1,idx) == highlight_patch
			num_pts = c(2,idx);
			x_pts = c(1,idx+(1:num_pts));
			y_pts = c(2,idx+(1:num_pts));
			h = patch(x_pts,y_pts,highlight_color);
			if ~isempty( highlight_alpha )
				alpha(h,highlight_alpha);
			end
			
		end
		idx = idx + c(2,idx) +1;
	end
	hold on
end

% Additional Contour Plotting Arguments
addtl_args = { 'linewidth',contour_width, 'color', contour_color};

if contour_filled
	[c,h] = contourf(ax, mesh_speed_rpm, mesh_torque_Nm, plot_data, contour_levels ,addtl_args{:});
else
	[c,h] = contour(ax, mesh_speed_rpm, mesh_torque_Nm, plot_data , contour_levels ,addtl_args{:});
end


colormap(contour_cmap);

if contours_labels
	clabel(c,h,'labelspacing',144)
end

hold on

if fill_above_full_throttle	
	% 	plot_size = axis;	
	max_x = 2* max(speeds_rpm);
	max_y = 2* max(torques_Nm);
	e1_ft = interp1(e1.full_throttle_speed_radps * convert.radps2rpm,  e1.full_throttle_torque_Nm, speeds_rpm(:) );
	e2_ft = interp1(e2.full_throttle_speed_radps * convert.radps2rpm,  e2.full_throttle_torque_Nm, speeds_rpm(:) );
	
	fill([max_x; max_x; 0; 0; speeds_rpm(:) ],[0; max_y; max_y; 0; max(e1_ft, e2_ft)],[0.8 0.8 1.0])	
end

if plot_na_curve
	plot(ax, e1.naturally_aspirated_speed_radps * convert.radps2rpm, e1.naturally_aspirated_torque_Nm,'r--','linewidth',full_throttle_width)
	plot(ax, e2.naturally_aspirated_speed_radps * convert.radps2rpm, e2.naturally_aspirated_torque_Nm,'b--','linewidth',full_throttle_width)
end

if plot_min_bsfc
	[min_consumption_radps, min_consumption_torque_Nm, ~] = REVS_calc_engine_min_bsfc(e1);
	plot(min_consumption_radps * convert.radps2rpm, min_consumption_torque_Nm, 'm-', 'linewidth', 3);
	[min_consumption_radps, min_consumption_torque_Nm, ~] = REVS_calc_engine_min_bsfc(e2);
	plot(min_consumption_radps * convert.radps2rpm, min_consumption_torque_Nm, 'c-', 'linewidth', 3);
end

% Full throttle Curves
plot(ax, e1.full_throttle_speed_radps * convert.radps2rpm, e1.full_throttle_torque_Nm,'r-','linewidth',full_throttle_width)
plot(ax, e2.full_throttle_speed_radps * convert.radps2rpm, e2.full_throttle_torque_Nm,'b-','linewidth',full_throttle_width)

% Closed throttle Curves
plot(ax, e1.closed_throttle_speed_radps * convert.radps2rpm, e1.closed_throttle_torque_Nm,'r-','linewidth',full_throttle_width)
plot(ax, e2.closed_throttle_speed_radps * convert.radps2rpm, e2.closed_throttle_torque_Nm,'b-','linewidth',full_throttle_width)


xlim([0.8*min([e1.idle_speed_radps.table(:);e2.idle_speed_radps.table(:)]), max( e1.full_throttle_speed_radps(end), e2.full_throttle_speed_radps(end) ) ] .* convert.radps2rpm);
xlabel('Speed ( RPM )');
ylabel('Torque ( Nm )');


if isempty(e1.name)
	e1.name = inputname(1);
end

if isempty(e2.name)
	e2.name = inputname(2);
end

switch lower(compare_mode)
	case 'efficiency'
		title_str = sprintf('Efficiency Difference {\\color{red}%s} vs. {\\color{blue}%s}', strrep(e1.name,'_',' '), strrep(e2.name,'_',' '));
	case 'bsfc'
		title_str = sprintf('BSFC Difference {\\color{red}%s} vs. {\\color{blue}%s}', strrep(e1.name,'_',' '), strrep(e2.name,'_',' '));
	case 'bsfc_pct'
		title_str = sprintf('BSFC %% Difference {\\color{red}%s} vs. {\\color{blue}%s}', strrep(e1.name,'_',' '), strrep(e2.name,'_',' '));
	case {'fuel','gps'}
		title_str = sprintf('Fuel Flow Difference {\\color{red}%s} vs. {\\color{blue}%s}', strrep(e1.name,'_',' '), strrep(e2.name,'_',' '));
	case {'fuel_pct','gps_pct'}
		title_str = sprintf('Fuel Flow %% Difference {\\color{red}%s} vs. {\\color{blue}%s}', strrep(e1.name,'_',' '), strrep(e2.name,'_',' '));
		
		
end


if ~strcmpi(scaling_mode,'none')
	title_str = sprintf( '%s scaled to %0.3f L', title_str, e2.displacement_L);
end




title(title_str);

if plot_power_lines
	plot_size = axis;
	powerlines(plot_size(1),plot_size(2),plot_size(3),plot_size(4),'increment',20)
end

end


function [gps, eff, bsfc] = interp_engine( e, mesh_speed_rpm, mesh_torque_Nm )

gps = interp2( e.fuel_map_speed_radps * convert.radps2rpm, e.fuel_map_torque_Nm, e.fuel_map_gps , mesh_speed_rpm, mesh_torque_Nm);

% Nan out points above WOT
ft = interp1(e.full_throttle_speed_radps*convert.radps2rpm, e.full_throttle_torque_Nm, mesh_speed_rpm) + 0.05*max(e.full_throttle_torque_Nm);
gps( mesh_torque_Nm > ft) = nan;

bsfc = gps * 3600 ./ max(eps, (mesh_speed_rpm .* mesh_torque_Nm) ./ 9539 );


eff = 100 * ((mesh_speed_rpm .* mesh_torque_Nm) ./ 9539 ) ./ ( gps * e.fuel.energy_density_MJpkg );


end




