%% Calculate Total Cycle Results
results(REVS.simulation_index).cycle_name = drive_cycle.name;

% ... calculate percent time trace missed by 2mph (%) 
vehicle_speed_error_mph     						= (datalog.drive_cycle_speed_mps - datalog.vehicle.speed_mps) * convert.mps2mph;
datalog.vehicle.distance_mi                     	= datalog.vehicle.distance_m / convert.mi2mtr;
results(REVS.simulation_index).cycle_error_time_pct = integrate((vehicle_speed_error_mph >= 2), time, 1) / time(end) * 100;
results(REVS.simulation_index).distance_mi          = datalog.vehicle.distance_mi(end);

if REVS_current_model_type == enum_model_type.GEM
    % for GEM, calculate grams CO2 to calculate gallons or gallons
    % equivlanet
    gCO2pgal                                 = engine.fuel_gCO2pgal;
    datalog.vehicle.fuel_gCO2                = datalog.engine.fuel_consumed_g * engine.fuel_carbon_weight_frac * (44.0095/12.0107); % (44.0095/12.0107) from 1065 molar masses of CO2 and C
    datalog.vehicle.fuel_consumed_gallons    = datalog.vehicle.fuel_gCO2 / gCO2pgal;
    datalog.vehicle.fuel_gCO2ps              = datalog.engine.fuel_rate_gps * engine.fuel_carbon_weight_frac * (44.0095/12.0107);   % (44.0095/12.0107) from 1065 molar masses of CO2 and C
    datalog.vehicle.fuel_rate_galps          = datalog.vehicle.fuel_gCO2ps / gCO2pgal;
    datalog.vehicle.fuel_consumed_L          = datalog.vehicle.fuel_consumed_gallons * convert.gal2lit;
    datalog.vehicle.fuel_consumed_g         = datalog.engine.fuel_consumed_g;
else
    % for GEM, calculate gallons from mass and density then CO2
    % equivlanet
    if (vehicle.fuel_type == enum_fuel_type.measured)
        gCO2pgal = vehicle.fuel_gCO2pgal;
    else
        gCO2pgal = engine.fuel_gCO2pgal;
    end
    
    temp.engine_to_vehicle_MJpL_ratio        = (engine.fuel_density_kgpL_15C * engine.fuel_energy_density_MJpkg) / (vehicle.fuel_density_kgpL_15C * vehicle.fuel_energy_density_MJpkg);
    datalog.vehicle.fuel_consumed_gallons    = datalog.engine.fuel_consumed_g / 1000 / engine.fuel_density_kgpL_15C / convert.gal2lit * temp.engine_to_vehicle_MJpL_ratio;
    datalog.vehicle.fuel_gCO2                = datalog.vehicle.fuel_consumed_gallons * gCO2pgal;
    datalog.vehicle.fuel_rate_galps          = datalog.engine.fuel_rate_gps / 1000 / engine.fuel_density_kgpL_15C / convert.gal2lit * temp.engine_to_vehicle_MJpL_ratio;
    datalog.vehicle.fuel_gCO2ps              = datalog.vehicle.fuel_rate_galps * gCO2pgal;
    try
        % experimental fuel penalties (for Chevy Malibu, initially)
        datalog.vehicle.accel_penalty_galps  = datalog.engine.accel_penalty_gps / 1000 / engine.fuel_density_kgpL_15C / convert.gal2lit * temp.engine_to_vehicle_MJpL_ratio;
        datalog.vehicle.tipin_penalty_galps  = datalog.engine.tipin_penalty_gps / 1000 / engine.fuel_density_kgpL_15C / convert.gal2lit * temp.engine_to_vehicle_MJpL_ratio;
    end
    datalog.vehicle.fuel_consumed_L          = datalog.vehicle.fuel_consumed_gallons * convert.gal2lit;
    datalog.vehicle.fuel_consumed_g          = datalog.vehicle.fuel_consumed_L * vehicle.fuel_density_kgpL_15C * 1000;
end

% ... calculate total fuel consumption (mpg)
results(REVS.simulation_index).fuel_g       = datalog.engine.fuel_consumed_g(end);
results(REVS.simulation_index).fuel_gallons = datalog.vehicle.fuel_consumed_gallons(end) ;
results(REVS.simulation_index).FE_mpg       = datalog.vehicle.distance_mi(end) / datalog.vehicle.fuel_consumed_gallons(end); 
results(REVS.simulation_index).gCO2         = datalog.vehicle.fuel_gCO2(end);
results(REVS.simulation_index).gCO2pmi      = results(REVS.simulation_index).gCO2 / datalog.vehicle.distance_mi(end);

% GEM specific results
if isfield(vehicle,'payload_ton') && ~is_unspecified(vehicle.payload_ton)
        results(REVS.simulation_index).gCO2ptm =  results(REVS.simulation_index).gCO2pmi / vehicle.payload_ton;
end

try 

% Signals from audit
if exist('audit','var')
    results(REVS.simulation_index).engine_output_energy_kJ = audit.mech.engine.crankshaft_gross_kJ(end);
end

end
%% Calculate Phase Results

for p = 1:max(datalog.drive_cycle_phase)

    start_idx = find(datalog.drive_cycle_phase == p);
    start_idx = setdiff( start_idx, start_idx+1 );

    end_idx = find(datalog.drive_cycle_phase == p);
    end_idx = setdiff( end_idx, end_idx-1 );

    results(REVS.simulation_index).phase_time_secs(p)       = sum(time(end_idx) - time(start_idx));
    results(REVS.simulation_index).phase_distance_mi(p)     = sum( datalog.vehicle.distance_mi(end_idx))  - sum( datalog.vehicle.distance_mi(start_idx));
    results(REVS.simulation_index).idle_time_secs(p)        = integrate(datalog.vehicle.speed_mps(start_idx:end_idx) < 0.1, time(start_idx:end_idx), 1);
    results(REVS.simulation_index).phase_avg_speed_mph(p)   = results(REVS.simulation_index).phase_distance_mi(p)/(results(REVS.simulation_index).phase_time_secs(p) - results(REVS.simulation_index).idle_time_secs(p))*3600;
	results(REVS.simulation_index).phase_fuel_g(p)          = sum( datalog.engine.fuel_consumed_g(end_idx)) - sum( datalog.engine.fuel_consumed_g(start_idx));
    results(REVS.simulation_index).phase_fuel_gallons(p)    = sum( datalog.vehicle.fuel_consumed_gallons(end_idx)) - sum( datalog.vehicle.fuel_consumed_gallons(start_idx));
    results(REVS.simulation_index).phase_gCO2(p)            = sum( datalog.vehicle.fuel_gCO2(end_idx)) - sum( datalog.vehicle.fuel_gCO2(start_idx));
    results(REVS.simulation_index).phase_gCO2pmi(p)         = results(REVS.simulation_index).phase_gCO2(p) / results(REVS.simulation_index).phase_distance_mi(p);
    results(REVS.simulation_index).phase_FE_mpg(p)          = results(REVS.simulation_index).phase_distance_mi(p) / results(REVS.simulation_index).phase_fuel_gallons(p);
    
	try
	results(REVS.simulation_index).phase_N_rpm(p)           = integrate(audit.mech.engine.crankshaft_gross_rpm .* (datalog.drive_cycle_phase == p), time, 1)/results(REVS.simulation_index).phase_time_secs(p);
	results(REVS.simulation_index).phase_NV_ratio(p)        = results(REVS.simulation_index).phase_N_rpm(p) / 60 / (results(REVS.simulation_index).phase_avg_speed_mph(p) * convert.mph2mps);
        
    [~, corrected_power_W] = REVS_calc_corrected_torque_and_power(time, audit.mech.engine.crankshaft_gross_Nm, audit.mech.engine.crankshaft_gross_radps, datalog.engine.crankshaft_inertia_kgm2);
    results(REVS.simulation_index).phase_crankshaft_pos_work_hphr(p) = integrate(max(0, corrected_power_W / 1000 * convert.kW2hp) .* (datalog.drive_cycle_phase == p), time / 3600, 1);

    [~, corrected_power_W] = REVS_calc_corrected_torque_and_power(time, audit.mech.gearbox.output_Nm, audit.mech.gearbox.output_radps, datalog.transmission.output_inertia_kgm2);
    results(REVS.simulation_index).phase_powertrain_pos_work_hphr(p) = integrate(max(0, corrected_power_W / 1000 * convert.kW2hp) .* (datalog.drive_cycle_phase == p), time / 3600, 1);
    
	clear corrected_power_W;
    
    if exist('audit','var')
        results(REVS.simulation_index).phase_engine_output_energy_kJ = sum( audit.mech.engine.crankshaft_gross_kJ(end_idx))  - sum( audit.mech.engine.crankshaft_gross_kJ(start_idx));
    end

	end
   
    if isfield(vehicle,'payload_ton') && ~is_unspecified(vehicle.payload_ton)
        results(REVS.simulation_index).phase_gCO2ptmi(p) = results(REVS.simulation_index).phase_gCO2pmi(p) / vehicle.payload_ton;
    end
end

%% Display Post-Processing Results
% ... display percent time trace missed by 2mph (%)
fprintf(1,'\n-- Summary data for %s drive cycle --\n\n',drive_cycle.name);
fprintf(1,'   Percent Time Missed by 2mph   = %6.2f %%\n',      results(REVS.simulation_index).cycle_error_time_pct)

for p = 1:max(datalog.drive_cycle_phase)
    if max(datalog.drive_cycle_phase) ~= min(datalog.drive_cycle_phase)
        fprintf(1,'   Phase %d -------------------------------\n',p);
        if isfield(drive_cycle, 'phase_name');
            fprintf(1,'   >> %s\n',drive_cycle.phase_name{p});
        end
        fprintf(1,'   Distance            = %6.3f mi\n',        results(REVS.simulation_index).phase_distance_mi(p));
        fprintf(1,'   Fuel Consumption    = %6.4f gallons\n',   results(REVS.simulation_index).phase_fuel_gallons(p));
        fprintf(1,'   Fuel Consumption    = %6.4f grams\n',     results(REVS.simulation_index).phase_fuel_g(p));
        fprintf(1,'   Fuel Economy        = %6.3f mpg\n',       results(REVS.simulation_index).phase_FE_mpg(p));
        fprintf(1,'   Fuel Consumption    = %6.3f g/mile\n',    results(REVS.simulation_index).phase_fuel_g(p) / results(REVS.simulation_index).phase_distance_mi(p));
        fprintf(1,'   CO2 Emission        = %6.2f g/mile\n',    results(REVS.simulation_index).phase_gCO2pmi(p));

        if isfield(vehicle,'payload_ton') && ~is_unspecified(vehicle.payload_ton)
            fprintf(1,'   CO2 Emission (Phase %d)         = %6.2f g/ton-mile\n', p, results(REVS.simulation_index).phase_gCO2ptmi(p));
        end
        fprintf(1,'\n');
    end
end

% Print Total Results
fprintf(1,'   Total ----------------------------------\n');
fprintf(1,'   >> %s\n',drive_cycle.name);
fprintf(1,'   Distance            = %6.3f mi\n', datalog.vehicle.distance_mi(end));
fprintf(1,'   Fuel Consumption    = %6.4f gallons\n', datalog.vehicle.fuel_consumed_gallons(end));
fprintf(1,'   Fuel Consumption    = %6.4f grams\n', datalog.engine.fuel_consumed_g(end));
fprintf(1,'   Fuel Economy        = %6.3f mpg\n', results(REVS.simulation_index).FE_mpg)
fprintf(1,'   Fuel Consumption    = %6.3f g/mile\n', datalog.engine.fuel_consumed_g(end) / datalog.vehicle.distance_mi(end));
fprintf(1,'   CO2 Emission        = %6.2f g/mile\n', results(REVS.simulation_index).gCO2pmi)

if isfield(vehicle,'payload_ton') && ~is_unspecified(vehicle.payload_ton)
    fprintf(1,'   CO2 Emission (Total)          = %6.2f g/ton-mile\n', results(REVS.simulation_index).gCO2ptm);
end

% Run Shift Statistics if Transmission Data
if isfield(datalog,'transmission')
%    run_shift_stats;
end

%% cleanup
clear gCO2pgal start_idx end_idx p meters_per_kg_fuel vehicle_speed_error_mph pts;
