classdef class_REVS_vehicle
    %class_REVS_vehicle
    %	Vehicle description class for REVS model for single axle drive
    
    properties       
        name;							% vehicle Description
        class;							% vehicle Type Specifier
        fuel;							% vehicle Fuel Specifier - Specified via class_REVS_fuel
        
        mass_static_kg;                 % Vehicle static mass (curb weight)
        mass_dynamic_kg;                % vehicle equivalent inertia mass, e.g. static mass * 1.03
        
        delta_mass_static_kg@class_REVS_dynamic_lookup;       % dynamic lookup for altering static mass during a drive cycle
        delta_mass_dynamic_kg@class_REVS_dynamic_lookup;;     % dynanic lookup for altering dynamic mass during a drive cycle
        
        use_abc_roadload@logical;               % Use ABC roadload vs Cd & Crr
        
        coastdown_target_A_N;                   % coastdown target "A" term, SI units, Newtons
        coastdown_target_B_Npms;                % coastdown target "B" term, SI units, Newtons / (meter / second)
        coastdown_target_C_Npms2;               % coastdown target "C" term, SI units, Newtons / (meter / second)^2
        
        coastdown_adjust_A_N = 0;               % coastdown adjustment for double counting "A" term, SI units, Newtons
        coastdown_adjust_B_Npms = 0;            % coastdown adjustment for double counting "B" term, SI units, Newtons / (meter / second)
        coastdown_adjust_C_Npms2 = 0;           % coastdown adjustment for double counting "C" term, SI units, Newtons / (meter / second)^2       
        
        frontal_area_m2;                        % vehicle frontal area, square meters
        aerodynamic_drag_coeff;                 % vehicle aerodynamic drag coefficient
        
        speed_limit_mps = inf;                  % vehicle speed limiter, truncates drive trace (?)
        
        drive_axle1 = class_REVS_drive_axle;	% Drive Axle Definition Class
        
        max_brake_force_N;                      % Total combined braking torque of all axles        
    end
    
    properties ( Dependent )
        coastdown_target_A_lbf;                % coastdown target "A" term, SAE units, pounds force
        coastdown_target_B_lbfpmph;            % coastdown target "B" term, SAE units, pounds force / mph
        coastdown_target_C_lbfpmph2;           % coastdown target "C" term, SAE units, pounds force / mph^2
    
        coastdown_adjust_A_lbf;                % coastdown adjustment for double counting "A" term, SAE units, pounds force
        coastdown_adjust_B_lbfpmph;            % coastdown adjustment for double counting "B" term, SAE units, pounds force / mph
        coastdown_adjust_C_lbfpmph2;           % coastdown adjustment for double counting "C" term, SAE units, pounds force / mph^2
    end
    
    methods
        
        function obj = class_REVS_vehicle( )
			
			obj.delta_mass_static_kg = class_REVS_dynamic_lookup;
            obj.delta_mass_static_kg.axis_1.signal       = 'cycle_phase_num';
            obj.delta_mass_static_kg.axis_1.breakpoints  = [0 1];
            obj.delta_mass_static_kg.table               = [0 0];
            
			obj.delta_mass_dynamic_kg = class_REVS_dynamic_lookup;
            obj.delta_mass_dynamic_kg.axis_1.signal      = 'cycle_phase_num';
            obj.delta_mass_dynamic_kg.axis_1.breakpoints = [0 1];
            obj.delta_mass_dynamic_kg.table              = [0 0];
        end
        
        function val = get.max_brake_force_N(obj)
            val = REVS_class_default( obj.max_brake_force_N , obj.mass_static_kg * 9.8 * 1.75);  % changed from 2.0 to 1.4 for REVS1 compatible brake gain and alternator regen...
        end
        
        
        function val = get.drive_axle1( obj )
            
            %Set axle parameters that require vehicle level inputs
            val = obj.drive_axle1;
            
            val.brake.max_torque_Nm = REVS_class_default( val.brake.max_torque_Nm, obj.max_brake_force_N * val.brake.proportion_norm * val.tire.radius_m);
            
            val.tire.wheel_slip_torque_Nm = REVS_class_default( val.tire.wheel_slip_torque_Nm, obj.mass_static_kg * val.tire.vehicle_weight_norm * 9.81 * val.tire.radius_m);
            
        end
        
        
        function val = get.coastdown_target_A_N(obj)
            if obj.use_abc_roadload
                val = obj.coastdown_target_A_N;
            else
                val = REVS_class_default(obj.coastdown_target_A_N,0);
            end
        end
        
        function val = get.coastdown_target_B_Npms(obj)
            if obj.use_abc_roadload
                val = obj.coastdown_target_B_Npms;
            else
                val = REVS_class_default(obj.coastdown_target_B_Npms,0);
            end
        end
        
        function val = get.coastdown_target_C_Npms2(obj)
            if obj.use_abc_roadload
                val = obj.coastdown_target_C_Npms2;
            else
                val = REVS_class_default(obj.coastdown_target_C_Npms2,0);
            end
        end
        
        function val = get.aerodynamic_drag_coeff(obj)
            if obj.use_abc_roadload
                val = REVS_class_default(obj.aerodynamic_drag_coeff,0);
            else
                val = obj.aerodynamic_drag_coeff;
            end
        end
        
        function val = get.frontal_area_m2(obj)
            if obj.use_abc_roadload
                val = REVS_class_default(obj.frontal_area_m2,0);
            else
                val = obj.frontal_area_m2;
            end
        end
        
        % setter/getters for dependent properties
        function obj = set.coastdown_target_A_lbf( obj, val )
            obj.coastdown_target_A_N = convert.lbf2N * val ;
        end
        
        function val = get.coastdown_target_A_lbf( obj )
            val = convert.N2lbf * obj.coastdown_target_A_N;
        end

        function obj = set.coastdown_target_B_lbfpmph( obj, val )
            obj.coastdown_target_B_Npms = convert.lbf2N / convert.mph2mps * val ;
        end
        
        function val = get.coastdown_target_B_lbfpmph( obj )
            val = convert.N2lbf / convert.mps2mph * obj.coastdown_target_B_Npms;
        end

        function obj = set.coastdown_target_C_lbfpmph2( obj, val )
            obj.coastdown_target_C_Npms2 = convert.lbf2N / convert.mph2mps / convert.mph2mps * val ;
        end
        
        function val = get.coastdown_target_C_lbfpmph2( obj )
            val = convert.N2lbf / convert.mps2mph / convert.mps2mph * obj.coastdown_target_C_Npms2;
        end
        
        function obj = set.coastdown_adjust_A_lbf( obj, val )
            obj.coastdown_adjust_A_N = convert.lbf2N * val ;
        end
        
        function val = get.coastdown_adjust_A_lbf( obj )
            val = convert.N2lbf * obj.coastdown_adjust_A_N;
        end

        function obj = set.coastdown_adjust_B_lbfpmph( obj, val )
            obj.coastdown_adjust_B_Npms = convert.lbf2N / convert.mph2mps * val ;
        end
        
        function val = get.coastdown_adjust_B_lbfpmph( obj )
            val = convert.N2lbf / convert.mps2mph * obj.coastdown_adjust_B_Npms;
        end

        function obj = set.coastdown_adjust_C_lbfpmph2( obj, val )
            obj.coastdown_adjust_C_Npms2 = convert.lbf2N / convert.mph2mps / convert.mph2mps * val ;
        end
        
        function val = get.coastdown_adjust_C_lbfpmph2( obj )
            val = convert.N2lbf / convert.mps2mph / convert.mps2mph * obj.coastdown_adjust_C_Npms2;
        end
        
        
    end
    
end

