#-----------------------------------------------------------------------------------#
# DNT-NAM AEDs: Section 2.3.6 in Issue Paper
# March 24, 2020
# last updated 14 May 2020
# author: paul-friedman.katie@epa.gov
#-----------------------------------------------------------------------------------#

#-----------------------------------------------------------------------------------#
# HTTK Data and Model Availability
#-----------------------------------------------------------------------------------#

#-----------------------------------------------------------------------------------#
# load libraries

rm(list = ls())

library(cowplot)
library(data.table)
library(dplyr)
library(ggforce)
library(ggplot2)
library(httk)
library(magrittr)
library(openxlsx)
library(reshape2)
library(stringr)
library(viridis)

print(sessionInfo())

#R version 3.6.1 (2019-07-05)
#Platform: x86_64-w64-mingw32/x64 (64-bit)
#Running under: Windows >= 8 x64 (build 9200)

#attached base packages:
 # [1] stats     graphics  grDevices utils     datasets  methods   base     

#other attached packages:
#  [1] cowplot_1.0.0     viridis_0.5.1     viridisLite_0.3.0 ggforce_0.3.1     ggplot2_3.2.1     httk_2.0.1       
#[7] stringr_1.4.0     reshape2_1.4.3    magrittr_1.5      dplyr_0.8.3       openxlsx_4.1.4    data.table_1.12.8
#[13] RMySQL_0.10.17    DBI_1.1.0        

#loaded via a namespace (and not attached):
#  [1] zip_2.0.4         Rcpp_1.0.4        pillar_1.4.2      compiler_3.6.1    plyr_1.8.4        tools_3.6.1      
#[7] digest_0.6.23     lifecycle_0.1.0   tibble_2.1.3      gtable_0.3.0      lattice_0.20-38   pkgconfig_2.0.3  
#[13] rlang_0.4.1       Matrix_1.2-17     rstudioapi_0.10   mvtnorm_1.1-0     expm_0.999-4      gridExtra_2.3    
#[19] withr_2.1.2       mitools_2.4       grid_3.6.1        tidyselect_0.2.5  deSolve_1.27.1    glue_1.3.1       
#[25] R6_2.4.0          survival_2.44-1.1 polyclip_1.10-0   farver_2.0.1      tweenr_1.0.1      purrr_0.3.3      
#[31] MASS_7.3-51.4     scales_1.1.0      splines_3.6.1     assertthat_0.2.1  colorspace_1.4-1  stringi_1.4.3    
#[37] survey_3.37       lazyeval_0.2.2    munsell_0.5.0     msm_1.6.8         crayon_1.3.4

# load in information needed to evaluate model and data availability for OPs
setwd("./public_code") # set to working directory
load(file='./source/dnt_op_sap_httk_addv2.RData') #includes newly extracted data
load(file='./source/op_listinfo.RData')

#-----------------------------------------------------------------------------------#
# add new data on OPs extracted from MRIDs

dnt.httk.add[is.na(Fup), Fup := 0] # bug in HTTK is that it will not run 3 compss if is.na(Fup), so set to zero
dnt.httk.add.df <- as.data.frame(dnt.httk.add)

chem.physical_and_invitro.data <- add_chemtable(dnt.httk.add.df,
                                                current.table=chem.physical_and_invitro.data,
                                                data.list=list(Compound="Compound",
                                                               CAS="CAS",
                                                               DTXSID="DTXSID",
                                                               Clint="Clint",
                                                               #Clint.pValue="Human.Clint.pValue", # don't have it
                                                               Funbound.plasma="Fup",
                                                               LogP="logP",
                                                               MW="MW",
                                                               Species="species",
                                                               Reference="Reference"
                                                               #pKa_Accept="pKa_Accept", # don't have this
                                                               #pKa_Donor="pKa_Donor" # don't have this
                                                ),
                                                overwrite=T)

httk.chem.tbl <- as.data.table(chem.physical_and_invitro.data)

# examine OPs specifically

httk.op.tbl.data <- httk.chem.tbl[DTXSID %in% op.list[,DTXSID]]

httk.op.tbl.data <- unique(httk.op.tbl.data[,c('Compound','CAS','DTXSID','logP',
                                               'pKa_Accept','pKa_Donor',
                                               'Human.Clint', 'Human.Clint.Reference',
                                               'Human.Funbound.plasma','Human.Funbound.plasma.Reference',
                                               'Rat.Clint','Rat.Clint.Reference',
                                               'Rat.Funbound.plasma', 'Rat.Funbound.plasma.Reference')])

# verify which models can be run

pbtk.rat <- as.data.table(get_cheminfo(info='all',species='rat',model='pbtk'))
pbtk.human <- as.data.table(get_cheminfo(info='all',species='human',model='pbtk'))
comp3ss.rat <- as.data.table(get_cheminfo(info='all',species='rat',model='3compartmentss'))
comp3ss.human <- as.data.table(get_cheminfo(info='all',species='human',model='3compartmentss'))

httk.op.tbl.data[!DTXSID %in% pbtk.rat[,DTXSID],rat.pbtk :=0]
httk.op.tbl.data[DTXSID %in% pbtk.rat[,DTXSID],rat.pbtk :=1]
httk.op.tbl.data[!DTXSID %in% pbtk.human[,DTXSID],human.pbtk :=0]
httk.op.tbl.data[DTXSID %in% pbtk.human[,DTXSID],human.pbtk :=1]

httk.op.tbl.data[!DTXSID %in% comp3ss.rat[,DTXSID],rat.3compss :=0]
httk.op.tbl.data[DTXSID %in% comp3ss.rat[,DTXSID],rat.3compss :=1]
httk.op.tbl.data[!DTXSID %in% comp3ss.human[,DTXSID],human.3compss :=0]
httk.op.tbl.data[DTXSID %in% comp3ss.human[,DTXSID],human.3compss :=1]

httk.op.tbl.data[is.na(rat.3compss), rat.3compss := 0]
httk.op.tbl.data[is.na(human.3compss), human.3compss := 0]
httk.op.tbl.data[is.na(rat.pbtk), rat.pbtk := 0]
httk.op.tbl.data[is.na(human.pbtk), human.pbtk := 0]

# add BMD values for completeness of table

httk.op.tbl.data$BMD10 <- op.bmd$BMD10[match(httk.op.tbl.data$DTXSID,
                                                op.bmd$DTXSID)]
httk.op.tbl.data$BMDL10 <- op.bmd$BMDL10[match(httk.op.tbl.data$DTXSID,
                                                op.bmd$DTXSID)]

colnames(httk.op.tbl.data)
setcolorder(httk.op.tbl.data,
            c(1:3,19:20,4:7,9,11,13, 15:18, 8,10,12,14))
httk.op.tbl.data <- httk.op.tbl.data[order(Compound)]

# empirical work by Barbara Wetmore demonstrated that Chlorethoxyfos was degraded/unstable in solution
# drop the Wambaugh et al 2019 data for Chlorethoxyfos

httk.op.tbl.data[Compound=='Chlorethoxyfos', c('Human.Clint', 'Rat.Clint', 
                                               'Human.Funbound.plasma','Rat.Funbound.plasma') := as.numeric(NA)]
httk.op.tbl.data[Compound=='Chlorethoxyfos', c('Human.Clint.Reference',
                                               'Human.Funbound.plasma.Reference') := 'Wetmore unpublished 2020']



httk.op.tbl.pbtk.comp <- httk.op.tbl.data[DTXSID %in% op.list[,DTXSID] & !is.na(Rat.Clint) & !is.na(Human.Clint)] # 10 substances
httk.op.tbl.pbtk.comp[,c('Compound','Rat.Funbound.plasma', 'Human.Funbound.plasma', 'Rat.Clint', 'Human.Clint')] # only 6 have Rat Fup and all other parameters
httk.op.tbl.pbtk.comp <- httk.op.tbl.pbtk.comp[!(Rat.Funbound.plasma==0) & !(Human.Funbound.plasma==0)]

suppapp.tbl4.httk.op.data.df <- as.data.frame(httk.op.tbl.data)
suppapp.tbl2.httk.op.pbtk.df <- as.data.frame(httk.op.tbl.pbtk.comp)

list_op <- list('SuppApp_Tbl4_httk_data_OPs' = suppapp.tbl4.httk.op.data.df, 
                'SuppApp_Tbl2_httk_data_OPs_pbtk' = suppapp.tbl2.httk.op.pbtk.df)
write.xlsx(list_op,
           keepNA=TRUE,
           file='./output/SuppApp_Tbl2_Tbl4_HTTK_data_and_models.xlsx') # see Supplemental Appendix Tables 2 and 4

#-----------------------------------------------------------------------------------#
# now compare the impact of rat vs. human data (with rat physiology) 
# See Supplemental Appendix D

test.chems <- merge(get_cheminfo(info="all",species="rat",model="pbtk"),
                    get_cheminfo(model="pbtk", info="all"))

test.chems <- subset(test.chems,Rat.Funbound.plasma!=0.005 &
                       Human.Funbound.plasma!=0.005)

test.chems <- subset(test.chems, CAS!="101193-40-2") # this substance quintolast has all parameters=0

#151 substances for this comparison

for (this.chem in test.chems$CAS)
{
  out <- calc_stats(chem.cas = this.chem, 
                    species="Rat", 
                    route="oral",
                    days=28,
                    daily.dose=1,
                    doses.per.day = 1,
                    output.units = "uM",
                    concentration = "plasma",
                    tissue = "plasma",
                    model = "pbtk",
                    default.to.human=FALSE,
                    adjusted.Funbound.plasma = TRUE,
                    regression = TRUE,
                    restrictive.clearance = TRUE)
  index <- test.chems$CAS==this.chem
  test.chems[index,"Rat.Cmax"] <- out$peak
  test.chems[index,"Rat.Cmean"] <- out$mean
  test.chems[index,"Rat.AUC"] <- out$AUC
}

chem.physical_and_invitro.data <- add_chemtable(
  test.chems,
  current.table=chem.physical_and_invitro.data,
  data.list=list(
    Compound="Compound",
    CAS="CAS",
    DTXSID="DTXSID",
    Funbound.plasma="Human.Funbound.plasma",
    Clint="Human.Clint"),
  species="Rat",
  reference="Human",
  overwrite=TRUE)

for (this.chem in test.chems$CAS)
{
  out <- calc_stats(chem.cas = this.chem, species="Rat",
                    route="oral",
                    days=28,
                    daily.dose=1,
                    doses.per.day = 1,
                    output.units = "uM",
                    concentration = "plasma",
                    tissue = "plasma",
                    model = "pbtk",
                    default.to.human=FALSE,
                    adjusted.Funbound.plasma = TRUE,
                    regression = TRUE,
                    restrictive.clearance = TRUE)
  index <- test.chems$CAS==this.chem
  test.chems[index,"Human.Cmax"] <- out$peak
  test.chems[index,"Human.Cmean"] <- out$mean
  test.chems[index,"Human.AUC"] <- out$AUC
}

fit1 <- lm(log(Rat.Cmax) ~ log(Human.Cmax),data=test.chems)
summary(fit1)

colnames(test.chems)
str(test.chems)

test.chems <- as.data.table(test.chems)
test.chems[!DTXSID %in% op.list[,DTXSID], is.op :=0]
test.chems[DTXSID %in% op.list[,DTXSID], is.op :=1]
test.chems[is.op==1]

Fig.Cmax <- ggplot(data=test.chems) +
  geom_point(aes(x=Human.Cmax, y=Rat.Cmax,color=factor(is.op)),size=3)+ 
  xlab(expression(paste("Rat PBTK ", C[max], " & Human ",italic("In vitro")," Data (uM)"))) +
  ylab(expression(paste("Rat PBTK ", C[max], " & Rat ",italic("In vitro")," Data (uM)"))) +
  scale_y_log10(limits=c(10^-2,1000)) +
  scale_x_log10(limits=c(10^-2,1000)) +
  scale_color_manual(name='Chemical type',
                     breaks=c("0","1"),
                     values=c("#999999",
                              "#009E73"),
                     labels=c("Is NOT OP","Is OP"))+
  geom_abline(slope=1, intercept=0) +
  geom_abline(slope=1, intercept=1/2, linetype="dashed") +
  geom_abline(slope=1, intercept=-1/2, linetype="dashed") +
  theme_bw()
print(Fig.Cmax)

fit2 <- lm(log(Rat.AUC) ~ log(Human.AUC),data=test.chems)
summary(fit2)

Fig.AUC <- ggplot(data=test.chems) +
  geom_point(aes(x=Human.AUC, y=Rat.AUC,color=factor(is.op)),size=3)+ 
  xlab(expression(paste("Rat PBTK AUC & Human ", italic("In vitro")," Data (uM*Days)"))) +
  ylab(expression(paste("Rat PBTK AUC & Rat ",italic("In vitro")," Data (uM*Days)"))) +
  scale_y_log10(limits=c(10^-1,10^4)) +
  scale_x_log10(limits=c(10^-1,10^4)) +
  scale_color_manual(name='Chemical type',
                     breaks=c("0","1"),
                     values=c("#999999",
                              "#009E73"),
                     labels=c("Is NOT OP","Is OP"))+
  geom_abline(slope=1, intercept=0) +
  geom_abline(slope=1, intercept=1, linetype="dashed") +
  geom_abline(slope=1, intercept=-1, linetype="dashed") +
  theme_bw()
print(Fig.AUC)

str(test.chems) # some of the columns that should be num are actually chr
# make numeric columns as.numeric
columns <- c('pKa_Donor', 'pKa_Accept', 'Rat.Clint', 'Human.Clint', 'Rat.Funbound.plasma','Human.Funbound.plasma')
test.chems[, (columns) := lapply (.SD, as.numeric), .SDcols = columns ]

# limit the sig figs on the values
httk.avail <- test.chems %>% mutate_if(is.numeric, ~signif(.,3)) %>% data.table()
httk.avail <- httk.avail[, c('Compound', 'DTXSID', 
               'logP', 'pKa_Donor', 'pKa_Accept',
               'Rat.Clint', 'Human.Clint', 
               'Rat.Funbound.plasma', 'Human.Funbound.plasma',
               'Rat.Cmax', 'Human.Cmax','Rat.Cmean', 'Human.Cmean','Rat.AUC','Human.AUC', 'is.op')]

setnames(httk.avail, c('Compound','pKa_Donor', 'pKa_Accept',
                       'Rat.Clint', 'Human.Clint', 
                       'Rat.Funbound.plasma', 'Human.Funbound.plasma',
                       'Rat.Cmax', 'Human.Cmax','Rat.Cmean', 'Human.Cmean','Rat.AUC','Human.AUC'),
         c('Name', 'pKa Donor', 'pKa Acceptor', 
           'Rat Clint', 'Human Clint', 'Rat Fup', 'Human Fup', 
           'Rat Cmax', 'Human Cmax', 'Rat Cmean', 'Human Cmean','Rat AUC', 'Human AUC'))

httk.avail.op <- httk.avail[is.op==1]
httk.avail.df <- as.data.frame(httk.avail)
httk.avail.op.df <- as.data.frame(httk.avail.op)
list.httk.data <- list(
  "ratphys.rat.hum.httk.comp" = httk.avail.df,
  "ratphys.rat.hum.httk.comp.ops" = httk.avail.op.df)
write.xlsx(list.httk.data, file='SuppApp_httk_ratphys_rat2hum_comp.xlsx')

fig.auc.cmax.interspecies <- plot_grid(Fig.Cmax, Fig.AUC, align='hv', 
                                       nrow=1, ncol=2, 
                                       rel_widths = c(1,1),
                                       labels=c("A", "B"),
                                       label_size=18,
                                       label_fontface = 'bold',
                                       label_x=0.15,
                                       label_y=0.95)
file.dir <- paste("./output/figures/", sep="")
file.name <- paste("/SuppApp_Fig2__auc_cmax_interspecies", Sys.Date(), ".png", sep="")
file.path <- paste(file.dir, file.name, sep="")
dir.create(path=file.dir, showWarnings = FALSE, recursive = TRUE)
png(file.path, width = 7000, height = 2500, res=600)
fig.auc.cmax.interspecies
dev.off()

#-----------------------------------------------------------------------------------#
# HTTK AED calculations
#-----------------------------------------------------------------------------------#

# load libraries

rm(list = ls())

library(data.table)
library(openxlsx)
library(dplyr)
library(magrittr)
library(reshape2)
library(stringr)
library(httk)
library(cowplot)
library(ggplot2)
library(ggforce)
library(viridis)

# load in information needed

setwd("./public_code") # set working directory
load(file='./source/dnt_op_sap_httk_addv2.RData') # data to add to HTTK
load(file='./source/op_mea_hci_potency_summary.RData') # in vitro potency info
load(file='./source/op_listinfo.RData') # op-relevant info
load('./source/HCI_13Apr2020.RData') # assay information
load('./source/NHEERL_MEA_dev_13Apr2020.RData') # assay information

dnt.httk.add[is.na(Fup), Fup := 0] # did this to get around bug in httk that requires a non-NA value for Fup

dnt.httk.add.df <- as.data.frame(dnt.httk.add)

chem.physical_and_invitro.data <- add_chemtable(dnt.httk.add.df,
                                                current.table=chem.physical_and_invitro.data,
                                                data.list=list(Compound="Compound",
                                                               CAS="CAS",
                                                               DTXSID="DTXSID",
                                                               Clint="Clint",
                                                               #Clint.pValue="Human.Clint.pValue", # don't have it
                                                               Funbound.plasma="Fup",
                                                               LogP="logP",
                                                               MW="MW",
                                                               Species="species",
                                                               Reference="Reference"
                                                               #pKa_Accept="pKa_Accept", # don't have this
                                                               #pKa_Donor="pKa_Donor" # don't have this
                                                ),
                                                overwrite=T)

httk.chem.tbl <- as.data.table(chem.physical_and_invitro.data)

httk.op.tbl <- httk.chem.tbl[DTXSID %in% op.list[,DTXSID]]

#----------------------------------------------------------------------------#
# Human comparison
#----------------------------------------------------------------------------#

httk.op.tbl[is.na(Human.Clint)] # only naled and z-tetrachlorvinphos

op.com.pos <- op.com[hitc==1 & use.me==1]
op.com.pos[hitc == 1, modl.ga.uM := ifelse(!is.na(modl_ga), 10^modl_ga, NA)]
setnames(op.com.pos, c('dsstox_substance_id'), c('DTXSID'))
op.com.pos <- op.com.pos[!(DTXSID %in% c('DTXSID1024209','DTXSID1032648'))]

httk.human.3css.df=data.frame()
for (i in 1:nrow (op.com.pos))
{
  aed.human.3css.50 <-(calc_mc_oral_equiv(conc=op.com.pos$modl.ga.uM[i], 
                                    chem.cas=op.com.pos$casn[i], 
                                    which.quantile=c(0.50), 
                                    species='Human',
                                    restrictive.clearance=T, 
                                    output.units='mgpkgpday',
                                    model='3compartmentss'))
  aed.human.3css.95 <-(calc_mc_oral_equiv(conc=op.com.pos$modl.ga.uM[i], 
                                         chem.cas=op.com.pos$casn[i], 
                                         which.quantile=c(0.95), 
                                         species='Human',
                                         restrictive.clearance=T, 
                                         output.units='mgpkgpday',
                                         model='3compartmentss'))
  
  httk.human.3css.df<-rbind(httk.human.3css.df,
                    cbind(op.com.pos$aeid[i],
                          op.com.pos$DTXSID[i], 
                          op.com.pos$spid[i],
                          op.com.pos$modl.ga.uM[i], 
                          op.com.pos$casn[i],
                          aed.human.3css.50,
                          aed.human.3css.95))
}
save(httk.human.3css.df, file='./source/human_3compss_aed_data_23apr2020.RData')

httk.human.3css.dt <- as.data.table(httk.human.3css.df)
setnames(httk.human.3css.dt,
         c('V1','V2','V3','V4','V5'),
         c('aeid','DTXSID','spid','modl.ga.uM','casn'))

col.num <- c('aeid', 'modl.ga.uM', 'aed.human.3css.50', 'aed.human.3css.95')
httk.human.3css.dt[, (col.num) := lapply (.SD, as.character), .SDcols = col.num ]
httk.human.3css.dt[, (col.num) := lapply (.SD, as.numeric), .SDcols = col.num ]

col.chr <- c('DTXSID','spid','casn')
httk.human.3css.dt[, (col.chr) := lapply (.SD, as.character), .SDcols = col.chr ]
httk.human.3css.dt[, (col.chr) := lapply (.SD, as.character), .SDcols = col.chr ]

op.com.pos[,key := paste0(aeid,'_',DTXSID,'_',spid,'_',modl.ga.uM, '_', casn)]
httk.human.3css.dt[,key := paste0(aeid,'_',DTXSID,'_',spid,'_',modl.ga.uM, '_', casn)]

op.com.pos$aed.human.3css.50 <- httk.human.3css.dt$aed.human.3css.50[match(op.com.pos$key,
                                                                            httk.human.3css.dt$key)]
op.com.pos$aed.human.3css.95 <- httk.human.3css.dt$aed.human.3css.95[match(op.com.pos$key,
                                                                            httk.human.3css.dt$key)]

#-------------------------------------------------------------------------------------#
# Rat and huRat comparison
#-------------------------------------------------------------------------------------#

# now do the rat calculations for 3compss median
# confirm the same number of chemicals have rat Clint
op.com.pos.rat <- as.data.frame(subset(op.com.pos, casn %in% get_cheminfo(species='Rat'))) 

unique(op.com.pos.rat$chnm) # 9 substances
#[1] "Dimethoate"        "Malathion"         "Diazinon"          "Chlorpyrifos oxon" "Chlorpyrifos"      "Bensulide"        
#[7] "Omethoate"         "Ethoprop"          "Diazoxon" 

unique(httk.chem.tbl[!is.na(Rat.Clint) & DTXSID %in% op.list[,DTXSID], c('Compound','Rat.Clint')])
# 10 substances, but malaoxon has no positive AEID DNT NAM data, so 9 substances is the total list

aed.rat.css.df=data.frame()
for (i in 1:nrow (op.com.pos.rat))
{
  aed.rat.css.50 <-(calc_mc_oral_equiv(conc=op.com.pos.rat$modl.ga.uM[i], 
                                    chem.cas=op.com.pos.rat$casn[i], 
                                    which.quantile=c(0.50),
                                    species='Rat',
                                    restrictive.clearance=T, 
                                    output.units='mgpkgpday'))
  
  aed.rat.css.95 <-(calc_mc_oral_equiv(conc=op.com.pos.rat$modl.ga.uM[i], 
                                       chem.cas=op.com.pos.rat$casn[i], 
                                       which.quantile=c(0.95),
                                       species='Rat',
                                       restrictive.clearance=T, 
                                       output.units='mgpkgpday'))
  aed.rat.css.df<-rbind(aed.rat.css.df,
                    cbind(op.com.pos.rat$aeid[i],
                          op.com.pos.rat$DTXSID[i], 
                          op.com.pos.rat$spid[i],
                          op.com.pos.rat$modl.ga.uM[i], 
                          op.com.pos.rat$casn[i],
                          aed.rat.css.50, 
                          aed.rat.css.95))
}
save(aed.rat.css.df, file='./source/aed_from_rat_css_httk_23apr2020.RData')

# now overwrite HTTK data with human to do the huRat

op.chems <- as.data.table(get_cheminfo(info="all", species="Human", model="3compartmentss"))

op.chems.test <- as.data.frame(op.chems[DTXSID %in% op.list[,DTXSID]]) #25 of 27 have human 3 compss httk data
setdiff(op.list$PREFERRED_NAME, op.chems.test$Compound)
#[1] "Z-Tetrachlorvinphos" "Naled" these two substances do not have human empirical data

chem.physical_and_invitro.data <- add_chemtable(
  op.chems.test,
  current.table=chem.physical_and_invitro.data,
  data.list=list(
    Compound="Compound",
    CAS="CAS",
    DTXSID="DTXSID",
    Funbound.plasma="Human.Funbound.plasma",
    Clint="Human.Clint"),
  species="Rat",
  reference="Human",
  overwrite=TRUE)

op.com.pos.hurat <- as.data.frame(subset(op.com.pos, casn %in% get_cheminfo(species='Rat'))) # this is huRat because the HTTK data are now rat

length(unique(op.com.pos.hurat$chnm)) #24 have enough data and rat phys to run model
#[1] "Methamidophos"     "Dimethoate"        "Malathion"         "Trichlorfon"       "Diazinon"          "Chlorpyrifos oxon"
#[7] "Chlorpyrifos"      "Coumaphos"         "Bensulide"         "Dichlorvos"        "Phosmet"           "Omethoate"        
#[13] "Ethoprop"          "Pirimiphos-methyl" "Terbufos"          "Phorate"           "Tribufos"          "Chlorethoxyfos"   
#[19] "Acephate"          "Profenofos"        "Fosthiazate"       "Tebupirimfos"      "Diazoxon"          "Dicrotophos" 

aed.hurat.css.df=data.frame()
for (i in 1:nrow (op.com.pos.hurat))
{
  aed.hurat.css.50 <-(calc_mc_oral_equiv(conc=op.com.pos.hurat$modl.ga.uM[i], 
                                     chem.cas=op.com.pos.hurat$casn[i], 
                                     which.quantile=c(0.50),
                                     species='Rat',
                                     restrictive.clearance=T, 
                                     output.units='mgpkgpday'))
  aed.hurat.css.95 <-(calc_mc_oral_equiv(conc=op.com.pos.hurat$modl.ga.uM[i], 
                                         chem.cas=op.com.pos.hurat$casn[i], 
                                         which.quantile=c(0.95),
                                         species='Rat',
                                         restrictive.clearance=T, 
                                         output.units='mgpkgpday'))
  aed.hurat.css.df<-rbind(aed.hurat.css.df,
                     cbind(op.com.pos.hurat$aeid[i],
                           op.com.pos.hurat$DTXSID[i], 
                           op.com.pos.hurat$spid[i],
                           op.com.pos.hurat$modl.ga.uM[i], 
                           op.com.pos.hurat$casn[i],
                           aed.hurat.css.50,
                           aed.hurat.css.95))
}

save(aed.hurat.css.df, file='./source/aed_from_hurat_css_httk_23apr2020.RData')

# add the rat css values to the human values and mc5 data

aed.rat.css.dt <- as.data.table(aed.rat.css.df)
aed.hurat.css.dt <- as.data.table(aed.hurat.css.df)

setnames(aed.rat.css.dt,
         c('V1','V2','V3','V4','V5'),
         c('aeid','DTXSID','spid','modl.ga.uM','casn'))

setnames(aed.hurat.css.dt,
         c('V1','V2','V3','V4','V5'),
         c('aeid','DTXSID','spid','modl.ga.uM','casn'))

col.num <- c('aeid', 'modl.ga.uM', 'aed.rat.css.50', 'aed.rat.css.95')
aed.rat.css.dt[, (col.num) := lapply (.SD, as.character), .SDcols = col.num ]
aed.rat.css.dt[, (col.num) := lapply (.SD, as.numeric), .SDcols = col.num ]

col.num <- c('aeid', 'modl.ga.uM', 'aed.hurat.css.50', 'aed.hurat.css.95')
aed.hurat.css.dt[, (col.num) := lapply (.SD, as.character), .SDcols = col.num ]
aed.hurat.css.dt[, (col.num) := lapply (.SD, as.numeric), .SDcols = col.num ]

col.chr <- c('DTXSID','spid','casn')
aed.rat.css.dt[, (col.chr) := lapply (.SD, as.character), .SDcols = col.chr ]
aed.hurat.css.dt[, (col.chr) := lapply (.SD, as.character), .SDcols = col.chr ]

colnames(op.com.pos)
#setnames(op.com.pos, c('dsstox_substance_id'), c('DTXSID'))
#op.com.pos[,key := paste0(aeid,'_',DTXSID,'_',spid,'_',modl.ga.uM, '_', casn)] #already there

aed.rat.css.dt[,key := paste0(aeid,'_',DTXSID,'_',spid,'_',modl.ga.uM, '_', casn)]
aed.hurat.css.dt[,key := paste0(aeid,'_',DTXSID,'_',spid,'_',modl.ga.uM, '_', casn)]

op.com.pos$aed.rat.3css.50 <- aed.rat.css.dt$aed.rat.css.50[match(op.com.pos$key,
                                                                  aed.rat.css.dt$key)]
op.com.pos$aed.rat.3css.95 <- aed.rat.css.dt$aed.rat.css.95[match(op.com.pos$key,
                                                                  aed.rat.css.dt$key)]

op.com.pos$aed.hurat.3css.50 <- aed.hurat.css.dt$aed.hurat.css.50[match(op.com.pos$key,
                                                                  aed.hurat.css.dt$key)]
op.com.pos$aed.hurat.3css.95 <- aed.hurat.css.dt$aed.hurat.css.95[match(op.com.pos$key,
                                                                  aed.hurat.css.dt$key)]

op.com.pos$bmd10.rat <- op.bmd$BMD10[match(op.com.pos$DTXSID, op.bmd$DTXSID)]
op.com.pos$bmdl10.rat <- op.bmd$BMDL10[match(op.com.pos$DTXSID, op.bmd$DTXSID)]
op.com.pos[,chnm_spid := paste0(chnm,'_',spid)]
op.com.pos[,bmd10.hum := ifelse(!is.na(bmd10.rat), bmd10.rat/10, NA)]
op.com.pos[,bmdl10.hum := ifelse(!is.na(bmdl10.rat), bmdl10.rat/10, NA)]

load('./source/HCI_13Apr2020.RData')
load('./source/NHEERL_MEA_dev_13Apr2020.RData')

mea.tbl <- mea.tbl[order(-aeid)]

op.com.pos[aeid %in% c(2494:2527,2529:2530), selectivity := round(mea.min.cyt - modl_ga,2)]
op.com.pos[aeid %in% c(2789:2792), selectivity := round(hci.hn2.nog.cyt - modl_ga,2)]# hn2 nog
op.com.pos[aeid %in% c(2777:2780), selectivity := round(hci.rat.nog.cyt - modl_ga, 2)] # rat nog
op.com.pos[aeid %in% c(2781:2788), selectivity := round(hci.synap.cyt - modl_ga, 2)]
op.com.pos[aeid %in% c(2795:2797), selectivity := round(hci.hNP1.cyt - modl_ga, 2)]
op.com.pos[aeid %in% c(2793:2794), selectivity := 2]
op.com.pos[selectivity >0.3, selectivity.score := 1]
op.com.pos[selectivity <0.3, selectivity.score := 0]
op.com.pos[selectivity==2, selectivity.score := 2]

op.com.pos.long <- melt.data.table(op.com.pos,
                                   id.vars = c('spid','casn','chnm','DTXSID',
                                               'aeid','aenm','modl_ga','modl.ga.uM','selectivity','selectivity.score',
                                               'bmd10.rat', 'bmdl10.rat','bmd10.hum','bmdl10.hum',
                                               'chnm_spid'),
                                   measure.vars = c('aed.human.3css.50',
                                                    'aed.human.3css.95',
                                                    'aed.rat.3css.50',
                                                    'aed.rat.3css.95',
                                                    'aed.hurat.3css.50',
                                                    'aed.hurat.3css.95'),
                                   variable.name = 'AED_type',
                                   value.name = 'AED')

op.com.pos.long[aeid %in% mea.tbl[,aeid], species := 'rat.assay']
op.com.pos.long[aeid %in% hci.tbl[activity %in% c('NOG initiation, rat', 'Synaptogenesis/maturation, rat'),aeid],species := 'rat.assay']
op.com.pos.long[aeid %in% hci.tbl[activity %in% c('Apoptosis/viability, hNP1',
                                                  'NOG initiation, hN2',
                                                  'Proliferation, hNP1'),aeid], species := 'human.assay']
op.com.pos.long[,aed.type.species := paste0(AED_type, '_', species)]
unique(op.com.pos.long$aed.type.species)

op.com.pos.long[species=='rat.assay', bmd10.hum := as.numeric(NA)]
op.com.pos.long[species=='rat.assay', bmdl10.hum := as.numeric(NA)]
op.com.pos.long[species=='human.assay', bmd10.rat := as.numeric(NA)]
op.com.pos.long[species=='human.assay', bmdl10.rat := as.numeric(NA)]

op.com.pos.long$aed.type.species <- factor(op.com.pos.long$aed.type.species, levels=c('aed.human.3css.50_human.assay',
                                                                                      'aed.human.3css.50_rat.assay',
                                                                                      'aed.human.3css.95_human.assay',
                                                                                      'aed.human.3css.95_rat.assay',
                                                                                      'aed.rat.3css.50_human.assay',
                                                                                      'aed.rat.3css.50_rat.assay',
                                                                                      'aed.rat.3css.95_human.assay',
                                                                                      'aed.rat.3css.95_rat.assay',
                                                                                      'aed.hurat.3css.50_human.assay',
                                                                                      'aed.hurat.3css.50_rat.assay',
                                                                                      'aed.hurat.3css.95_human.assay',
                                                                                      'aed.hurat.3css.95_rat.assay'))

aed.plot <- op.com.pos.long[aed.type.species %in% c('aed.human.3css.50_human.assay',
                                        'aed.rat.3css.50_rat.assay',
                                        'aed.hurat.3css.50_rat.assay') & !(DTXSID %in% c('DTXSID1024209',
                                                                                         'DTXSID1032648',
                                                                                         'DTXSID2032344'))]
unique(aed.plot$species)
aed.plot[species=='rat.assay',species.name := 'rat']
aed.plot[species=='human.assay', species.name := 'human']
aed.plot[,chnm_species := paste0(chnm, ", ", species.name)]

#save(aed.plot, file='aed_plot_data_3may2020.RData') # also saved as Fig6_DNT_NAM_AED_data.xlsx
aed.plot2 <- aed.plot[!is.na(AED)]

colors <- c('#39568CFF',
            '#1f968BFF',
            '#73D055FF',
            'black',
            'red',
            'gray')

dev.off()

pdf(file=paste('aed_human&rat_all_thinline',
               format(Sys.Date(),
                      "%y%m%d.pdf"),
               sep="_"),
    height=9,
    width=11,
    pointsize=10)

for (i in 1:8) {
  print(
    ggplot(data=aed.plot2,
           aes(y=AED, x=factor(aed.type.species)))+
      geom_boxplot(outlier.shape = NA, width = 0.33, 
                   aes(fill=factor(aed.type.species)))+
      geom_jitter(alpha=0.8, color='black',aes(shape=factor(selectivity.score)))+
      facet_grid_paginate(chnm~species.name, 
                          switch='y',
                          scales = 'free_x', 
                          ncol=2, 
                          nrow=3, 
                          page=i)+
      geom_hline(data = aed.plot2, 
                 mapping = aes(yintercept=bmd10.hum, linetype='huBMD10'), color='red', size=0.5,show.legend =TRUE)+
      geom_hline(data = aed.plot2, 
                 mapping = aes(yintercept=bmdl10.hum, linetype='huBMDL10'), color='blue', size=0.5,show.legend=TRUE)+
      geom_hline(data = aed.plot2, 
                 mapping = aes(yintercept=bmd10.rat, linetype='BMD10'), color='darkorange3', size=0.5, show.legend=TRUE)+
      geom_hline(data = aed.plot2, 
                 mapping = aes(yintercept=bmdl10.rat, linetype='BMDL10'), color='purple', size=0.5,show.legend=TRUE)+
      scale_linetype_manual(name = 'BMD Values',
                            values = c('BMD10' = 1,
                                       'BMDL10' = 2,
                                       'huBMD10' = 1,
                                       'huBMDL10' = 2
                                       ),
                            guide = guide_legend(override.aes = list(colour = c('darkorange3',
                                                                                'purple',
                                                                                'red',
                                                                                'blue'
                                                                             ))))+
      scale_y_continuous(trans='log10',
                         breaks= c(0.0001,
                                   0.001,
                                   0.01,
                                   0.1,
                                   1,
                                   10,
                                   100,
                                   1000),
                         labels = c(0.0001,
                                    0.001,
                                    0.01,
                                    0.1,
                                    1,
                                    10,
                                    100,
                                    1000
                                    ))+
      scale_fill_manual(name = 'AED estimates',
                        breaks = c('aed.human.3css.50_human.assay',
                                   'aed.rat.3css.50_rat.assay',
                                   'aed.hurat.3css.50_rat.assay'),
                        labels = c('Hum, AED50, hum cells',
                                   'Rat, AED50, rat cells',
                                   'huRat, AED50, rat cells'),
                        values = c('#39568CFF',
                                   '#1f968BFF',
                                   '#73D055FF'))+
      scale_shape_manual(name = 'selectivity',
                         breaks = c('0',
                                    '1',
                                    '2'),
                         labels = c('not selective',
                                    'selective',
                                    'NA'),
                         values = c(21,17,15))+
      guides(fill = guide_legend(override.aes = list(linetype = 0), 
                                 nrow=4, byrow=TRUE),
             shape = guide_legend(override.aes = list(linetype=0)))+
      coord_cartesian(ylim=c(0.0001,3000))+
      xlab("IVIVE Comparison Method") +
      ylab('Administered equivalent doses (mg/kg/day)')+
      theme_bw() +
      theme(
        axis.line = element_line(colour = "black"),
        axis.text = element_text(size=10),
        axis.title = element_text(size=12, face='bold'))+
      theme(axis.text = element_text(family = "sans", face = "bold", size=12),
            axis.text.x = element_blank(),
            axis.ticks.x = element_blank())+
      theme(legend.position="right", legend.title=element_blank())+
      theme(
        strip.background = element_rect(
          color="black", 
          fill="white", linetype="solid"
        ),
        strip.text.x = element_text(
          size = 12, color = "black", face = "bold"
        ),
        strip.text.y = element_text(
          size = 12, color = "black", face = "bold"
        )
      )
  )
  
}
dev.off()



op.com.pos[aeid %in% mea.tbl[,aeid], species := 'rat.assay']
op.com.pos[aeid %in% hci.tbl[activity %in% c('NOG initiation, rat', 'Synaptogenesis/maturation, rat'),aeid],species := 'rat.assay']
op.com.pos[aeid %in% hci.tbl[activity %in% c('Apoptosis/viability, hNP1',
                                                  'NOG initiation, hN2',
                                                  'Proliferation, hNP1'),aeid], species := 'human.assay']

tbl14 <- op.com.pos[ , list(
  DTXSID = DTXSID,
  pos.rat = sum(hitc[species=='rat.assay']),
  rat.min.ac50 = min(modl.ga.uM[species=='rat.assay'], na.rm=TRUE),
  rat.med.ac50 = median(modl.ga.uM[species=='rat.assay'], na.rm=TRUE),
  pos.hum = sum(hitc[species=='human.assay']),
  hum.min.ac50 = min(modl.ga.uM[species=='human.assay'], na.rm=TRUE),
  hum.med.ac50 = median(modl.ga.uM[species=='human.assay'], na.rm=TRUE)
), by = list(chnm)]
tbl14 <- unique(tbl14)



tbl14.summ <- tbl14 %>% mutate_at(vars(rat.min.ac50,
                                  rat.med.ac50,
                                  hum.min.ac50,
                                  hum.med.ac50), ~round(.,2)) %>% data.table()

write.csv(tbl14.summ, file='../output/tbl14_dnt_nam_potency_ranges_for_AEDs.csv')

#-------------------------------------------------------------------------------------#
# Rat and huRat comparison to PBPK-AEDs
#-------------------------------------------------------------------------------------#

# read in the pbpk-derived AEDs
aed.pbpk <- fread('./source/dnt_nam_pbpk_aed_info.csv')

# create the subset of data for the 3 substances with PBPK-derived AEDs
aed.plot.subset <- aed.plot2[chnm %in% aed.pbpk[,chnm]]
aed.plot.subset[,c('bmd10.hum','bmdl10.hum'):= NULL]
aed.plot.subset <- aed.plot.subset[!aed.type.species=='aed.human.3css.50_human.assay']
aed.plot.sub.pbpk <- rbind(aed.plot.subset, aed.pbpk)

# create comparison figure
fig.aed.pbpk <- ggplot(data=aed.plot.sub.pbpk,
           aes(y=AED, x=factor(aed.type.species)))+
      geom_boxplot(outlier.shape = NA, 
                   width = 0.33, 
                   aes(fill=factor(aed.type.species)))+
      geom_jitter(alpha=0.8, color='black',aes(shape=factor(selectivity.score)))+
      facet_wrap(~chnm, 
                 scales = 'free_y', 
                 ncol=3, 
                 nrow=1)+
      geom_hline(data = aed.plot.sub.pbpk, 
                 mapping = aes(yintercept=bmd10.rat, linetype='BMD10'), color='darkorange3', size=0.5, show.legend=TRUE)+
      geom_hline(data = aed.plot.sub.pbpk, 
                 mapping = aes(yintercept=bmdl10.rat, linetype='BMDL10'), color='purple', size=0.5,show.legend=TRUE)+
      scale_linetype_manual(name = 'BMD Values',
                            values = c('BMD10' = 1,
                                       'BMDL10' = 2),
                            guide = guide_legend(override.aes = list(colour = c('darkorange3',
                                                                                'purple'
                            ))))+
      scale_y_continuous(trans='log10',
                         breaks= c(0.01,
                                   0.1,
                                   1,
                                   10,
                                   100,
                                   1000),
                         labels = c(0.01,
                                    0.1,
                                    1,
                                    10,
                                    100,
                                    1000
                         ))+
      scale_fill_manual(name = 'AED estimates',
                        breaks = c('aed.rat.3css.50_rat.assay',
                                   'aed.hurat.3css.50_rat.assay',
                                   'aed.rat.pbpk.plasma_rat.assay',
                                   'aed.rat.pbpk.brain_rat.assay'),
                        labels = c(
                                   'Rat, AED50, rat cells',
                                   'huRat, AED50, rat cells',
                                   'Rat, PBPK-AED-plasma, rat cells',
                                   'Rat, PBPK-AED-brain, rat cells'),
                        values = c(
                                   '#1f968BFF',
                                   '#73D055FF',
                                   '#FDE725FF',
                                   '#440154FF'))+
      scale_shape_manual(name = 'selectivity',
                         breaks = c('0',
                                    '1',
                                    '2'),
                         labels = c('not selective',
                                    'selective',
                                    'NA'),
                         values = c(21,17,15))+
      guides(fill = guide_legend(override.aes = list(linetype = 0), 
                                 nrow=4, byrow=TRUE),
             shape = guide_legend(override.aes = list(linetype=0)))+
      coord_cartesian(ylim=c(0.01,3000))+
      xlab("IVIVE Approach") +
      ylab('Administered equivalent doses (mg/kg/day)')+
      theme_bw() +
      theme(
        axis.line = element_line(colour = "black"),
        axis.text = element_text(size=10),
        axis.title = element_text(size=12, face='bold'))+
      theme(axis.text = element_text(family = "sans", face = "bold", size=12),
            axis.text.x = element_blank(),
            axis.ticks.x = element_blank())+
      theme(legend.position="right", legend.title=element_blank())+
      theme(
        strip.background = element_rect(
          color="black", 
          fill="white", linetype="solid"
        ),
        strip.text.x = element_text(
          size = 12, color = "black", face = "bold"
        ),
        strip.text.y = element_text(
          size = 12, color = "black", face = "bold"
        )
      )
  
file.dir <- paste("./output/figures/", sep="")
file.name <- paste("/Fig7_aed_rat&pbpk_subset", Sys.Date(), ".png", sep="")
file.path <- paste(file.dir, file.name, sep="")
dir.create(path=file.dir, showWarnings = FALSE, recursive = TRUE)
png(file.path, width = 7000, height = 2500, res=600)
fig.aed.pbpk
dev.off()