package pviScreen;

import java.util.ArrayList;

import uncertaintyAnalysis.StatisticalDistribution;

/**
 * <p>Title: Building</p>
 *
 * <p>Description: building stores and calculates building parameters</p>
 *
 * <p>Copyright: no copyright claimed</p>
 *
 * <p>Company: US EPA</p>
 *
 * @author Jim Weaver
 * @version 1.0
 *
 * 3-25-2011
 *
 */
public class Building extends PVIScreenData
{
  private double airExchangeRate;
  private double mixingZoneHeight;
  private double h;
  private double qs;
  private double eta;
  private double thCrack;
  private double lengthCrack;
  private double dCrack;
  private double wCrack;
  private double aCrack;
  private double aB;
  private double sq;
  private double qf;
  private double maxExponentialValue;
  private boolean bDirtFloor;
  
  private double dWidth;
  private double dLength;
  private double dDepthBelowGrade;
  private double dXBuildingCenter;
  private double dYBuildingCenter;
 
  

  public Building()
  {
    super();  
	super.setName("building");

    h=0.0;
    
    //create distributions
    super.createNewDistribution("Building","AirExchangeRate");
    super.createNewDistribution("Building","CeilingHeight");
    super.createNewDistribution("Building","Width");
    super.createNewDistribution("Building","Length");
    super.createNewDistribution("Building","FoundationDepthBelowGrade");
    super.createNewDistribution("Building","FoundationThickness");
    super.createNewDistribution("Building","CrackWidth");
    super.createNewDistribution("Building","Qsoil");
    super.createNewDistribution("Building","AirFlowBelowBuilding");
  }


  //set mass transfer coefficient
  public void setMassTransferCoefficient()
  {
	  
	  
	//assure that the current values of the parameters are used.  
	wCrack = this.getCrackWidth();
	lengthCrack = this.getLCrack();
	aB = this.getAb();
	thCrack = this.getThCrack();
	qs = this.getQs();
	mixingZoneHeight = this.getMixingZoneHeight();
	airExchangeRate = this.getAirExchangeRate();
	
    if (!bDirtFloor)
    {
      //area of the cracks
      aCrack = wCrack*lengthCrack;
      //relative area of cracks
      eta = aCrack/aB;
      if (qs <= 0)
      {
        h = eta*dCrack/thCrack;
      }
      else
      {
        sq=(qs/aB)*(thCrack/dCrack)*(1.0/eta);
        if(sq<maxExponentialValue)
        {
          //low sq estimate of h
          double k2=Math.exp(sq);
          double k3=1.0/k2;
          double k4=mixingZoneHeight*airExchangeRate;
          double k5=((k2-1.0)/k2);
          h=k4/(k3+k4*(aB/qs)*k5-1.0);
        }
        else
        {
          //large sq estimate of h:
          h=aB/qs-(1.0/mixingZoneHeight)*(1.0/airExchangeRate);
          h=1./h;
        }
      }
    }
    else
    {
      //mass transfer coefficient for a dirt floor.
      h = 10000000.;
    }
   }
 

  //simple sets
  public void setLocation(double dX, double dY){this.dXBuildingCenter = dX; this.dYBuildingCenter = dY;}
  public void setAirExchangeRate(double dValue){this.airExchangeRate = dValue;this.setConstantDistribution(dValue, "AirExchangeRate");}
  public void setMixingZoneHeight(double dValue){this.mixingZoneHeight = dValue;this.setConstantDistribution(dValue,"CeilingHeight");}
  //air flow rate into the buiding
  public void setQs(double dValue){this.qs = dValue;this.setConstantDistribution(dValue,"Qsoil");}
  //thickness of foundation
  public void setThCrack(double dValue){this.thCrack = dValue;this.setConstantDistribution(dValue,"FoundationThickness");}
  //width of crack
  public void setWCrack(double dValue){this.wCrack = dValue;this.setConstantDistribution(dValue,"CrackWidth");}
  //flow below building
  public void setAirFlowBelowBuilding(double dValue){this.qf = dValue;this.setConstantDistribution(dValue,"AirFlowBelowBuilding");}
  //width
  public void setWidth(double dValue){this.dWidth = dValue;this.setConstantDistribution(dValue,"width");}
  //length
  public void setLength(double dValue){this.dLength = dValue;this.setConstantDistribution(dValue,"length");}
  //depth below grade
  public void setDepthBelowGrade(double dValue){this.dDepthBelowGrade = dValue;this.setConstantDistribution(dValue,"FoundationDepthBelowGrade");}
 
  //all derived parameters -- no statistical distributions for these
  //(as they are recalcualted for each Monte Carlo run)
  //foundation area in contact with soil
  public void setAb(double dValue){this.aB = dValue;}  
  //effective diffusion coefficient of crack (usually assumed to be the same as the underlying soil)
  public void setDCrack(double dValue){this.dCrack = dValue;}
  //length of crack
  public void setLCrack(double dValue){this.lengthCrack = dValue;}
  //maximum exponential value
  public void setMaxExponentialValue(double maxExponentialValue){this.maxExponentialValue = maxExponentialValue;}
  //dirt floor or not
  public void setIsItADirtFloor(boolean bDirtFloor){this.bDirtFloor = bDirtFloor;}

  //simple gets
  public double getXLocation(){return this.dXBuildingCenter;}
  public double getYLocation(){return this.dYBuildingCenter;}
  public double getAirExchangeRate(){return getCurrentRandomValue("airExchangeRate",this.getName());}
  public double getMixingZoneHeight(){return getCurrentRandomValue("CeilingHeight",this.getName());}
  public double getWidth(){return getCurrentRandomValue("Width",this.getName());}
  public double getLength(){return getCurrentRandomValue("Length",this.getName());}
  public double getFoundationDepthBelowGrade(){return getCurrentRandomValue("FoundationDepthBelowGrade",this.getName());}
  public double getFoundationThickness(){return getCurrentRandomValue("FoundationThickness", this.getName());}
  public double getCrackWidth(){return getCurrentRandomValue("CrackWidth",this.getName());}
  public double getThCrack(){return getCurrentRandomValue("FoundationThickness",this.getName());}
  public double getAirFlowBelowBuilding(){return getCurrentRandomValue("AirFlowBelowBuilding",this.getName());}
  public double getQs(){return getCurrentRandomValue("QSoil",this.getName());}
  
  public double getParameter(String sName){return getCurrentRandomValue(sName,this.getName());}
  public double getMiddleValue(String sName){return getMiddleValue(sName,this.getName());}  
  
  //derived parameters 
  public double getFoundationMassTransferCoefficient(){return h;}
  public double getLCrack(){return lengthCrack;}
  public double getAb(){return aB;}
  public boolean getIsItADirtFloor(){return bDirtFloor;}
}
