package pviScreen;

import util.OutputDigits;
import util.Stripper;
import util.UnitConverter;
import uncertaintyAnalysis.StatisticalDistribution;

/**
 * <p>Title: Chemical</p>
 *
 * <p>Description: container for chemical information
 *
 * George E. Devaull, 2007, Indoor Vapor Intrusion with Oxygen-Limited Biodegradation for
 * a Subsurface Gasoline Phase Concentration, Environmental Science and Technology, 41, 3241-3248.
 *
 *
 * </p>
 *
 * <p>
 *  Calculation for the depth of the aerobic zone:
 *  Equation 29 of Devaull, 2007
 *
 *
 *
 *
 *
 *
 * </p>
 *
 *
 * <p>Copyright: no copyright claimed</p>
 *
 * <p>Company: US EPA</p>
 *
 * @author Jim Weaver
 * @version 1.0
 * 3-1-2011
 */
public class Chemical extends PVIScreenData
{
	//Phase Concentration data Phase Concentration
	protected String sInputType;
	//is the chemical used in the simulation?
	protected boolean bIsChemicalUsed;
	//Phase Concentration concentration
	protected double cs;
	//Phase Concentration concentration multiplier
	protected double mf;
	//surface concentration
	//the surface concentration is used to specify the oxygen boundary condition (Oxygen as a chemical)
	protected double cSurface;
	//surface flux
	//the surface flux is used to specify the oxygen flux boundary condition (oxygen as a chemical)
	protected double jSurface;
	//indoor air concentration, ce
	//foundation concentration, cf
	//transition point concentration (calculated from solution of model equations), ct
	protected double ce,cf,ct;
	//transition point flux (calculated from solution of model equations)
	protected double jt;
	//foundation flux;
	protected double jf;
	//saturated vapor concentration
	protected double svc;
	//fuel phase concentration
	protected double dFuelPhaseConcentrationByVolume, dFuelPhaseConcentrationByMass;
	//henrys constant
	protected double kh;
	//oil-water partition coefficient
	protected double dKo;
	//diffusion coefficient in water
	protected double dw;
	//diffusion coefficient in air
	protected double da;
	//first order rate loss coefficient, kw
	protected double kw;
	//molecular weight
	protected double dMW;
	//density
	protected double dDensity;
	//solubility
	protected double dSolubility;
	//oragnic carbon partition coefficient
	protected double dKoc;
	//moisture content
	protected double moistureContent;
	//effective diffusion coefficient
	protected double edf;
	//diffusive-reaction length
	protected double dDiffusiveReactionLength;
	//parameter phi mass ratio of oxygen to chemical comsumption
	//(mg-chemical/mg-oxygen)
	protected double phi;
	//parameter alpha
	protected double alpha;
	//parameters of equations 21, 27, 29
	protected double cOne;
	//vadose zone
	protected VadoseZone vz;
	//aquifer
	protected AquiferData aq;
	//oil lens
	protected OilLens old;
	//unit conversion factors
	protected UnitConverter uc;

	//user input risks
	protected ChemicalRisk cr;
	//is user risk set?
	protected boolean bUserRiskSet=false;



	//parameters for henry's coefficient temperature correction

	//reference temperature
	protected double dHReferenceTemp;
	//reference critical temperature
	protected double dHCriticalTemp;
	//normal boiling point
	protected double dHBoilingTemp;
	//reference henrys constant
	protected double dHReference;
	//reference
	protected double dHEnthalpy;
	//temperature-corrected henrys coefficient
	protected double dHCorrected;


	//risk parameters

	//cancer risk:
	//risk level and concentration are used together:    10-6 risk level for a concentration of 0.29 ug/m3
	//                                                   "dRiskLevel"  risk level for a concentration of "dRiskConcentration"
	//concentration for a specified risk level
	protected double dRiskConcentration;
	//specified risk level
	protected double dRiskLevel;
	//has risk level been set ?
	protected boolean bRiskLevelSet;
	//risk multiplication factor  =  0.29 ug/m3 is risk at 10-6, risk at 10-5 is 0.29 ug/m3 * (10-5)/(10-6)
	protected double dRiskConcentrationMultiplier;

	//non-cancer risk
	//reference concentration for inhalation
	protected double dRfC;
	//reference concentration set?
	protected boolean bRfCSet;

	//risk evaluated through screening level
	//screening level
	protected double dScreeningLevel;
	//screening level set?
	protected boolean bScreeningLevelSet;

	//concentration adjustment for ground water
	protected double dConcentrationAdjustmentGroundWater;
	//concentration adjustment for soil sample
	protected double dConcentrationAdjustmentSoilSample;


	public Chemical()
	{
		super();
		//Phase Concentration concentration multiplier (unless changed later)
		mf = 1.0;
		//chemical are used if an concentration input is entered
		//more chemicals are included in the database than may be used in an individual simulation
		this.bIsChemicalUsed = false;
		//has camcer risk level been set?
		this.bRiskLevelSet = false;
		//has non-cancer reference concentration been set?
		this.bRfCSet = false;
		//has a screening level been set
		this.bScreeningLevelSet = false;
	}



	public void calculateAlpha(){}



	/**
	 * calculateAirPhaseConcentration  determine the air phase concentration from either a fuel composition
	 *    or a water phase concentration 
	 * 
	 */
	public void setAirPhaseConcentration()
	{
		Stripper s = new Stripper();

		//disable chemicals that are not used in the simulation
		//all chemicals in the database are by default given active statistical distributions
		//this will deactivate unused distributions jww 4-21-2016
		for (StatisticalDistribution sd: sdPVIScreen)
		{
			if (s.areTheyEqual(sd.getDistributionType(),"Chemical"))
			{	
				if (s.areTheyEqual(sd.getName(),this.getName()))
				{
					sd.setIsThisDistributionActuallyUsed(true);
				}
			}  
		}

		boolean bAirPhase=false;
		boolean bSoilSample=false;
		boolean bFuelPhaseConcentrationByVolume=false;
		boolean bFuelPhaseConcentrationByMass=false;  
		boolean bWaterPhase=false;

		// fuel Phase Concentration by mass or by volume (the distinction will be made later ) 
		if (s.areTheyEqual("FuelPhaseConcentrationByMass", this.sInputType, ' ')
				|| s.areTheyEqual("FuelPhaseConcentrationByVolume", this.sInputType, ' ') )
		{
			//EPA lab notebook ADA JWW-01 page 175, 186 and 194
			//multiphase partitioning

			//use the top of the contaminant distribution 
			double dElevation = old.getOilZoneTopElevation();


			double dSo = old.getOilSaturation(dElevation);
			double dSa = vz.getTrappedAirSaturation();
			double dSw = 1. - dSo - dSa;

			this.determineKo();
			String sMassOrVolumeConc = "FuelPhaseConcentrationByMass";
			bFuelPhaseConcentrationByMass=true;
			bFuelPhaseConcentrationByVolume=false;
			if (this.sInputType.equalsIgnoreCase("FuelPhaseConcentrationByVolume") || this.sInputType.equalsIgnoreCase("Fuel Phase Concentration by Volume"))
			{
				sMassOrVolumeConc = "FuelPhaseConcentrationByVolume";
				bFuelPhaseConcentrationByMass = false;
				bFuelPhaseConcentrationByVolume=true;
			}
			StatisticalDistribution sdFuel = new StatisticalDistribution();
			for (StatisticalDistribution sd: sdPVIScreen)
			{
				if (sd.getName().equalsIgnoreCase(sMassOrVolumeConc))
				{
					//this is the StatisticalDistribution we need
					sdFuel = sd;
					break;
				}
			}


			int iNumberOfEntries = sdFuel.getNumberOfPointsDefiningDistribution();
			for (int i=0;i<iNumberOfEntries;i++)
			{
				double dFrequency = sdFuel.getFrequency(i);
				double f1 = sdFuel.getConcentration(i);		
				double f2 = old.getOilDensity();
				double f3 = dSo;
				double f4 = vz.getPorosity();

				//difference in determining the mass per total volume from a mass fraction and a volume fraction Phase Concentration
				double dCw = f1*old.getOilDensity()*dSo*vz.getPorosity();
				if (this.sInputType.equalsIgnoreCase("FuelPhaseConcentrationByVolume") || this.sInputType.equalsIgnoreCase("Fuel Phase Concentration by Volume"))
				{
					dCw = f1*this.getDensity()*dSo*vz.getPorosity();
				}


				double denom = vz.getPorosity()*(dSw 
						+ this.getKo()*dSo 
						+ this.getHenrysConstant()*dSa
						+ this.getKoc()*vz.getBulkDensity()*vz.getFractionOrganicCarbon()/vz.getPorosity());
				dCw = dCw/denom;
				double dCa = this.getHenrysConstant()*dCw;

				//add this concentration to the statisticalDistribution for the Phase Concentration concentration
				//this.createNewDistribution("Concentration");
				for (StatisticalDistribution sd: sdPVIScreen)
				{
					if (sd.getName().equalsIgnoreCase("AirPhaseConcentration"))
					{
						sd.addAPoint(dCa, dFrequency);
						break;
					}
				}		  
				this.setCalculationUnit("AirPhaseConcentration",uc.getModelUnit("ConcentrationAir"));
			}
		}
		else if (s.areTheyEqual("SoilSample", this.sInputType))
		{

			//input as soil sample, jww, 4-18-2016, notebook jww-04, page 30-31
			double dSw = vz.getMoistureContent()*vz.getPorosity();
			double dSo = 0.05;
			double dSa = 1.-dSw-dSo;
			if (dSa<0.0)
			{	  
				//do not allow negative air saturations
				dSa = 0.01;
				dSo = 0.04;
			}  

			bSoilSample=true;

			StatisticalDistribution sdSoilSample = new StatisticalDistribution();
			for (StatisticalDistribution sd: sdPVIScreen)
			{
				if (sd.getName().equalsIgnoreCase("SoilSample"))
				{
					//this is the StatisticalDistribution we need
					sdSoilSample = sd;
					break;
				}
			}

			double denom = vz.getPorosity()*(dSw 
					+ this.getKo()*dSo 
					+ this.getHenrysConstant()*dSa
					+ this.getKoc()*vz.getBulkDensity()*vz.getFractionOrganicCarbon()/vz.getPorosity());	  	
			double dBulkHat = vz.getBulkDensity() + dSw*vz.getPorosity()*1.0 + dSo*vz.getPorosity()*0.75;

			int iNumberOfEntries = sdSoilSample.getNumberOfPointsDefiningDistribution();
			for (int i=0;i<iNumberOfEntries;i++)
			{

				//add an air phase concentration for each point on the input distribution
				double dFrequency = sdSoilSample.getFrequency(i); 
				double dCw = dBulkHat*sdSoilSample.getConcentration(i)/denom;
				double dCa = this.getHenrysConstant()*dCw;

				//adjust concentration of soil sample
				//JWW 3-16-2017
				dCa = dCa * this.dConcentrationAdjustmentSoilSample;

				for (StatisticalDistribution sd: sdPVIScreen)
				{
					if (sd.getName().equalsIgnoreCase("AirPhaseConcentration"))
					{
						sd.addAPoint(dCa, dFrequency);
						break;
					}
				}

			}

			this.setCalculationUnit("AirPhaseConcentration",uc.getModelUnit("ConcentrationAir"));



		}
		else if (s.areTheyEqual(this.sInputType,"WaterPhaseConcentration",' '))
		{
			//water phase concentration input
			//not implemented, jww 4-18-2016
			//included 2/22/2017
			StatisticalDistribution sdWaterSample = new StatisticalDistribution();
			for (StatisticalDistribution sd: sdPVIScreen)
			{
				if (sd.getName().equalsIgnoreCase("WaterPhaseConcentration"))
				{
					//this is the StatisticalDistribution we need
					sdWaterSample = sd;
					break;
				}
			}



			int iNumberOfEntries = sdWaterSample.getNumberOfPointsDefiningDistribution();


			for (int i=0;i<iNumberOfEntries;i++)
			{

				//add an air phase concentration for each point on the input distribution
				double dFrequency = sdWaterSample.getFrequency(i); 

				//adjust the volume units:  the standard air phase concentration in PVIScreen is per cm3
				//whilst the standard water phase concentration in PVIScreen in per L
				//thus the water phase concentration should be multiplied by (1/1000)
				double dCa = this.getHenrysConstant()*sdWaterSample.getValue(dFrequency)*(1.0/1000.0);


				//adjust the air phase concentration by the adjustment factor
				//JWW 3-16-2017
				dCa = dCa * this.dConcentrationAdjustmentGroundWater;

				for (StatisticalDistribution sd: sdPVIScreen)
				{
					if (sd.getName().equalsIgnoreCase("AirPhaseConcentration"))
					{
						sd.addAPoint(dCa, dFrequency);
						break;
					}
				}

			}		

			this.setCalculationUnit("AirPhaseConcentration",uc.getModelUnit("ConcentrationAir"));

			bWaterPhase = true;
		}
		else
		{
			//air input no conversion needed
			bAirPhase = true;

		}




		//indicate unused statistical distributions, jww 4-21-2016
		for (StatisticalDistribution sd: sdPVIScreen)
		{
			if (s.areTheyEqual(sd.getName(),"AirPhaseConcentration"))
			{
				if (bAirPhase==false){sd.setIsThisDistributionActuallyUsed(false);} 
			}
			if (s.areTheyEqual(sd.getName(),"SoilSample"))
			{ 
				if(bSoilSample==false){sd.setIsThisDistributionActuallyUsed(false);}
			}
			if (s.areTheyEqual(sd.getName(),"FuelPhaseConcentrationByVolume"))
			{
				if(bFuelPhaseConcentrationByVolume==false){sd.setIsThisDistributionActuallyUsed(false);}
			}
			if (s.areTheyEqual(sd.getName(),"FuelPhaseConcentrationByMass"))
			{
				if(bFuelPhaseConcentrationByMass==false){sd.setIsThisDistributionActuallyUsed(false);}
			}
			if (s.areTheyEqual(sd.getName(),"WaterPhaseConcentration"))
			{
				if(bWaterPhase==false){sd.setIsThisDistributionActuallyUsed(false);}
			}

		}

	}


	/**
	 * determineKo  determine the value of the oil-water partition coefficient
	 */
	public void determineKo()
	{
		//Weaver et al., 1994, EPA 600/R-94//039a page 175, equation 45
		//molecular weight of gasoline 105 g/mole = 105000 mg/mole == but this is entered as input
		dKo = old.getOilDensity()/old.getOilMolecularWeight();
		double dMw =  this.getMolecularWeight();
		double dS = this.getSolubility();
		double denom =  (this.getSolubility())/this.getMolecularWeight();
		dKo = dKo/denom;
	}

	/**
	 *
	 * @param name String
	 */
	public void determineEffectiveDiffusionCoefficient()
	{
		double porosity;
		double moistureContent;
		double airFilledPorosity;
		//get the porosity and moisture content
		porosity = vz.getPorosity();
		moistureContent = vz.getMoistureContent();
		airFilledPorosity = porosity - moistureContent;
		//chemical properties (forces gets to go through statistical distributions)
		da = this.getDa();
		dw = this.getDw();
		kh = this.getHenrysConstant();


		//effective diffusion coefficient
		//Devaull, 2007, eq 2
		//air phase contribution
		edf = (Math.pow(airFilledPorosity,10./3.)/porosity)*(da/porosity);

		//water phase contribution
		edf = edf + (Math.pow(moistureContent,10./3.)/porosity)*(1./porosity)*(dw/kh);
	}

	//sets
	public void setAlpha(double d)
	{
		// diffusion reaction length
		//this must be divided by the depth:
		//this function gives alpha for a depth of one
		//
		// alpha = sqrt(  (deff/kw)*(kh/moistureContent) * length  )
		//
		//
		alpha = d/getDiffusiveReactionLength();
	}



	/**
	 * diffusive Reaction length
	 * @return double
	 */
	public void setDiffusiveReactionLength()
	{
		//update the parameters (if using monte carlo, these could change from run to run)  
		kw = this.getKw();
		kh = this.getHenrysConstant();
		moistureContent = vz.getMoistureContent();
		edf = this.getEffectiveDiffusionCoefficient();
		//diffusion-reaction length
		dDiffusiveReactionLength = Math.sqrt(  (edf/kw) * (kh/moistureContent));
	}

	/**
	 * convertMassFractionToVolumeFraction
	 *   convert the mass fraction concentration to volume fraction concentration
	 *   Weaver EPA Lab Notebook Ada-jww-01 page 187
	 */
	public void convertMassFractionToVolumeFraction()
	{
		this.dFuelPhaseConcentrationByVolume = this.dFuelPhaseConcentrationByMass*old.getOilDensity()/this.getDensity();
	}

	/**
	 * convertVolumeFractionToMassFraction
	 *   convert the mass fraction concentration to volume fraction concentration
	 *   Weaver EPA Lab Notebook Ada-jww-01 page 187
	 */
	public void convertVolumeFractionToMassFraction()
	{
		this.dFuelPhaseConcentrationByMass = this.dFuelPhaseConcentrationByVolume*this.getDensity()/old.getOilDensity();
	}

	public void correctHenrysConstantForTemperature()
	{  
		double dRc = 1.9872;

		//temperature conversion from USEEPA OSWER 2001


		double dTempSoil = vz.getSoilTemperature() + 273.15;

		double dTempRatio = (this.dHBoilingTemp + 273.15)/(this.dHCriticalTemp +  273.15);
		double dExponent = 0.30;
		if (dTempRatio<0.57) {dExponent = 0.30;}
		else if (0.57<=dTempRatio && dTempRatio<=0.71){dExponent = 0.74*dTempRatio - 0.116;}
		else if (dTempRatio>0.71){dExponent = 0.41;}


		//no values of dTempRatio are allowed--quick fix (3-13-2018, jww)
		if(dTempRatio>1.0){dTempRatio=0.99;}


		double dDeltaHVTS = this.dHEnthalpy*Math.pow(  (1-dTempSoil/(this.dHCriticalTemp+273.15))/(1. - dTempRatio)  ,dExponent);
		this.dHCorrected = Math.exp(- (dDeltaHVTS/dRc) * (  1.0/dTempSoil - 1.0/(this.dHReferenceTemp+273.15)))*this.dHReference;



	}
	
	//write out the parameter values
	public void writeParameters()
	{
		String sOut="";
		
		sOut = sOut + " degadation, " + kw + ", ";
		sOut = sOut + " diff air, " + da + ", ";
		sOut = sOut + " diff water, " + dw + ", ";
		sOut = sOut + " henry's, " + kh + ", ";
		sOut = sOut + " corrected henry's, " + dHCorrected + ", ";
		
	    System.out.println(sOut);
		
	}

	//simple sets
	public void setInputType(String sInputType){this.sInputType=sInputType;}
	public void setIsChemicalUsed(boolean bIsChemicalUsed){this.bIsChemicalUsed = bIsChemicalUsed;}
	//public void setPhase ConcentrationConcentration(double cs){this.cs = cs;}
	public void setSourceConcentrationConcentrationMultiplier(double mf){this.mf = mf;}
	public void setSurfaceConcentration(double cSurface){this.cSurface = cSurface;}
	public void setSurfaceFlux(double jSurface){this.jSurface = jSurface;}

	public void setSourceConcentration(double dValue){this.setConstantDistribution(dValue, "AirPhaseConcentration");}
	public void setSaturatedVaporConcentration(double dValue){this.svc = dValue;this.setConstantDistribution(dValue,"SaturatedVaporConcentration");}  
	public void setFuelPhaseConcentrationByVolume(double dValue){this.dFuelPhaseConcentrationByVolume = dValue;this.setConstantDistribution(dValue,"FuelPhaseConcentrationByVolume");}
	public void setFuelPhaseConcentrationByMass(double dValue){this.dFuelPhaseConcentrationByMass = dValue;this.setConstantDistribution(dValue,"FuelPhaseConcentrationByMass");}

	public void setHenrysConstant(double dValue){this.kh = dValue;this.setConstantDistribution(dValue,"HenrysConstant");}  
	public void setDa(double dValue){this.da = dValue;this.setConstantDistribution(dValue,"DiffusionInAir");}
	public void setDw(double dValue){this.dw = dValue;this.setConstantDistribution(dValue,"DiffusionInWater");}
	public void setKw(double dValue){this.kw = dValue;this.setConstantDistribution(dValue,"degradation");}
	public void setPhi(double dValue){this.phi = dValue;this.setConstantDistribution(dValue,"UtilizationFactor");}
	public void setMolecularWeight(double dValue){this.dMW = dValue;this.setConstantDistribution(dValue,"MolecularWeight");}
	public void setDensity(double dValue){this.dDensity = dValue;this.setConstantDistribution(dValue,"Density");}
	public void setSolubility(double dValue){this.dSolubility = dValue;this.setConstantDistribution(dValue,"Solubility");}
	public void setKoc(double dValue){this.dKoc = dValue;this.setConstantDistribution(dValue,"Koc");}

	//objects
	public void setVadoseZone(VadoseZone vz){this.vz = vz;}
	public void setAquifer(AquiferData aq){this.aq = aq;}
	public void setOilLensData(OilLens old){this.old = old;}
	public void setUnitConverter(UnitConverter uc){this.uc = uc;}

	//results
	public void setTransitionPointConcentration(double ct){this.ct = ct;}
	public void setTransitionPointFlux(double jt){this.jt = jt;}
	public void setIndoorAirConcentration(double ce){this.ce = ce;}
	public void setFoundationConcentration(double cf){this.cf = cf;}
	public void setFoundationFlux(double jf){this.jf = jf;}


	//set henry's constant temperature conversion factors
	public void setReferenceHenrysConstant(double dValue){this.dHReference = dValue;}
	public void setReferenceHenrysConstantTemp(double dValue){this.dHReferenceTemp = dValue;}
	public void setNormalBoilingPoint(double dValue){this.dHBoilingTemp = dValue;}
	public void setCritialTemp(double dValue){this.dHCriticalTemp = dValue;}
	public void setEnthalpy(double dValue){this.dHEnthalpy = dValue;}


	//risk levels
	public void setRiskLevel(double dRiskLevel){this.dRiskLevel = dRiskLevel;}
	public void setRiskConcentration(double dRiskConcentration){this.dRiskConcentration = dRiskConcentration;}
	public void setRiskLevelSet(boolean bRiskLevelSet){this.bRiskLevelSet = bRiskLevelSet;}
	public void setReferenceConcentration(double dRfC){this.dRfC = dRfC;}
	public void setReferenceConcentrationSet(boolean bRfCSet){this.bRfCSet = bRfCSet;}

	public void setUserRisk(ChemicalRisk cr){this.cr = cr;}
	public void setUserRiskSet(boolean bUserRiskSet){this.bUserRiskSet = bUserRiskSet;}

	public void setScreeningLevel(double dScreeningLevel){this.dScreeningLevel = dScreeningLevel;}
	public void setSreeningLevelSet(boolean b){this.bScreeningLevelSet = b;}

	//concentration adjustment factors
	public void setConcentrationAdjustmentGroundWater(double d){this.dConcentrationAdjustmentGroundWater = d;}
	public void setConcentrationAdjustmentSoilSample(double d){this.dConcentrationAdjustmentSoilSample = d;}


	//gets



	public double getUCFmgM3()
	{
		//internal calculation in mg/cm3
		//unit conversion factor to mg/m3
		//mg/m3 = (mg/cm3)*(100 cm/m)*(100cm/m)*(100cm/m)
		return (100.*100.*100.);
	}

	public void determineLeachedConcentration(double dTime)
	{
		//leaching calculation
		double dOilZone = old.getOilZoneTopElevation() - (vz.getDepthToBottom()-vz.getDepthToHistoricWaterTable());
		double dVolumeTotal = old.getLensArea()*dOilZone;

		//average oil saturation
		double dSatAtTop = old.getOilSaturation(old.getOilZoneTopElevation());
		double dSoAvg = 0.5*(dSatAtTop + old.getResidualOilSaturation());
		double dMobileOilZone = old.getOilZoneTopElevation() - old.getMobileOilZoneBottomElevation();
		double dZResidual = old.getMobileOilZoneBottomElevation() -  (vz.getDepthToBottom()-vz.getDepthToHistoricWaterTable());
		double dSo = (dZResidual*old.getResidualOilSaturation() + dSoAvg*dMobileOilZone)/(dZResidual+dMobileOilZone);

		double dSw = 1. - dSo;
		double dBCoeff = dSw;
		dBCoeff = dBCoeff + dSo*this.getKo();
		dBCoeff = dBCoeff + (vz.getBulkDensity()/vz.getPorosity())*vz.getFractionOrganicCarbon()*this.getKoc();

		//ground water velocity
		double dqw = Math.abs(aq.getGradient()*aq.getHydraulicConductivity());
		//adjustment
		dqw = 0.90*dqw;

		double dQw = dqw * old.getLensWidth()*(dZResidual+dMobileOilZone);

		//coefficient
		double dK;

		dK = 1/vz.getPorosity();
		dK = dK/dVolumeTotal;
		dK = dK/dBCoeff;
		dK = dK*dQw;
		dK = -1.0*dK;

		//convert volume concentration input to mass of chemical per volume oil
		double dFuelPhase = this.getFuelSourceByVolume();
		double dCInitial = this.getFuelSourceByVolume()*this.getDensity();
		//convert mass of chemical per volume of oil to water phase concentration
		double dKo = this.getKo();
		dCInitial = dCInitial/this.getKo();

		//water phase concentration at time dTime
		double dFactor = dK*dTime;

		double dMult = Math.exp(dFactor);
		dMult = Math.pow(2.7178, dFactor);


		double dCw = dCInitial*Math.exp(dFactor);

		dFvi = this.getKo()*dCw/this.getDensity();

	}
	double dFvi;

	//dRiskLevelDesired = risk level desired
	public void setRiskConcentrationMultiplier(double dRiskLevelDesired)
	{
		this.dRiskConcentrationMultiplier = dRiskLevelDesired/this.dRiskLevel;
	}

	//update default risk levels (from systemData/03ChemicalProperties.csv) with user inputs (from main input file)
	public void updateRisk()
	{
		//did the user update the risk levels?
		if (bUserRiskSet)
		{	  
			//yes, so change risk levels
			this.dRiskLevel = cr.getRiskLevel();
			this.dRiskConcentration = cr.getRiskConcentration()*uc.getUCF("RiskConcentration",cr.getRiskUnit());
			this.dRfC = cr.getHQConcentration()*uc.getUCF("RiskConcentration", cr.getHQUnit());
		}
	}


	public double getLeachedVolumeFraction(){return dFvi;}

	//simple gets
	public String getInputType(){return sInputType;}
	public boolean getIsChemicalUsed(){return this.bIsChemicalUsed;}
	//public double getPhase ConcentrationConcentration(){return cs*mf;}
	public double getConcentrationMultiplier(){return mf;}
	public double getSurfaceConcentration(){return cSurface;}
	public double getSurfaceFlux(){return jSurface;}

	public double getPhi(){return getCurrentRandomValue("UtilizationFactor",this.getName());}
	public double getDa(){return getCurrentRandomValue("DiffusionCoefficientInAir",this.getName());}
	public double getDw(){return getCurrentRandomValue("DiffusionCoefficientInWater",this.getName());}
	//public double getHenrysConstant(){return getCurrentRandomValue("HenrysConstant",this.getName());}
	public double getHenrysConstant(){return this.dHCorrected;}
	public double getKw(){return getCurrentRandomValue("Degradation",this.getName());}
	public double getMolecularWeight(){return getCurrentRandomValue("MolecularWeight",this.getName());}
	public double getDensity(){return getCurrentRandomValue("Density",this.getName());}
	public double getSolubility(){return getCurrentRandomValue("Solubility",this.getName());}
	public double getKoc(){return getCurrentRandomValue("Koc",this.getName());}
	public double getSourceConcentration(){return getCurrentRandomValue("AirPhaseConcentration",this.getName());}
	public double getSourceConcentration(String sType){return getCurrentRandomValue(sType,this.getName());}
	public double getFuelSourceByVolume(){return getCurrentRandomValue("FuelPhaseConcentrationByVolume",this.getName());}
	public double getFuelSourceByMass(){return getCurrentRandomValue("FuelPhaseConcentrationByMass",this.getName());} 


	public double getEffectiveDiffusionCoefficient(){return edf;}
	public double getKo(){return dKo;}
	public double getDiffusiveReactionLength(){return dDiffusiveReactionLength;}
	public double getAlpha(){return alpha;}
	public double getTransitionPointConcentration(){return ct;}
	public double getTransitionPointFlux(){return jt;}
	public double getIndoorAirConcentration(){return ce;}
	public double getFoundationConcentration(){return cf;}
	public double getFoundationFlux(){return jf;}


	//get risk information
	//cancer
	public double getRiskLevel(){return this.dRiskLevel;}
	public double getRiskConcentration(){return this.dRiskConcentration;}
	public boolean getRiskLevelSet(){return this.bRiskLevelSet;}
	public double getRiskConcentrationMultiplier(){return this.dRiskConcentrationMultiplier;}
	//non-cancer
	public double getReferenceConcentration(){return this.dRfC;}
	public boolean getReferenceConcentrationSet(){return this.bRfCSet;}
	//risk via screening level
	public double getScreeningLevel(){return this.dScreeningLevel;}
	public boolean getScreeningLevelSet(){return this.bScreeningLevelSet;}
}
