package pviScreen;



import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;



import uncertaintyAnalysis.MonteCarloModel;
import uncertaintyAnalysis.StatisticalDistribution;
import util.CalendarClock;
import util.FileLogger;
import util.OutputDigits;
import util.Stripper;
import util.UnitConverter;
import util.UnitDefinition;

import numerics.Bisection.Bisection;
import numerics.Bisection.Bisectable;


/**
 * <p>Title: EPAPVIScreen</p>
 *
 * <p>Description:  EPAPVIScreen implements the BioVapor Equations as summarized in the API publication:
 *
 * Users Manual BioVapor A 1-D Vapor Intrusion Model with Oxygen-Limited Aerobic Biodegradation, GSI Environmental Inc,
 * 2211 Norfolk, Suite 1000, Houston, Texas 77098-4054, Jan 8, 2010.
 *
 * Appendix B:
 *
 * See Also EPA NoteBook of James William Weaver, Ada-01
 *
 *  </p>
 *
 * <p>Copyright: no copyright claimed</p>
 *
 * <p>Company: US EPA</p>
 *
 * @author Jim Weaver
 * @version 1.0
 * 
 * 
 * internal calculation units:   distance in cm, time in seconds
 * 
 */
public class EPAPVIScreen extends MonteCarloModel implements Bisectable
{
	private String sName;
	private String sShortFileName;
	private int iIteration;
	protected Oxygen oxygen;
	protected VadoseZone vz;
	protected Building bld;
	protected Chemical c;
	//protected Vector vzChemicals;
	protected ArrayList<Chemical> alChemical;
	private boolean boolSolutionFound, boolSolutionFoundFlux, boolSolutionFoundConcentration;
	protected double aerobicZoneThickness;
	protected double anaerobicZoneThickness;
	protected double aerobicZoneThicknessConcentration,aerobicZoneThicknessFlux;
	protected double dTotalThickness;
	protected double dTransitionPointDepth;
	//protected double totalDepth;
	protected double soilRespiration;
	protected double efdo2;
	protected double jeo2;
	protected double ceo2;
	protected double eq29LHS;
	protected double eq37;
	private FileLogger fl;
	private FileLogger fl2;
	private String sOutput;
	private String sOutputHeading;
	private String sSolutionType;
	private Bisection bisection;
	//flag for model run:  true = Concentration equation, false = flux equation
	private boolean bConcentration;
	//unit converter (to unpack from bvcd)
	private UnitConverter uc;
	//OilLens
	private OilLens old;
	//statistical distributions
	private ArrayList<StatisticalDistribution> sDistribution;
	//BioVaporControlData
	private PVIScreenControlData pvisCD;
	//max exponential for calculation
	private double dMaxAlpha;
	//linked parameters
	private LinkedParameters lp;



	public EPAPVIScreen(){}

	public void initial()
	{
		//name and shortfilename string for output file	 
		sName = "";
		sShortFileName = "";
		//output string - start with a blank
		sOutput = ""; 
		boolSolutionFoundFlux = false;
		boolSolutionFoundConcentration = false;
		//arraylist for all the chemicals (they appear in the vadose zone)
		alChemical = new ArrayList<Chemical>();
		//array lists for the statisticalDistributions
		sDistribution = new ArrayList<StatisticalDistribution>();
		//iteration counter
		iIteration = 0;
		//create bisection object
		bisection = new Bisection();
		//default bisection parameters
	}

	//method for Bisectable Interface
	public double getFunctionValue (double aerobicZoneDepth)
	{
		double returnValue;
		double aZDConc;
		double aZDFlux;
		//double aZDFluxConc;
		returnValue = 0.;

		if (bConcentration)
		{
			//concentration only specified
			return determineAerobicZoneThicknessFromConc(aerobicZoneDepth);
		}
		else
		{
			//flux only specified
			return determineAerobicZoneThicknessFromFlux(aerobicZoneDepth);
		}

	}

	//methods required for the MonteCarloModel 

	/**
	 * setMonteCarloParameters  == generate sets of parameters for later use in running simulations
	 * parameters are set here in arrays, then run through the model
	 */
	public void setMonteCarloParameters(int iInterval)
	{
		//set up the intervals for Monte Carlo Latin Hypercube Sampling
		//for each statistical distribution
		for (StatisticalDistribution sd: sDistribution)
		{
			//if (sd.getIsThisDistributionActuallyUsed())
			{	
				sd.setNumberOfSimulations(iInterval);
				sd.setUpIntervals(iInterval);
				sd.generateOrdering(iInterval);
			}
		}

		//remove any data from the output string
		blankTheOutputString();
	}


	/**
	 * setVariableParameter -- set each MonteCarlo variable value
	 * called from each instance of the Monte Carlo LHS processor
	 * interval = index of interval:
	 *    these will return values at frequencies from 0 to 1.0
	 *    that were generated by setMonteCarloParameters
	 */
	public void setVariableParametersForAMonteCarloRun(int interval)
	{
		//set parameter values for every variable parameter and put in output string
		for (StatisticalDistribution sd: sDistribution)
		{
			sd.setValue(interval);
		}

		//modify certain required variables
		modifyVariables();

		//put variables  in output string
		for (StatisticalDistribution sd: sDistribution)
		{
			if(sd.getIsThisDistributionActuallyUsed()){addToOutputString(", " + sd.getCurrentRandomValue());}
		}

	}

	//modify certain variables for consistency within the model
	private void modifyVariables()
	{
		double dAER = 0;
		double dWidth = 0;
		double dLength = 0;
		double dMixingZone = 0;
		double dQSoil = 0;
		double dBuildingFlow;

		//link the building air flow to the parameter Qsoil
		//gather the needed inputs
		for (StatisticalDistribution sd: sDistribution)
		{
			if (sd.getName().equalsIgnoreCase("AirExchangeRate")){dAER = sd.getCurrentRandomValue();}
			if (sd.getName().equalsIgnoreCase("CeilingHeight")){dMixingZone = sd.getCurrentRandomValue();}
			if (sd.getName().equalsIgnoreCase("Length")){dLength = sd.getCurrentRandomValue();}
			if (sd.getName().equalsIgnoreCase("Width")){dWidth = sd.getCurrentRandomValue();}
			if (sd.getName().equalsIgnoreCase("Qsoil")){dQSoil = sd.getCurrentRandomValue();}
		}

		//test and reset the value
		dBuildingFlow = dMixingZone*dLength*dWidth*dAER;
		//the Qsoil cannot be higher than the building flow:
		if (dBuildingFlow<dQSoil)
		{
			dQSoil = dBuildingFlow;

			//save the modified value
			for (StatisticalDistribution sd: sDistribution)
			{
				if (sd.getName().equalsIgnoreCase("Qsoil")){sd.overrideStatisticalValue(dQSoil);}
			}   	
		}



		//adjust ground water concentrations


		String sIndependent, sDependent;
		StatisticalDistribution sdIndependent;
		//adjust linked parameters
		int iNumber = lp.getLinkedParameterCount();

		for (int i=0;i<iNumber;i++)
		{
			sIndependent = lp.getIndependentParameter(i);
			sDependent = lp.getDependentParameter(i);
			for (StatisticalDistribution sd1: sDistribution)
			{
				//find the statistical distribution for the independent value
				if (sd1.getName().equalsIgnoreCase(sIndependent))
				{
					//find the statistical distribution for the dependent value
					for (StatisticalDistribution sd:  sDistribution)
					{
						if (sd.getName().equalsIgnoreCase(sDependent))
						{
							//match the value of the dependent variable to that of the independent variable
							sd.overrideStatisticalValue(sd1.getCurrentRandomValue());
							break;
						}
					}
				}
			}	
		} 	
	}

	//parameters for the averaged-parameter run of the model
	public void setAveragedParametersForAMonteCarloRun()
	{
		//set parameter values for every variable parameter and put in output string
		for (StatisticalDistribution sd: sDistribution)
		{
			//sd.setAveragedParameterValue();
			sd.setAveragedParameterValue();
		}

		//modify certain required variables
		modifyVariables();

		//put variables  in output string
		for (StatisticalDistribution sd: sDistribution)
		{
			if (sd.getIsThisDistributionActuallyUsed())
			{	
				addToOutputString(", " + sd.getCurrentRandomValue());
			}
		}    	
	}


	//parameters for the median-parameter run of the model   //JWW  3-25-2020
	//provides a bench mark that is closer to the results of the monte carlo runs
	public void setMedianParametersForAMonteCarloRun()
	{
		//set parameter values for every variable parameter and put in output string
		for (StatisticalDistribution sd: sDistribution)
		{
			sd.setMedianParameterValue();
		}

		//modify certain required variables
		modifyVariables();

		//put variables  in output string
		for (StatisticalDistribution sd: sDistribution)
		{
			if (sd.getIsThisDistributionActuallyUsed())
			{	
				addToOutputString(", " + sd.getCurrentRandomValue());
			}
		}    	
	}


	public void blankTheOutputString(){sOutput = "";sOutputHeading="";}
	public void reNewConcentrationDistributions(){}


	int iRun=0;
	/**
	 * runModel -- execute the BioVaporAPI model either from a deterministic call or
	 * when called by the MonteCarlo Latin Hypercube processor
	 */
	public void runModel()
	{
		double oldOilZone = 0;
		double dLensThickness = 0;

		//set the correct values of the parameters
		//(will change from Monte Carlo run to Monte Carlo run)
		this.setParametersForEachRun();

		//total thickness for calculation of aerobic and anaerobic zone thicknesses

		double vzDepth = vz.getDepthToSample();   //jww  4-12-2016, previously sample was assumed to be at the water table
		double  bldFound = bld.getFoundationDepthBelowGrade();

		
		if (old.bLensUsed)
		{
			oldOilZone = old.getOilZoneThickness();
			dLensThickness = old.getOilZoneThickness();
			
		}
		else
		{
			dLensThickness = 0;  //jww  3-31-2020
		}

		

		dTotalThickness = (vz.getDepthToSample() - dLensThickness) - bld.getFoundationDepthBelowGrade(); //jww  4-15-2016, previously sample was assumed to be at the water table; 3-20-2017 JWW oil lens thickness made separate variable

		if(dTotalThickness<0.0) {System.out.println("****Sample or Oil in building****");}




		//the initial aerobic zone thickness is the total possible
		aerobicZoneThickness = dTotalThickness;

		//use bisection to solve BioVapor Model
		//solve for the aerobic zone depth using both the flux and the concentration equations

		//select model to run (this flag will determine if the concentration or the flux model is executed)

		//run the concentration equation model
		bConcentration = true;

		//set up bisection calculation
		bisection = new Bisection();
		bisection.setLimits(pvisCD.getBisectionLowerLimit(),dTotalThickness);
		bisection.setMaximumIterations(pvisCD.getBisectionMaxIterations());
		bisection.setTolerance(pvisCD.getBisectionTolerance());	  
		//value to match is 0.0, because the net oxygen concentration is moved to the right hand side of equation 4.9 inside the bisection routine
		bisection.setValueToMatch(0.0);   

		//solve concentration-basis model
		bisection.bisect(this);

		boolSolutionFoundConcentration = bisection.getSolutionFound();
		System.out.println("Concentration Solution " + this.getBisectionResult() + " in " + this.getNumberOfBisectionIterations() + " iterations");

		if (boolSolutionFoundConcentration)
		{
			//get aerobic zone depth as the result and the iteration count
			aerobicZoneThicknessConcentration = bisection.getResult();
		}
		else
		{
			//if no solution found for aerobic zone depth, then the whole vadose zone is aerobic
			aerobicZoneThicknessConcentration = dTotalThickness;
		}



		//set the correct values of the parameters
		//(will change from Monte Carlo run to Monte Carlo run)
		this.setParametersForEachRun(); 
		//run the flux model
		bConcentration = false;
		//set up bisection calculation
		bisection = new Bisection();
		bisection.setLimits(pvisCD.getBisectionLowerLimit(),dTotalThickness);
		bisection.setMaximumIterations(pvisCD.getBisectionMaxIterations());
		bisection.setTolerance(pvisCD.getBisectionTolerance());	  
		//value to match is 0.0, because the net oxygen flux is moved to the right hand side of equation 4.6 inside the bisection routine
		bisection.setValueToMatch(0.0);  

		//solve flux-basis model
		bisection.bisect(this);
		boolSolutionFoundFlux = bisection.getSolutionFound();
		System.out.println("Flux          Solution " + this.getBisectionResult() + " in " + this.getNumberOfBisectionIterations() + " iterations");



		if (boolSolutionFoundFlux)
		{
			//get aerobic zone depth as the result and the iteration count
			aerobicZoneThicknessFlux = bisection.getResult();
		}
		else
		{
			//if no solution found for aerobic zone depth, then the whole vadose zone is aerobic
			aerobicZoneThicknessFlux = dTotalThickness;
		}


		//for the constructed floor:
		if (bld.getIsItADirtFloor()==false)
		{ 	 

			//constructed floor    	 
			if ( (boolSolutionFoundFlux && boolSolutionFoundConcentration) ||
					(boolSolutionFoundFlux || boolSolutionFoundConcentration) )
			{
				boolSolutionFound = true;
				//take the smallest aerobic zone depth as the solution
				if (aerobicZoneThicknessConcentration <= aerobicZoneThicknessFlux)
				{
					sSolutionType = "Concentration Solution";
					aerobicZoneThickness = aerobicZoneThicknessConcentration;
				}
				else
				{
					sSolutionType = "Flux Solution";
					aerobicZoneThickness = aerobicZoneThicknessFlux;		   
				}
			}
			else if (boolSolutionFoundConcentration)
			{	   
				//dirt floor
				sSolutionType = "ConcentrationSolution";
				aerobicZoneThickness = aerobicZoneThicknessConcentration;

			}
			else
			{
				//solution not found
				aerobicZoneThickness = dTotalThickness;
			}


		}

		//aerobic/anaerobic transition point depth
		//add the aerobic zone thickness to the depth of the foundation bottom
		this.dTransitionPointDepth = bld.getFoundationDepthBelowGrade() + aerobicZoneThickness;

		//aerobicZoneDepth = aerobicZoneDepthConcentration;
		boolSolutionFound = boolSolutionFoundConcentration;
		iIteration ++;
	}


	/**
	 *
	 * solve equation B.4.9 of API, 2010 as described on page B-6 of API Guide and
	 * page 50 to 55 of JWW Ada-01, equation summary on page 55
	 *
	 *
	 */
	public double determineAerobicZoneThicknessFromConc(double aerobicZoneDepth)
	{
		double eqB49Value;
		double[] sum;
		sum = new double[4];

		this.aerobicZoneThickness = aerobicZoneDepth;
		this.anaerobicZoneThickness = this.dTotalThickness - this.aerobicZoneThickness;

		//soil respiration term constant
		soilRespiration = - vz.getBulkDensity()*vz.getSoilRespirationRate()*vz.getFractionOrganicCarbon();

		sum = calculateSums();

		//surface oxygen concentration
		eqB49Value = oxygen.getNetSurfaceConcentration();

		//demand from hydrocarbons
		eqB49Value = eqB49Value - sum[0]/efdo2;

		//soil respiration demand
		eqB49Value = eqB49Value + 0.5*(soilRespiration/efdo2)*this.aerobicZoneThickness*this.aerobicZoneThickness;

		//contribution of contaminant flux
		eqB49Value = eqB49Value - sum[1]*this.aerobicZoneThickness/efdo2;

		Stripper s = new Stripper();
		for (Chemical chemical: alChemical)
		{
			if (s.areTheyEqual(chemical.getName(),"Benzene"))
			{
				chemical.writeParameters();
			}	

		}


		//when the correct aerobic zone depth has been found eqB49Value is equal to zero
		return eqB49Value;
	}




	/**
	 *
	 * solve equation B.4.6 of API, 2010 as described on page B-6 of API Guide and
	 * page 50 to 64 of JWW Ada-01, equation summary on page 64
	 *
	 *
	 */
	public double determineAerobicZoneThicknessFromFlux(double aerobicZoneDepth)
	{
		double eqB46Value;
		double[] sum;
		sum = new double[4];

		this.aerobicZoneThickness = aerobicZoneDepth;
		this.anaerobicZoneThickness = this.dTotalThickness - this.aerobicZoneThickness;

		//soil respiration term constant
		double bulkDensity = vz.getBulkDensity();
		double sRR = vz.getSoilRespirationRate();
		double foc = vz.getFractionOrganicCarbon();
		soilRespiration =  - vz.getBulkDensity()*vz.getSoilRespirationRate()*vz.getFractionOrganicCarbon();

		sum = calculateSums();

		//surface oxygen concentration
		double surfaceFlux = 0.0;
		double dSurfaceOxygenConc = oxygen.getSurfaceConcentration();
		double dMinBioConc = oxygen.getMinBiodegradationConcentration();
		surfaceFlux = - ((bld.getQs() + bld.getAirFlowBelowBuilding())/bld.getAb())  *  (oxygen.getSurfaceConcentration() - oxygen.getMinBiodegradationConcentration());


		eqB46Value = surfaceFlux;

		//demand from hydrocarbons
		eqB46Value = eqB46Value - sum[2];

		//soil respiration demand
		eqB46Value = eqB46Value - soilRespiration*this.aerobicZoneThickness;


		Stripper s = new Stripper();
		for (Chemical chemical: alChemical)
		{
			if (s.areTheyEqual(chemical.getName(),"Benzene"))
			{
				chemical.writeParameters();
			}	

		}

		//when the correct aerobic zone depth has been found eqB49Value is equal to zero
		return eqB46Value;
	}


	private double[] calculateSums()
	{
		double[] sum;
		double A,B;
		double csi,cei,cti,cfi,jti,jfi;
		double edc;
		double h;
		double k2,k2a,k3,k4;
		double lr;
		double phi;
		double tfe;

		double alpha;
		double denom;
		double cSum;

		sum = new double[4];

		double dBioRateBenzene = 0;


		//sum needed for equation B.4.9:  (deff,i/phi,i)*(cfi - cti)
		sum[0] = 0.;
		//sum needed for equation B.4.9:  jti/phi,i
		sum[1] = 0.;
		//sum needed for equation B.4.6:  (jfi - jti)/phi,i
		sum[2] = 0.;
		//sum needed for simultaneous imposition of boundary conditions:  For the alternate
		//calculation of B.4.9:   jfi/phi,i  (to be used to determine jti/phi,i)
		sum[3] = 0.;

		k2a = bld.getMixingZoneHeight()*bld.getAirExchangeRate();




		for (Chemical chemical: alChemical)
		{
			if (chemical.getName().equalsIgnoreCase("benzene"))
			{
				dBioRateBenzene = chemical.getKw();
				//System.out.println(chemical.getName() + ", " + dBioRateBenzene);
			}

			//only calculate if the source concentration is greater than zero  
			if (chemical.getSourceConcentration()>0.0)
			{	 

				chemical.setAlpha(aerobicZoneThickness);
				alpha = chemical.getAlpha();




				//get the diffusion coefficient
				edc = chemical.getEffectiveDiffusionCoefficient();
				phi = chemical.getPhi();

				//set the diffusion coefficient for the building foundation crack
				bld.setDCrack(edc);
				bld.setMassTransferCoefficient();

				//building foundation mass transfer coefficient
				h = bld.getFoundationMassTransferCoefficient();
				k2 = 1./k2a + 1./h;
				k3 = aerobicZoneThickness/edc;

				if (alpha < dMaxAlpha)
				{

					//equations b.3.5 and b.3.6 of API Biovapor Guide
					A = 0.5*(Math.exp(-alpha) + Math.exp(alpha));
					B = 0.5*(Math.exp(alpha) - Math.exp(-alpha))/alpha;

					//determine the aerobic/anaerobic zone concentration from equation B.3.4
					csi = chemical.getSourceConcentration();
					cti = csi*(A*k2 + B*k3);

					denom = (A + (anaerobicZoneThickness/aerobicZoneThickness)*((A*A -1)/B))*k2;
					denom = denom + (B*aerobicZoneThickness + A*anaerobicZoneThickness)/edc;
					cti = cti/denom;

					//determine the [below the] foundation concentration from equation B.3.3
					cfi = cti* (k2 / (A*k2 + B*k3));

					//term for the flux equation
					tfe = (A-1)/B;
				}
				else
				{
					//alternatives to equations b.3.3 and b.3.4 for high alpha
					//equation b.3.4a
					csi = chemical.getSourceConcentration();
					lr = chemical.getDiffusiveReactionLength();
					k4 = lr/edc;
					cti = csi*(k2 + k4);
					denom = (1. + anaerobicZoneThickness/lr)*k2 + (lr + anaerobicZoneThickness)/edc;
					cti = cti/denom;
					//equation b.3.3a
					cfi = cti*(k2/(k2+k4))*2.0*Math.exp(-alpha);

					//term for the flux equation
					tfe = alpha;
				}

				chemical.setTransitionPointConcentration(cti);
				chemical.setFoundationConcentration(cfi);
				//determine the indoor air concentration equation from equation b.2.1
				cei = cfi*(1.0/k2a)/k2;
				chemical.setIndoorAirConcentration(cei);

				//determine the [below the] foundation contaminant flux from equation b.3.9
				jfi = h*(cfi - cei);
				chemical.setFoundationFlux(jfi);

				//determine the contaminant flux at the aerobic/anaerobic zone transition point (equation B.3.8)
				//tfe is the "term for the flux equation"
				//tfe = (A-1)/B for low alpha (equation B.3.8)
				//tfe = alpha for high alpha (see Weaver labnotebook Ada-01, page 58-59
				jti = jfi + (1./k3)*tfe*(cti + cfi);
				chemical.setTransitionPointFlux(jti);

				//accumulate sums needed for equation b.4.9
				//from equation b.4.9
				//(note:  the divide by the oxygen diffusion coefficient is done at the end--in a different function)
				sum[0] = sum[0] + (edc/phi)*(cfi-cti);

				//2nd sum
				sum[1] = sum[1] + jti/phi;

				//sum needed for equation b.4.6
				double dTest = -(edc/aerobicZoneThickness)*tfe*(cti + cfi);
				double dTest2 = dTest - (jfi - jti);
				double dDeltaFlux = - tfe * (edc/aerobicZoneThickness) * (cti + cfi);

				//sum[2] = sum[2] + (jfi -  jti)/phi;
				sum[2] = sum[2] + dDeltaFlux/phi;

				//sum needed for alternate calculation of B.4.9
				sum[3] = sum[3] + jfi/phi;

				//a debugging check that should always equal zero
				cSum = sum[2] - (sum[3] - sum[1]);
			}
		}



		String sOut = "";
		sOut = sOut + "AZT, "+ aerobicZoneThickness + ", ";

		for (int i=0;i<sum.length;i++)
		{
			sOut = sOut + i + ", " + sum[i] + ", ";

			//System.out.println(" aerobic zone thickness, Benzene bio rate, count, sum, " + aerobicZoneThickness + ", " + dBioRateBenzene + ", " + i + ", " + sum[i]);
		}
		System.out.println(sOut);

		return sum;

	}

	//write results
	public void writeResult(){this.writeResultBase("Result");}
	public void writeAveragedParameterResult(){this.writeResultBase("AveragedParameterResult");}
	public void writeMedianParameterResult(){this.writeResultBase("MedianParameterResult");}

	/**
	 * writeResult  write summary of each monte carlo run to the output file
	 * @param fl
	 */
	private void writeResultBase(String sFirstColumnText)
	{
		//Inputs were previously recorded in the sOutput string

		//gather remaining outputs in the sOutput string compiled by addToOutputString()
		String sTemp = "";
		double dUCF = 1.0;


		dUCF = uc.getOutputUCFFromType("length");	
		sTemp = sTemp + ",** , " + this.aerobicZoneThicknessConcentration*dUCF + ", " + this.aerobicZoneThicknessFlux*dUCF + ", " + this.aerobicZoneThickness*dUCF;
		addToOutputString(sTemp);

		addToOutputString(", " + sSolutionType);
		addToOutputString(", " + this.dTransitionPointDepth*dUCF);
		addToOutputString(", **");



		dUCF = uc.getOutputUCFFromType("concentrationAir");
		sTemp = "";
		for (Chemical c: alChemical)
		{
			if(c.getIsChemicalUsed())
			{	 
				sTemp = "";
				sTemp = ", " + c.getIndoorAirConcentration()*dUCF 
						+ ", " + c.getFoundationConcentration()*dUCF 
						+ ", " + c.getTransitionPointConcentration()*dUCF 
						+ ", "  + c.getSourceConcentration()*dUCF;
				addToOutputString(sTemp);
			}
		}

		//now write result
		fl.logMessage(sFirstColumnText + "," + iIteration + sOutput);
	}


	/**
	 * writeOutputHeadings   write headings for the output file
	 */
	public void writeOutputHeadings()
	{
		iIteration=0;
		writeOutputHeadingsRisk();
		writeOutputHeadingsBase("Results");
	}

	//headings for the averaged-parameter result
	public void writeAveragedParameterOutputHeadings()
	{
		iIteration=0;
		writeOutputHeadingsBase("AveragedParameterResult");
	}
	
	//headings for the averaged-parameter result
	public void writeMedianParameterOutputHeadings()
	{
		iIteration=0;
		writeOutputHeadingsBase("MedianParameterResult");
	}

	public void writeOutputHeadingsRisk()
	{

		//first write out several lines for the risk level outputs 
		//these were found to be conviently placed just ahead of the montecarlo results
		//then write headings for the monte carlo results

		//risk levels (appears just ahead of the montecarlo results) 
		//separated risk output from MonteCarlo output (JWW 1-25-2016)
		sOutputHeading = "";
		addToOutputHeading("Risk Level,Heading");
		//input statistical distribution values
		for (StatisticalDistribution sd: sDistribution)
		{
			if (sd.getIsThisDistributionActuallyUsed()){addToOutputHeading(",");}
		}

		//results
		addToOutputHeading(",,");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");

		for (Chemical c: alChemical)
		{
			if (c.getIsChemicalUsed())
			{	
				//write out four titles per chemical
				addToOutputHeading("," + "Risk Level"
						+ ","
						+ ","
						+ "," );
			}
		}	
		fl.logMessage(sOutputHeading); 

		//chemical name for risk level
		sOutputHeading = "";
		addToOutputHeading("Risk Level,Chemical");
		//input statistical distribution values
		for (StatisticalDistribution sd: sDistribution)
		{
			if(sd.getIsThisDistributionActuallyUsed()){addToOutputHeading(",");}
		}

		//results
		addToOutputHeading(",,");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");	 	


		for (Chemical c: alChemical)
		{
			if(c.getIsChemicalUsed())
			{	
				//write out four titles per chemical
				addToOutputHeading("," + c.getName()
				+ ","
				+ ","
				+ "," );
			}
		}	
		fl.logMessage(sOutputHeading);

		sOutputHeading = "";
		addToOutputHeading("Result,Value");
		//input statistical distribution values
		for (StatisticalDistribution sd: sDistribution)
		{
			if(sd.getIsThisDistributionActuallyUsed()){addToOutputHeading(", " + sd.getParentName() + " " +  sd.getName() );}
		}	 

		//cancer risk level 	
		sOutputHeading = "";
		addToOutputHeading("Risk Level,Value");
		//input statistical distribution values
		for (StatisticalDistribution sd: sDistribution)
		{
			if(sd.getIsThisDistributionActuallyUsed()){addToOutputHeading(",");}
		}

		//results
		addToOutputHeading(",,");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");	 	 	

		for (Chemical c: alChemical)
		{
			if(c.getIsChemicalUsed())
			{	
				//write out four titles per chemical
				addToOutputHeading("," + c.getRiskConcentration()*uc.getOutputUCFFromType("RiskConcentration")*c.getRiskConcentrationMultiplier()
						+ ","
						+ ","
						+ "," );
			}
		}	
		fl.logMessage(sOutputHeading);

		//cancer risk level (1x10-6, ie.)
		sOutputHeading = "";
		addToOutputHeading("Risk Level,Risk Level");
		//input statistical distribution values
		for (StatisticalDistribution sd: sDistribution)
		{
			if(sd.getIsThisDistributionActuallyUsed()){addToOutputHeading(",");}
		}

		//results
		addToOutputHeading(",,");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");	 	 	

		for (Chemical c: alChemical)
		{
			if (c.getIsChemicalUsed())
			{	
				//write out four titles per chemical
				addToOutputHeading("," + c.getRiskLevel()*c.getRiskConcentrationMultiplier()
						+ ","
						+ ","
						+ "," );
			}  
		}	
		fl.logMessage(sOutputHeading);	

		//reference concentration for non-cancer risk
		sOutputHeading = "";
		addToOutputHeading("Risk Level,Rfc");
		//input statistical distribution values
		for (StatisticalDistribution sd: sDistribution)
		{
			if(sd.getIsThisDistributionActuallyUsed()){addToOutputHeading(",");}
		}

		//results
		addToOutputHeading(",,");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");
		addToOutputHeading(",");	 	 	

		for (Chemical c: alChemical)
		{
			if(c.getIsChemicalUsed())
			{	
				//write out four titles per chemical
				addToOutputHeading("," + c.getReferenceConcentration()*uc.getOutputUCFFromType("RiskConcentration")
						+ ","
						+ ","
						+ "," );
			}
		}	
		fl.logMessage(sOutputHeading);	
	}

	public void writeOutputHeadingsBase(String sColumnOneHeading)
	{	
		//headings for the montecarlo outputs
		sOutputHeading = "";
		addToOutputHeading(sColumnOneHeading + "," + "Heading");
		//input statistical distribution values
		for (StatisticalDistribution sd: sDistribution)
		{
			if(sd.getIsThisDistributionActuallyUsed()){addToOutputHeading(", " + sd.getParentName() + " " +  sd.getName() );}
		}

		//results
		addToOutputHeading(",**,Aerobic Zone Thickness From Concentration");
		addToOutputHeading(",Aerobic Zone Thickness From Flux");
		addToOutputHeading(",Aerobic Zone Thickness");
		addToOutputHeading(",Result from: ");
		addToOutputHeading(",TransitionPointDepth");
		addToOutputHeading(",**");

		for (Chemical c: alChemical)
		{
			if(c.getIsChemicalUsed())
			{	
				//write out four titles per chemical
				addToOutputHeading("," + c.getName() + " Indoor Air Concentration" 
						+ "," + c.getName() + " Foundation Concentration"
						+ "," + c.getName() + " Transition Point Concentration" 
						+ "," + c.getName() + " Source Concentration");
			}
		}

		fl.logMessage(sOutputHeading);


		//write another line with just the units
		sOutputHeading = "";
		addToOutputHeading(sColumnOneHeading + "," + "Unit/Count");
		for (StatisticalDistribution sd: sDistribution)
		{ 
			if(sd.getIsThisDistributionActuallyUsed()){addToOutputHeading(", " +  " ("  + sd.getCalculationUnit() + ") ");}
		}

		String sLengthUnit = "";
		UnitConverter uc;
		uc = pvisCD.getUnitConverter();

		sLengthUnit = uc.getOutputNameFromType("length");	
		addToOutputHeading(",** ");
		addToOutputHeading(", (" + sLengthUnit + "), (" + sLengthUnit + " ), (" + sLengthUnit + " ), ** ,(" + sLengthUnit + " )" );
		addToOutputHeading(",** ");


		String sConcentrationUnit;
		sConcentrationUnit = uc.getOutputNameFromType("concentrationAir");

		for (Chemical c: alChemical)
		{
			//write out four units per chemical
			if(c.getIsChemicalUsed()){addToOutputHeading(", (" + sConcentrationUnit + "), (" + sConcentrationUnit + "), (" + sConcentrationUnit + "), (" + sConcentrationUnit + ")");}
		}

		fl.logMessage(sOutputHeading);
	}





	public void writeResult(FileLogger fl)
	{
		double A,B;
		double alpha, beta;
		double co2;
		double ce,cf,ct;
		double cz;
		double dz,z;
		double edc;
		double h;
		double jf,jt;
		double k2,k2a,k3;
		double tfe;
		double ucf;
		double zPrime,zTotal;
		double[] sumO2;


		OutputDigits od;

		String stringOut,stringOut2;


		//prepare for output
		od = new OutputDigits();
		sumO2 = new double[2];

		//calculate solution values of various paramters and equations
		setAnaerobicZoneDepth();

		//ucf = unit conversion factor to produce output in mg/M3

		//use unit conversion factor from the first chemical
		ucf = alChemical.get(0).getUCFmgM3();



		//concentration profile
		//z is measured from the transition point upward
		dz = -0.01*(aerobicZoneThickness);
		//starting at z=aerobicZoneDepth, starts at the top
		//and using a negative dz causes the calculation to work downward
		z = aerobicZoneThickness;

		//soil respiration term constant
		soilRespiration = vz.getBulkDensity()*vz.getSoilRespirationRate();

		//headings
		stringOut = "depth, oxygen, ";
		stringOut2 = "";

		for (Chemical chemical: alChemical)
		{	  
			if(c.getIsChemicalUsed()){stringOut2 = stringOut2 + chemical.getName() + ", ";}
		}
		//duplicate the headings for the nondimensional output
		stringOut2 = stringOut2 + " , , " + "oxygen, " + stringOut2;
		fl.logMessage(stringOut + stringOut2);



		//indoor air concentration
		stringOut = "";
		for (Chemical chemical: alChemical)	  
		{
			if(c.getIsChemicalUsed()){stringOut = stringOut + chemical.getIndoorAirConcentration()*ucf + ", ";}
		}
		//oxygen boundary condition
		stringOut = oxygen.getSurfaceConcentration()*ucf + ", " + stringOut;
		stringOut = bld.getThCrack() + ", " + stringOut;

		//normalized indoor air concentration
		stringOut = stringOut + " , , ";
		stringOut2 = "";

		for (Chemical chemical: alChemical)
		{	  
			if(c.getIsChemicalUsed()){stringOut2 = stringOut2 + chemical.getIndoorAirConcentration()/chemical.getSourceConcentration() + ", ";}
		}
		stringOut2 = oxygen.getSurfaceConcentration()/oxygen.getSurfaceConcentration() + ", " + stringOut2;
		fl.logMessage(stringOut + stringOut2);

		//foundation concentration
		stringOut = "";

		for (Chemical chemical: alChemical)
		{	  
			if(c.getIsChemicalUsed()){stringOut = stringOut + chemical.getFoundationConcentration()*ucf + ", ";}
		}
		//jww-7-8-2011 stringOut = oxygen.getSurfaceConcentration()*ucf + ", " + stringOut;
		//change made because the surface concentration is not the foundation oxygen concentration
		stringOut = " " + ", " + stringOut;
		stringOut = 0.0 + ", " + stringOut;

		//normalized foundation concentration
		stringOut = stringOut + " , , ";
		stringOut2 = "";

		for (Chemical chemical: alChemical)
		{	  
			if(c.getIsChemicalUsed()){stringOut2 = stringOut2 + chemical.getFoundationConcentration()/chemical.getSourceConcentration() + ", ";}
		}
		stringOut2 = oxygen.getSurfaceConcentration()/oxygen.getSurfaceConcentration() + ", " + stringOut2;
		fl.logMessage(stringOut + stringOut2);






		//zTotal runs from the surface down (zTotal = 0 at surface)
		zTotal = 0;

		k2a = bld.getMixingZoneHeight()*bld.getAirExchangeRate();

		//concentrations through the aerobic zone
		while (0<=z && z<=aerobicZoneThickness)
		{
			stringOut = "";
			//dimensional concentraion
			stringOut = stringOut + getConcentration(z,true);
			//nondimensional concentration
			stringOut = stringOut + " , , " + getConcentration(z,false);
			//dimensional flux
			stringOut = stringOut + " , , " + getFlux(z);

			od.setValue(-zTotal);
			stringOut = od.num2String() + ", " + stringOut;
			fl.logMessage(stringOut);

			z = z + dz;
			zTotal = zTotal - dz;
		}

		//below the aerobic zone
		zTotal = aerobicZoneThickness;

		//transition point concentration
		stringOut = "";
		for (Chemical chemical: alChemical)
		{	  
			if(c.getIsChemicalUsed())  
			{		
				cz = chemical.getTransitionPointConcentration();
				stringOut = stringOut + cz*ucf + ", ";
			}
		}
		co2 = 0.;
		stringOut = co2*ucf + ", " + stringOut;

		//normalized transition point concentration
		stringOut = stringOut + " , , ";
		stringOut2 = "";

		for (Chemical chemical: alChemical)
		{	  
			if(c.getIsChemicalUsed())
			{
				stringOut2 = stringOut2 + chemical.getTransitionPointConcentration()/chemical.getSourceConcentration() + ", ";
			}
		}
		stringOut2 = co2/oxygen.getSurfaceConcentration() + ", " + stringOut2;

		od.setValue(-zTotal);
		stringOut = od.num2String() + ", " + stringOut;
		fl.logMessage(stringOut + stringOut2);


		zTotal = aerobicZoneThickness + anaerobicZoneThickness;

		stringOut = "";
		for (Chemical chemical:  alChemical)
		{	
			if (c.getIsChemicalUsed())
			{	
				cz = chemical.getSourceConcentration();
				stringOut = stringOut + cz*ucf + ", ";
			}
		}
		co2 = 0.;
		stringOut = co2*ucf + ", " + stringOut;


		//normalized source concentration
		stringOut = stringOut + " , , ";
		stringOut2 = "";
		for (Chemical c: alChemical)
		{	  
			if(c.getIsChemicalUsed())
			{	
				stringOut2 = stringOut2 + c.getSourceConcentration()/c.getSourceConcentration() + ", ";
			}
		}
		stringOut2 = co2/oxygen.getSurfaceConcentration() + ", " + stringOut2;

		od.setValue(-zTotal);
		stringOut = od.num2String() + ", " + stringOut + stringOut2;

		fl.logMessage(stringOut);

	}

	private String getFlux(double z)
	{
		double A,B;
		double alpha, beta;
		double co2;
		double ce,cf,ct;
		double cz;
		double dz;
		double edc;
		double h;
		double jf,jt,jz;
		double k2,k2a,k3;
		double tfe;
		double ucf;
		double zPrime,zTotal;
		double[] sumO2;

		String stringOut;
		sumO2 = new double[2];

		stringOut="";
		//ucf = unit conversion factor to produce output in mg/M3
		//use first chemical to get unit conversion factor for all
		ucf = alChemical.get(0).getUCFmgM3();

		sumO2[0]=0.;
		sumO2[1]=0.;

		for (Chemical chemical:  alChemical)
		{ 	
			chemical.setAlpha(aerobicZoneThickness);

			alpha=chemical.getAlpha();
			beta=chemical.getFoundationConcentration()/chemical.getTransitionPointConcentration();

			//these quantities were set during calculation of the aerobicZoneDepth
			jt=chemical.getTransitionPointFlux();
			zPrime=z/aerobicZoneThickness;
			if(alpha<dMaxAlpha){
				jz=(Math.exp(-alpha)-beta)*Math.exp(alpha*zPrime);
				jz=jz+(beta-Math.exp(alpha))*Math.exp(-alpha*zPrime);
				jz=jt*jz/(Math.exp(-alpha)-Math.exp(alpha)-2.0*beta);
			}
			else{
				//alternate form of concentration equation Weaver lab notebook Ada-01, page 61
				//assumes that the form for the concentration can be used for the flux since exp(alpha) >> 2*beta
				jz=-Math.exp(alpha*(zPrime-2.0));
				jz=jz+beta*Math.exp(alpha*(zPrime-1.0));
				jz=jz-beta*Math.exp(-alpha*(zPrime+1.0));
				jz=jz+Math.exp(alpha*(-zPrime));
				jz=jt*jz;
			}


			//prepare output

			stringOut=stringOut + jz + ", ";


		}


		return stringOut;
	}

	/**
	 *   getConcentration   -- determine the concentration of each chemical for a given depth in the aerobic zone
	 *
	 * @param z double  depth for reqsult
	 * @param dimensional boolean   should the results be dimensional? yes if true
	 * @return String    string containing results for oxygen and each chemical
	 */
	private String getConcentration(double z, boolean dimensional)
	{
		double A,B;
		double alpha, beta;
		double co2;
		double ce,cf,ct;
		double cz;
		double dz;
		double edc;
		double h;
		double jf,jt;
		double k2,k2a,k3;
		double tfe;
		double ucf;
		double zPrime,zTotal;
		double[] sumO2;

		String stringOut;
		sumO2 = new double[2];

		stringOut="";
		//ucf = unit conversion factor to produce output in mg/M3

		//use unit conversion factor from the first chemical for all chemicals
		ucf = alChemical.get(0).getUCFmgM3();



		sumO2[0]=0.;
		sumO2[1]=0.;


		for (Chemical chemical: alChemical)
		{	   
			chemical.setAlpha(aerobicZoneThickness);

			alpha=chemical.getAlpha();
			beta=chemical.getFoundationConcentration()/chemical.getTransitionPointConcentration();
			edc=chemical.getEffectiveDiffusionCoefficient();

			//these quantities were set during calculation of the aerobicZoneDepth
			ct=chemical.getTransitionPointConcentration();
			zPrime=z/aerobicZoneThickness;
			if(alpha<dMaxAlpha){
				cz=(Math.exp(-alpha)-beta)*Math.exp(alpha*zPrime);
				cz=cz+(beta-Math.exp(alpha))*Math.exp(-alpha*zPrime);
				cz=ct*cz/(Math.exp(-alpha)-Math.exp(alpha));
			}
			else{
				//alternate form of concentration equation Weaver lab notebook Ada-01, page 61

				cz=-Math.exp(alpha*(zPrime-2.0));
				cz=cz+beta*Math.exp(alpha*(zPrime-1.0));
				cz=cz-beta*Math.exp(-alpha*(zPrime+1.0));
				cz=cz+Math.exp(alpha*(-zPrime));
				cz=ct*cz;
			}

			jt=chemical.getTransitionPointFlux();

			//accumulate sums
			//from equation b.4.9
			//(note:  the divide by the oxygen diffusion coefficient is done at the end)
			sumO2[0]=sumO2[0]+(edc/chemical.getPhi())*(cz-ct);

			//2nd sum
			sumO2[1]=sumO2[1]+jt/chemical.getPhi();

			//prepare output

			if (!dimensional)
			{
				stringOut = stringOut + cz/ct + ", ";
			}
			else
			{
				stringOut=stringOut + cz*ucf + ", ";
			}

		}

		//oxygen concentration from equation b.7.8
		co2=(sumO2[0]+z*sumO2[1])/efdo2-0.5*(soilRespiration/efdo2)*z*z;

		if (!dimensional)
			//nondimensional
		{co2=co2/oxygen.getSurfaceConcentration();}
		else
			//dimensional with unit conversion to mg/m3
		{co2 = co2*ucf;}

		stringOut = co2 + ", " +stringOut;


		return stringOut;
	}


	//adds
	public void addChemical(Chemical c){alChemical.add(c);}

	//sets
	public void setAnaerobicZoneDepth()
	{
		anaerobicZoneThickness = vz.getDepthToWaterTable() - bld.getFoundationDepthBelowGrade() - aerobicZoneThickness;
	}







	//initialize parameters which might change from run to run
	//i.e., during monte carlo simulations
	public void setParametersForEachRun()
	{
		//control parameters
		dMaxAlpha = pvisCD.getMaxAlpha();

		//vadose zone calculations
		vz.setBulkDensity();

		//building calculations
		double dAreaBasement;
		double dWidth = bld.getWidth();
		double dLength = bld.getLength();
		//area of the basement equals footprint area + the area of the basement walls below grade
		dAreaBasement = dLength*dWidth;
		dAreaBasement = dAreaBasement + bld.getFoundationDepthBelowGrade()*(2.0*dWidth + 2.0*dLength);
		bld.setAb(dAreaBasement);
		//crack is the length of the perimeter crack
		//the perimeter crack for each lenght:
		// length of crack--based on the footprint ofthe foundation, 
		//  but accounting for the width of the walls 
		// two times the (width minus two times the wall thickness) plus
		// two times the (length minus two times the wall thickness)
		double dWallThickness = bld.getThCrack();
		bld.setLCrack(2.0*(dWidth-2.0*dWallThickness + dLength-2.0*dWallThickness));
		bld.setMaxExponentialValue(pvisCD.getMaxAlpha());

		//oil volume calculations
		if (old.isLensUsed())
		{	  
			old.setElevations();
			old.findTopOfOilZone();
		}

		//old.generateOilProfile();
		//old.writeProfile(fl);

		//give vadose zone to the oxygen object
		oxygen.setVadoseZone(vz);
		//oxygen calculations
		oxygen.correctHenrysConstantForTemperature();  
		oxygen.setSurfaceFlux( - oxygen.getNetSurfaceConcentration()*((bld.getAirFlowBelowBuilding()+bld.getQs())/bld.getAb()) );
		oxygen.determineEffectiveDiffusionCoefficient();
		efdo2 = oxygen.getEffectiveDiffusionCoefficient();



		//chemical calculations
		for (Chemical c: alChemical)
		{
			//if test added 3-20-2017, jww
			if (c.bIsChemicalUsed)
			{	
				c.setVadoseZone(vz);
				c.correctHenrysConstantForTemperature();
				c.determineEffectiveDiffusionCoefficient();
				c.setDiffusiveReactionLength();
				//setting source concentration made last, 3-20-2017, jww
				c.setSourceConcentration(c.getSourceConcentration());
			}  
		}

	}

	/**
	 * loadData  take data BioVaporControlData that was read in BioVaporReader
	 *           embed in the BioVaporAPI object
	 * @param pvisCD
	 */
	public void loadData(PVIScreenControlData pvisCD)
	{
		//save the data
		this.pvisCD = pvisCD;

		//assign output file
		fl = pvisCD.getFileLogger();

		//assure that all data are available
		vz = pvisCD.getVadoseZone();
		//building
		bld = pvisCD.getBuilding();
		//oxygen
		oxygen = pvisCD.getOxygen();
		//oil lens
		old = pvisCD.getOilLens();
		//chemicals
		alChemical = pvisCD.getChemicals();
		//statistical distributions
		sDistribution = pvisCD.getStatisticalDistribution();
		//unit conversions
		uc = pvisCD.getUnitConverter();
		//linked parameter definitions
		lp = pvisCD.getLinkedParameters();
	}

	//helpers:
	private void addToOutputString(String sOutput){this.sOutput = this.sOutput + sOutput;}   
	private void addToOutputHeading(String sOutputHeading){this.sOutputHeading = this.sOutputHeading + sOutputHeading;}


	//simple sets
	public void setAerobicZoneDepth(double aerobicZoneDepth){this.aerobicZoneThickness = aerobicZoneDepth;}
	public void setMaxExponential(double alphaMax){this.dMaxAlpha = alphaMax;}
	public void setFileLoggerForInputs(FileLogger fl2){this.fl2 = fl2;}


	//simple gets
	public double getDepthToWater(){return vz.getDepthToWaterTable();}
	public boolean getIsMonteCarlo(){return pvisCD.getIsMonteCarlo();}
	public int getNumberOfRuns(){return pvisCD.getNumberOfRuns();}
	public int getNumberOfBisectionIterations(){return bisection.getCount();}
	public double getBisectionResult(){return bisection.getResult();}

	//get-like functions
	public boolean isSolutionFound(){return boolSolutionFound;}



}
