package pviScreen;

import java.util.ArrayList;

import uncertaintyAnalysis.StatisticalDistribution;
import util.FileLogger;
import util.OutputDigits;
import util.Stripper;
import util.UnitConverter;


//take raw input and place in data PVIScreen DataContainers
public class FillDataContainer 
{
	private Stripper s;
	private UnitConverter uc;
	private String sErrorOutput;

	private Building building;
	private VadoseZone vz;
	private AquiferData ad;
	private Oxygen oxygen;
	private OilLens oil;
	private ArrayList<StatisticalDistribution> sDistribution;
	private RawInput ri;
	private ChemicalDataReader cdr;

	private RunIdentifier runID;
	private PVIScreenControlData pvisCD;
	ArrayList<Chemical> alChemical;
	private LinkedParameters lp;

	private FileLogger fl;


	public FillDataContainer()
	{


		building = new Building();
		vz = new VadoseZone();
		ad = new AquiferData();
		oxygen = new Oxygen();
		oil = new OilLens();

		sDistribution = new ArrayList<StatisticalDistribution>();
		lp = new LinkedParameters();

		s = new Stripper();
	}


	public String addData(PVIScreenData pvisData, String key2, String[] lineArray)
	{
		String sOutput = "";
		String sOutputError = "";
		String key3 = lineArray[2];

		//add data to RawInput container


		if (key3.equalsIgnoreCase("Constant"))
		{
			sOutput = addDataItem(pvisData,key2,lineArray[3],lineArray[4],getUnitType(key2));
			if (sOutput.equalsIgnoreCase(" "))
			{
				sOutputError = ", Error in Input: " + arrayToString(lineArray);
				System.out.println(sOutputError);
			}

		}
		else if (key3.equalsIgnoreCase("Variable"))
		{
			sOutput = addDataItemVariable(pvisData,key2,lineArray[3],lineArray[4],getUnitType(key2),lineArray[5]);
			if (sOutput.equalsIgnoreCase(" "))
			{
				sOutputError = ", Error in Input: " + arrayToString(lineArray);
				System.out.println(sOutputError);
			} 
		}

		this.addToKnownErrorOutput(sOutputError);		 
		return sOutput + sOutputError;				
	}
	
	
	//add data for a chemical
	public String addChemicalData(String[] lineArray)
	{

		String sEol = System.getProperty("line.separator");
		sEol = "";
		String key3 = lineArray[2];
		String sOutput;

		sOutput = "";
		s = new Stripper();

		String[] sNameCondensed = new String[6];
		String[] sUnitName = new String[6];

		//five possible types of input data
		sNameCondensed[0] = "AirPhaseConcentration"; sUnitName[0]="ConcentrationAir";
		sNameCondensed[1] = "FuelPhaseConcentrationByMass"; sUnitName[1]="MassFraction";
		sNameCondensed[2] = "FuelPhaseConcentrationByVolume"; sUnitName[2]="VolumeFraction";
		sNameCondensed[3] = "WaterPhaseConcentration"; sUnitName[3]="ConcentrationWater";	
		sNameCondensed[4] = "SoilSample"; sUnitName[4]="SoilSample";  
		sNameCondensed[5] = "GroundWaterConcentration"; sUnitName[5] = "ConcentrationWater";  //JWW  9-9-2020


		//chemical name 
		Boolean bChemicalFound = false;

		//match the input chemical name to a chemical data set
		//while selecting the appropriate input data type
		for (Chemical chemical: alChemical)
		{
			if (s.areTheyEqual(chemical.getName(),lineArray[3],' '))
			{
				chemical.setIsChemicalUsed(true);

				//select proper type of input data
				for (int j=0;j<sNameCondensed.length;j++)
				{ 	  
					if (s.areTheyEqual(sNameCondensed[j], lineArray[1],' '))
					{	  
						chemical.setInputType(sNameCondensed[j]);  
						bChemicalFound = true;
						sOutput = sOutput + ", " + sNameCondensed[j] + "," + lineArray[4] + ", " + lineArray[5];
						if (s.areTheyEqual("constant", lineArray[2]))
						{	  
							//chemical found in chemical data
							sOutput = sOutput + addDataItem(chemical,sNameCondensed[j],lineArray[4],lineArray[5],sUnitName[j]) + sEol;
						}
						else if (s.areTheyEqual("variable", lineArray[2]))
						{  
							//chemical found in chemical data
							sOutput = sOutput + addDataItemVariable(chemical,sNameCondensed[j],lineArray[4],lineArray[5],sUnitName[j],lineArray[6]) + sEol;  
						} 
						break;
					}	
				}
			}
		}

		if (bChemicalFound!=true){sOutput = sOutput + "Data not found for input chemical" + lineArray[3];}  

		return sOutput;  	  
	}   



	//set individual bit of Variable data
	private String addDataItemVariable(PVIScreenData bvd, String key2m, String sValue, String sUnit, String sUnitType, String sFrequency)
	{
		String sReturn="";;
		OutputDigits od = new OutputDigits();
		//key2m = modified key2 (i.e., no spaces included)
		//sValue = string version of data item value
		//sUnit = string unit (i.e. cm, cm3/sec, 1/hr, ft, etc)
		//sUnitType = name for type of unit (i.e., length, mass, concentration)
		//  (No spaces allowed use "camel case" e.g., RateConstant)
		//sFrequency = string version of frequency value
		double value = Double.parseDouble(sValue)*uc.getUCF(sUnitType,sUnit);
		double frequency = Double.parseDouble(sFrequency);
		bvd.setCumulativeDistributionValue(key2m, value, frequency);
		bvd.setCurrentRandomValue(key2m,0.5);
		bvd.setCalculationUnit(key2m,uc.getModelUnit(sUnitType));
		od.setValue(value);
		sReturn = ", converted to: " + od.num2String() + " " + uc.getModelUnit(sUnitType);
		return sReturn;  
	}  

	//set individual bit of constant data
	private String addDataItem(PVIScreenData bvd, String key2m, String sValue, String sUnit, String sUnitType)
	{
		String sReturn = "";
		OutputDigits od = new OutputDigits();
		//key2m = modified key2 (i.e., no spaces included
		//sValue = string version of data item value
		//sUnit = string unit
		//sUnitTupe = name for type of unit (i.e., length, mass, concentration) 
		//  (No spaces allowed use "camel case" e.g., RateConstant)
		double value = Double.parseDouble(sValue)*uc.getUCF(sUnitType,sUnit);
		bvd.setStatisticalDistributionValue(key2m, value);
		bvd.setCurrentRandomValue(key2m,0.5);
		bvd.setCalculationUnit(key2m,uc.getModelUnit(sUnitType));
		//limit the number of output digits
		od.setValue(value);
		sReturn = ", converted to " + od.num2String() + " " + uc.getModelUnit(sUnitType);
		return sReturn;
	}   
	/**
	 * getUnitType         return the type of unit for the specified input parameter  (i.e., CrackWidth is a "Length")
	 * @param sInputName   name of input parameter ("key2" -- second entry on input line)
	 * @return
	 */
	private String getUnitType(String sInputName)
	{

		String sReturn;
		RawInputLine ril = new RawInputLine();

		//the unit conversions are in the RawInputLine, use this class for uniformity
		sReturn = ril.getUnitType(sInputName,uc);

		return sReturn;
	}

	private String arrayToString(String[] lineArray)
	{
		String sOut="";
		for (int i=0;i<lineArray.length;i++){sOut = sOut + lineArray[i] + ", ";}
		return sOut;
	}


	//add	
	public void addToKnownErrorOutput(String sError)
	{
		if (!sError.equalsIgnoreCase(""))
		{
			sErrorOutput = sErrorOutput + sError + "\n";
		}
	}



	//assemble all model input data
	public void assembleModelData()
	{

		//gather all the StatisticalDistributions for use in the Monte Carlo processor
		this.collectStatisticalDistributions();

		//some calculations are needed to prepare for running the  model
		//or for each monte carlo run
		//since the model will run in an inherent monte carlo mode 
		//(i.e., deterministic = 1 monte carlo run)
		//the initialization of parameters is done after selecting
		//each run's set of parameters by calling setParametersForEachRun()

		//add data to the oil lens object
		if (!pvisCD.getOilDistributionType().equalsIgnoreCase("none"))
		{	
			oil.setIsLensUsed(true);
			oil.setVadoseZone(vz);
			oil.findTopOfOilZone();
			oil.setFieldData(pvisCD.getFieldData());
			oil.setElevations();
			oil.generateOilProfile();
			oil.writeProfile(fl);
		}
		else
		{
			oil.setIsLensUsed(false);
		}

		//set statistical distributions for linked parameters
		this.setLinkedParameterDistributions();


		//revise chemical list
		ArrayList<Chemical> alChemicalRevised = new ArrayList<Chemical>();
		//process chemical inputs if necessary
		for (Chemical chemical:  alChemical)
		{
			chemical.setOilLensData(oil);
			chemical.setVadoseZone(vz);
			chemical.setAquifer(ad);
			chemical.setUnitConverter(uc);
			chemical.correctHenrysConstantForTemperature();
			//setting the following value to 1.0 removes archaic use of a risk concentration multiplier (jww 12-20-2013)
			chemical.setRiskConcentrationMultiplier(1.0);
			if (chemical.getIsChemicalUsed())
			{
				chemical.setAirPhaseConcentration();
				//only keep chemicals that are actually used in the simulation
				alChemicalRevised.add(chemical);
			}  
		}

		//load up the data sets for use in the PVIScreenControlData object
		pvisCD.setVadoseZone(vz);
		pvisCD.setBuilding(building);
		pvisCD.setOxygen(oxygen);
		pvisCD.setAquiferData(ad);	
		pvisCD.setOilLensData(oil);
		pvisCD.setChemicals(alChemicalRevised);
		pvisCD.setStatisticalDistribution(sDistribution);
		//pvisCD.setOutputUnitDefinition(uDefinition);	
		pvisCD.setUnitConverter(uc);
		pvisCD.setRawInput(ri);
		pvisCD.setRunIdentifier(runID);
		pvisCD.setLinkedParameters(lp);

		//check to see if all unit conversions are supported
		Boolean bUnsupportedUnit = uc.getBooleanUnsupportedUnit();

		//collect up the errors and determine if model can run
		pvisCD.addToKnownErrorOutput(this.getKnownErrorOutput());
		pvisCD.addToKnownErrorOutput(cdr.getKnownErrorOutput());
		pvisCD.setCanRunModel();	   
	}


	//assemble all the statistical distributions into one container (sDistribution)
	private void collectStatisticalDistributions()
	{   
		//gather all statistical distributions from the building, vadose zone, chemical and oxygen objects

		//the building object
		for (StatisticalDistribution sd: building.getStatisticalDistribution()){sDistribution.add(sd);}

		//the vadose zone object
		for (StatisticalDistribution sd: vz.getStatisticalDistribution()){sDistribution.add(sd);}

		//the oxygen object
		for (StatisticalDistribution sd: oxygen.getStatisticalDistribution()){sDistribution.add(sd);}

		//the oil lens object
		for (StatisticalDistribution sd: oil.getStatisticalDistribution()){sDistribution.add(sd);}

		//the chemical objects
		for (Chemical c: alChemical)
		{ 
			for (StatisticalDistribution sd: c.getStatisticalDistribution())
			{
				//only add statistical distributions for chemicals that are actually used
				if (c.getIsChemicalUsed()){sDistribution.add(sd);}
			}
		}
	}   


	//link parameters
	public void linkParameters(String sGroup, String[] lineArray) 
	{
		double dMultiplier = 1.0;
		//Building	AirFlowBelowBuilding	Linked	Qsoil	times	1
		try{dMultiplier = Double.parseDouble(lineArray[5]);}
		catch(Exception e){dMultiplier=1.0;}
		lp.addLinkedParameter(lineArray[1],lineArray[3],lineArray[4],dMultiplier);
	}

	//set statistical distribution for linked parameters
	private void setLinkedParameterDistributions()
	{
		int iNumber, iCount;
		String sIndependent, sDependent;
		double dMultiplier;

		iNumber = lp.getLinkedParameterCount();


		//run through all linked parameters
		for (int i=0;i<iNumber;i++)
		{
			sIndependent = lp.getIndependentParameter(i);
			sDependent = lp.getDependentParameter(i);
			dMultiplier = lp.getMultiplier(i);
			for (StatisticalDistribution sd1: sDistribution)
			{
				//find the statistical distribution for the independent value
				if (sd1.getName().equalsIgnoreCase(sIndependent))
				{
					//find the statistical distribution for the dependent value
					for (StatisticalDistribution sd:  sDistribution)
					{
						if (sd.getName().equalsIgnoreCase(sDependent))
						{
							//set the same statistical distribution in the dependent variable
							//as the independent variable
							iCount = sd1.getNumberOfPointsDefiningDistribution();
							sd.setNumberOfPointsDefiningDistribution(sd1.getNumberOfPointsDefiningDistribution());

							//apply multiplier
							double[] dValue = sd1.getConcentrationDistribution();
							for (int j=0;j<iCount;j++)
							{
								dValue[j] = dValue[j]*dMultiplier;
							}
							sd.setConcentrationDistribution(dValue);
							sd.setFrequencyDistribution(sd1.getFrequencyDistribution());
							sd.setCalculationUnit(sd1.getCalculationUnit());
							sd.setCurrentRandomValue(sd1.getCurrentRandomValue());
							sd.setChosenFrequency(sd1.getChosenFrequency());
							break;
						}
					}
					break;
				}
			}

		}		 


	}

	//sets
	public void setUnitConverter(UnitConverter uc){this.uc = uc;}
	public void setPVIScreenControlData(PVIScreenControlData pvisCD){this.pvisCD = pvisCD;}
	public void setRunIdentifier(RunIdentifier runID){this.runID = runID;}
	public void setChemicals(ArrayList<Chemical> alChemical){this.alChemical = alChemical;}
	public void setFileLogger(FileLogger fl){this.fl = fl;}
	public void setRawInput(RawInput ri){this.ri = ri;}
	public void setChemicalDataReader(ChemicalDataReader cdr){this.cdr = cdr;}

	//gets
	public Building getBuilding(){return this.building;}
	public VadoseZone getVadoseZone(){return this.vz;}
	public AquiferData getAquiferData(){return this.ad;}
	public Oxygen getOxygen(){return this.oxygen;}
	public OilLens getOilLens(){return this.oil;}
	public String getKnownErrorOutput(){return this.sErrorOutput;}
	public ArrayList<StatisticalDistribution> getStatisticalDistributions(){return this.sDistribution;}

}
