package pviScreen;

import java.util.ArrayList;

import util.FileLogger;

import numerics.Bisection.Bisectable;
import numerics.Bisection.Bisection;

public class OilLens extends PVIScreenData implements Bisectable
{
	//elevation of the top of the oil lens (determined by internal calculation)
	private double dFuelVolume;
	private double dGasolineDensity;
	private double dOilMolecularWeight;
	private double dResidualOilSaturation;
	private double dOilWaterInterfacialTension;
	private double dLensWidth;
	private double dLensLength;
    private double dXLensCenter;
    private double dYLensCenter;
    private double dSurfaceTension;
    
    protected double dLensArea;
    protected double dMaxOilSaturation;
    protected double dOilZoneThickness;
    protected double dFuelVolumePerUnitArea;
    
    //elevations
    protected double dElevTopOilZone;
    protected double dElevBottomMobileOilZone;
    protected double dElevWaterTable;
    protected double dElevHistoricWaterTable;
    protected double dElevSurface;
    
    //control parameters
    protected boolean bLensUsed;
    
    //
 
    
    
    //necessary data
    protected VadoseZone vz;
    protected FieldData fd;
  
    
	
	public OilLens()
	{
	   super();  
	   super.setName("oilLens");
	    
	   //create distributions
	   super.createNewDistribution("oilLens","FuelVolume");
       super.createNewDistribution("oilLens","MaxOilSaturation");
       super.createNewDistribution("oilLens","ResidualOilSaturation");
       super.createNewDistribution("oilLens","LensWidth");
       super.createNewDistribution("oilLens","LensLength");
       super.createNewDistribution("oilLens","OilWaterInterfacialTension");
       super.createNewDistribution("oilLens","OilMolecularWeight");
       super.createNewDistribution("oilLens","OilDensity");
       
       bLensUsed = false;
	}
	
	public void setElevations()
	{
		dElevWaterTable = vz.getDepthToBottom()-vz.getDepthToWaterTable();
		dElevHistoricWaterTable = vz.getDepthToBottom()-vz.getDepthToHistoricWaterTable();
		dElevSurface = vz.getDepthToBottom();
	}
	
	private double dZ1;
	private double dZWT;
	/**
	 * findTopOfOilZone  -- determine the top of the oil zone,
	 *    given:  oil volume, area, residual saturation, maximum oil saturation,
	 *            historical water table depth, water table depth
	 */
	public void findTopOfOilZone()
	{
		Bisection bisection = new Bisection();
	 
		//James W. Weaver, US EPA lab notebook Ada JWW-01, page 168-173
		 
		//set the elevations relative to the datum
		this.setElevations();
		
		dLensArea = this.getLensWidth() * this.getLensLength();
				
		//The air entry head 
		double dHce = vz.getAirEntryHead();
		//modify the air entry head
		dHce = dHce*this.getOilWaterInterfacialTension()/vz.getWaterSurfaceTension();
		//bottom of the oil zone elevation
		double dZ1 = dElevHistoricWaterTable;
		//maximum elevation
		double dZ2 = dElevSurface;
		
		
		double test1 = getFunctionValue(this.dElevWaterTable);
		
		bisection.setMaximumIterations(1000);
		bisection.setTolerance(1.e-4);
		bisection.setValueToMatch(0.0);
		bisection.setLimits(dZ1, dZ2);	
		bisection.bisect(this);
		
		double result = bisection.getResult();
		this.dElevTopOilZone = result; 
		
		//oil zone thickness
		this.dOilZoneThickness = this.dElevTopOilZone - this.dElevWaterTable;
		this.dElevBottomMobileOilZone = this.dElevTopOilZone - this.dOilZoneThickness;
		
		//fuel volume per unit area
		double dVol = this.getFuelVolume();
		double dArea= this.dLensArea;
		this.dFuelVolumePerUnitArea = this.getFuelVolume()/this.dLensArea;
		
		//set the maximum oil saturation
		dMaxOilSaturation = getOilSaturation(dElevTopOilZone);
	}
	 

	/**
	 * getFuncationValue  -- required function for Bisectable interface
	 *                       used for calculating the top of the oil zone
	 * @param independentVariable
	 * @return
	 */
	public double getFunctionValue(double dElevation) 
	{
       double dReleasedOil, dFormationOil; 
       double dPor = vz.getPorosity();
       double dSor = this.getResidualOilSaturation();
       double dHce = vz.getAirEntryHead();
       double dSwr = vz.getResidualWaterSaturation();
       double dSar = vz.getTrappedAirSaturation();
       double dLambda = vz.getLambda();

       //volume of oil in oil zone
       //US EPA lab notebook Ada JWW-01, page 190:  
       
   	   //modify the air entry head for the difference in interfacial tension
	   dHce = dHce*this.getOilWaterInterfacialTension()/vz.getWaterSurfaceTension();
			
       //balance the released volume of fuel with the amount in the oil zone
       
       //released volume of fuel
       dReleasedOil = this.getFuelVolume();
       
       //volume of oil in the formation:
       dFormationOil = 0.0;
       
       if (dElevation<dElevHistoricWaterTable){}
       else if (dElevHistoricWaterTable<=dElevation && dElevation<=dElevWaterTable)
       {
    	   dFormationOil = dPor*dSor*dLensArea*(dElevation-dElevHistoricWaterTable);
       }
       else if (dElevWaterTable<=dElevation && dElevation<=dElevWaterTable+dHce)
       {
    	   dFormationOil = dPor*dSor*dLensArea*((dElevWaterTable-dElevHistoricWaterTable) + (dElevWaterTable + dHce - dElevWaterTable));
       }
       else if (dElevWaterTable+dHce<=dElevation && dElevation<=dElevSurface)
       {
    	   dFormationOil = dSor*( (dElevation-dElevHistoricWaterTable) )
    			         + (1. - dSar - dSwr - dSor)*(dElevation-dElevWaterTable - dHce )
    			         - (1. - dSar - dSwr - dSor)*( Math.pow(dHce, dLambda)/(1.0-dLambda) )
    			                *( Math.pow(dElevation-dElevWaterTable,1.0-dLambda) - Math.pow(dHce, 1.0-dLambda) );
    	   dFormationOil = dFormationOil*dPor*dLensArea;
       }
       
       //when the top elevation is found the released oil equals the formation oil
	   return dReleasedOil - dFormationOil;
	}			
	
	ArrayList<ProfilePoint> alProfilePoint;
	public void generateOilProfile()
	{
		double dSo, dSw, dSa;
		
		alProfilePoint = new ArrayList<ProfilePoint>();
		//oil profile points
		ProfilePoint pp = new ProfilePoint();
	    double dElev = this.dElevHistoricWaterTable;
	    dSo = this.getOilSaturation(dElev);
	    dSw = this.getWaterSaturation(dElev);
	    dSa = this.getAirSaturation(dElev);
	    pp.setPoint(dElev, dSo, dSw, dSa);
	    alProfilePoint.add(pp);
	    
	    
	    dElev = this.dElevWaterTable;
	    pp = new ProfilePoint();
	    dSo = this.getOilSaturation(dElev);
	    dSw = this.getWaterSaturation(dElev);
	    dSa = this.getAirSaturation(dElev);
	    pp.setPoint(dElev, dSo, dSw, dSa);
	    alProfilePoint.add(pp);
	    
	    double dHce = vz.getAirEntryHead();
	    //modify the air entry head for the difference in interfacial tension
	    dHce = dHce*this.getOilWaterInterfacialTension()/vz.getWaterSurfaceTension();
	    dElev = this.dElevWaterTable + dHce;
	    dSo = this.getOilSaturation(dElev);
	    dSw = this.getWaterSaturation(dElev);
	    dSa = this.getAirSaturation(dElev);
	    pp.setPoint(dElev, dSo, dSw, dSa);
	    alProfilePoint.add(pp);
	    
	    double dZ = (this.dElevTopOilZone - dElev)/20.0;
	    double z = dElev;
	    while (z<this.dElevTopOilZone)
	    {
	    	pp = new ProfilePoint();
	    	dSo = this.getOilSaturation(z);
	 	    dSw = this.getWaterSaturation(z);
	 	    dSa = this.getAirSaturation(z);
	 	    pp.setPoint(z, dSo, dSw, dSa);
	 	    alProfilePoint.add(pp);
	 	    z = z + dZ;
	    }
	    
	    //generate some more points for plotting (above top of oil zone)
	    z = this.dElevTopOilZone;
	    dZ = (this.dElevSurface - dElev)/20.0;
	    while (z<this.dElevSurface)
	    {
	    	pp = new ProfilePoint();
	    	dSo = this.getOilSaturation(z);
	 	    dSw = this.getWaterSaturation(z);
	 	    dSa = this.getAirSaturation(z);
	 	    pp.setPoint(z, dSo, dSw, dSa);
	 	    alProfilePoint.add(pp);
	 	    z = z + dZ;
	    }
	    
	    
		
	}
	
	public void writeProfile(FileLogger fl)
	{
	  String sOut = "";
	  
	  //heading
	  sOut = "elevation, oil saturation, water saturation, air saturation";
	  
	  ///write out the profile
      for (ProfilePoint pp: alProfilePoint)
      {	  
    	sOut = "";
    	sOut = pp.getElevation() + "," + pp.getOilSaturation() + "," + pp.getWaterSaturation() + "," + pp.getAirSaturation();
	    fl.logMessage(sOut);	
      }
	  
	
	}
	
	public double getOilSaturation(double dElevation) 
	{
       double dSor = this.getResidualOilSaturation();
       double dHce = vz.getAirEntryHead();
       double dSwr = vz.getResidualWaterSaturation();
       double dSar = vz.getTrappedAirSaturation();
       double dLambda = vz.getLambda();
       double dSo,dSw;

       //volume of oil in oil zone
       //US EPA lab notebook Ada JWW-01, page 190:  
       
   	   //modify the air entry head for the difference in interfacial tension
	   dHce = dHce*this.getOilWaterInterfacialTension()/vz.getWaterSurfaceTension();
			
	   //starting possibility
       dSo = 0.0;
 
        
       if (dElevation<dElevHistoricWaterTable){}
       else if (dElevHistoricWaterTable<=dElevation && dElevation<=dElevWaterTable)
       {
    	   dSo = dSor;
       }
       else if (dElevWaterTable<=dElevation && dElevation<=dElevWaterTable+dHce)
       {
    	   dSo = dSor;
       }
       else if (dElevWaterTable+dHce<=dElevation && dElevation<=this.dElevTopOilZone)
       {
    	   dSw = (dSwr + (1.0-dSar-dSwr-dSor)*Math.pow((dHce/(dElevation-dElevWaterTable)),dLambda));
           dSo = 1.0 - dSar - dSw;
       }
       else if (this.dElevTopOilZone<dElevation)
       {
    	   dSo = 0.0;
       }
       
	   return dSo;
	}	
	
	public double getWaterSaturation(double dElevation) 
	{
       double dSor = this.getResidualOilSaturation();
       double dHce = vz.getAirEntryHead();
       double dSwr = vz.getResidualWaterSaturation();
       double dSar = vz.getTrappedAirSaturation();
       double dLambda = vz.getLambda();
       double dSw;

       //volume of oil in oil zone
       //US EPA lab notebook Ada JWW-01, page 190:  
       
   	   //modify the air entry head for the difference in interfacial tension
	   dHce = dHce*this.getOilWaterInterfacialTension()/vz.getWaterSurfaceTension();
			
	   //starting possibility
       dSw = 0.0;
 
        
       if (dElevation<dElevHistoricWaterTable) {dSw=1.0;}
       else if (dElevHistoricWaterTable<=dElevation && dElevation<=dElevWaterTable)
       {
    	   dSw = 1.0 - dSor;
       }
       else if (dElevWaterTable<=dElevation && dElevation<=dElevWaterTable+dHce)
       {
    	   dSw = 1.0 -  dSor;
       }
       else if (dElevWaterTable+dHce<=dElevation && dElevation<=this.dElevTopOilZone)
       {
           dSw =  (dSwr + (1.0-dSar-dSwr-dSor)*Math.pow((dHce/(dElevation-dElevWaterTable)),dLambda));
       }
       else if (this.dElevTopOilZone < dElevation)
       {	   
    	   dSw =  (dSwr + (1.0-dSar-dSwr)*Math.pow((dHce/(dElevation-dElevWaterTable)),dLambda));
       }
       
	   return dSw;
	}
	
	public double getAirSaturation(double dElevation)
	{
		double dSa;
		double dHce = vz.getAirEntryHead();
		double dSar = vz.getTrappedAirSaturation();
		double dSwr = vz.getResidualWaterSaturation();
		double dLambda = vz.getLambda();
	    double dSw;


	       //volume of oil in oil zone
	       //US EPA lab notebook Ada JWW-01, page 190:  
	       
	   	   //modify the air entry head for the difference in interfacial tension
		   dHce = dHce*this.getOilWaterInterfacialTension()/vz.getWaterSurfaceTension();
				
		   //starting possibility
	       dSa = 0.0;
	       
	       if (dElevation<=this.dElevWaterTable + dHce){dSa = 0.0;}
	       else if (this.dElevWaterTable + dHce < dElevation && dElevation <= this.dElevTopOilZone)
	       {
	    	   dSa = dSar;
	       }
	       else if (this.dElevTopOilZone < dElevation)
	       {	   
	    	   dSw =  (dSwr + (1.0-dSar-dSwr)*Math.pow((dHce/(dElevation-dElevWaterTable)),dLambda));
	    	   dSa = 1.-dSw;
	       }	
	       
	       return dSa;
		
	}
	
 
	
	//sets
	public void setVadoseZone(VadoseZone vz){this.vz = vz;}
	public void setFieldData(FieldData fd){this.fd = fd;}
	public void setGasolineDensity(double dGasolineDensity){this.dGasolineDensity = dGasolineDensity;}
	public void setFuelVolume(double dValue){this.dFuelVolume=dValue;this.setConstantDistribution(dValue, "FuelVolume");}
	public void setMaxOilSaturation(double dValue){this.dMaxOilSaturation=dValue;this.setConstantDistribution(dValue, "MaxOilSaturation");}
	public void setResidualOilSaturation(double dValue){this.dResidualOilSaturation=dValue;this.setConstantDistribution(dValue, "ResidualOilSaturation");}
	public void setLensWidth(double dValue){this.dLensWidth=dValue;this.setConstantDistribution(dValue, "LensWidth");}
	public void setLensLength(double dValue){this.dLensLength=dValue;this.setConstantDistribution(dValue, "LensLength");}
	public void setOilWaterInterfacialTension(double dValue){this.dOilWaterInterfacialTension=dValue;this.setConstantDistribution(dValue, "OilWaterInterfacialTension");}
	public void setOilMolecularWeight(double dValue){this.dOilMolecularWeight = dValue;this.setConstantDistribution(dValue, "OilMolecularWeight");}
	public void setOilDensity(double dValue){this.dGasolineDensity=dValue;this.setConstantDistribution(dValue,"OilDensity");}
	public void setIsLensUsed(boolean b){this.bLensUsed = b;}
	
	
	//gets
	public double getFuelVolume(){return getCurrentRandomValue("FuelVolume",this.getName());}	
	public double getMaxOilSaturation(){return getCurrentRandomValue("MaxOilSaturation",this.getName());}		
	public double getResidualOilSaturation(){return getCurrentRandomValue("ResidualOilSaturation",this.getName());}	
	public double getLensWidth(){return getCurrentRandomValue("LensWidth",this.getName());}	
	public double getLensLength(){return getCurrentRandomValue("LensLength",this.getName());}		
	public double getOilWaterInterfacialTension(){return getCurrentRandomValue("OilWaterInterfacialTension",this.getName());}
    public double getOilMolecularWeight(){return getCurrentRandomValue("OilMolecularWeight",this.getName());}
	public double getOilDensity(){return getCurrentRandomValue("OilDensity",this.getName());}
	public double getLensArea(){return dLensArea;}
    public double getSurfaceTension(){return dSurfaceTension;}
    public double getOilZoneTopElevation(){return this.dElevTopOilZone;}
    public double getMobileOilZoneBottomElevation(){return this.dElevBottomMobileOilZone;}
    public double getSurfaceElevation(){return this.dElevSurface;}
    public double getOilZoneThickness()
    {
    	if (this.bLensUsed){return this.dOilZoneThickness;}
    	else {return 0.0;}
    }
    public double getOilVolumePerUnitArea(){return this.dFuelVolumePerUnitArea;}
    public ArrayList<ProfilePoint> getProfilePoints(){return alProfilePoint;}
    public boolean isLensUsed(){return this.bLensUsed;}

	
   
}
