package pviScreen;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;

import numerics.Bisection.Bisection;
import uncertaintyAnalysis.StatisticalDistribution;
import util.CalendarClock;
import util.FileLogger;
import util.FileModifier;
import util.NameConstants;
import util.OutputDigits;
import util.Stripper;
import util.UnitConverter;
import util.UnitDefinition;

/**
 * BioVaporReader:   class for reading BioVaporAPI input
 * @author JWEAVER
 *  8-18-0212
 */
public class PVIScreenReader 
{
	private String sTitle;
	private String sShortFileName;
	private String sOutputFileName;
	private String sOutputFileTime;
	private String sStatisticsFileName;
	private String sErrorOutput;
	protected Oxygen oxygen;
	protected VadoseZone vz;
	protected Building bld;
	protected Chemical c;
	protected AquiferData ad;
	protected OilLens old;
	protected RunIdentifier runID;
	protected Risk risk;
	protected ArrayList<Chemical> alChemical;
	protected double aerobicZoneDepth;
	protected double anaerobicZoneDepth;
	protected double aerobicZoneDepthConcentration,aerobicZoneDepthFlux;
	protected double totalDepth;
	protected double soilRespiration;
	protected double efdo2;
	protected double jeo2;
	protected double ceo2;
	protected double eq29LHS;
	protected double eq37;
	protected double dMaxAlpha;
	private FileLogger fl;
	private boolean bWriteFile;


	private String sOutput;
	private Bisection bisection;
	private double dBisectionLowerLimit;
	private int iBisectionMaxIterations;
	private double dBisectionTolerance;
	//flag for model run:  true = Concentration equation, false = flux equation
	private boolean bConcentration;
	private boolean bMonteCarlo;
	private int iNumberOfRuns;
	//unit converter
	private UnitConverter uc;
	//chemical data reader
	private ChemicalDataReader cdr;
	//suggested value reader
	private SuggestedValueReader svr;
	//statistical distributions
	private FieldData fd;
	private ArrayList<StatisticalDistribution> sDistribution;
	//output units
	//private ArrayList<UnitDefinition> uDefinition;
	//BioVaporControlData container object
	private PVIScreenControlData pvisCD;
	private boolean bUnsupportedUnit;
	//raw input container
	private RawInput ri;
	//file modifier (to change extension)
	private FileModifier fm;
	//container for linked parameters
	private LinkedParameters lp;
	//stripper to remove unwanted characters (usually spaces)
	private Stripper s;
	//container for inputs and converted/processed inputs
	private FillDataContainer fdc;
	//container for certain parameter definitions choices 
	//(browsers, oil distribution, dirt floor, risk levels, hazard quotients)
	private ParameterChoices pc;
	//parameter choice reader
	private ParameterChoiceDataReader pcdr;
	//output unit reader
	private OutputUnitReader our;
	//constants
	private NameConstants nc;


	public PVIScreenReader(){}

	public void initial() throws IOException
	{
		//shortfilename string for output file	 
		sShortFileName = "";
		//output string - start with a blank
		sOutput = ""; 
		sOutputFileTime = "";
		//output string for errors
		sErrorOutput = "";

		//array lists for the statisticalDistributions
		sDistribution = new ArrayList<StatisticalDistribution>();
		//uDefinition = new ArrayList<UnitDefinition>();
		//all PVIScreen simulations are stochastic:
		bMonteCarlo = true;
		//iteration counter
		//iIteration = 1;
		//create bisection object
		bisection = new Bisection();
		//default bisection parameters
		dBisectionLowerLimit = 0.0001;
		iBisectionMaxIterations = 100;
		dBisectionTolerance = 0.e-6;


		//unit conversions
		uc = new UnitConverter();
		uc.readUnits("src/systemData/01PVIScreenUnitConversion.csv");
		uc.readUnitTypes("src/systemData/01PVIScreenUnitDefinitions.csv");

		//chemical data reader
		cdr = new ChemicalDataReader();
		cdr.setUnitConverter(uc);

		//suggested value reader
		svr = new SuggestedValueReader();

		//output unit reader
		our = new OutputUnitReader();


		old = new OilLens();
		//by default lens is not used
		old.setIsLensUsed(false);
		//aq = new AquiferData();

		//data container for control data
		pvisCD = new PVIScreenControlData();
		//unsupported unit flag
		bUnsupportedUnit = false;
		//actually write out echo print of inputs
		//true to write out for actual model runs
		//false to read for user interface
		bWriteFile = true;

		//data container for raw input data
		ri = new RawInput();

		//file modifier
		fm = new FileModifier();

		//run id container
		runID = new RunIdentifier();
		runID.setDate("");

		//linked parameter container
		lp = new LinkedParameters();

		//the stripper
		s = new Stripper();

		//the raw data input container
		fdc = new FillDataContainer();

		//list of input line types (strings)
		nc = new NameConstants();

		//list of parameter choices 
		//(browsers, oil distribution, dirt floor, risk levels, hazard quotients)
		pcdr = new ParameterChoiceDataReader();

		//risk container
		risk = new Risk();

	}


	/**
	 * readFile  	reads and processes the input data for BioVaporAPI
	 * 
	 * @param pathName   file name of BioVaporAPI input data in comma separated value (csv) format
	 * @throws IOException
	 */
	public void readFile(String pathName) throws IOException
	{
		FileReader fr = new FileReader(pathName);
		BufferedReader br = new BufferedReader(fr);

		String currentLine;
		String[] lineArray = {""};
		String sErrorOutput = "";
		String sConvertedOutput = "";

		//read the first line of the input file.
		currentLine = br.readLine();
		lineArray = currentLine.split(",",0);

		//read the chemical data
		cdr.readChemicalData(nc.file_ChemicalData);
		cdr.readHenrysConstantData(nc.file_HenrysConstantData);
		cdr.readChemicalRiskData(nc.file_ChemicalRiskData);	




		sOutputFileName = fm.getOutputFileAndPathName(pathName, nc.i_Extension_Input_Dots, nc.s_Extension_Results, true);

		fl = new FileLogger(sOutputFileName);
		//bWriteFile is true by default; must be set to be false for it to be false
		fl.setWriteFile(bWriteFile);
		//echo the file name into the output file
		fl.logMessage("Input File:," + pathName);
		uc.setFileLogger(fl);
		cdr.setFileLoggerOutput(fl);
		pvisCD.setFileLogger(fl);
		//all simulations are stochastic:
		pvisCD.setIsMonteCarlo(true);



		//get the data from the chemical data
		alChemical = cdr.getChemicalData();

		//read parameter choices
		pcdr.readParameterData(nc.file_ParameterChoices);
		pc = pcdr.getParameterDefinitions();



		fdc.setUnitConverter(uc);
		fdc.setPVIScreenControlData(pvisCD);
		fdc.setRunIdentifier(runID);
		fdc.setChemicals(alChemical);

        //read the output units file
		our.setUnitConverter(uc);
		our.readOutputUnits("src/systemData/07OutputUnits.csv");


		//read each line in the file
		while ((currentLine = br.readLine()) != null) 
		{
			String key, key2, key3;

			key = ""; key2 = ""; key3 = "";
			sErrorOutput = "";
			sConvertedOutput = "";

			lineArray = currentLine.split(",",0);
			key = lineArray[0];


			//the control parameters
			if (key.equalsIgnoreCase("Control") || key.equalsIgnoreCase("controll")) 
			{
				key2=lineArray[1];

				//site and analyst identification
				if (key2.equalsIgnoreCase("Analyst"))
				{
					runID.setAnalyst(lineArray[2]);
					this.saveRawInput(nc.s_CONTROL2STRING, lineArray);
				}
				else if (key2.equalsIgnoreCase("Affiliation"))
				{
					runID.setAffiliation(lineArray[2]); 
					this.saveRawInput(nc.s_CONTROL2STRING,lineArray);	 
				}
				else if (key2.equalsIgnoreCase("Site"))
				{
					runID.setSite(lineArray[2]);
					this.saveRawInput(nc.s_CONTROL2STRING,lineArray);
				}
				else if (key2.equalsIgnoreCase("SiteLocation"))
				{
					runID.setLocation(lineArray[2]);
					this.saveRawInput(nc.s_CONTROL2STRING,lineArray);
				}
				else if (key2.equalsIgnoreCase("City"))
				{
					runID.setCity(lineArray[2]);
					this.saveRawInput(nc.s_CONTROL2STRING,lineArray);
				}
				else if (key2.equalsIgnoreCase("State"))
				{
					runID.setState(lineArray[2]);
					this.saveRawInput(nc.s_CONTROL2STRING, lineArray);
				} 
				else if (s.areTheyEqual("NumberOfSimulations",key2))
				{
					pvisCD.setNumberOfRuns((int)Double.parseDouble(lineArray[2]));
					this.saveRawInput(nc.s_CONTROLSTRINGVALUE, lineArray);	 
				}
				else if (s.areTheyEqual("NumberOfFrequencyDistributionIntervals",key2))
				{
					pvisCD.setNumberOfFrequencyDistributionIntervals((int)Double.parseDouble(lineArray[2]));
					this.saveRawInput(nc.s_CONTROLSTRINGVALUE,lineArray);
				}
				else if (s.areTheyEqual("MaxAlpha", key2))			
				{
					pvisCD.setMaxAlpha(Double.parseDouble(lineArray[2]));
					this.saveRawInput(nc.s_CONTROLSTRINGVALUE,lineArray);
				}
				else if (key2.equalsIgnoreCase("bisection"))
				{
					//bisection routine limits
					key3 = lineArray[2];
					if (key3.equalsIgnoreCase("lowerLimit")|| key3.equalsIgnoreCase("Lower limit"))
					{
						pvisCD.setBisectionLowerLimit(Double.parseDouble(lineArray[3]));
						this.saveRawInput(nc.s_CONTROL2STRINGVALUE,lineArray);
					}
					else if (key3.equalsIgnoreCase("MaxIterations") || key3.equalsIgnoreCase("max iterations"))
					{
						//double dTest = Double.parseDouble(lineArray[3]);
						//int iTest = (int)dTest;
						pvisCD.setBisectionMaxIterations((int)Double.parseDouble(lineArray[3]));
						this.saveRawInput(nc.s_CONTROL2STRINGVALUE,lineArray);
					}
					else if (key3.equalsIgnoreCase("tolerance"))
					{
						pvisCD.setBisectionTolerance(Double.parseDouble(lineArray[3]));
						this.saveRawInput(nc.s_CONTROL2STRINGVALUE,lineArray);
					}

				}
				else if (s.areTheyEqual("GroundwaterConcentrationFactor", key2))
				{
					pvisCD.setGroundWaterConcentrationAdjustmentFactor(Double.parseDouble(lineArray[2]));
					this.saveRawInput(nc.s_CONTROLSTRINGVALUE,lineArray);
				}
				else if (s.areTheyEqual("SoilSampleConcentrationFactor", key2))
				{
					pvisCD.setSoilSampleConcentrationAdjustmentFactor(Double.parseDouble(lineArray[2]));
					this.saveRawInput(nc.s_CONTROLSTRINGVALUE,lineArray);
				}
				else if (s.areTheyEqual(key2, "OilDistribution"))
				{
					//oil distribution type
					key3 = lineArray[2];
					if (s.areTheyEqual(key3,"field"))
					{
						pvisCD.setOilDistributionType("Field");
						this.saveRawInput(nc.s_CONTROL2STRING,lineArray);
					}
					else if (s.areTheyEqual(key3,"capillary"))
					{
						pvisCD.setOilDistributionType("capillary");
						this.saveRawInput(nc.s_CONTROLSTRINGCHOICE,lineArray);
					} 
					else if (s.areTheyEqual(key3,"FileName"))
					{
						//pvisCD.setOilDistributionType("FileName");
						//pvisCD.setOilDistributionFileName(lineArray[3]);
						//this.saveRawInput(nc.s_CONTROL2STRINGVALUE,lineArray);
					}	 
					else if (s.areTheyEqual(key3,"none"))
					{
						pvisCD.setOilDistributionType("none");
						this.saveRawInput(nc.s_CONTROL2STRING,lineArray);
					}
				}
				else if (key2.equalsIgnoreCase("Risk Level") || key2.equalsIgnoreCase("RiskLevel"))
				{
					//risk level
					pvisCD.addRiskLevel(Double.parseDouble(lineArray[2]));
					this.saveRawInput(nc.s_CONTROLSTRINGCHOICE,lineArray);
				}
				else if (key2.equalsIgnoreCase("Hazard Quotient") || key2.equalsIgnoreCase("HazardQuotient"))
				{
					//risk level
					pvisCD.addHazardQuotient(Double.parseDouble(lineArray[2]));
					this.saveRawInput(nc.s_CONTROLSTRINGCHOICE, lineArray);
				}
				/*
			 else if ("LinkedParameters".equalsIgnoreCase(s.removeCharacter(' ',key2)))
			 {
			    //link an independent and dependent parameter 
				//dependent parameter value = independent parameter times multiplier
				String sDependentParameter = lineArray[2];
				String sIndependentParameter = lineArray[4];
				String sOperator = lineArray[5];
				double dMultiplier = Double.parseDouble(lineArray[6]);
				lp.addLinkedParameter(sDependentParameter, sIndependentParameter, sOperator, dMultiplier);


			 }
				 */
				else if ("browser".equalsIgnoreCase(s.removeCharacter(' ',key2)))
				{
					//selection of browser
					pvisCD.setBrowserName(lineArray[2]);
					this.saveRawInput(nc.s_CONTROLSTRINGCHOICE, lineArray);
				}
				/*
				else if (s.areTheyEqual(key2,"OutputUnit",' '))
				{
					//output unit choices 
					uc.addOutputUnit(lineArray[2], lineArray[3]);
					this.saveRawInput(nc.s_CONTROL3STRING,lineArray);
				}
				*/
			}
			else if (s.areTheyEqual(key,"VadoseZone",' ') || s.areTheyEqual(key,"UnsaturatedZone",' '))
			{
				key2 = lineArray[1];


				if (key2.equalsIgnoreCase("Begin")){vz = new VadoseZone();}
				else if (key2.equalsIgnoreCase("End")){}
				else 
				{
					//attempt to add data to the vadose zone
					sConvertedOutput = fdc.addData(vz,key2,lineArray);
					sConvertedOutput = addData(vz,key2,lineArray);
					saveRawInput(nc.s_VALUEINPUT,lineArray);
				}
			}

			else if (key.equalsIgnoreCase("Aquifer"))
			{
				key2 = lineArray[1];
				//add the aquifer to the bvd

				if (key2.equalsIgnoreCase("Begin")){ad = new AquiferData();}
				else if (key2.equalsIgnoreCase("End")){}
				else
				{
					//attempt to add data to the aquiferData object
					sConvertedOutput = fdc.addData(ad, key2, lineArray);
					sConvertedOutput = addData(ad,key2,lineArray);
					//add raw input to RawInput object
					saveRawInput(nc.s_VALUEINPUT,lineArray);
				}

			}

			else if (key.equalsIgnoreCase("OilLens") || key.equalsIgnoreCase("Oil Lens"))
			{
				key2 = lineArray[1];
				//add the oil lens to the bvd

				if (key2.equalsIgnoreCase("Begin")){ /*old = new OilLens();*/}
				else if (key2.equalsIgnoreCase("End")){}
				else
				{
					//attempt to add data to the aquiferData object
					sConvertedOutput = fdc.addData(old, key2, lineArray);
					sConvertedOutput = addData(old,key2,lineArray);
					saveRawInput(nc.s_VALUEINPUT,lineArray);
				}

			}	

			else if (key.equalsIgnoreCase("Building") || key.equalsIgnoreCase("bld"))
			{
				key2 = lineArray[1];
				try{key3 = lineArray[2];}
				catch(Exception e){key3="";}
				//add building data to the bvd

				if (key2.equalsIgnoreCase("DirtFloor") || key2.equalsIgnoreCase("Dirt Floor"))
				{
					if (lineArray[2].equalsIgnoreCase("yes") || lineArray[2].equalsIgnoreCase("y")){bld.setIsItADirtFloor(true);}
					else if (lineArray[2].equalsIgnoreCase("no") || lineArray[2].equalsIgnoreCase("n")){bld.setIsItADirtFloor(false);}
					else {bld.setIsItADirtFloor(false);}
					this.saveRawInput(nc.s_CONTROLSTRINGCHOICE,lineArray);	 
				}
				else if (key2.equalsIgnoreCase("Begin")){bld = new Building();}
				else if (key2.equalsIgnoreCase("End"))
				{
					//add a "hard coded" link between the airflowbelowbuilding to the qsoil 
					lp.addLinkedParameter("AirFlowBelowBuilding","Qsoil","times",1.0);
				}
				else if (key3.equalsIgnoreCase("Distribution")){this.readDistribution("Building",lineArray);}
				else if (key3.equalsIgnoreCase("Linked")){fdc.linkParameters("Building",lineArray);}
				else
				{
					sConvertedOutput = fdc.addData(bld, key2, lineArray);
					sConvertedOutput = addData(bld,key2,lineArray);  
					saveRawInput(nc.s_VALUEINPUT,lineArray);
				} 				
			}	  

			else if (s.areTheyEqual(key, "ScreeningLevel"))
			{	  
				key2 = lineArray[1];
				if (s.areTheyEqual(key2, "Begin"))
				{
					risk = new Risk();
				    pvisCD.setDoesScreeningExist(true);
				}
				else if (s.areTheyEqual(key2, "End")){}
				else
				{
					try
					{
						ChemicalRisk cr = new ChemicalRisk();
						cr.setChemicalName(lineArray[1]);
						cr.setScreeningLevel(Double.parseDouble(lineArray[3]));
						cr.setScreeningLevelUnit(lineArray[4]);

						risk.addRiskLevel(cr);
						saveRawInput(nc.s_CONTROLSCREENINGLEVELINPUT,lineArray);
					}
					catch(Exception e){}
				}
			}

			else if (key.equalsIgnoreCase("Chemical"))
			{
				key2 = lineArray[1];

				if (key2.equalsIgnoreCase("Begin")){}
				if (key2.equalsIgnoreCase("Heading")){}
				else if (s.areTheyEqual("airPhaseConcentration", key2, ' ')
						||   s.areTheyEqual("FuelPhaseConcentrationByMass", key2, ' ') 
						||   s.areTheyEqual("FuelPhaseConcentrationByVolume", key2, ' ') 
						||   s.areTheyEqual("WaterPhaseConcentration", key2, ' ') 	
						||   s.areTheyEqual("GroundWaterConcentration", key2, ' ')   //JWW  9-9-2020
						||   s.areTheyEqual("SoilSample", key2, ' ') 
						)
				{
					sConvertedOutput = fdc.addChemicalData(lineArray);
					saveRawInput(nc.s_CHEMICALINPUT,lineArray);
				}


			}
			
			/*
			//jww oxygen moved to suggested value inputs
			else if (key.equalsIgnoreCase("Oxygen"))
			{
				key2 = lineArray[1];
				if (key2.equalsIgnoreCase("Begin")){oxygen = new Oxygen();}
				else if (key2.equalsIgnoreCase("End")){}
				else
				{
					sConvertedOutput = addData(oxygen,key2,lineArray);
					saveRawInput(nc.s_VALUEINPUT,lineArray);
				}

			}
			*/

			//echo input data to output file
			String sOut = arrayToString(lineArray);
			fl.logMessage(sOut + sConvertedOutput + sErrorOutput); 
			this.addToKnownErrorOutput(sErrorOutput);
		}

		//read field data
		if (pvisCD.getOilDistributionType().equalsIgnoreCase("field"))
		{		
			fd = new FieldData();
			fd.setFileLogger(fl);
			fd.setUnitConverter(uc);
			fd.readFile(pvisCD.getOilDistributionFileName());
			pvisCD.setFieldData(fd);
		}

		//read the suggested values (can be over read below)
		//(any value read from the suggested value file is written back to the suggested value file)
		//(regardless of where it appears on the interface)
		svr.setPVISControlData(pvisCD);
		svr.setRawInputContainer(ri);
		svr.setNameConstants(nc);
		svr.setUnitConverter(uc);
		svr.setBuildingObject(bld);
		svr.setVadoseZoneObject(vz);
		svr.setMainOutputFileLogger(fl);
		svr.setFillDataContainer(fdc);
		svr.setExistingErrorOutput(sErrorOutput);
		svr.readSuggestedValues(nc.file_SuggestedValues);		

        oxygen = svr.getOxygenObject();

		//gather all the StatisticalDistributions for use in the Monte Carlo processor
		this.collectStatisticalDistributions();

		//some calculations are needed to prepare for running the  model
		//or for each monte carlo run
		//since the model will run in an inherent monte carlo mode 
		//(i.e., deterministic = 1 monte carlo run)
		//the initialization of parameters is done after selecting
		//each run's set of parameters by calling setParametersForEachRun()

		//add data to the oil lens object
		if (pvisCD.getOilDistributionType().equalsIgnoreCase("lens"))
		{	
			old.setIsLensUsed(true);
			old.setVadoseZone(vz);
			old.findTopOfOilZone();
			old.setFieldData(fd);
			old.setElevations();
			old.generateOilProfile();
			old.writeProfile(fl);
		}
		else
		{
			old.setIsLensUsed(false);
		}

		//set statistical distributions for linked parameters
		this.setLinkedParameterDistributions();


		//revise chemical list
		ArrayList<Chemical> alChemicalRevised = new ArrayList<Chemical>();
		//process chemical inputs if necessary
		for (Chemical chemical:  alChemical)
		{
			chemical.setOilLensData(old);
			chemical.setVadoseZone(vz);
			chemical.setAquifer(ad);
			chemical.setUnitConverter(uc);
			chemical.correctHenrysConstantForTemperature();

			//setting the following value to 1.0 removes archaic use of a risk concentration multiplier (jww 12-20-2013)
			chemical.setRiskConcentrationMultiplier(1.0);
			//update default risk levels with user inputs
			String sName = chemical.getName();
			ChemicalRisk crCheck = risk.getChemicalRisk(sName);
			if (crCheck!=null)
			{
				chemical.setUserRisk(crCheck);	
				chemical.setUserRiskSet(true);
				chemical.updateRisk();
				//set screening levels from risk object to chemical object, jww, 4-23-2016
				if(crCheck.getScreeningLevel()>0.0)
				{
					chemical.setSreeningLevelSet(true);
					chemical.setScreeningLevel(crCheck.getScreeningLevel());
				}
			}
			alChemicalRevised.add(chemical);
			if (chemical.getIsChemicalUsed())
			{
				chemical.setConcentrationAdjustmentGroundWater(pvisCD.getGroundWaterConcentrationAdjustmentFactor());
				chemical.setConcentrationAdjustmentSoilSample(pvisCD.getSoilSampleConcentrationAdjustmentFactor());
				chemical.setAirPhaseConcentration();  
			}  
		}

		//load up the data sets for use in the PVIScreenControlData object
		pvisCD.setVadoseZone(vz);
		pvisCD.setBuilding(bld);
		pvisCD.setOxygen(oxygen);
		pvisCD.setAquiferData(ad);	
		pvisCD.setOilLensData(old);
		pvisCD.setChemicals(alChemicalRevised);
		pvisCD.setStatisticalDistribution(sDistribution);
		pvisCD.setParameterChoices(pc);
		//pvisCD.setOutputUnitDefinition(uDefinition);	
		pvisCD.setUnitConverter(uc);
		pvisCD.setRawInput(ri);
		pvisCD.setRunIdentifier(runID);
		pvisCD.setLinkedParameters(lp);
		//pvisCD.setRisk(risk);



		//check to see if all unit conversions are supported
		bUnsupportedUnit = uc.getBooleanUnsupportedUnit();

		//collect up the errors and determine if model can run
		pvisCD.addToKnownErrorOutput(this.getKnownErrorOutput());
		pvisCD.addToKnownErrorOutput(cdr.getKnownErrorOutput());
		pvisCD.setCanRunModel();

		//close input file
		fr.close();


	}

	//read an external distribution
	private void readDistribution(String sGroup, String[] lineArray) throws IOException
	{
		String key, key2, key3;
		String sParameterName;
		String sConvertedOutput;
		String currentLine;

		String sFileNameDistribution = lineArray[3];
		sParameterName = lineArray[1];
		//input data are from a distribution (to be read in from file)

		//open file
		FileReader frDistribution = new FileReader(sFileNameDistribution);
		BufferedReader brDistribution = new BufferedReader(frDistribution);



		//read each line
		while ((currentLine = brDistribution.readLine()) != null) 
		{
			lineArray = currentLine.split(",",0);  
			//add each line of building data
			key = lineArray[0];
			key2 = lineArray[1];
			key3 = lineArray[2];
			sConvertedOutput = "";


			if (key2.equalsIgnoreCase("Distribution") && key2.equalsIgnoreCase("Begin")){}
			else if (key2.equalsIgnoreCase("name"))
			{
				String sDistributionName = lineArray[2];
				//save the parameter name (key2) and the data file name (key4)
				if (sGroup.equalsIgnoreCase("Building")){bld.addDataSource(sParameterName, sFileNameDistribution, sDistributionName);}

			}

			else if (key2.equalsIgnoreCase("Distribution") && key2.equalsIgnoreCase("end")){}
			else if (key3.equalsIgnoreCase("variable"))
			{

				sConvertedOutput = addData(bld,key2,lineArray);
				saveRawInput(nc.s_VALUEINPUT,lineArray);
			}


			//echo input data to output file
			String sOut = arrayToString(lineArray);
			fl.logMessage(sOut + sConvertedOutput + sErrorOutput); 
			this.addToKnownErrorOutput(sErrorOutput);
		}			   
	}

	//link parameters
	private void linkParameters(String sGroup, String[] lineArray) 
	{
		double dMultiplier = 1.0;
		//Building	AirFlowBelowBuilding	Linked	Qsoil	times	1
		try{dMultiplier = Double.parseDouble(lineArray[5]);}
		catch(Exception e){dMultiplier=1.0;}
		lp.addLinkedParameter(lineArray[1],lineArray[3],lineArray[4],dMultiplier);
	}

	//set statistical distribution for linked parameters
	private void setLinkedParameterDistributions()
	{
		int iNumber, iCount;
		String sIndependent, sDependent;
		double dMultiplier;

		iNumber = lp.getLinkedParameterCount();


		//run through all linked parameters
		for (int i=0;i<iNumber;i++)
		{
			sIndependent = lp.getIndependentParameter(i);
			sDependent = lp.getDependentParameter(i);
			dMultiplier = lp.getMultiplier(i);
			for (StatisticalDistribution sd1: sDistribution)
			{
				//find the statistical distribution for the independent value
				if (sd1.getName().equalsIgnoreCase(sIndependent))
				{
					//find the statistical distribution for the dependent value
					for (StatisticalDistribution sd:  sDistribution)
					{
						if (sd.getName().equalsIgnoreCase(sDependent))
						{
							//set the same statistical distribution in the dependent variable
							//as the independent variable
							iCount = sd1.getNumberOfPointsDefiningDistribution();
							sd.setNumberOfPointsDefiningDistribution(sd1.getNumberOfPointsDefiningDistribution());

							//apply multiplier
							double[] dValue = sd1.getConcentrationDistribution();
							for (int j=0;j<iCount;j++)
							{
								dValue[j] = dValue[j]*dMultiplier;
							}
							sd.setConcentrationDistribution(dValue);
							sd.setFrequencyDistribution(sd1.getFrequencyDistribution());
							sd.setCalculationUnit(sd1.getCalculationUnit());
							sd.setCurrentRandomValue(sd1.getCurrentRandomValue());
							sd.setChosenFrequency(sd1.getChosenFrequency());
							break;
						}
					}
					break;
				}
			}

		}		 


	}



	private String addData(PVIScreenData pvisData, String key2, String[] lineArray)
	{
		String sOutput = "";
		String sOutputError = "";
		String key3 = lineArray[2];

		//add data to RawInput container


		if (key3.equalsIgnoreCase("Constant"))
		{
			sOutput = addDataItem(pvisData,key2,lineArray[3],lineArray[4],getUnitType(key2));
			if (sOutput.equalsIgnoreCase(" "))
			{
				sOutputError = ", Error in Input: " + arrayToString(lineArray);
				System.out.println(sOutputError);
			}

		}
		else if (key3.equalsIgnoreCase("Variable"))
		{
			sOutput = addDataItemVariable(pvisData,key2,lineArray[3],lineArray[4],getUnitType(key2),lineArray[5]);
			if (sOutput.equalsIgnoreCase(" "))
			{
				sOutputError = ", Error in Input: " + arrayToString(lineArray);
				System.out.println(sOutputError);
			} 
		}

		this.addToKnownErrorOutput(sOutputError);		 
		return sOutput + sOutputError;				
	}




	//set individual bit of Variable data
	private String addDataItemVariable(PVIScreenData bvd, String key2m, String sValue, String sUnit, String sUnitType, String sFrequency)
	{
		String sReturn="";;
		OutputDigits od = new OutputDigits();
		//key2m = modified key2 (i.e., no spaces included)
		//sValue = string version of data item value
		//sUnit = string unit (i.e. cm, cm3/sec, 1/hr, ft, etc)
		//sUnitType = name for type of unit (i.e., length, mass, concentration)
		//  (No spaces allowed use "camel case" e.g., RateConstant)
		//sFrequency = string version of frequency value
		double value = Double.parseDouble(sValue)*uc.getUCF(sUnitType,sUnit);
		double frequency = Double.parseDouble(sFrequency);
		bvd.setCumulativeDistributionValue(key2m, value, frequency);
		bvd.setCurrentRandomValue(key2m,0.5);
		bvd.setCalculationUnit(key2m,uc.getModelUnit(sUnitType));
		od.setValue(value);
		sReturn = ", converted to: " + od.num2String() + " " + uc.getModelUnit(sUnitType);
		return sReturn;  
	}  

	//set individual bit of constant data
	private String addDataItem(PVIScreenData bvd, String key2m, String sValue, String sUnit, String sUnitType)
	{
		String sReturn = "";
		OutputDigits od = new OutputDigits();
		//key2m = modified key2 (i.e., no spaces included
		//sValue = string version of data item value
		//sUnit = string unit
		//sUnitTupe = name for type of unit (i.e., length, mass, concentration) 
		//  (No spaces allowed use "camel case" e.g., RateConstant)
		double value = Double.parseDouble(sValue)*uc.getUCF(sUnitType,sUnit);
		bvd.setStatisticalDistributionValue(key2m, value);
		bvd.setCurrentRandomValue(key2m,0.5);
		bvd.setCalculationUnit(key2m,uc.getModelUnit(sUnitType));
		//limit the number of output digits
		od.setValue(value);
		sReturn = ", converted to " + od.num2String() + " " + uc.getModelUnit(sUnitType);
		return sReturn;
	}

	//save RawInput data to data container
	private void saveRawInput(String sRawInputType, String[] sRawInputLine)
	{
		RawInputLine ril = new RawInputLine();
		//put the raw input into a RawInputLine object
		ril.setRawInputLine(sRawInputType, sRawInputLine);
		//add raw input line to RawInputLine collection
		ri.add(ril);
	}



	//assemble all the statistical distributions into one container (sDistribution)
	private void collectStatisticalDistributions()
	{   
		//gather all statistical distributions from the building, vadose zone, chemical and oxygen objects

		//the building object
		for (StatisticalDistribution sd: bld.getStatisticalDistribution()){sDistribution.add(sd);}

		//the vadose zone object
		for (StatisticalDistribution sd: vz.getStatisticalDistribution()){sDistribution.add(sd);}

		//the oxygen object
		for (StatisticalDistribution sd: oxygen.getStatisticalDistribution()){sDistribution.add(sd);}

		//the oil lens object
		for (StatisticalDistribution sd: old.getStatisticalDistribution()){sDistribution.add(sd);}

		//the chemical objects
		for (Chemical c: alChemical)
		{ 
			for (StatisticalDistribution sd: c.getStatisticalDistribution())
			{
				//only add statistical distributions for chemicals that are actually used
				if (c.getIsChemicalUsed()){sDistribution.add(sd);}
			}
		}
	}


	/**
	 * getUnitType         return the type of unit for the specified input parameter  (i.e., CrackWidth is a "Length")
	 * @param sInputName   name of input parameter (lineArray[0] (a.k.a. "key2") -- second entry on input line)
	 * @return
	 */
	private String getUnitType(String sInputName)
	{
		String sReturn;
		RawInputLine ril = new RawInputLine();

		//the unit conversions are in the RawInputLine, use this class for uniformity
		sReturn = ril.getUnitType(sInputName,uc);

		return sReturn;
	}

	private String arrayToString(String[] lineArray)
	{
		String sOut="";
		for (int i=0;i<lineArray.length;i++){sOut = sOut + lineArray[i] + ", ";}
		return sOut;
	}


	//add	
	public void addToKnownErrorOutput(String sError)
	{
		if (!sError.equalsIgnoreCase(""))
		{
			sErrorOutput = sErrorOutput + sError + "\n";
		}
	}			

	//sets
	public void setUnitConverter(UnitConverter uc){this.uc = uc;}
	public void setWriteFile(boolean b){this.bWriteFile = b;}
	public void setFileModifier(FileModifier fm){this.fm = fm;}

	//simple gets
	public ArrayList<StatisticalDistribution> getStatisticalDistributions(){return sDistribution;}
	public PVIScreenControlData getPVIScreenControlData(){return pvisCD;}
	public boolean getBooleanUnsupportedUnit(){return bUnsupportedUnit;}
	public String getOutputFileName(){return sOutputFileName;}
	public String getStatisticsFileName(){return this.sStatisticsFileName;}
	public UnitConverter getUnitConverter(){return uc;}
	public FileLogger getFileLogger(){return fl;}
	public String getKnownErrorOutput(){return this.sErrorOutput;}

	public void setOutputFileTime(String sOutputFileTime) {this.sOutputFileTime = sOutputFileTime;}

}
