package pviScreen;

import java.io.IOException;
import java.util.ArrayList;
import statistics.ResistantStatisticalMeasures;
import uncertaintyAnalysis.MonteCarloLHS;
import util.CalendarClock;
import util.FileLogger;
import util.OutputProcessor;
import util.UnitConverter;
import util.VersionCompilation;

public class RUNPVIScreen 
{
	private String sModelName;
	private String sOutputFileName;
	private String sStatisticsFileName;
	private String sOutputFileTime;
	private MonteCarloLHS mcLHS;
    private EPAPVIScreen pvis;
    private PVIScreenControlData pvisCD;
    private UnitConverter uc;
    private PVIScreenReader pvisr;
    private OutputProcessor op;
    private ResistantStatisticalMeasures rsm;
    private FieldData fd;
    private String sInputFileName;
	private int iNumberOfRuns;
	private VersionCompilation vc;
	
	 
    
    public RUNPVIScreen(){}
    
   
    
    public void runPVIScreen() throws IOException
    {
    	
		boolean bMonteCarlo;
		boolean bUnsupportedUnit;
		boolean bRun;
		
		//version information compilation
		vc = new VersionCompilation();
	
		
		sModelName = "PVIScreen";
		
		
		//boolean flag   model will run only if the BioVaporControlData sets this to true:  i.e., no input errors.
		bRun = false;
		
		FileLogger fl = new FileLogger();
		FileLogger flSummary = new FileLogger();
		
		//create the PVIScreen object
	    pvis = new EPAPVIScreen();
	    pvis.initial();
	     
	    pvisr = new PVIScreenReader();
	    pvisr.initial();
	    
	    System.out.println("Simulating: " + sInputFileName);
	    pvisr.setOutputFileTime(sOutputFileTime);
	    pvisr.readFile(sInputFileName);
	    
	     
	    pvisCD = new PVIScreenControlData();
	    pvisCD = pvisr.getPVIScreenControlData();
	    
	    sOutputFileName = pvisr.getOutputFileName();
	    sStatisticsFileName = pvisr.getStatisticsFileName();
	    bUnsupportedUnit = false;
	    bUnsupportedUnit = pvisr.getBooleanUnsupportedUnit();
	    this.uc = pvisCD.getUnitConverter();
	    fl = pvisCD.getFileLogger();
	    fl.logMessage("Control,Results");
	    

	    
	    
	    bRun = pvisCD.getCanRunModel();
	    if (bRun==true)
	    {
	      //give the data to the biovapor model
	      pvis.loadData(pvisr.getPVIScreenControlData());  
	      bMonteCarlo = pvis.getIsMonteCarlo();
	 
	    
	    
	      if (!bMonteCarlo && !bUnsupportedUnit)
	      {
	        //deterministic run	
	        pvis.runModel();
	      }
	      else if (bMonteCarlo && !bUnsupportedUnit)
	      {  
            //MonteCarlo Simulation
        
            CalendarClock ccStart = new CalendarClock();	
            System.out.println("Monte Carlo Latin Hypercube Simulation Beginning");
          
            mcLHS = new MonteCarloLHS();
            mcLHS.setFileLoggers(fl,flSummary);
            mcLHS.setMonteCarloModel(pvis);
            mcLHS.setMonteCarloParameters(pvis.getNumberOfRuns());
            mcLHS.simulate();
          
        
            CalendarClock ccEnd = new CalendarClock();
            System.out.println("Monte Carlo Latin Hypercube Simulation Ending");
          
            //get run time
            String sUnit = " seconds";
            double dSimulationTime = ccStart.timeDifferenceSeconds(ccEnd);
            if (dSimulationTime>60){dSimulationTime = dSimulationTime/60.; sUnit = " minutes";}
            else if (dSimulationTime>3600.){dSimulationTime = dSimulationTime/3600.; sUnit = " hours";}
            System.out.println("Run Time " + dSimulationTime + sUnit);
            
           
            System.out.println("Processing and sorting outputs");
            processMonteCarloOutput(sModelName,sInputFileName,sOutputFileName,sStatisticsFileName);
            System.out.println("Post-processing complete");
            
            CalendarClock ccPost = new CalendarClock();
            dSimulationTime = ccEnd.timeDifferenceSeconds(ccPost);
            if (dSimulationTime>60){dSimulationTime = dSimulationTime/60.; sUnit = " minutes";}
            else if (dSimulationTime>3600.){dSimulationTime = dSimulationTime/3600.; sUnit = " hours";}
            
            System.out.println("Post-processing Time " + dSimulationTime + sUnit);            
                  
  	      }
	 
	    }
	    else
	    {
	    	//model cannot run
	    	System.out.println(pvisCD.getKnownErrors());
	    }
	    System.out.println("Completed Simulation: " + sInputFileName);
	    
    }
    /**
     * processMonteCarloOutput  post process the Monte Carlo output of the RiverSimulation
     *                          using the OutputProcessor and Resistant Statistical Measures classes
     *                          
     *  sModelName                  String containing model name
     *  sInputFileName              String containing main input file name
     *  sOutputFileName             String containing main output file name (to be read by the output processor)
     *  sStatisticsFileName         String containing statistics output file name (to be filled in by the output processor)
     *                          
     * @throws IOException
     */
    public void processMonteCarloOutput(String sModelName,
  		                              String sInputFileName,
  		                              String sOutputFileName,
  		                              String sStatisticsFileName) throws IOException
    {
        op = new OutputProcessor();
        //process output data for statistical analysis
        op = new OutputProcessor();
        op.initial();
        op.setVersionInformation(vc);
        op.setIntervalFrequency(pvisCD.getNumberOfFrequencyDistributionIntervals());
        op.setResultsFileName(sOutputFileName);
        op.setUnitConverter(uc);
        op.setNumberOfResults(pvis.getNumberOfRuns());
       
        //resistant statistical measures analysis
        rsm = new ResistantStatisticalMeasures();
        //set version information
        rsm.setVersionInformation(vc);
        //set the output file
        rsm.setFileLogger(op.getOutputFileLogger());
        //give to output processor
        op.setResistantStatisticalMeasures(rsm);
        
        op.openOutputFile(sOutputFileName);
       
      //consolidating the statistics output and results output by entering sOutputFileName for both entries
        op.writeRunFileInformation(sModelName,sInputFileName,sOutputFileName,sOutputFileName);        
    
        
        
        //start evaluation of parameter significance
        //sort the results and write
        
        op.readResults("Benzene Indoor Air Concentration");
        op.sortByMergeSortAndReWrite();
        
     
        //read sorted results to an ArrayList of ResultsColumns
        op.readSortedResultsToArrayList();
        //prepare data for statistical measures calculation
        //  simple statistics:  min, max, average, variance, range
        //  resistant statistical measures:   5th percentile, first quartile/25th percentile, median, third quartile/75th percentile,
        //                                        median + 1.5 InterQuartileRange, 95th percentile
              
        //generate the statistics of the output
        op.generateStatistics();        
        
        //characterize the data:  lowest, low, middle, high, highest
        op.characterizeAndWriteData();
            
        //write statistics
        op.writeStatistics();      
        
        ArrayList<Chemical> alChemical = pvisCD.getChemicals();
   
      
        op.sortResultsColumn("Aerobic Zone Thickness");
        op.makeResultsColumnLogHistogram("Aerobic Zone Thickness","Aerobic");
        op.writeLogHistogram("Aerobic Zone Thickness");

        
        //create the log histograms
        for (Chemical chemical: alChemical)
        {
          if (chemical.getSourceConcentration()>0.0)
          {	  
            String sChemicalName = chemical.getName();
            String sChemicalTitle = sChemicalName + " indoor air concentration";
            op.sortResultsColumn(sChemicalTitle);
            op.makeResultsColumnLogHistogram(sChemicalTitle,sChemicalName);
            op.writeLogHistogram(sChemicalTitle);
          }  
        }   
        
        
        
    
    }

    
    
    //sets
    public void setDataFileName(String sInputFileName){this.sInputFileName = sInputFileName;}
    public void setOutputFileTime(String sOutputFileTime){this.sOutputFileTime = sOutputFileTime;}
    
    
    //gets
    public String getOutputFileName(){return this.sOutputFileName;}
    public String getOutputFileNameStatistics(){return this.sStatisticsFileName;}
    public ArrayList<Chemical> getChemicals(){return pvisCD.getChemicals();}
}
