package statistics;

import util.FileLogger;

/**
c     ******************************************************************
c     *
c     * lfit uses linear regression to fit the function
c     *      y = b0 + b1x
c     *      to a set of x,y pairs
c     *
c     *
c     *
c     * input arguments..............................................
c     * x()    array of times
c     * y()    array of concentration
c     * ix     number of time-concentration pairs
c     * output arguments.............................................
c     * b0     regression coefficient
c     * b1     regression coefficient
c     * s2     estimated error variance
c     * r2     "goodness of fit"--fraction of total variability
c     *         explained by linear regression
c     *
c     * jim weaver
c     * ecosystems research division
c     * national exposure research laboratory
c     * united states environmental protection agency
c     * athens, georgia  30605
c     *
c     * double precision fortran 77
c     * required routines:  none 
c     *
c     * reference:  Statistics, Principles and Methods, Richard A. Johnson
c     *             Gouri K. Bhattacharyya, 1996, 3ed


*/
public class DataFitting 
{
	private int iCount;
	private double[] dX,dY;
    private double dFittedA,dFittedB;
    private double dSumSquaredError, dS2, dR2;
	private String sFittingType;
    
	public DataFitting(){}
	
	public void fitTheData()
	{
		if (this.sFittingType.equalsIgnoreCase("linear")){this.linearFit();}
		else if (this.sFittingType.equalsIgnoreCase("power")){this.powerFit();}
		else { System.out.println("Error in specifying fitting type: " + sFittingType);}
	}
	
	/**
	 * powerFit   fit the function  y = ax^b  to data sets dX[] and dY[]
	 *            by taking the log of the data, then using a linear fit:
	 *            log(y) = log(a) + b log(x)
	 *            a = anti-log (log(a))
	 *            specifically:
	 *              = 10^(log10(a)), if log base 10
	 *              = e^(ln(a)), if natural logrithm
	 *            b = b
	 */
	public void powerFit()
	{
		double[] dLogX = new double[iCount];
		double[] dLogY = new double[iCount];
		
		for (int i=0;i<iCount;i++)
		{
			dLogX[i] = Math.log10(dX[i]);
			dLogY[i] = Math.log10(dY[i]);
		}
		
		linearFit(dLogX,dLogY);
		this.dFittedA = Math.pow(10,this.dFittedA);
	}
	
	
	/**
	 * powerFit   fit the function  y = ax^b  to data sets dX[] and dY[]	
	 *            here the internally stored dX and dY are used
	 */ 
	public void linearFit(){linearFit(this.dX, this.dY);}
	
	/**
	 * linearFit   fit the function  y = ax^b  to data sets dX[] and dY[]	
	 *            here dX and dY are passed
	 *            
	 *            
	 *            see. USEPA Laboratory Notebook Ada JWW-02 pp. 137-141
	 *            
	 *            based on Statistics, Fifth Edition, 1991, 
	 *            James T. McClave and Frank H. Dietrich II
	 *            Dellen Publishing/MacMillan Publishing, New York, 928pp.
	 *            see pages 640-670
	 */ 	
	public void linearFit(double[] dX, double[] dY)
	{
	  double dSumX,dSumY;
	  double dSumXY, dSumXX, dSumYY;	
	  double dXBar, dYBar;
	  double dSXX, dSYY, dSXY;
      //accumulate sums
	  dSumX = 0; dSumY = 0; dSumXX = 0; dSumYY = 0; dSumXY = 0;
	  for (int i=0;i<iCount;i++)
	  {	  
		  dSumX = dSumX + dX[i];
	      dSumY = dSumY + dY[i];
	      dSumXY = dSumXY + dX[i]*dY[i];
	      dSumXX = dSumXX + dX[i]*dX[i];
	      dSumYY = dSumYY + dY[i]*dY[i];
	  }
	  
	  //averages
	  dXBar = dSumX/(double)iCount;
	  dYBar = dSumY/(double)iCount;

	  //sums of "squares" (the first two are squares, the third the product of x and y)
	  dSXX = dSumXX - dSumX*dSumX/(double)iCount;
	  dSYY = dSumYY - dSumY*dSumY/(double)iCount;		  
	  dSXY = dSumXY - dSumX*dSumY/(double)iCount;  		
	
	  //calculate regression intercept(dFittedA) and slope(dFittedB)
	  dFittedB = dSXY/dSXX;		
	  dFittedA = dYBar - dFittedB*dXBar;
	
      //calculate the estimate of error variance
	  dSumSquaredError = dSYY - dSXY*(dSXY/dSXX);
	  dS2 = dSumSquaredError/(double)(iCount-2);
	   
	  //calculate the goodness of fit
	  dR2 = 1.0 - dSumSquaredError/dSYY;
	}
	
	/**
	 * getEstimate   determine the value of the dependent variable ("Y")
	 *               given the fitting parameters and fitting type
	 */
	public double getEstimate(double dValue)
	{
	    double dFY=0;	
		if (this.sFittingType.equalsIgnoreCase("linear"))
		{
			dFY = this.dFittedA + this.dFittedB*dValue;
		}
		else if (this.sFittingType.equalsIgnoreCase("power"))
		{
			//log(y) = log(a) + b log(x)
			 
	
		    dFY = this.dFittedA*Math.pow(dValue, this.dFittedB);
		}
		return dFY;
	}
	
	/*
	 * writeResults   write results to a supplied FileLogger file
	 * @flOut         FileLogger to write to
	 */
	public void writeResults(FileLogger flOut)
	{
		//write out the variables, fitted values and errors
		flOut.logMessage("");
		flOut.logMessage("Fitted Results");
		flOut.logMessage("Fitted Results,Independent Variable,Dependent Variable,Fitted Result,Error");
		for (int i=0;i<iCount;i++)
		{
		   double dFittedY,dError;
		   dFittedY = this.getEstimate(dX[i]);
		   dError = this.dY[i]-dFittedY;
		   flOut.logMessage("Fitted Results," + this.dX[i] + "," + this.dY[i] + "," + dFittedY + "," + dError);	   
		}
		
		//write out the parameters
		flOut.logMessage("Fitted Results," + this.getFittingType() + ",parameter a:," + this.getIntercept() + ",parameter b:," + this.getSlope() + ",goodness of fit (r2)," + this.getGoodnessOfFit() );
		
	}
	
	
    //sets
	public void setXValues(double[] dX){this.dX = dX;}
	public void setYValues(double[] dY){this.dY = dY;}
	public void setNumberOfValues(int iCount){this.iCount = iCount;}
	public void setFittingType(String sFittingType){this.sFittingType = sFittingType;}
	
	//gets
	public double getIntercept(){return this.dFittedA;}
	public double getSlope(){return this.dFittedB;}
	public double getEstimateOfErrorVariance(){return this.dS2;}
	public double getGoodnessOfFit(){return this.dR2;}
	public String getFittingType(){return this.sFittingType;}
	
}
