package statistics;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;

import util.FileLogger;
import util.Stripper;

 
/**
 * Histogram   class to generate histograms for monte carlo output
 *             Interval class used for storing the intervals of the histogram
 *             Histogram is used in the OutputProcessor
 *             
 *             
 * @author JWEAVER
 *
 */
public class Histogram 
{
	private boolean bHistogramMade;
	private int iTotalCount;
	private int iFrequencyInterval;
	private String sHistogramTitle;
	private String sHistogramChemical;
	private double dHistogramLevelOfConcern;
	private String sOutputUnit;
	private double dHistogramLevelOfConcernCumulativeFrequency;
	private boolean bLevelOfConcernSet;
	private double dRiskLevel;
	private double dRfC;
	private boolean bRfCSet;
	ArrayList<Interval> alInterval;
	private double[] dValue2;
	private boolean bCoupled;
	private boolean bHistogramLinear;
	private RiskContainer rc;
	
	public Histogram()
	{
		alInterval = new ArrayList<Interval>();
		bHistogramLinear = true;
		bHistogramMade = false;
		bCoupled = false;
		
		//risk (or level of concern) level
		sHistogramChemical = "";
		dHistogramLevelOfConcern = 0.0;
		sOutputUnit = "";
		bLevelOfConcernSet = false;
		dRiskLevel = 0.0;
		bRfCSet = false;
	}
	
	double dV;
	double dMinHistogram;
	double dMaxHistogram;
	
	public void generateHistogram(double[] dValue, int iCount)
	{
	  //develop a histogram for the selected output
	
	  bHistogramMade = false;
	  bHistogramLinear = true;
	
      //storage container
	  alInterval = new ArrayList<Interval>();
	
	  dMinHistogram = 1.e20;
	  dMaxHistogram = -1.e20;
	  for (int i=0;i<iCount;i++)
	  {
	  	if (dValue[i]>dMaxHistogram){dMaxHistogram=dValue[i];}
		if (dValue[i]<dMinHistogram){dMinHistogram=dValue[i];}
	  }
	 
	  //the intervals must be very slightly larger than the true min to max
	  //this is necessary because when creating the bins min and max will be at an endpoint
	  
	  dMaxHistogram = dMaxHistogram*1.000001;
	  dMinHistogram = dMinHistogram*0.999999;
	  
	  dV = (dMaxHistogram-dMinHistogram)/(double)(iFrequencyInterval-1);

	  //make the histogram only if the interval is above zero
	  //(all entries not the same)
	  if (dV>0.0)
	  {	
	    //histogram interval	
	    bHistogramMade = true;
	    double d1;
	    double d2;
	    d1 = dMinHistogram;
	    d2 = d1 + dV;
	    for (int i=0;i<iFrequencyInterval;i++)
	    {	
	      Interval interval = new Interval(); 
	      interval.setLowPointValue(d1);
	      interval.setMidPointValue(0.5*(d2+d1));
	      interval.setHighPointValue(d2);
	      
  	      //divide results among histogram compartments
	      for (int j=0;j<iCount;j++)
	      {
	  	    if (d1<=dValue[j] && dValue[j]<=d2)
		    {
		      interval.addOneMember(j);
		      //add coupled data point (if coupled histogram)
		      if (bCoupled){ interval.addCoupledDataPoint(dValue2[j]);}
		    }
	      }
	  
	      //generate coupled histogram
	      if (bCoupled){interval.generateCoupledHistogram();}
	      
	      alInterval.add(interval);
	      d1 = d2;
	      d2 = d1 + dV;
	    }
	  }
	  else
	  {
		if (iCount>0)
		{	
		  //for one value dV = 0.0 (i.e., dMin = dMax)
		  Interval interval = new Interval(); 
		  interval.setLowPointValue(dMinHistogram);
	      interval.setMidPointValue(0.5*(dMaxHistogram+dMinHistogram));
	      interval.setHighPointValue(dMaxHistogram);
	      interval.addOneMember(0);
	      if (bCoupled){System.out.println("Error in Coupled Histogram:  only one primary histogram interval");}
	      alInterval.add(interval);
		}  
	  }
	  
	  //get total interval count
	  iTotalCount = 0;
	  for (Interval i: alInterval){iTotalCount = iTotalCount + i.getIntervalCount();}
    }

	
		public void generateLogHistogram(double[] dValue, int iCount)
		{
			//develop a histogram for the selected output
			bHistogramMade = false;
			bHistogramLinear = false;
			
			//log-scale bins
			double[] dLogLevel = { -4.5,-4.25,-4.0,-3.75,-3.5,-3.25,-3.0,-2.75,-2.5,-2.25,-2.0,-1.75,-1.5,-1.25,-1.0,-0.75,-0.5,-0.25,
					0.0,
					0.25,0.5,0.75,1.0,1.25,1.5,1.75,2.0,2.25,2.5,2.75,3.0,3.25,3.5,3.75,4.0,4.25,4.5,4.75,5.0,5.25,5.5,5.75,6.0};
			
		    //storage container
			//alLogInterval = new ArrayList<Interval>();
			bHistogramMade = true;
			this.iFrequencyInterval = dLogLevel.length;
	        
			//number of intervals
			int iInterval=dLogLevel.length;
			
	        //first interval
			Interval interval = new Interval();
		
			interval.setMidPointValue(dLogLevel[0] + 0.5*dLogLevel[0]);
			interval.setHighPointValue(dLogLevel[1]);
			for (int j=0;j<dValue.length;j++)
			{ 
				  if (Math.log10(dValue[j]) < dLogLevel[0])
				  {
				    interval.addOneMember(j); 	
				  }
			}
			//alLogInterval.add(interval);
			alInterval.add(interval);
			

			  for (int i=1;i<iInterval-1;i++)
			  {	
			    interval = new Interval(); 
			    interval.setLowPointValue(dLogLevel[i]);
			    interval.setMidPointValue(0.5*(dLogLevel[i] + dLogLevel[i+1]));
			    interval.setHighPointValue(dLogLevel[i+1]);
			  
		  	    //divide results among histogram compartments
			    for (int j=0;j<dValue.length;j++)
			    {
			      double dLogVal = Math.log10(dValue[j]);
				  if (dLogLevel[i]<=dLogVal && dLogVal<=dLogLevel[i+1])
				  {
				    interval.addOneMember(j); 	
				  }
			    }
			
			    //alLogInterval.add(interval);
			    alInterval.add(interval);
			   
			  }
			
			  //points above the largest interval
			  interval = new Interval();
			  interval.setLowPointValue(dLogLevel[iInterval-1]);
			  interval.setMidPointValue(1.5*dLogLevel[iInterval-1]);
			  for (int j=0;j<dValue.length;j++)
			  { 
					  if (Math.log10(dValue[j]) > dLogLevel[iInterval-1])
					  {
					    interval.addOneMember(j); 	
					  }
			  }
			  //alLogInterval.add(interval);	
			  alInterval.add(interval);
			  
			  //get total interval count
			  this.iTotalCount = 0;
			  //for (Interval i: alLogInterval){this.iTotalCount = this.iTotalCount + i.getIntervalCount();}
			  for (Interval i: alInterval){this.iTotalCount = this.iTotalCount + i.getIntervalCount();}		
		}		
	
	public void writeLinearHistogram(FileLogger flOut ){writeHistogram(flOut,alInterval,"Linear Intervals");}
	public void writeLogHistogram(FileLogger flOut){writeHistogram(flOut,alInterval,"Log Intervals");}
		
	/**
	 * writeHistogram   write out histogram (either log or linear)
	 * @param sTitle
	 */
	public void writeHistogram(FileLogger flOut, ArrayList<Interval> alHistogram, String sMessage)
	{
		flOut.logMessage("");
		double dSum = 0.0;
		
		
		//headings
		flOut.logMessage("Histogram,begin");
		flOut.logMessage("Histogram," +  this.getHistogramTitle() + "," + sMessage +   ",histogram beginning point," + dMinHistogram + ",histogram ending," + dMaxHistogram +  ",histogram interval," + dV);
		flOut.logMessage("Histogram, Valid?," + this.bHistogramMade);
		flOut.logMessage("Histogram,chemical," + this.sHistogramChemical + "," + this.dHistogramLevelOfConcern + "," + this.sOutputUnit + "," + this.dRiskLevel + "," + this.dRfC + "," + this.sOutputUnit);
		flOut.logMessage("Histogram,Interval Begining Point,Interval Mid-Point, Interval EndingPoint,  Count, Frequency, Cumulative Frequency");
		
	
		
		//histogram values
		for (Interval i: alHistogram)
		{
			double dLowPointValue = i.getLowPointValue();
			double dMidPointValue = i.getMidPointValue();
			double dHighPointValue = i.getHighPointValue();
			int iCount = i.getIntervalCount();
			double dFrequency = (double)iCount/(double)this.iTotalCount;
			dSum = dSum + dFrequency;
			flOut.logMessage("Histogram," + dLowPointValue + "," + dMidPointValue + "," + dHighPointValue + "," + iCount + "," +  dFrequency + "," + dSum);
		}
		
		//cumulative count and frequency
		flOut.logMessage("Histogram,,,Total Count," + this.iTotalCount + ",Total Cumulative Frequency," + dSum);
		
		flOut.logMessage("Histogram,end");
		
	}	
	
   public void writeCoupledHistgram(FileLogger flOut)
   {
		flOut.logMessage("");
		double dSum = 0.0;
		ArrayList<Interval> alHistogram;
		
		//headings
		flOut.logMessage("Summary Coupled Histogram for:, " + this.getHistogramTitle());
		flOut.logMessage("Summary Coupled Histogram, Interval Mid-Point, Count, Frequency, Cumulative Frequency,Coupled Histograms,");
		
		alHistogram = this.getHistogramIntervals();
		
		//histogram values
		for (Interval i: alHistogram)
		{
			double dMidPointValue = i.getMidPointValue();
			int iCount = i.getIntervalCount();
			double dFrequency = (double)iCount/(double)this.iTotalCount;
			dSum = dSum + dFrequency;
			
			//write out the coupled histogram on several lines:
			Histogram h = i.getCoupledHistogram();
			ArrayList<Interval> alCoupledIntervals = h.getHistogramIntervals();
			double dSumCoupled = 0.0;
			String sOut = "";
			String sOut2 = "Summary Coupled Histogram,,,,,,";
			String sOut3 = "Summary Coupled Histogram,,,,,,";
			String sOut4 = "Summary Coupled Histogram,,,,,,";
			int iIntervalCount = 0;
			for (Interval iCoupled: alCoupledIntervals)
			{	
				sOut = sOut + iCoupled.getMidPointValue() + ",";
				sOut2 = sOut2 + iCoupled.getIntervalCount() + ",";
				int iCountCoupled = iCoupled.getIntervalCount();
				int iTotalCountCoupled = h.getTotalCount();
				//probability density
				double dFrequencyCoupled = (double)iCountCoupled/(double)iTotalCountCoupled;
				sOut3 = sOut3 + "";
				//cumulative probability
				dSumCoupled = dSumCoupled + dFrequencyCoupled;
				sOut3 = sOut3 + dFrequencyCoupled + ",";
				sOut4 = sOut4 + dSumCoupled + ",";
				iIntervalCount++;
			} 
			
			
			if (iIntervalCount!=0)
			{
				//add spacing for histograms of a single interval
				if (iIntervalCount==1)
			    {
			       for (int j=0;j<h.getIntervalFrequency()-1;j++)
			       {	  
			          sOut = sOut + ",";
			          sOut2 = sOut2 + ",";
			          sOut3 = sOut3 + ",";
			          sOut4 = sOut4 + ",";
			        }
			    } 
			
			    //add row Titles for all histograms (having 1 or more intervals)
			    sOut = sOut + "Intervals";
			    sOut2 = sOut2 + "Counts";
			    sOut3 = sOut3 + "Frequency";
			    sOut4 = sOut4 + "Cumulative Frequency";
			}
			
			flOut.logMessage("Summary Histogram," + dMidPointValue + "," +  iCount + "," +  dFrequency + "," + dSum + ",," + sOut);
            flOut.logMessage(sOut2);
            flOut.logMessage(sOut3);
            flOut.logMessage(sOut4);
		}
		
		//cumulative count and frequency
		flOut.logMessage("Summary Histogram,," + this.iTotalCount + ",," + dSum);	   
   }
   
   
    /**
     * read previously written histogram
     * @param sFilePathAndName    file name of previously written histogram
     * @param sType               histogram type to file (typically the chemical name)
     * @param sType2              histogram type:  "Log intervals" or "linear intervals"
     * @throws IOException 
     *                            
     */
	public void readHistogram(String sFilePathAndName, String sType, String sType2) throws IOException
	{
		boolean bHistogramOne = false;
		boolean bHistogramTwo = false;
		boolean bHistogramThree = false;
		boolean bHistogramFour = false;
		boolean bHistogramFive = false;
		boolean bHistogramSix = false;
		Stripper s = new Stripper();
		
		
		FileReader fr = new FileReader(sFilePathAndName);
		BufferedReader br = new BufferedReader(fr);

		
		String currentLine;
		String[] lineArray = {""};
		  		
		//read the first line of the input file.
		//check to see if the file is an output file
		currentLine = br.readLine();
		lineArray = currentLine.split(",",0);
		
		
		 
		 while ((currentLine = br.readLine()) != null) 
		 {
			 lineArray = currentLine.split(",",0);
			 if (lineArray[0].equalsIgnoreCase("Histogram"))
			 {	 
				
				if (bHistogramSix==true && s.removeCharacter(' ', lineArray[1]).equalsIgnoreCase("end"))
				{	
					bHistogramOne = false;
					bHistogramTwo = false;
					bHistogramThree = false;
					bHistogramFour = false;
					bHistogramFive = false;
					
					 
					//correct histogram has been found, exit now.
					return;
				}
	
				else if (bHistogramFive==true && 
						s.removeCharacter(' ', lineArray[1])!="")
				{
					//read the selected histogram data
					Interval i = new Interval();
					System.out.println("histogram entry "+lineArray[1] + lineArray[2] + lineArray[3]);
					try
					{
					  i.setLowPointValue(Double.parseDouble(lineArray[1]));
					  i.setMidPointValue(Double.parseDouble(lineArray[2]));
					  i.setHighPointValue(Double.parseDouble(lineArray[3]));
				      i.setIntervalCount(Integer.parseInt(lineArray[4]));
				      i.setFrequency(Double.parseDouble(lineArray[5]));
				      i.setCumulativeFrequency(Double.parseDouble(lineArray[6]));
						
				      //save this histogram interval
					  alInterval.add(i);
					}
					catch (Exception e)
					{
						
					}
					
					 
					bHistogramSix = true;
				}
				/*
				 * original code
				 * else if (bHistogramThree==true)
					{
						//skip one line
						bHistogramTwo = false;
						bHistogramThree = false;
						bHistogramFour = true;
					}

				 */
					else if (bHistogramFour==true)
					{
						//skip one line
						bHistogramThree = false;
						bHistogramFour = false;
						bHistogramFive = true;
					}
					
					else if (bHistogramThree==true && lineArray[1].equalsIgnoreCase("chemical"))
					{
						//"chemical" and level of concern
						this.sHistogramChemical = lineArray[2];
						try
						{
							this.dHistogramLevelOfConcern = Double.parseDouble(lineArray[3]);
							this.sOutputUnit = lineArray[4];
							this.dRiskLevel = Double.parseDouble(lineArray[5]);
							this.dRfC = Double.parseDouble(lineArray[6]);
                            this.bLevelOfConcernSet=true;
                            this.bRfCSet = true;
						}
						catch (Exception e)
						{
							this.dHistogramLevelOfConcern = 0.0;
							this.dRiskLevel = 0.0;
							this.dRfC = 0.0;
						}
						bHistogramThree=false;
						bHistogramFour = true;
						
					}
					else if (bHistogramTwo==true && s.areTheyEqual("valid?", lineArray[1]))							 
					{	
					  //is the histogram valid? 
						 try
						 {
						   if (s.areTheyEqual("false",lineArray[2])){this.bHistogramMade = false;}
						   else if (s.areTheyEqual("true",lineArray[2])){this.bHistogramMade = true;}
						 }
						 catch (Exception e)
						 {
							 
						 }
						 if (bHistogramMade==true)
						 {	
						   //continue reading the histogram	 
						   bHistogramTwo = false;
						   bHistogramThree = true;
						 }
						 else
						 {
							 //histogram is invalid, exit
							 return;
						 } 
					}							
					else if (bHistogramOne==true  &&  
							s.removeCharacter(' ',lineArray[1]).equalsIgnoreCase(s.removeCharacter(' ', sType)) && 
							s.removeCharacter(' ',lineArray[2]).equalsIgnoreCase(s.removeCharacter(' ', sType2)))
					{
						//use the second entry on the line (lineArray[1]) as a title
						this.sHistogramTitle = lineArray[1];
						//select a linear or log histogram
						//linear histogram
						if (sType2.equalsIgnoreCase("LinearIntervals")){bHistogramLinear = true;}
						//log histogram
						else {bHistogramLinear = false;}
						//histogram is starting for the selected output (sType) (typically a chemical) and histogram type (sType2)
						bHistogramTwo = true;
						bHistogramOne = false;
						
					}
					else if (s.removeCharacter(' ',lineArray[1]).equalsIgnoreCase("begin")) 
				    {
				   	  //a histogram is beginning
					
					  //it may or may not be the correct one
					  bHistogramOne = true;
				    }
			 }
		 }
	 

	}	
	public void makeLogHistogram(double[] dValue, int iCount)
	{
		//develop a histogram for the selected output
		//check range of data
		//JWW 2-10-2016
		double[] dLogLevel=null;
		int iAdd = 0;
		
		boolean bNew = true;
		bHistogramMade = false;
		
		//check for meaningful results, jww 4-11-2016
		for (int i=0;i<dValue.length;i++)
		{
			//no histogram if any NaN values
			if (Double.isNaN(dValue[i])){return;}
			//no histogram if all zeros
			if (dValue[i]==0){iAdd++;}
		}
		if (iAdd==dValue.length){return;}
		
		
		if (bNew)
		{	
		   
		  double xMin, xMax;
		  xMin = 1.e+308;
		  xMax = 1.e-308;
		  for (int i=0;i<dValue.length;i++)
		  {
			  //do not allow zero as the minimum value jww 4-11-2016
			  if (0 < dValue[i] && dValue[i]<xMin){xMin=dValue[i];}
			  if (dValue[i]>xMax){xMax=dValue[i];}
		  }
			  
		  double dLogIntervalLength = Math.log10(xMax) - Math.log10(xMin);
		  
		  int iNumber = 1 + (int)(dLogIntervalLength/0.25);
		  dLogLevel = new double[iNumber];
		  dLogLevel[0] = Math.log10(xMin);
		  for (int i=1;i<iNumber;i++)
		  {
			  dLogLevel[i] = dLogLevel[i-1] + 0.25;
		  }
		  
		}
		else
		{	
		  //log-scale bins
		  //double[] dLogLevel = { -4.5,-4.25,-4.0,-3.75,-3.5,-3.25,-3.0,-2.75,-2.5,-2.25,-2.0,-1.75,-1.5,-1.25,-1.0,-0.75,-0.5,-0.25,
	      //	0.0,
		  //		0.25,0.5,0.75,1.0,1.25,1.5,1.75,2.0,2.25,2.5,2.75,3.0,3.25,3.5,3.75,4.0,4.25,4.5,4.75,5.0,5.25,5.5,5.75,6.0};
		}
		
	    //storage container
		//alInterval = new ArrayList<Interval>();
		bHistogramMade = true;
		this.bHistogramLinear = false;
        
		//number of intervals
		int iInterval=dLogLevel.length;
		
        //first interval
		Interval interval = new Interval();
		interval.setLowPointValue(dLogLevel[0]);
		interval.setMidPointValue(dLogLevel[0] + 0.5*dLogLevel[0]);
		interval.setHighPointValue(1.5*dLogLevel[0]);
		for (int j=0;j<iCount;j++)
		{ 
			  if (Math.log10(dValue[j]) < dLogLevel[0])
			  {
			    interval.addOneMember(j); 	
			  }
		}
		alInterval.add(interval);
		

		  for (int i=1;i<iInterval-1;i++)
		  {	
		    interval = new Interval(); 
		    interval.setLowPointValue(dLogLevel[i]);
		    interval.setMidPointValue(0.5*(dLogLevel[i] + dLogLevel[i+1]));
		    interval.setHighPointValue(dLogLevel[i+1]);
		  
	  	    //divide results among histogram compartments
		    for (int j=0;j<iCount;j++)
		    {
		      double dLogVal = Math.log10(dValue[j]);
			  if (dLogLevel[i]<dLogVal && dLogVal<=dLogLevel[i+1])
			  {
			    interval.addOneMember(j); 	
			  }
		    }
		
		    alInterval.add(interval);
		   
		  }
		
		  
		  if (iInterval>3)
		  {	  
			  //points above the largest interval
			  interval = new Interval();
			  double dPreviousInterval = dLogLevel[iInterval-2]-dLogLevel[iInterval-3];
			    interval.setLowPointValue(dLogLevel[iInterval-1]);
				interval.setMidPointValue(dLogLevel[iInterval-1] + 0.5*dPreviousInterval);
				interval.setHighPointValue(dLogLevel[iInterval-1] + 1.0*dPreviousInterval);
				
				for (int j=0;j<iCount;j++)
				{ 
					  if (Math.log10(dValue[j]) > dLogLevel[iInterval-1])
					  {
					    interval.addOneMember(j); 	
					  }
				}
				alInterval.add(interval);
		  }	
			
			//get total interval count
			iTotalCount = 0;
			for (Interval i: alInterval){iTotalCount = iTotalCount + i.getIntervalCount();}
	}		
	
	/**
	 * determine the level of concern (usually used for cancer risk)
	 */
	public void determineLevelOfConcern(String sHistogramChemical, double [] dRiskConcentration, double[] dRiskValue, String[] sChemicalName, int iNumberRisk)
	{
	   //data were collected in the outputProcessor as the main output file was read (iNumberRisk, dRiskValue, sChemicalName)
	   for (int i=0;i<iNumberRisk;i++)
	   {
		   if (sHistogramChemical.equalsIgnoreCase(sChemicalName[i]))
		   {
			   this.dHistogramLevelOfConcern = dRiskConcentration[i];
			   this.dRiskLevel = dRiskValue[i];
			   this.bLevelOfConcernSet = true;
			   break;
		   }
	   }
	}
	
	public void determineRfC(String sHistogramChemical, double [] dRfC,  String[] sChemicalName, int iNumberRfC)
	{
		 //data were collected in the outputProcessor as the main output file was read (iNumberRisk, dRiskValue, sChemicalName)
		   for (int i=0;i<iNumberRfC;i++)
		   {
			   if (sHistogramChemical.equalsIgnoreCase(sChemicalName[i]))
			   {
				   this.dRfC = dRfC[i];
				   this.bRfCSet = true;
				   break;
			   }
		   }
	}
	
	//determine cumulative probability of a concentration
	//concentration is in normal units, bNormalConcentrationUnit = true
	//concentration is log unit , bNormalConcentrationUnit = false;
	public double getCumulativeProbability(double dConcentrationToCheck, boolean bNormalConcentrationUnit)
	{
		 double dReturn=0;
		 double dLow=0,dCFLow=0;
		 double dHigh=0,dCFHigh=0;
		 double dLowOld=0,dCFLowOld=0;
		 double dLogLOC;
	
		 //input concentration (dConcentrationToCheck) is in normal units
		 //convert to log base 10
		 dLogLOC = dConcentrationToCheck;
		 if (bNormalConcentrationUnit)
		 {	 
		   dLogLOC = Math.log10(dConcentrationToCheck);
		 }
		 
		 //is the test value within the range of the histogram?
		 //is the test value below the lowest value?
		 if (dLogLOC<alInterval.get(0).getMidPointValue())
		 {
		    return  0.0; 
		 }
		 //is the test value above the highest value?
		 if (dLogLOC>alInterval.get(alInterval.size()-1).getMidPointValue())
		 {
			 return  1.0;
		 }
		 
		 //step through intervals to find bounding interval
		 for (Interval i:  alInterval)
		 {
			 dLow = i.getMidPointValue();
			 dCFLow = i.getCumulativeFrequency();
			 if (dLow>dLogLOC)
			 {
				 dHigh = i.getMidPointValue();
				 dCFHigh = i.getCumulativeFrequency();
				 dLow = dLowOld;
				 dCFLow = dCFLowOld;
				 break;
			 }
			 dLowOld = dLow;
			 dCFLowOld = dCFLow;
		 }
		 
		 //estimate the cumulative frequency of the level of concern
		 if (dLow<=dLogLOC && dLogLOC<=dHigh)
		 {
			 double sl = (dCFHigh-dCFLow)/(dHigh-dLow);
			 double dCumulativeFrequency = dCFLow + (dLogLOC-dLow)*sl;
			 if (dCumulativeFrequency<0.00001){dCumulativeFrequency = 0.0;}
			 else if (dCumulativeFrequency>1.0){dCumulativeFrequency = 1.0;}
			 dReturn = dCumulativeFrequency;
		 }
		 
		 return dReturn;
		 
	}	
	
	//determine the cumulative probability of the Level Of Concern
	public void determineCumulativeProbabilityOfTheLevelOfConcern(double dConcentrationToCheck)
	{
		 double dLow=0,dCFLow=0;
		 double dHigh=0,dCFHigh=0;
		 double dLowOld=0,dCFLowOld=0;
	
		 //input concentration (dConcentrationToCheck) is in normal units
		 //convert to log base 10
		 double dLogLOC;
		 dLogLOC = Math.log10(dConcentrationToCheck);
		 
		 
		 //is the test value within the range of the histogram?
		 //is the test value below the lowest value?
		 if (dLogLOC<alInterval.get(0).getMidPointValue())
		 {
			dHistogramLevelOfConcernCumulativeFrequency = 0.0; 
			return;
		 }
		 //is the test value above the highest value?
		 if (dLogLOC>alInterval.get(alInterval.size()-1).getMidPointValue())
		 {
			 dHistogramLevelOfConcernCumulativeFrequency = 1.0;
			 return;
		 }
		 
		 //step through intervals to find bounding interval
		 for (Interval i:  alInterval)
		 {
			 dLow = i.getMidPointValue();
			 dCFLow = i.getCumulativeFrequency();
			 if (dLow>dLogLOC)
			 {
				 dHigh = i.getMidPointValue();
				 dCFHigh = i.getCumulativeFrequency();
				 dLow = dLowOld;
				 dCFLow = dCFLowOld;
				 break;
			 }
			 dLowOld = dLow;
			 dCFLowOld = dCFLow;
		 }
		 
		 //estimate the cumulative frequency of the level of concern
		 if (dLow<=dLogLOC && dLogLOC<=dHigh)
		 {
			 double sl = (dCFHigh-dCFLow)/(dHigh-dLow);
			 double dCumulativeFrequency = dCFLow + (dLogLOC-dLow)*sl;
			 if (dCumulativeFrequency<0.00001){dCumulativeFrequency = 0.0;}
			 else if (dCumulativeFrequency>1.0){dCumulativeFrequency = 1.0;}
			 dHistogramLevelOfConcernCumulativeFrequency = dCumulativeFrequency;
		 }
		 
		 
		 
	}
	
	double dHighestProbabilityDensity=-1.0;
	//histogram values
	
	//histogram interval with highest probability density
	//minimum value of histogram specified as dMinValueToReport
	//set dMinValueToReport = 0 for not restriction
	//otherwise this is used with graphing and a desire to set a minimum plot value
	public double getIntervalWithHighestProbabilityDensity()
	{
		int iCount, iTotalCount;
		double dMaxInterval;
		double dFrequency;
		double dTest;
	 
		iTotalCount = 0;
		iCount = 0;
		dHighestProbabilityDensity = 0.0;
		dMaxInterval = 0.0;
		
		//get the total count of points (sum of the count in each interval)
		for (Interval i: this.alInterval)
		{ 
			iCount = i.getIntervalCount();
		    iTotalCount = iTotalCount + i.getIntervalCount();
		} 
		
	    //find the maximum probability density
		dFrequency = 0.0;
		for (Interval i: this.alInterval)
		{ 
			iCount = i.getIntervalCount();
			dFrequency = (double)iCount/(double)iTotalCount;
		    
		    //if (dFrequency>=dHighestProbabilityDensity && i.getMidPointValue()>=dMinValueToReport)
		    if (dFrequency>=dHighestProbabilityDensity)
		    {
		    	dHighestProbabilityDensity = dFrequency; 
		    	dMaxInterval=i.getMidPointValue();
		    }
		} 
		
		return dMaxInterval;
	}
	
	/*
   public double getHighestProbabilityDensity()
   {
	   if (dHighestProbabilityDensity<0.0){double dMx = this.getIntervalWithHighestProbabilityDensity();}
	   return dHighestProbabilityDensity;
   }
   */
   
   //get the concentration (or value) for a given frequency
   public double getValueForCumulativeFrequency(double dFrequency)
   {	   
		double dValue=0.0;
	
		//check first interval first
		int iInterval = alInterval.size();
		
		if (iInterval>0)
		{	
			Interval iFirst = alInterval.get(0);
			Interval iLast = alInterval.get(iInterval-1);
			double dFreq0 = iFirst.getCumulativeFrequency();
			double dFreqN = iLast.getCumulativeFrequency();
			if (dFrequency<dFreq0)
			{
				dValue = iFirst.getMidPointValue();
			}
			else if (dFrequency>dFreqN)
			{
				dValue = iLast.getMidPointValue();
			}
			else
			{
			  //search between the intervals to estimate the concentration at the specified frequency (dFrequency) 
		      for (int i=0;i<iInterval-1;i++)
		      {
		        Interval interval1 = alInterval.get(i);
		        Interval interval2 = alInterval.get(i+1);
		        double dFreq1 = interval1.getCumulativeFrequency();
		        double dFreq2 = interval2.getCumulativeFrequency();
		        if (dFreq1<= dFrequency && dFrequency <= dFreq2)
		        {
		    	  //found it
		    	  double dConc1 = interval1.getMidPointValue();
		    	  double dConc2 = interval2.getMidPointValue();
		    	  dValue = dConc1 + (dFrequency-dFreq1)*(dConc2-dConc1)/(dFreq2-dFreq1);
		    	  break;
		        }
		      
		      }
			}
		}
	   return dValue;
   }
	
   //sets
   public void setIntervalFrequency(int iFrequencyInterval){this.iFrequencyInterval = iFrequencyInterval;}
   public void setHistogramTitle(String sHistogramTitle){this.sHistogramTitle =  sHistogramTitle;}
   public void setHistogramChemical(String sHistogramChemical){this.sHistogramChemical = sHistogramChemical;}
   public void setRiskLevel(double dRiskLevel){this.dRiskLevel = dRiskLevel;}
   public void setLevelOfConcern(double dHistogramLevelOfConcern){this.dHistogramLevelOfConcern = dHistogramLevelOfConcern;this.bLevelOfConcernSet=true;}
   public void setCoupledHistogram(boolean bCoupled){this.bCoupled = bCoupled;}
   public void setCoupledHistogramData(double[] dValue2){this.dValue2 = dValue2;}
   public void setOutputUnit(String s){this.sOutputUnit = s;}
   public void setRiskContainer(RiskContainer rc){this.rc = rc;}
   
   //gets
   public int getIntervalFrequency(){return this.iFrequencyInterval;}
   public boolean getIsHistogramMake(){return this.bHistogramMade;}
   public ArrayList<Interval> getHistogramIntervals(){return alInterval;}
   public String getHistogramTitle(){return this.sHistogramTitle;}
   public String getHistogramChemical(){return this.sHistogramChemical;}
   public double getLevelOfConcern(){return this.dHistogramLevelOfConcern;}
   public boolean getLevelOfConcernSet(){return this.bLevelOfConcernSet;}
   public int getTotalCount(){return this.iTotalCount;}
   public double getHistogramLevelOfConcernCumulativeFrequency(){return this.dHistogramLevelOfConcernCumulativeFrequency;}
   public double getRiskLevel(){return this.dRiskLevel;}
   public double getRfC(){return this.dRfC;}
   public String getOutputUnit(){return this.sOutputUnit;}
   public boolean doesHistogramExist(){return bHistogramMade;}
   public double getHighestProbabilityDensity(){return this.dHighestProbabilityDensity;}
   
}
