package statistics;

import util.FileLogger;
import util.VersionCompilation;
import numerics.Bisection.DiscreteBisection;

/**
 * Resistant Statistical Measures
 * 
 * For calculation of resistant statistical measures of a dataset:
 *   minimum, maximum, median, 1st quantile, 3rd quantile, interquantile range,
 *   5th percentile, 95th percentile.
 * 
 *   Introduction to the Practice of Statistics, 2nd ed., 1993
 *   David S. Moore and George P. McCabe
 *   W.H. Freeman & Co., New York, 854 pp.
 *   See pages 37-43.
 *   
 *   See also US EPA laboratory Notebook   Ada JWW-02, pp 128-130.
 * 
 * @author Jim Weaver
 * 
 * 
 * Created 10-29-2012
 *
 */
public class ResistantStatisticalMeasures 
{
	//name
	private String sName;
	//data points
    double[] dX;
    //median value
    double dMedian;
    //minimum and maximum values
    double dMinimum, dMaximum;
    //first and third quartiles
    double dFirstQuartile, dThirdQuartile;
    //interquartile range
    double dInterQuartileRange;
    //5th and 95th percentiles
    double d5thPercentile, d95thPercentile;
    //index of median if odd number of points
    int iMedian;
    //file logger for output 
    private FileLogger fl;
    
	
	public ResistantStatisticalMeasures()
	{
		this.sName = "RSM";
	}
	
	
	/**
	 * determine the resistant statistical measures
	 */
	public void determineMeasures()
	{
		//clear old results
		this.clear();
		
		//sort data from low to high
		this.sort();
		//find the minimum and maximum values
		this.findMinMax();
		//find the median
		this.findMedian();
		//find the first quartile/25th percentile
		this.findFirstQuartile();
		//find the third quartile/75th percentile
	    this.findThirdQuartile();
	    //find the interquartile range
	    this.findInterQuartileRange();
        //find the approximate 5th percentile
	    this.find5thPercentile();
	    //find the approximate 95th percentile
		this.find95thPercentile();
	}
	
	
	/**
	 * sort the data
	 */
	public void sort()
	{
		double dValueTemp;
		boolean bChanged;
		    
		bChanged = true;
		//now the sorting starts:
		while (bChanged == true) 
		{
			 bChanged = false;
		     for (int i=0;i<dX.length-1;i++)
		     {
			    //sort from lowest to highest 
			    if (dX[i]>dX[i+1])
			    {
				  dValueTemp = this.dX[i+1];
				  this.dX[i+1] = this.dX[i];
				  this.dX[i] = dValueTemp;
				  bChanged = true;
			    }
		      }
		   }  	
	}
	
	/**
	 * determine the median of the data, using private calculateMedian function
	 */
	public void findMedian()
	{
		dMedian = calculateMedian(this.dX);
	}
	
	/**
	 * determine the median of the data
	 * 
	 *  (data are passed explicitly so the method can be used to determine quartiles
	 *  which are medians of the lower and upper half of the data)
	 *  
	 */
	public double calculateMedian(double[] dX)
	{
		boolean bEven;
		int iCount;
		double dMedianInternal;
		
		iCount = dX.length;
		//determine if the number of points is odd or even
		bEven = isEven(iCount);
		
		if (bEven)
		{
			//there are an even number of data points
			//the median is average of the two middle points
			dMedianInternal = 0.5*(dX[(iCount/2) - 1] + dX[iCount/2]); 
		}
		else
		{
			//there are an odd number of data points
			//the median is the middle point
			dMedianInternal = dX[(iCount-1)/2];
		}
		return dMedianInternal;
	}
	
	
	
	/**
	 * find the first quartile of the data
	 *
	 */
	public void findFirstQuartile()
	{
		int iEnd,iLength;
		iLength = this.dX.length;
		double[] dTemp;
	 
		
		//find the median of the partial data set below the median of the entire data set
		boolean bIsEven = isEven(iLength);
		
		
		if (bIsEven)
		{
			//even number of points
			//ending index of array
			iEnd = iLength/2 - 1; 
		}
		else
		{
		    //odd number of points
			//ending index of array
			iEnd = (iLength-1)/2 - 1;
		}
		
		dTemp = new double[iEnd+1];
		for (int i=0;i<iEnd+1;i++){dTemp[i] = dX[i];}
		dFirstQuartile = calculateMedian(dTemp);
	}
	
	/**
	 * find the first quartile of the data
	 *
	 */
	public void findThirdQuartile()
	{
		int iBegin,iLength;
		iLength = this.dX.length;
		double[] dTemp;
	 
		//find the median of the partial data set below the median of the entire data set
		boolean bIsEven = isEven(iLength);
		
		if (bIsEven)
		{
			//even number of points
			iBegin = iLength/2; 
		}
		else
		{
		    //odd number of points
			iBegin = (iLength-1)/2 + 1;
		}
		
		int iDimension = iLength-iBegin;
		dTemp = new double[iDimension];
		for (int i=iBegin;i<iLength;i++){dTemp[i-iBegin] = dX[i];}
		dThirdQuartile = calculateMedian(dTemp);
	}
	
	/**
	 * find the interquartile range (IQR)
	 * IQR = Q3 - Q1
	 */
	public void findInterQuartileRange()
	{
	   this.dInterQuartileRange = this.dThirdQuartile - this.dFirstQuartile;
	}
	
	/**
	 * find value for the fifth percentile
	 */
	public void find5thPercentile(){this.d5thPercentile = findPercentile(5);}
	 
	/**
	 * find value for the 95th percentile
	 */
	public void find95thPercentile(){this.d95thPercentile = findPercentile(95);}
	
	/**
	 * find a specified percentile
	 * @param dPercentile  percentile desired
	 * @return  value at specified percentile
	 */
	public double findPercentile(double dPercentile)
	{
		DiscreteBisection db = new DiscreteBisection();
		
		double[] dFrequency = new double[dX.length];
		for (int i=0;i<dFrequency.length;i++)
		{
			dFrequency[i] = (double)i/(double)dFrequency.length;
		}
		
		db.setDiscreteBisectableArray(dFrequency);
		db.setValue(dPercentile/100.);
		db.bisect();
		int iSolution = db.getIndexOfSolution();
		//discrete bisection returns the value just below the value requested
		//the percentile calculation is designed to give the value just above the requested value:
		//so that the percentile value gives the lowest value in the distribution that
		//is greater than the percentile indicated:  i.e., there is at least as much as requested
		iSolution = iSolution + 1;
		
		
		return dX[iSolution];
	}
	
	/**
	 * find the minimum and maximum of the data
	 */
	public void findMinMax()
	{
	   dMinimum = 1.e200;
	   dMaximum = -1.e200;
	   
	   for (int i=0;i<dX.length;i++)
	   {
		   if (dX[i]<dMinimum){dMinimum = dX[i];}
		   if (dX[i]>dMaximum){dMaximum = dX[i];}
	   }
	}
	
	
	/**
	 * determine if a number is even or odd
	 * @iNumber   integer to check
	 * @return  true if the input variable is even
	 */
	private boolean isEven(int iNumber)
	{
		int iTest;
		double dTest;
		double dTolerance = 0.0001;
		
		//test for even or odd by use of integer division
		dTest = 0.5*(double)iNumber;
		iTest = iNumber/2;
		
		if (dTest-(double)iTest > dTolerance)
		{
			//number is odd (difference is around 0.5)
			return false;
		}
		else
		{
			//number is even (difference is around 0.0)
			return true;
		}
	}
	
	/**
	 * clear all results and data
	 * 
	 */
	public void clear()
	{
		this.d5thPercentile = 0.;
		this.d95thPercentile = 0.;
		this.dMinimum = 0.0;
		this.dMaximum = 0.0;
		this.dMedian = 0.0;
		this.dFirstQuartile = 0.0;
		this.dThirdQuartile = 0.0;
		this.dInterQuartileRange = 0.0;
	}
	
	//output
	public void writeResistantMeasuresOutput()
	{
		double dOut;
		String sOut;
		//write the outputs describing the resistant statistical measures
		fl.logMessage("");
		
		sOut = "Resistant Statistical Measures Output";
		fl.logMessage(sOut);
		String sOutput = sOut + ",Minimum,";
		sOutput = sOutput + this.dMinimum;
		fl.logMessage(sOutput);
		
		sOutput = sOut + ",5th Percentile,";
		sOutput = sOutput + this.d5thPercentile;
		fl.logMessage(sOutput);
		
		sOutput = sOut + ",Median - 1.5 InterQuartileRange,";
		dOut = this.dMedian - 1.5*this.dInterQuartileRange;
		sOutput = sOutput + dOut;
		
		sOutput = sOut + ",First Quartile/25th Percentile,";
		sOutput = sOutput + this.dFirstQuartile;
		fl.logMessage(sOutput);
		
		sOutput = sOut + ",Median,";
		sOutput = sOutput + this.dMedian;
		fl.logMessage(sOutput);
		
		sOutput = sOut + ",Third Quartile/75th Percentile,";
		sOutput = sOutput + this.dThirdQuartile;
		fl.logMessage(sOutput);
		
		sOutput = sOut + ",Median + 1.5 InterQuartileRange,";
		dOut = this.dMedian + 1.5*this.dInterQuartileRange;
		sOutput = sOutput + dOut;
		fl.logMessage(sOutput);
		
		sOutput = sOut + ",95th Percentile,";
		sOutput = sOutput + this.d95thPercentile;
		fl.logMessage(sOutput);
		
		sOutput = sOut + ",Maximum,";
		sOutput = sOutput + this.dMaximum;
		fl.logMessage(sOutput);

	}
	
	  /**
	   * add version information to the models VersionCompilation object
	   * @param vc  VersionCompilation object
	   */
	  public void setVersionInformation(VersionCompilation vc)
	  {
		 vc.setFileData("ResistantStatisticalMeasures ",
				 this.sName,
				 "Development Version 0.00",
				 "c:\\Documents and Settings\\jweaver\\workspace\\RiverModel\\src\\util\\ResistantStatisticalMeasures.java",
                 "c:\\Documents and Settings\\jweaver\\workspace\\RiverModel\\bin\\util\\ResistantStatisticalMeasures.class");						 
	  }
	  	
	
	//sets
	public void setData(double[] dX){this.dX = dX;}
	public void setFileLogger(FileLogger fl){this.fl = fl;}
	
	
	//gets
	public double getMaximum(){return dMaximum;}
	public double getMinimum(){return dMinimum;}
	public double getMedian(){return dMedian;}
	public double getFirstQuartile(){return this.dFirstQuartile;}
	public double getThirdQuartile(){return this.dThirdQuartile;}
	public double getInterQuartileRange(){return this.dInterQuartileRange;}
	public double get5thPercentile(){return this.d5thPercentile;}
	public double get95thPercentile(){return this.d95thPercentile;}
	public double[] getData(){return this.dX;}
	
	
}
