package statistics;

import java.util.ArrayList;

import statistics.Histogram;
import util.MergeSort;
import util.Stripper;


/**
 * class ResultsColumn
 * 
 * class to hold 1 column of Output (heading, unit, column of values)
 * and the results of the simple statistical analysis:  min, max, average, variance, range
 * @author JWEAVER
 *
 */
public class ResultsColumn 
{

	    //is the data input? (alternatively output)
	    private boolean bInput;
		//does the data column consist of only double values
		private boolean bDoubleDataColumn;
		//have the statistical tests been performed on the double data?
		private boolean bStatisticsSet;
		//has the Cumulative frequency distribution been set?
		private boolean bCumulativeDistributionSet;
		//column heading (alt. title)
		private String sHeading;
		//extension to column heading (used for subordinate results column)
		private String sHeadingExtension;
		//unit of measure for column, if applicable
		private String sUnit;
		//number of elements in array
		private int iNumber;
		//number of points actually used in statistical analysis
		private int iNumberStatistics;
		//index of current element
		private int iCount;
		//double value of current element
		private double[] dValue;
		//string value of current element // if not a number set to the control character // jww 4-5-2016
		private String[] sValue;
		//string value of the original entry, not to be changed // jww 4-5-2016
		private String[] sOriginalValue;
		//control character:  this character indicates corresponding double is to be ignored
		//typically * is used as the control character
		private String sControlCharacter;
		//double value of validated data 
		//(same as entries in dValue, but only those with control character in sValue)
		//dimensioning of dValidated and dValidatedFrequency exactly match number of elements
		private double[] dValidated;
		//cumulative distribution frequencies for validated data (dValidated)
		private double[] dValidatedFrequency;
		//normalized data
		private double[] dValidatedNormalized;
		//container for subordinate results columns
		ArrayList<ResultsColumn> alSubordinateResultsColumn;
		//stripper utility for string comparisons
		private Stripper s;
		
		
		//simple statistical measures
		double dMin;
		double dMax;
		double dAverage;
		double dTotal;
		double dVariance;
		double dRange;
		
		//resistant statistical measures
		double d5thPercentile;
		double d95thPercentile;
		double dFirstQuartile;
		double dThirdQuartile;
		double dMedian;
		double dInterQuartileRange;
		double dRSMMinimum;
		double dRSMMaximum;
		
		//confidence intervals
		double d90LowConfidenceInterval, d90HighConfidenceInterval;
		double d95LowConfidenceInterval, d95HighConfidenceInterval;
		double d99LowConfidenceInterval, d99HighConfidenceInterval;

		//Histogram
		Histogram histogram;
		
		public ResultsColumn()
		{
			bInput = false;
			//prepare for the first data item
			iCount = 0;
			//does the data column truly contain data?
			this.bDoubleDataColumn = true;
			//have the statistics been calculated?
			this.bStatisticsSet = false;
			//have the cumulative distribution been set?
			this.bCumulativeDistributionSet = false;
			//set up container for subordinate results columns
			this.alSubordinateResultsColumn = new ArrayList<ResultsColumn>();
			//string comparison utility
			s = new Stripper();
		}
		
		//initialize data array
		public void initializeArray()
		{
			this.dValue = new double[1];
		    this.sValue = new String[1];
		    this.sOriginalValue = new String[1];
		    //need this array initialized, but we will not fill until later
		    this.dValidatedFrequency = new double[1];
		}
		
		//add a data value
		public void addValue(double dValueNew, String sValueNew, String sOriginalValue)
		{
		   
		  if (iCount  < dValue.length)	
		  {
			  this.dValue[iCount] = dValueNew;
		      this.sValue[iCount] = sValueNew;
		      this.sOriginalValue[iCount] = sOriginalValue;
		      iCount ++;
		  }
		  else
		  {
			  //increase dimensioning by one
			  //this should happen for each point added
			  //the consequences will be that the dimensioning of
			  //dValue and sValue will exactly match the number of elements
			  double[] dTemp = new double[dValue.length];
			  String[] sTemp = new String[sValue.length];
			  String[] sTempO = new String[this.sOriginalValue.length];
			  int iLength = dValue.length;
			  //save all existing array values
			  for (int i=0;i<iLength;i++)
			  {
				  dTemp[i] = this.dValue[i];
				  sTemp[i] = this.sValue[i];
				  sTempO[i] = this.sOriginalValue[i];
			  }
			  //add one new entry in the array
			  this.dValue = new double[iLength+1];
			  this.sValue = new String[iLength+1];
			  this.sOriginalValue = new String[iLength+1];
			  //make the dimensioning of the frequency variable match the data variable dimension
			  //(values added later)
			  this.dValidatedFrequency = new double[iLength+1];
			  //restore all existing array values
			  for (int i=0;i<iLength;i++)
			  {
				  this.dValue[i] = dTemp[i];
				  this.sValue[i] = sTemp[i];
				  this.sOriginalValue[i] = sTempO[i];
			  }
			  //add new array value to the newly created array element
			  this.dValue[this.dValue.length-1] = dValueNew;
			  this.sValue[this.sValue.length-1] = sValueNew;	
			  this.sOriginalValue[this.sOriginalValue.length-1] = sOriginalValue;
			  iCount++;
		  }
		}
		
		/**
		 * validateData    reduce the data to an array that is dimensioned to the number of 
		 *                 data points
		 */
		public void validateData()
		{
			//check to see if this is a data column
			//data columns have control strings set to the control character
			
			//only validate if the column has not been previously determined to contain invalid data
			//(i.e., a column of labels may already be invalid, as is the first column which all say result)
			//JWW  1-15-2016
			if (bDoubleDataColumn)
			{	
				int iValidCount = 0;
				for (int i=0;i<dValue.length;i++)
				{
					if (sValue[i].equalsIgnoreCase(sControlCharacter)){iValidCount++;}
				}
				
				//set flag for data column
				if (iValidCount>0){this.bDoubleDataColumn=true;}
				else{this.bDoubleDataColumn=false;}
				
				//condense valid data into new array, but save the old
				//dimension of new array exactly matches number of entries
				this.dValidated = new double[iValidCount];
				this.dValidatedFrequency = new double[iValidCount];
				
				//pack valid data into new array
				int iCountHere = 0;
				for (int i=0;i<dValue.length;i++)
				{
					if (sValue[i].equalsIgnoreCase(sControlCharacter))
					{
						this.dValidated[iCountHere] = dValue[i];
					    iCountHere++;
					}    
				}
			}
		}
		
		
		public void addSubordinateResult(String sHeadingExtension)
		{
			this.sHeadingExtension = sHeadingExtension;
			ResultsColumn rc = new ResultsColumn();	
			rc.setHeading(this.getHeading() + sHeadingExtension );
			rc.setHeadingExtension(sHeadingExtension);
			rc.setControlCharacter(this.getControlCharacter());
			rc.initializeArray();
			rc.setIsThisADoubleDataColumn(this.getIsThisADoubleDataColumn());
			 	
			alSubordinateResultsColumn.add(rc);
		}
		
		public void addToSubordinateResultColumn(String sHeadingExtension, double dValue)
		{
			for (ResultsColumn rc: alSubordinateResultsColumn)
			{
				if (s.areTheyEqual(rc.getHeading(), this.getHeading()+sHeadingExtension))
				{	
					rc.addValue(dValue, Double.toString(dValue),Double.toString(dValue));
					break;
				}	
			}
		}
		
		public ArrayList<ResultsColumn> getSubordinateResultsColumns(){return this.alSubordinateResultsColumn;} 
		public ResultsColumn getSubordinateResultsColumn(String sHeadingExtension)
		{
			ResultsColumn rcReturn = new ResultsColumn();
			for (ResultsColumn rc:  this.alSubordinateResultsColumn)
			{
				if (s.areTheyEqual(sHeadingExtension, rc.getHeadingExtension()))
				{
				  rcReturn = rc;
				  break;
				}
			}
			return rcReturn;
		}
		
		//generate a qualitative characterization of the range of inputs jww 4-1-2016
		private String[] sCharacterization = {"Lowest","Low","Middle","High","Highest"};
		private String[] sCharacterizedData;
		private double[] dCharacterizationBreakPoints = {0.1,0.25,0.75,0.9};
        private boolean bCharacterized = false;
		public void characterizeData()
		{
			if (this.bDoubleDataColumn && this.bStatisticsSet)
			{
				if (this.dMax!=0.0)
				{
					dValidatedNormalized = new double[dValidated.length];
					sCharacterizedData = new String[dValidated.length];
					for (int i=0;i<dValidatedNormalized.length;i++)
					{
						dValidatedNormalized[i] = dValidated[i]/dMax;
					}
					
					//characterize the Normalized data
					for (int i=0;i<dValidatedNormalized.length;i++)
					{
						if (dValidatedNormalized[i]<=dCharacterizationBreakPoints[0]){sCharacterizedData[i]=sCharacterization[0];}
						for (int j=1;j<sCharacterization.length-1;j++)
						{
							if (dCharacterizationBreakPoints[j-1] < dValidatedNormalized[i]  &&
								dValidatedNormalized[i] <= dCharacterizationBreakPoints[j] )
							{ 
								sCharacterizedData[i] = sCharacterization[j];	
							}
						}
						if (dCharacterizationBreakPoints[dCharacterizationBreakPoints.length-1] < dValidatedNormalized[i])
						{
							sCharacterizedData[i] = sCharacterization[sCharacterization.length-1];
						}
					}
					this.bCharacterized=true;
					
				}
			}
		}
		
		public String getCharacterizedValue(int iIndex)
		{
			if (this.bCharacterized){return sCharacterizedData[iIndex];}
			else {return "";}
		}
		
		public void sortByExternalData(double[] dSortBy)
		{
			MergeSort ms = new MergeSort();
			ms.sortWithDependentDouble(dSortBy,this.dValidated);
		}
		
		
		//sets
		public void setInputOrNot(boolean bInput){this.bInput = bInput;}
		public void setHeading(String sHeading){this.sHeading = sHeading;}
		public void setHeadingExtension(String sHeadingExtension){this.sHeadingExtension = sHeadingExtension;}
		public void setUnit(String sUnit){this.sUnit = sUnit;}
		public void setNumber(int iNumber){this.iNumber = iNumber;}
		public void setNumberOfPointsUsedInStatisticalAnalysis(int i){this.iNumberStatistics = i;}
		public void setValue(int iIndex, double dValue){this.dValue[iIndex]=dValue;}
		public void setAllDoubleValues(double[] dValue){this.dValue = dValue;}
		public void setAllStringValues(String[] sValue){this.sValue = sValue;}
		public void setControlCharacter(String sControlCharacter){this.sControlCharacter = sControlCharacter;}
		public void setIsThisADoubleDataColumn(boolean bDoubleDataColumn){this.bDoubleDataColumn = bDoubleDataColumn;}
		public void setMax(double dMax){this.dMax = dMax;}
		public void setMin(double dMin){this.dMin = dMin;}		
		public void setAverage(double dAverage){this.dAverage = dAverage;}	
		public void setTotal(double dTotal){this.dTotal = dTotal;}
		public void setVariance(double dVariance){this.dVariance = dVariance;}	
		public void setRange(double dRange){this.dRange = dRange;}	
		public void setHaveTheStatisticsBeenCalculated(boolean bStatisticsSet){this.bStatisticsSet = bStatisticsSet;}
		public void setMedian(double dMedian){this.dMedian = dMedian;}
		public void setFirstQuartile(double dFirstQuartile){this.dFirstQuartile = dFirstQuartile;}
		public void setThirdQuartile(double dThirdQuartile){this.dThirdQuartile = dThirdQuartile;}
		public void setInterQuartileRange(double dInterQuartileRange){this.dInterQuartileRange = dInterQuartileRange;}
		public void set5thPercentile(double d5thPercentile){this.d5thPercentile = d5thPercentile;}
		public void set95thPercentile(double d95thPercentile){this.d95thPercentile = d95thPercentile;}		
		public void setRSMMinimum(double dRSMMinimum){this.dRSMMinimum = dRSMMinimum;}
		public void setRSMMaximum(double dRSMMaximum){this.dRSMMaximum = dRSMMaximum;}
		public void set90LowConfidenceInterval(double d90LowConfidenceInterval){this.d90LowConfidenceInterval = d90LowConfidenceInterval;}
		public void set90HighConfidenceInterval(double d90HighConfidenceInterval){this.d90HighConfidenceInterval = d90HighConfidenceInterval;}	
		public void set95LowConfidenceInterval(double d95LowConfidenceInterval){this.d95LowConfidenceInterval = d95LowConfidenceInterval;}
		public void set95HighConfidenceInterval(double d95HighConfidenceInterval){this.d95HighConfidenceInterval = d95HighConfidenceInterval;}	
		public void set99LowConfidenceInterval(double d99LowConfidenceInterval){this.d99LowConfidenceInterval = d99LowConfidenceInterval;}
		public void set99HighConfidenceInterval(double d99HighConfidenceInterval){this.d99HighConfidenceInterval = d99HighConfidenceInterval;}	
		public void setHistogram(Histogram h){this.histogram = h;}	
		public void setIsTheCumulativeDistributionSet(boolean b){this.bCumulativeDistributionSet=b;}
        public void setCumulativeDistributionFrequencies(double[] dFreq){this.dValidatedFrequency = dFreq;}
        public void setValidatedData(double[] dValidated){this.dValidated = dValidated;}
		
		//gets
        public boolean getInputOrNot(){return this.bInput;}
		public String getHeading(){return this.sHeading;}
		public String getUnit(){return this.sUnit;}
		public int getNumber(){return iNumber;}
		public int getNumberOfPointsUsedInStatisticalAnalysis(){return iNumberStatistics;}
		public double getValue(int iIndex){return dValue[iIndex];}
		public double[] getAllDoubleValues(){return dValue;}
		public String[] getAllStringValues(){return sValue;}
		public String getControlCharacter(){return sControlCharacter;}
		public boolean getIsThisADoubleDataColumn(){return this.bDoubleDataColumn;}
		public double getMin(){return this.dMin;}
		public double getMax(){return this.dMax;}		
		public double getAverage(){return this.dAverage;}	
		public double getTotal(){return this.dTotal;}
		public double getVariance(){return this.dVariance;}		
		public double getRange(){return this.dRange;}
		public boolean getHaveTheStatisticsBeenSet(){return bStatisticsSet;}
		public double getMedian(){return dMedian;}
		public double getFirstQuartile(){return this.dFirstQuartile;}
		public double getThirdQuartile(){return this.dThirdQuartile;}
		public double getInterQuartileRange(){return this.dInterQuartileRange;}
		public double get5thPercentile(){return this.d5thPercentile;}
		public double get95thPercentile(){return this.d95thPercentile;}
		public double getRSMMinimum(){return this.dRSMMinimum;}
		public double getRSMMaximum(){return this.dRSMMaximum;}
		public double get90LowConfidenceInterval(){return this.d90LowConfidenceInterval;}
		public double get90HighConfidenceInterval(){return this.d90HighConfidenceInterval;}
		public double get95LowConfidenceInterval(){return this.d95LowConfidenceInterval;}
		public double get95HighConfidenceInterval(){return this.d95HighConfidenceInterval;}
		public double get99LowConfidenceInterval(){return this.d99LowConfidenceInterval;}
		public double get99HighConfidenceInterval(){return this.d99HighConfidenceInterval;}	
	    public Histogram getHistogram(){return this.histogram;}
	    public boolean getIsTheCumulativeDistributionSet(){return this.bCumulativeDistributionSet;}
	    public double[] getValidatedData(){return this.dValidated;}
	    public double[] getCumulativeDistributionFrequencies(){return this.dValidatedFrequency;}	
	    public String getHeadingExtension(){return this.sHeadingExtension;}
	    public String getOriginalValue(int i){return this.sOriginalValue[i];}
	}
