package ui;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import javax.imageio.ImageIO;

import javafx.application.Platform;
import javafx.collections.ObservableList;
import javafx.concurrent.Task;
import javafx.embed.swing.SwingFXUtils;
import javafx.scene.Scene;
import javafx.scene.SnapshotParameters;
import javafx.scene.control.Tab;
import javafx.scene.control.TabPane;
import javafx.scene.image.WritableImage;
import javafx.scene.layout.BorderPane;
import javafx.stage.Stage;
import pviScreen.Building;
import pviScreen.Chemical;
import pviScreen.OilLens;
import pviScreen.PVIScreenControlData;
import pviScreen.RawInput;
import pviScreen.RawInputLine;
import pviScreen.RunIdentifier;
import pviScreen.VadoseZone;
import statistics.RiskContainer;
import util.CalendarClock;
import util.ClipPathOrFile;
import util.FileLogger;
import util.NameConstants;
import util.OutputDigits;
import util.Stripper;
import util.UnitConverter;

public class ReportWriter 
{
	private Repository repository;
	private HandlerStatisticsPlot hsp;
	private PVIScreenControlData pvisCD;
	private RunIdentifier runID;
	private RiskContainer rc;
	private String sOutputFileAndPath,sInputFileAndPath;
	private FileLogger flReport;
	private String sReportFileName;
	private int iFigureCount;
	private int iTableCount;
	private OutputDigits od;
	private Stripper s;
	private NameConstants nc;
	private UnitConverter uc;
	private double[] dColorLow;
	private double[] dColorHigh;
	private HandlerSitePlot hSitePlot;
	
	public ReportWriter(){}
    
	public void prepareGraphics()
	{
		BorderPane border = repository.getBorder();
 	    final TabPane tp2 = (TabPane)border.getCenter();
  	    Task task = new Task<Void>() {
          @Override
          public Void call() {

              Platform.runLater(
                      new Runnable() {
                          public void run() {
                              try {

                                 
                                  ObservableList<Tab> olTab = tp2.getTabs();
                                  int iCount = 0;
                            	  for (Tab tab: olTab)
                            	  {	
                            		if (tab.getText().equalsIgnoreCase("benzene"))
                            		{
	                            	    WritableImage snapshot = tab.getContent().snapshot(new SnapshotParameters(), null);
	                            	    String sFilePNG = "test2Name" + iCount + ".png";
	                                    File outFile = new File(sFilePNG);
	                                    try 
	                                    {
	                                      ImageIO.write(SwingFXUtils.fromFXImage(snapshot, null),"png", outFile);
	                                    } 
	                                    catch (IOException ex) 
	                                    {
	                                      //System.out.println(ex.getMessage());
	                                    }
	                          	        repository.addOutputImage(snapshot);
	                          	        System.out.println("graphic for: " + tab.getText());
	                          	        repository.addOutputImageName(tab.getText());
	                          	        iCount++;
                            		}
                            	   }
                            		
                                  
                              } catch (Exception s) {}

                              System.out.println("finished");

                          }
                      });

              return null;
          }
      };
      Thread th = new Thread(task);
      th.start();		
	}
	
    public void startWritingReport()
    {
    	s = new Stripper();
    	nc = new NameConstants();
    	pvisCD = repository.getPVIScreenControlData();
    	hsp = repository.getHandlerStatisticsPlot();
    	rc = repository.getRiskContainer();
    	runID = pvisCD.getRunIdentifier();
		sInputFileAndPath = repository.getInputFileAndPath();
		sOutputFileAndPath = repository.getFilePathAndNameResults();
		dColorLow = repository.getColorLow();
		dColorHigh = repository.getColorHigh();
		iFigureCount = 1;
		iTableCount = 1;
		od = new OutputDigits();
    }
    
    
 
    //components of the report
    
    //write the headings
    public void writeReportHeadings()
    {
       String sOut = "";
       flReport.logMessage("<!DOCTYPE html>");
 	   flReport.logMessage("<html>");
 	   flReport.logMessage("<head>");
 	   sOut = "<style>table, th, td{border-collapse:collapse;}</style>";
       flReport.logMessage(sOut);
 	   sOut = "<style>table, th, td{border:1px solid green;}</style>";
 	   flReport.logMessage(sOut);
 	   sOut = "<style>td{text-align:center;}</style>";
 	   flReport.logMessage(sOut);
 	   sOut = "<style>th{background-color:green;color:white;}</style>";
 	   flReport.logMessage(sOut);
 	   sOut = "<style>caption {caption-side:top;}</style>";
 	   
 	   //The fancy risk table...
 	   //use with <table id=\'risk\'>
	   sOut = "";
	   sOut = sOut + "<style>#risk{font-family:\"Trebuchet MS\", Arial, Helvetica, sans-serif;width:80%;border-collapse:collapse;}";
	   //sOut = sOut + "#risk td, #customers th {font-size:1em;border:1px solid #98bf21;padding:3px 7px 2px 7px;}";
	   sOut = sOut + "#risk td {font-size:1em;padding:3px 7px 2px 7px;}";
	   sOut = sOut + "#risk th {font-size:1.1em;text-align:left;padding-top:5px;padding-bottom:4px;background-color:#A7C942;color:#ffffff;}";
	   sOut = sOut + "#risk tr.alt td {color:#000000;background-color:#EAF2D3;}";
	   sOut = sOut + "#risk tr.high td{color:#990033;background-color:#FFFFFF;}";
	   sOut = sOut + "</style>";
	   flReport.logMessage(sOut);
	   sOut = "";
	   sOut = sOut + "<style>#norisk{font-family:\"Trebuchet MS\", Arial, Helvetica, sans-serif;width:60%;border-collapse:collapse;}";
	   //sOut = sOut + "#risk td, #customers th {font-size:1em;border:1px solid #98bf21;padding:3px 7px 2px 7px;}";
	   sOut = sOut + "#norisk td {font-size:1em;padding:3px 7px 2px 7px;}";
	   sOut = sOut + "#norisk th {font-size:1.1em;text-align:left;padding-top:5px;padding-bottom:4px;background-color:#A7C942;color:#ffffff;}";
	   sOut = sOut + "#norisk tr.alt td {color:#000000;background-color:#EAF2D3;}";
	   sOut = sOut + "#norisk tr.high td{color:#990033;background-color:#FFFFFF;}";
	   sOut = sOut + "</style>";
	   flReport.logMessage(sOut);
	   sOut = "";
	   sOut = sOut + "<style>#source{font-family:\"Trebuchet MS\", Arial, Helvetica, sans-serif;width:80%;border-collapse:collapse;}";
	   //sOut = sOut + "#risk td, #customers th {font-size:1em;border:1px solid #98bf21;padding:3px 7px 2px 7px;}";
	   sOut = sOut + "#source td {font-size:1em;padding:3px 7px 2px 7px;}";
	   sOut = sOut + "#source th {font-size:1.1em;text-align:left;padding-top:5px;padding-bottom:4px;background-color:#A7C942;color:#ffffff;}";
	   sOut = sOut + "#source tr.alt td {color:#000000;background-color:#EAF2D3;}";
	   sOut = sOut + "#source tr.high td{color:#990033;background-color:#FFFFFF;}";
	   sOut = sOut + "</style>";      
	   flReport.logMessage(sOut);
	 
 	   
 	   
 	   flReport.logMessage("</head>");
 	   flReport.logMessage("<body>");
 	  
 	   flReport.logMessage("<FONT face=\"verdana\">");
 	   flReport.logMessage("<center>");
 	
 	   flReport.logMessage("<br>");
 	   flReport.logMessage("<h1>");
 	   flReport.logMessage("PVIScreen Model Report");
       flReport.logMessage("</h1>");  
 	   flReport.logMessage("<br>");
 	   flReport.logMessage("</center>");   
    }
    
    
    public void writeRunIdentification()
    {
 	   String sOut="";
 	   
 	   flReport.logMessage("<h2>Run Identification</h2>");
 	   
 	   sOut = "";
 	   sOut = sOut + "<br>";
	   sOut = sOut + "Site: " + runID.getSite();
	   sOut = sOut + "<br>";
	   flReport.logMessage(sOut);
 	   
 	   sOut = "";
	   sOut = sOut + "Location: " + runID.getLocation();
	   sOut = sOut + "<br>";
	   flReport.logMessage(sOut); 	   
	   
	   sOut = "";
	   sOut = sOut + "City: " + runID.getCity();
	   sOut = sOut + "<br>";
	   flReport.logMessage(sOut); 	   
	   
	   sOut = "";
	   sOut = sOut + "State: " + runID.getState();
	   sOut = sOut + "<br>";
	   flReport.logMessage(sOut); 	   	   
	   
 	   sOut = "";
 	   sOut = sOut + "Analyst: " + runID.getAnalyst();
 	   sOut = sOut + "<br>";
 	   flReport.logMessage(sOut);
 	   
 	   sOut = "";
 	   sOut = sOut + "Affiliation: " + runID.getAffiliation();
 	   sOut = sOut + "<br>";
 	   flReport.logMessage(sOut);
 	   
 	   sOut = "";
 	   String sDate = runID.getDate();
 	   if (sDate.equalsIgnoreCase(""))
 	   {
 		   CalendarClock cc = new CalendarClock();
 		   sDate = cc.getThreeLetterMonth() + " " + cc.getDay() + " " + cc.getYear();
 	   }
 	   sOut = sOut + "Date: " + sDate;
 	   sOut = sOut + "<br>";
 	   flReport.logMessage(sOut);
    }
    
    //write problem description
    public void writePVIScreenParagraph()
    {
    	String sOut;
    	flReport.logMessage("<h2>PVIScreen Background</h2>");
    	
    	sOut = "<p>";
    	sOut = sOut + "PVIScreen is a model for assessing impacts from petroleum vapors on residences. ";
    	sOut = sOut + "PVIScreen was designed for automatic uncertainty analysis using Monte Carlo simulation. ";
    	sOut = sOut + "The main result from the model is a probabiliy curve for indoor air concentration for each simulated chemical. ";
    	sOut = sOut + "Both cancer and non-cancer risk levels are indicated on the probability curves. ";
    	
    	sOut = sOut + "</p>";
    	
    	flReport.logMessage(sOut);
    	
    	sOut = "<p>";
    	sOut = sOut + "PVIScreen is based on the BioVapor model (Devaull, 2007; API, 2010). ";
    	sOut = sOut + "PVIScreen extends the capabilities of BioVapor by including automatic uncertainty analysis, " +
    			"flexible unit selection, and direct inclusion of liquid gasoline (NAPL). ";
    	sOut = sOut + "Major assumptions of the model include:";
    	flReport.logMessage(sOut);
    	
    	sOut = "<ul>";
    	sOut = sOut + addListItem("Oxygen supply permits/limits biodegradation of petroleum vapors");
    	sOut = sOut + addListItem("Multiple components of fuel contribute to oxygen demand");
    	sOut = sOut + addListItem("Soil Respiration contributes to oxygen demand");
    	sOut = sOut + addListItem("Homogeneous vadose zone");
    	sOut = sOut + addListItem("Steady state conditions");
    	sOut = sOut + "</ul>";
    	
    	
    	sOut = sOut + "</p>";
    		
    	flReport.logMessage(sOut);
    	
    }
    
    
    //write input/output file information
    public void writeIOInformation()
    {
    	String sOut;
        ClipPathOrFile cpof = new ClipPathOrFile();
    	
    	//clip the file name apart from each other
    	String sIFile = cpof.clipPathOrFile(this.sInputFileAndPath,"file");
    	String sOFile = cpof.clipPathOrFile(this.sOutputFileAndPath,"file");
    	String sRFile = cpof.clipPathOrFile(this.sReportFileName, "file");
    	String sDirectory = cpof.clipPathOrFile(this.sInputFileAndPath,"path");
    	
    	sOut = "<p>";
    	sOut = sOut + "The results in this report were generated from: ";
    	sOut = sOut + sIFile;
    	sOut = sOut + ".  ";
    	sOut = sOut + "</p>";
    	flReport.logMessage(sOut);
    	sOut = "<p>";
    	sOut = sOut + "The result file is: ";
    	sOut = sOut + sOFile;
    	sOut = sOut + ".  ";
    	flReport.logMessage(sOut);
    	sOut = "<p>";
    	sOut = sOut + "This file is:  ";
    	sOut = sOut + sRFile;
    	sOut = sOut + ".  ";
    	sOut = sOut + "</p>";
    	flReport.logMessage(sOut);
    	sOut = "<p>";
    	sOut = sOut + "All of these are located in directory:  ";
    	sOut = sOut + sDirectory;
    	sOut = sOut + ".  ";
    	sOut = sOut + "</p>";
    	flReport.logMessage(sOut);
    }
    
 
    
    //write site schematic information
    public void writeSiteSchmatic()
    {
    	String sOut;
    	Building bldg = pvisCD.getBuilding();
    	VadoseZone vz = pvisCD.getVadoseZone();
    	OilLens ol = pvisCD.getOilLens();
    	UnitConverter uc = pvisCD.getUnitConverter();
    	
    	
    	double dUCF = uc.getOutputUCFFromType("Length");
    	String sUCF = uc.getOutputNameFromType("Length");
    	
    	flReport.logMessage("<h2>Site Description</h2>");
   	   
    	sOut = "<p>";
    	sOut = sOut + "The depths of the foundation, water table, historic water table, top of contamination are shown in Figure ";
    	sOut = sOut + iFigureCount + ".  ";
    	flReport.logMessage(sOut);
    	sOut = "";
    	sOut = sOut + "The depth of the foundation is ";
    	sOut = sOut + od.num2String(bldg.getFoundationDepthBelowGrade()*dUCF);
    	sOut = sOut + " ";
    	sOut = sOut + sUCF;
    	sOut = sOut + ", and ";
    	flReport.logMessage(sOut);
    	sOut = "";
    	sOut = "the depth to the water table is ";
    	sOut = sOut + od.num2String(vz.getDepthToWaterTable()*dUCF);
    	sOut = sOut + " ";
    	sOut = sOut + sUCF;
    	sOut = sOut + ".  ";
    	flReport.logMessage(sOut);
    	
    	sOut = "";
    	sOut = sOut + "The depth to the historic water table is ";
    	sOut = sOut + od.num2String(vz.getDepthToHistoricWaterTable()*dUCF);
    	sOut = sOut + " ";
    	sOut = sOut + sUCF;
    	sOut = sOut + ".  ";
    	sOut = sOut + "</p>";
    	flReport.logMessage(sOut);
    	
    	sOut = "";
    	sOut = sOut + "The distance between the bottom of the foundation and the sample that defines the contamination is ";
    	double dFoundationElevation =  ol.getSurfaceElevation() - bldg.getFoundationDepthBelowGrade();
    	sOut = sOut +  od.num2String((dFoundationElevation - vz.getDepthToSample())*dUCF);
    	sOut = sOut + " ";
    	sOut = sOut + sUCF;
    	sOut = sOut + ".  ";
    	sOut = sOut + "</p>";
    	flReport.logMessage(sOut);
    	
    	sOut = "";
    	sOut = sOut + "The depth of sample is ";
    	sOut = sOut +  od.num2String(vz.getDepthToSample()*dUCF);
    	sOut = sOut + " ";
    	sOut = sOut + sUCF;
    	sOut = sOut + ".  ";
    	sOut = sOut + "</p>";
    	flReport.logMessage(sOut);
    	
    	
    	try
    	{
    		
    		sOut = "<div class=\"figure\">";
    		
	    	WritableImage wiSite = repository.getSiteSchematic();
	    	if (wiSite==null)
	    	{	
	    		//schematic not yet drawn
	    		//draw schematic without displaying (i.e., argument of false)
	    		try
	    		{
	    		  hSitePlot.drawSchematic(false);
	    		  wiSite = repository.getSiteSchematic();
	    		}
	    		catch(Exception ee){}
	    	}
	    	
	    	
	        // Write snapshot to file system as a .png image
	    	String sFilePNG = this.sReportFileName + ".png";
	        File outFile = new File(sFilePNG);
	        try 
	        {
	            ImageIO.write(SwingFXUtils.fromFXImage(wiSite, null),"png", outFile);
	        } 
	        catch (IOException ex) 
	        {
	            System.out.println(ex.getMessage());
	        }
	        
	        //include file in report
	        sOut = "";
	        sOut = sOut + "<p>";
	        sOut = sOut + "<img src=\"" + sFilePNG + "\" alt=\"Site Schematic\" ></img>";
	        sOut = sOut + "</p>";
	        flReport.logMessage(sOut);
	        
	        sOut = "";
	        sOut = sOut + "<p>";
	        sOut = sOut + "Figure " + iFigureCount + "   Site Schematic.";
	        iFigureCount++;
	        flReport.logMessage(sOut);
	        
	        sOut = "</p>";
	        sOut = sOut + "</div>";
	    	
	        flReport.logMessage(sOut);
    	}
    	catch(Exception e){}
    }
    
    public void writeSourceDefinition()
    {
    	String sOut;
    	String sUnitType;
    	UnitConverter uc = pvisCD.getUnitConverter();
    
    	
    	sOut = "";
    	sOut = sOut + "<h2>Source Definition</h2>";
    	sOut = sOut + "<p>";
    	flReport.logMessage(sOut);
    	
    	sOut = "The source of contamination is defined by: " + this.getSourceDescription() + ".";
    	
    	
    	flReport.logMessage(sOut);
    	
    	//screening level used
    	sOut = "";
    	sOut = sOut + "<p>";
    	sOut = sOut + "<center>";
    	sOut = sOut + "<table id=\'source\'>";
    	flReport.logMessage(sOut);
    	
    	    	
    	sOut = "";
       	sOut = sOut + "<tr><th colspan=\'3\'>";
    	sOut = sOut + "Table " + iTableCount + " Source Definition.  ";
    	sOut = sOut + "</th></tr>";
    	flReport.logMessage(sOut); 
    	iTableCount++;
    	
    	sOut = "<tr>";
    	sOut = sOut + addHeadingCell("Name");
    	sOut = sOut + addHeadingCell("Source Data");
    	sOut = sOut + addHeadingCell("Concentration");
    	sOut = sOut + "</tr>";
    	flReport.logMessage(sOut);    
    	
    	
    	
    	boolean bFirst = false;
    	boolean bTRAlt = true;
    	
		//write out chemicals with source definitions
    	for (Chemical chemical: pvisCD.getChemicals())
    	{
    	  if (chemical.getIsChemicalUsed())	
    	  {	  
    		sOut = "";
    		if (bFirst)
    		{
    			//since risks are sorted, the first is the highest risk
    			sOut = sOut + "<tr class=\'high\'>";
    			bFirst = false;
    		}
    		else
    		{
	    		if (bTRAlt)
	    		{ 
	    			sOut = sOut + "<tr class=\'alt\'>";
	    			bTRAlt  = false;
	    		}
	    		else
	    		{	
	    		  sOut = sOut + "<tr>";
	    		  bTRAlt = true;
	    		}
    		}	
    		sOut = sOut + addDataCell(chemical.getName());
    		sOut = sOut + addDataCell(chemical.getInputType());
    		
    		//sUnitType = "concentrationAir";
    		sUnitType = chemical.getInputType();
    		
    		double dConc = chemical.getSourceConcentration(sUnitType);
    		double dUC = uc.getOutputUCFFromType(sUnitType);
    		
    		sOut = sOut + addDataCell(od.num2String(chemical.getSourceConcentration(sUnitType)*uc.getOutputUCFFromType(sUnitType)) + " " + uc.getOutputNameFromType(sUnitType) );
    		
    		sOut = sOut + "</tr>";
    		
    		flReport.logMessage(sOut);
    	  }	
    	}   	
    	
    	sOut = "";
    	sOut = sOut + "</table></center>";
    	flReport.logMessage(sOut);
    	
    	sOut = "<br>";
    	flReport.logMessage(sOut);
    		
    }
    
    private String getSourceDescription()
    {
    	String sReturn="";
    	String sInputType="";
    	int iAirPhase=0;
    	int iSoilSample=0;
    	int iNAPLMass=0;
    	int iNAPLVolume=0;
    	int iWater=0;
    	int iType=0;
    	
    	for (Chemical chemical: pvisCD.getChemicals())
    	{
    		sInputType = chemical.getInputType();
    		if (chemical.getIsChemicalUsed())
    		{
    			if (s.areTheyEqual("AirPhaseConcentration",sInputType)){iAirPhase++;}
    			else if (s.areTheyEqual("SoilSample",sInputType)){iSoilSample++;}
    			else if (s.areTheyEqual("WaterPhaseConcentration",sInputType)){iWater++;}
    			else if (s.areTheyEqual("FuelPhaseConcentrationByMass",sInputType)){iNAPLMass++;}
    			else if (s.areTheyEqual("FuelPhaseConcentrationByVolume",sInputType)){iNAPLVolume++;}
    		}
    	}
    	
    	if (iAirPhase>0)
    	{	
    	  sReturn = sReturn + " " + iAirPhase + " chemicals with soil gas data";
    	  iType++;
    	}
    	
    	if (iSoilSample>0)
    	{	
    	  sReturn = sReturn + " " + iSoilSample + " chemicals with soil sample data";
    	  iType++;
    	}
    	
    	if (iWater>0)
    	{	
    	  sReturn = sReturn + " " + iWater + " chemicals with water phase data";
    	  iType++;
    	}
    	
    	if (iNAPLMass>0)
    	{	
    	  sReturn = sReturn + " " + iNAPLMass + " chemicals with NAPL (concentration by mass) data";
    	  iType++;
    	}
    	
    	if (iNAPLVolume>0)
    	{	
    	  sReturn = sReturn + " " + iNAPLVolume + " chemicals with NAPL (concentration by volume) data";
    	  iType++;
    	}
    	
    	//more than one type of input concentration
    	if (iType>1)
    	{
    		sReturn = sReturn + " " + "Mixed input concentration sources, check these for consistency.";    	
    	}
    	
    	
    	return sReturn;
    }
    
    public void writeScreeningLevelTable()
    {
    	ArrayList<Chemical>alChemical = pvisCD.getChemicals();
    	String sOut;
    	boolean bScreeningSet=false;
    	int iCount=0;
    	String[] sColorText = {"Higher","Moderate","Lower"};
    	
    	for (Chemical c: alChemical)
    	{	
    		if(c.getScreeningLevelSet()){iCount++;}
    	}
    	
    	if (iCount==0){return;}
    	
    
    	sOut = "";
    	sOut = sOut + "<p>";
        sOut = sOut + "Modeled indoor air concentrations are compared to screening the following screening levels.";	
    	sOut = sOut + "</p>";
    	flReport.logMessage(sOut);
    	
    	//screening level used
    	sOut = "";
    	sOut = sOut + "<p>";
    	sOut = sOut + "<center>";
    	sOut = sOut + "<table id=\'risk\'>";
    	flReport.logMessage(sOut);
    	
    	    	
    	sOut = "";
       	sOut = sOut + "<tr><th colspan=\'3\'>";
    	sOut = sOut + "Table " + iTableCount + " Screening Level Results.  ";
    	sOut = sOut + "</th></tr>";
    	flReport.logMessage(sOut); 
    	iTableCount++;
    	
    	sOut = "<tr>";
    	sOut = sOut + addHeadingCell("Name");
    	sOut = sOut + addHeadingCell("Screening Level");
    	sOut = sOut + addHeadingCell("% Simulations Above Screening Level");
    	sOut = sOut + "</tr>";
    	flReport.logMessage(sOut);    	
    	
    	rc.sortScreeningByPerCentExceeds();
    	for (int i=0;i<rc.getTotalCount();i++)
    	{	
    		if (rc.getIsScreeningChemical(i))
    		{	
	    		sOut = "<tr>";
				sOut = sOut + addDataCell(rc.getScreeningdName(i));
				sOut = sOut + addDataCell(od.num2String(rc.getScreeningConcentration(i)));
				sOut = sOut + addDataCell(od.num2String(rc.getScreeningPerCent(i)));	
				
	    		sOut = sOut + "</tr>";
	    		flReport.logMessage(sOut);
    		}
    	} 
    	
    	sOut = "";
    	sOut = sOut + "</table>";
    	flReport.logMessage(sOut);
    	
    	sOut = "<br>";
    	flReport.logMessage(sOut);
    	
    	
    }
    
    public void writeRiskTable()
    {
    	String sOut;
    	UnitConverter uc = pvisCD.getUnitConverter();
    	String sRiskConcentrationOutput = uc.getOutputNameFromType("RiskConcentration");
    	
    	boolean bTRAlt;
    	boolean bFirst;
    	String[] sColorText = {"Higher","Moderate","Lower"};    	
    	
        if (rc.getRiskCount()>0)
        {	
	    	
	    	//cancer risk level
	    	sOut = "";
	    	sOut = sOut + "<p>";
	    	sOut = sOut + "<center>";
	    	sOut = sOut + "<table id=\'risk\'>";
	    	flReport.logMessage(sOut);
	    	
	    	    	
	    	sOut = "";
	       	sOut = sOut + "<tr><th colspan=\'4\'>";
	    	sOut = sOut + "Table " + iTableCount + " Cancer Risk Results.  ";
	    	sOut = sOut + "</th></tr>";
	    	flReport.logMessage(sOut); 
	    	iTableCount++;
	    	
	    	sOut = "<tr>";
	    	sOut = sOut + addHeadingCell("Name");
	    	sOut = sOut + addHeadingCell("Risk Level");
	    	sOut = sOut + addHeadingCell("Concentration (" + sRiskConcentrationOutput + ")");
	    	sOut = sOut + addHeadingCell("% Simulations Above Cancer Risk Level");
	    	sOut = sOut + "</tr>";
	    	flReport.logMessage(sOut);
	    	
	    	
	    	
	    	
	    	
	    	try
	    	{
	    		rc.sortCancerRiskByPerCentExceeds();
	    	 
	    		bTRAlt = true;
	    		bFirst = true;
	    		//write out chemicals with cancer risk data
		    	for (int i=0;i<rc.getTotalCount();i++)
		    	{
		    	  if(rc.getIsRiskChemical(i))
		    	  {	  
		    		sOut = "";
		    		if (bFirst)
		    		{
		    			//since risks are sorted, the first is the highest risk
		    			sOut = sOut + "<tr class=\'high\'>";
		    			bFirst = false;
		    		}
		    		else
		    		{
			    		if (bTRAlt)
			    		{ 
			    			sOut = sOut + "<tr class=\'alt\'>";
			    			bTRAlt  = false;
			    		}
			    		else
			    		{	
			    		  sOut = sOut + "<tr>";
			    		  bTRAlt = true;
			    		}
		    		}	
		    		sOut = sOut + addDataCell(rc.getCancerName(i));
		    		sOut = sOut + addDataCell(od.num2String(rc.getRiskLevel(i)));
		    		sOut = sOut + addDataCell(od.num2String(rc.getRiskConcentration(i)));
		    		sOut = sOut + addDataCell(od.num2String(rc.getRiskPerCent(i)));
		    		
		    		
		    		sOut = sOut + "</tr>";
		    		
		    		flReport.logMessage(sOut);
		    	  }	
		    	}
	    	}
	    	catch(Exception e){}
	    	
	    	sOut = "";
	    	sOut = sOut + "</table>";
	    	flReport.logMessage(sOut);
	    	
	    	sOut = sOut + "<br>";
	    	flReport.logMessage(sOut);
	    	
	    	sOut = sOut + "<table id=\'norisk\'>";
	    	flReport.logMessage(sOut);
	    	
	    	sOut = "<tr>";
	    	sOut = sOut + addHeadingCell("Chemicals with no Cancer Risk Results");
	    	
	    	sOut = sOut + "</tr>";
	    	flReport.logMessage(sOut);   	
	    	//write out chemicals without cancer risk data
	    	try
	    	{
	    		rc.sortCancerRiskByPerCentExceeds();
	    		bTRAlt = true;
	    		//write out chemicals without cancer risk data
	    		sOut = "";
		    	sOut = sOut + "<TR><TD>";
		    	for (int i=0;i<rc.getTotalCount();i++)
		    	{
		    	 
		    	  if(!rc.getIsRiskChemical(i))
		    	  {	  
		    		String sOutputName = rc.getCancerName(i);
		    		if (sOutputName==""){}
		    		else{sOut = sOut +  rc.getCancerName(i) + ", ";}
		    		/*  
		    		sOut = "";
		    		if (bTRAlt)
		    		{
		    			sOut = sOut + "<tr class=\'alt\'>";
		    			bTRAlt  = false;
		    		}
		    		else
		    		{	
		    		  sOut = sOut + "<tr>";
		    		  bTRAlt = true;
		    		}
		    		sOut = sOut + addDataCell(rc.getCancerName(i));
		    		sOut = sOut + "</tr>";
		    		
		    		flReport.logMessage(sOut);
		    		*/
		    	  }
		    	  
		    	}
		    	sOut = sOut + "</TD></TR>";
		    	  flReport.logMessage(sOut);	    	
		    	
	    	}
	    	catch(Exception e){}
	    	
	    	sOut = "";
	    	sOut = sOut + "</table>";
	    	sOut = sOut + "</center>";
	    	sOut = sOut + "</p>";
	    	flReport.logMessage(sOut);  	
        }
    	
    	
    	//non-cancer risk levels   
        if (rc.getHazardCount()>0)
        {	
	    	sOut = "";
	    	sOut = sOut + "<p>";
	    	sOut = sOut + "<center>";
	    	sOut = sOut + "<table id=\'risk\'>";
	    	flReport.logMessage(sOut);
	    	
	    	sOut = "";
	    	sOut = sOut + "<tr><th colspan = \'5\'>";
	    	sOut = sOut + "Table " + iTableCount + " Non-Cancer Hazard Results.  ";
	    	sOut = sOut + "</th></tr>";
	    	flReport.logMessage(sOut); 
	    	iTableCount++;
	    	
	    	sOut = "<tr>";
	    	sOut = sOut + addHeadingCell("Name");
	    	sOut = sOut + addHeadingCell("Hazard Quotient");
	    	sOut = sOut + addHeadingCell("Concentration (" + sRiskConcentrationOutput + ")");
	    	sOut = sOut + addHeadingCell("% Simulations Above Hazard Level");	
	    	sOut = sOut + "</tr>";
	    	flReport.logMessage(sOut);
	    	
	    	try
	    	{
	    		rc.sortHazardByPerCentExceeds();
	    		bTRAlt = true;
	    		bFirst = true;
		    	for (int i=0;i<rc.getTotalCount();i++)
		    	{
		    		//write out chemicals with hazard data
		    		if (rc.getIsHazardChemical(i))
		    		{	
		    		  sOut = "";
		    		  if (bFirst)
		    		  {
		    			  //first one has highest risk
		    			  sOut = sOut + "<tr class=\'high\'>"; 
		    			  bFirst = false;
		    		  }
		    		  else
		    		  {
			    		if (bTRAlt)
			    		{
			    			sOut = sOut + "<tr class=\'alt\'>";
			    			bTRAlt  = false;
			    		}
			    		else
			    		{	
			    		  sOut = sOut + "<tr>";
			    		  bTRAlt = true;
			    		}
		    		  }	
		    		  sOut = sOut + addDataCell(rc.getHazardName(i));
		    		  sOut = sOut + addDataCell(od.num2String(rc.getHazardQuotient(i)));
		    		  sOut = sOut + addDataCell(od.num2String(rc.getHazardConcentration(i)));
		    		  sOut = sOut + addDataCell(od.num2String(rc.getHazardPerCent(i)));
		    		  
		    		
		    		  
		    		  sOut = sOut + "</tr>";
		    		
		    		 flReport.logMessage(sOut);
		    		}	
		    	}
	    	}	
	    	catch(Exception e){}
	    	sOut = "";
	    	sOut = sOut + "</table>";
	    	sOut = sOut + "</center>";
	    	sOut = sOut + "<br>";
	    	flReport.logMessage(sOut);
	    	
	    	
	    	int iHazardCountNot=0;
	    	//write out the non-hazard chemicals
	    	try
	    	{
	    		rc.sortHazardByPerCentExceeds();
	    		bTRAlt = false;
	    		
	    		
	    		for (int i=0;i<rc.getTotalCount();i++)
		    	{
		    		//count the chemicals without hazard data
		    		if (!rc.getIsHazardChemical(i))
		    		{	
	    		       if (rc.getHazardName(i)!=""){ iHazardCountNot++;}
		    		}
		    	}
	    		
	    		
	    		if(iHazardCountNot>0)
	    		{	
	    			sOut = "";
	    			sOut = sOut + "<center>";
		    		sOut = sOut + "<table id=\'norisk\'>";
		        	flReport.logMessage(sOut);
		        	
		        	sOut = "<tr>";
		        	sOut = sOut + addHeadingCell("Chemicals with no Hazard Results.");
		        	
		        	sOut = sOut + "</tr>";
		        	flReport.logMessage(sOut);   	   		
		    		
		    		
			    	for (int i=0;i<rc.getHazardCount();i++)
			    	{
			    		//write out chemicals with hazard data
			    		if (!rc.getIsHazardChemical(i))
			    		{	
			    		  sOut = "";
			    		  sOut = sOut + "<tr>";
			    		  sOut = "";
				    		if (bTRAlt)
				    		{
				    			sOut = sOut + "<tr class=\'alt\'>";
				    			bTRAlt  = false;
				    		}
				    		else
				    		{	
				    		  sOut = sOut + "<tr>";
				    		  bTRAlt = true;
				    		}
			    		  sOut = sOut + addDataCell(rc.getHazardName(i));
			    		  sOut = sOut + "</tr>";
			    		  flReport.logMessage(sOut);
			    		}	
			    	}
			    	sOut = "";
			    	sOut = sOut + "</table>";
			    	sOut = sOut + "</center>";
			    	sOut = sOut + "</p>";
			    	flReport.logMessage(sOut);
	    		}
	    	}	
	    	catch(Exception e){}   	
        }
    	
    
       
    }
    
    
    public void writeRiskResults()
    {
    	String sOut;
      
    	flReport.logMessage("<h2>Risk Results</h2>");
   	   
    	sOut = "";
    	sOut = sOut + "The model determines cancer risk for carcinogenic constituents of gasoline, ";
    	sOut = sOut + "and non-cancer risks for as many consituents as possible.  ";
    	sOut = sOut + "<A href=\"http://www.epa.gov/iris\")> EPA's Integrated Risk Assessment System (IRIS) </A> ";
    	sOut = sOut + " was consulted for cancer risk levels and non-cancer reference concentrations (RfCs). ";
    	flReport.logMessage(sOut);
    	sOut = "";
    	sOut = sOut + "</p>";
    	flReport.logMessage(sOut);
    	
    	sOut = "";
    	sOut = sOut + "Results are given separately for cancer (Table " + iTableCount + ")";
    	int iTemp = iTableCount + 1;
    	sOut = sOut + ", and non-cancer risks (Table " + iTemp + ").";
    	flReport.logMessage(sOut);
    	
    	sOut = "";
    	sOut = sOut + "For each kind of risk, the chemicals are listed from highest to lowest risk, ";
    	sOut = sOut + "according to the percentage of simulations exceeding the specified risk levels.  ";
    	flReport.logMessage(sOut);
    	
    	sOut = "";
    	sOut = sOut + "For example, if 20% of simulations exceed a hazard quotient of 1.0 for xylenes, and 3% exceed a hazard ";
    	sOut = sOut + "quotient of 0.2 for toluene, then xylenes will appear first as the highest risk chemical and hazard quotient.  ";
    	sOut = sOut + "This happens because more simulations for xylenes (20%) exceed the hazard quotient, than do for toluene (3%).  ";
    	flReport.logMessage(sOut);
    }
    
    
    //write out all observed probability curves
    public void writeProbabilityCurves()
    {
    	String sFilePNG;
    	int iCount = 0;
    	WritableImage wi;
    	
    	
    	
    	 
    	TabPane tpStatistics = repository.getStatisticsTabPane();
   
  	    ObservableList<Tab> olTab = tpStatistics.getTabs();
  	    for (Tab tab: olTab)
  	    {	
  	       repository.addOutputImageName(tab.getText());
  	       WritableImage snapshot = (tab.getContent()).snapshot(new SnapshotParameters(), null);
	       repository.addOutputImage(snapshot);
	      
  	    }
     
    	 
    	ArrayList<WritableImage>alImages =  repository.getOutputImages();
    	ArrayList<String>alImageNames = repository.getOutputImageNames();
    	int iImageCount = alImages.size();
    	//for (int i=0;i<iImageCount;i++)
    	//only one output  because of problem in printing png files from javaFx
    	for (int i=0;i<1;i++)
    	{	
    	 	
          wi = alImages.get(i);
          // Write snapshot to file system as a .png image
    	  sFilePNG = this.sReportFileName + iCount + ".png";
          File outFile = new File(sFilePNG);
          try 
          {
            ImageIO.write(SwingFXUtils.fromFXImage(wi, null),"png", outFile);
            //include file in report
            String sOut = "<p>";
            flReport.logMessage(sOut);
            sOut = "";
            sOut = sOut + "<img src=" + sFilePNG + " alt=\"Statistical Output\">";
            flReport.logMessage(sOut);
            sOut = "";
            sOut = sOut + "<br>";
            sOut = sOut + "Figure " + iFigureCount + " Indoor air probabilities and risk levels for ";
            sOut = sOut + alImageNames.get(i);
            sOut = sOut + ".";
            sOut = sOut + "</p>";
            iFigureCount++;
            flReport.logMessage(sOut);
            iCount++;
          } 
          catch (IOException ex) 
          {
            //System.out.println(ex.getMessage());
          }
        
         
    	} 
    	 
    }
    
    //write the chemical inputs
    public void writeChemicalInput()
    {
    	 String sOut;
         boolean bTRAlt;
         ClipPathOrFile cpof = new ClipPathOrFile();
         
         sOut = "";
         sOut = sOut + "<h2>";
         sOut = sOut + "Chemical Input Data";
         sOut = sOut + "</h2>";
         flReport.logMessage(sOut);
         iTableCount++;     
         
         String sClippedFile = cpof.clipPathOrFile(this.sInputFileAndPath, "file");
         //the tables:  
         //variable parameter table first:
     	 RawInput ri = pvisCD.getRawInput();
     	 flReport.logMessage("<center>");
     	 flReport.logMessage("<table id=\'risk\'>");
     	 sOut = "";
     	 sOut = sOut + "<tr><th colspan=\'7\'>";
     	 sOut = sOut + "Table " + iTableCount + " Variable parameters from file:  " + sClippedFile + ".";
     	 sOut = sOut + "</th></tr>";
     	 flReport.logMessage(sOut);       
         
     	sOut = "<tr>";
    	sOut = sOut + addHeadingCell("Group");
    	sOut = sOut + addHeadingCell("InputType");
    	sOut = sOut + addHeadingCell("Distribution Type");
    	sOut = sOut + addHeadingCell("Chemical");
    	sOut = sOut + addHeadingCell("Value");
    	sOut = sOut + addHeadingCell("Unit");
    	sOut = sOut + addHeadingCell("Frequency");
    	sOut = sOut + "</tr>";
    	flReport.logMessage(sOut);
    	String sTest;
    	String sRawInputType;
    	double dThisValue, dNextValue, dTestValue;
    	String sThisName, sNextName;
    	String sAdditionalText;
    	boolean bWrite;
    	int iIndex;
    	int iSize;
    	
    	RawInputLine ril,rilNext;
    	ArrayList<RawInputLine> alRIL;
    	alRIL = ri.getAllRawInputLines();
    	iSize = alRIL.size();
    	bWrite = true;
    	String sPrevName="";
    	String sName="";
    	bTRAlt = true;
    	
    	for (int i=0;i<iSize-1;i++)
    	{
    		ril = alRIL.get(i);
    		
            sName = ril.getParameterName();
            sRawInputType = ril.getRawInputType();
        	bWrite = false;
        	if (s.areTheyEqual(nc.s_CHEMICALINPUT,sRawInputType)){bWrite=true;}
        	
    		
    		sTest = ril.getDistributionType();
    		
    		
    		//if (sTest.equalsIgnoreCase("Constant"))
    		{
    			//do not write constant values into this table
    		//	bWrite = false;
    		}	
    		//else
    		{
    		  //check next value to see if it is the same as the current value
    		  //if so, then the parameter is a defacto constant and should not be printed here
    		  dThisValue = ril.getValue();
    		  try
    		  {
    			  rilNext = alRIL.get(i+1);
    			  dNextValue = rilNext.getValue();
    			  dTestValue = (dThisValue-dNextValue)/(0.5*(dThisValue+dNextValue));
    			  if (Math.abs(dTestValue)<=1.e-6)
    			  {
    				  ///values are effectively the same
    				  bWrite = false;
    				  //have to skip the next one
    				  i++;
    			  }
    		  }
    		  catch(Exception e){}
    		}
    		
    		
    		
    		if (bWrite)
    		{  
    			sOut = "";
            	if (sPrevName.equalsIgnoreCase(sName))
            	{
            		//don't change row formatting
            	}
            	else
            	{
            		//change row formatting
            		if (bTRAlt){bTRAlt = false;}
            		else{bTRAlt = true;}
            	}
             
        		if (bTRAlt){sOut = sOut + "<tr class=\'alt\'>";}
        		else {sOut = sOut + "<tr>";}
    			
    			
    		  //row is being written:	
    		  //row content
        	  sOut = sOut + addDataCell(ril.getGroupName());
        	  sOut = sOut + addDataCell(ril.getParameterName2());
        	  sOut = sOut + addDataCell(ril.getDistributionType());
        	  sOut = sOut + addDataCell(ril.getParameterName());
        	  sOut = sOut + addDataCell(""+ril.getValue());
        	  sOut = sOut + addDataCell(ril.getUnit());
        	  sOut = sOut + addDataCell(""+ril.getFrequency());
        	  //sOut = sOut + addDataCell(ril.getComment());
        	
        	  sOut = sOut + "</tr>";	

        	  flReport.logMessage(sOut);
    		} 
    		else {sOut = "";}
    		
    		sPrevName = sName;
        }
    	
    	//last value in file
		if (bWrite)
		{
		  ril = alRIL.get(iSize-1);
		  if (!ril.getDistributionType().equalsIgnoreCase("Constant"))
		  {
			  sOut = sOut + addDataCell(ril.getGroupName());
      	      sOut = sOut + addDataCell(ril.getParameterName2());
      	      sOut = sOut + addDataCell(ril.getDistributionType());
      	      sOut = sOut + addDataCell(ril.getParameterName());
      	      sOut = sOut + addDataCell(""+ril.getValue());
      	      sOut = sOut + addDataCell(ril.getUnit());
      	      sOut = sOut + addDataCell(""+ril.getFrequency());
		  }   
		}    	
    	
    	flReport.logMessage("</table>");
    	flReport.logMessage("</CENTER>");
    	
    	sOut = "<BR><BR><BR>";
    	flReport.logMessage(sOut);    	 
    }
    
    //write input data
    public void writeInputData()
    {
        String sOut;
        boolean bTRAlt;
        ClipPathOrFile cpof = new ClipPathOrFile();
        
        sOut = "";
        sOut = sOut + "<h2>";
        sOut = sOut + "Input Data";
        sOut = sOut + "</h2>";
        flReport.logMessage(sOut);
        iTableCount++;
        
        sOut = "";
        sOut = sOut + "<p>";
        sOut = sOut + "Input parameters can be chosen to vary according to a statistical distribution or be set as constants.  ";
        int iC2 = iTableCount + 1;
        sOut = sOut + "Tables separately give the values of the variable (Table " + iTableCount + ") and constant (Table " + iC2 + ") parameters.  ";
        sOut = sOut + "</p>";
        flReport.logMessage(sOut);
        
       
        String sClippedFile = cpof.clipPathOrFile(this.sInputFileAndPath, "file");
        //the tables:  
        //variable parameter table first:
    	RawInput ri = pvisCD.getRawInput();
    	flReport.logMessage("<center>");
    	flReport.logMessage("<table id=\'risk\'>");
    	sOut = "";
    	sOut = sOut + "<tr><th colspan=\'6\'>";
    	sOut = sOut + "Table " + iTableCount + " Variable parameters from file:  " + sClippedFile + ".";
    	sOut = sOut + "</th></tr>";
    	flReport.logMessage(sOut); 
     
    	
    	sOut = "<tr>";
    	sOut = sOut + addHeadingCell("Group");
    	sOut = sOut + addHeadingCell("Parameter");
    	sOut = sOut + addHeadingCell("Distribution Type");
    	sOut = sOut + addHeadingCell("Value");
    	sOut = sOut + addHeadingCell("Unit");
    	sOut = sOut + addHeadingCell("Frequency");
    	//sOut = sOut + addHeadingCell("Comment");
    	sOut = sOut + "</tr>";
    	
    	flReport.logMessage(sOut);
    	String sTest;
    	String sRawInputType;
    	double dThisValue, dNextValue, dTestValue;
    	String sThisName, sNextName;
    	String sAdditionalText;
    	boolean bWrite;
    	int iIndex;
    	int iSize;
    	
    	RawInputLine ril,rilNext;
    	ArrayList<RawInputLine> alRIL;
    	alRIL = ri.getAllRawInputLines();
    	iSize = alRIL.size();
    	bWrite = true;
    	String sPrevName="";
    	String sName="";
    	bTRAlt = true;
    	
    	for (int i=0;i<iSize-1;i++)
    	{
    		ril = alRIL.get(i);
    		
            sName = ril.getParameterName();
            sRawInputType = ril.getRawInputType();
        	bWrite = false;
        	if (s.areTheyEqual(nc.s_VALUEINPUT,sRawInputType)){bWrite=true;}
        	
    		
    		sTest = ril.getDistributionType();
    		if (sTest.equalsIgnoreCase("Constant"))
    		{
    			//do not write constant values into this table
    			bWrite = false;
    		}	
    		else
    		{
    		  //check next value to see if it is the same as the current value
    		  //if so, then the parameter is a defacto constant and should not be printed here
    		  dThisValue = ril.getValue();
    		  try
    		  {
    			  rilNext = alRIL.get(i+1);
    			  dNextValue = rilNext.getValue();
    			  dTestValue = (dThisValue-dNextValue)/(0.5*(dThisValue+dNextValue));
    			  if (Math.abs(dTestValue)<=1.e-6)
    			  {
    				  ///values are effectively the same
    				  bWrite = false;
    				  //have to skip the next one
    				  i++;
    			  }
    		  }
    		  catch(Exception e){}
    		}
    		
    		
    		
    		if (bWrite)
    		{  
    			sOut = "";
            	if (sPrevName.equalsIgnoreCase(sName))
            	{
            		//don't change row formatting
            	}
            	else
            	{
            		//change row formatting
            		if (bTRAlt){bTRAlt = false;}
            		else{bTRAlt = true;}
            	}
             
        		if (bTRAlt){sOut = sOut + "<tr class=\'alt\'>";}
        		else {sOut = sOut + "<tr>";}
    			
    			
    		  //row is being written:	
    		  //row content
        	  sOut = sOut + addDataCell(ril.getGroupName());
        	  sOut = sOut + addDataCell(ril.getParameterName());
        	  sOut = sOut + addDataCell(ril.getDistributionType());
        	  sOut = sOut + addDataCell(""+ril.getValue());
        	  sOut = sOut + addDataCell(ril.getUnit());
        	  sOut = sOut + addDataCell(""+ril.getFrequency());
        	  //sOut = sOut + addDataCell(ril.getComment());
        	
        	  sOut = sOut + "</tr>";	

        	  flReport.logMessage(sOut);
    		} 
    		else {sOut = "";}
    		
    		sPrevName = sName;
        }
    	
    	//last value in file
		if (bWrite)
		{
		  ril = alRIL.get(iSize-1);
		  if (!ril.getDistributionType().equalsIgnoreCase("Constant"))
		  {
			  sOut = sOut + addDataCell(ril.getGroupName());
      	      sOut = sOut + addDataCell(ril.getParameterName());
      	      sOut = sOut + addDataCell(ril.getDistributionType());
      	      sOut = sOut + addDataCell(""+ril.getValue());
      	      sOut = sOut + addDataCell(ril.getUnit());
      	      sOut = sOut + addDataCell(""+ril.getFrequency());
      	      //sOut = sOut + addDataCell(ril.getComment());
		  }   
		}    	
    	
    	flReport.logMessage("</table>");
    	flReport.logMessage("</CENTER>");
    	
    	sOut = "<BR><BR><BR>";
    	flReport.logMessage(sOut);
    	
    	
    	
    	
    	flReport.logMessage("<center>");
    	flReport.logMessage("<table id=\'risk\'>");    	
    	iTableCount++; 	
    	
    	//now the constant parameters
    	sOut = "";
    	sOut = sOut + "<tr><th colspan=\'5\'>";
    	sOut = sOut + "Table " + iTableCount + " Constant parameters (and effectively constant parameters) from file:  " + sClippedFile + ".";
    	sOut = sOut + "</th></tr>";
    	flReport.logMessage(sOut);   	
    	
    	
    	sOut = "<tr>";
    	sOut = sOut + addHeadingCell("Group");
    	sOut = sOut + addHeadingCell("Parameter");
    	sOut = sOut + addHeadingCell("Distribution Type");
    	sOut = sOut + addHeadingCell("Value");
    	sOut = sOut + addHeadingCell("Unit");
    	//sOut = sOut + addHeadingCell("Frequency");
    	//sOut = sOut + addHeadingCell("Comment");
    	sOut = sOut + "</tr>";
    	
    	flReport.logMessage(sOut);
    	
    	boolean bAdditionalText = false;
    	String sDistribution = "";
    	bWrite = false;
    	bTRAlt = false;
    	for (int i=0;i<iSize-1;i++)
    	{
    		ril = alRIL.get(i);
    		sDistribution = ril.getDistributionType();
     
        	bWrite = false;
        	sOut = "";
        	sAdditionalText = "";
    		if (bTRAlt)
    		{
    			sOut = sOut + "<tr class=\'alt\'>"; 
    		}
    		else
    		{	
    		  sOut = sOut + "<tr>";
    		}
    		
    		sTest = ril.getDistributionType();
    		if (sTest.equalsIgnoreCase("Constant"))
    		{
    			//do not write constant values into this table
    			bWrite = true;
    		}	
    		else
    		{
    		  //check next value to see if it is the same as the current value
    		  //if so, then the parameter is a defacto consant and should not be printed here
    		  dThisValue = ril.getValue();
    		  sThisName = ril.getGroupName();
    		 
    		  try
    		  {
    			  rilNext = alRIL.get(i+1);
    			  dNextValue = rilNext.getValue();
    			  sNextName = rilNext.getGroupName();
    			  if (sThisName.equalsIgnoreCase(sNextName))
    			  {	  
    			    dTestValue = (dThisValue-dNextValue)/(0.5*(dThisValue+dNextValue));
    			    if (Math.abs(dTestValue)<=1.e-6)
    			    {
    				  ///values are effectively the same
    				  bWrite = true;
    				  sDistribution = "Effectively Constant";
    				  sAdditionalText = "(*)";
    				  bAdditionalText = true;
    			    }
    			  } 
    		  }
    		  catch(Exception e){}
    		}
    		
    		
    		
    		if (bWrite)
    		{  
    		  if (bTRAlt){bTRAlt=false;}
    		  else{bTRAlt = true;}
        	  sOut = sOut + addDataCell(ril.getGroupName());
        	  sOut = sOut + addDataCell(ril.getParameterName());
        	  sOut = sOut + addDataCell(sDistribution + " " + sAdditionalText);
        	  sOut = sOut + addDataCell(""+ril.getValue());
        	  sOut = sOut + addDataCell(ril.getUnit());
        	   
        	
        	  sOut = sOut + "</tr>";	

        	  flReport.logMessage(sOut);
    		}  
    		else {sOut = "";}
    		
    		
    		
        }
    	//last value in file
		if (bWrite)
		{
		  ril = alRIL.get(iSize-1);
		  if (ril.getDistributionType().equalsIgnoreCase("Constant"))
		  {
			  sOut = sOut + addDataCell(ril.getGroupName());
      	      sOut = sOut + addDataCell(ril.getParameterName());
      	      sOut = sOut + addDataCell(ril.getDistributionType());
      	      sOut = sOut + addDataCell(""+ril.getValue());
      	      sOut = sOut + addDataCell(ril.getUnit());
      	      //sOut = sOut + addDataCell(""+ril.getFrequency());
      	      //sOut = sOut + addDataCell(ril.getComment());
		  }   
		}
		
		if (bAdditionalText)
		{	
		  sOut = "";
	      sOut = sOut + "<tr>";
		  sOut = sOut + "<td colspan = \'5\'>";
		  sOut = sOut + "(*) Parameter is effectively constant because input was declared variable, but the two or more input values were the same.";
		  sOut = sOut + "</td>";
		  sOut = sOut + "</tr>";
		  flReport.logMessage(sOut);
		}
		
    	flReport.logMessage("</table>");
    	flReport.logMessage("</CENTER>");
    	
    }
    
    public void writeRiskData()
    {
        String sOut;
        boolean bTRAlt;
        boolean bWrite;
    	String sName="";
        boolean bExit;
        
        bExit=true;
        ArrayList<Chemical>alChemical=pvisCD.getChemicals();
        //check to see if all chemicals are evaluated by screening levels only
        //if all are, then exit and print nothing here
        for (Chemical chemical: alChemical)
    	{
    		//chemical = alChemical.get(i);
    		sName = chemical.getName();
    		if(chemical.getIsChemicalUsed() && !chemical.getScreeningLevelSet())
    		{
    			bExit = false;
    		}
    	}	
        if (bExit){return;}
        
        
        uc = pvisCD.getUnitConverter();
       
        sOut = "";
        sOut = sOut + "<h2>";
        sOut = sOut + "Toxicity Values";
        sOut = sOut + "</h2>";
        flReport.logMessage(sOut);
        iTableCount++;
        
        sOut = "";
        sOut = sOut + "<p>";
        sOut = sOut + "Toxicity parameters are found in the systemData file 03ChemicalProperties.cvs and Table (" + iTableCount + ")";
        sOut = sOut + "</p>";
        flReport.logMessage(sOut);
        
       
        
        //the tables:  
        //variable parameter table:
     
    	flReport.logMessage("<center>");
    	flReport.logMessage("<table id=\'risk\'>");
    	sOut = "";
    	sOut = sOut + "<tr><th colspan=\'4\'>";
    	sOut = sOut + "Table " + iTableCount + " Risk parameters from system data file 03ChemicalProperties.csv.";
    	sOut = sOut + "</th></tr>";
    	flReport.logMessage(sOut); 
     
    	
    	sOut = "<tr>";
    	sOut = sOut + addHeadingCell("Chemical");
    	sOut = sOut + addHeadingCell("Risk Level");
    	sOut = sOut + addHeadingCell("Risk Concentration " + uc.getOutputNameFromType("RiskConcentration"));
    	sOut = sOut + addHeadingCell("Reference Concentration "  + uc.getOutputNameFromType("RiskConcentration"));
     
    	sOut = sOut + "</tr>";
    	
    	flReport.logMessage(sOut);
    	 
        od = new OutputDigits();

        bWrite = true; 
    	bTRAlt = true;
 
    	 
    	for (Chemical chemical: alChemical)
    	{
    		//chemical = alChemical.get(i);
    		sName = chemical.getName();
    		if(chemical.getIsChemicalUsed() && !chemical.getScreeningLevelSet())
    		{	
	    		if (bWrite)
	    		{  
					sOut = "";
		        	 
		        	//change row formatting
		        	if (bTRAlt){bTRAlt = false;}
		        	else{bTRAlt = true;}
		        	 
		         
		    		if (bTRAlt){sOut = sOut + "<tr class=\'alt\'>";}
		    		else {sOut = sOut + "<tr>";}
	    			
	    			
	    		  //row is being written:	
	    		  //row content
	        	  sOut = sOut + addDataCell(sName);
	        	  sOut = sOut + addDataCell(od.num2String(chemical.getRiskLevel()));
	        	  sOut = sOut + addDataCell(od.num2String(chemical.getRiskConcentration()));
	        	  sOut = sOut + addDataCell(od.num2String(chemical.getReferenceConcentration()));
	        	  
	        	  sOut = sOut + "</tr>";	
	
	        	  flReport.logMessage(sOut);
	    		} 
	    		else {sOut = "";}
    		}
    	  }	
    		
    		 
        }
    	
        public void writeChemicalData()
        {
            String sOut;
            boolean bTRAlt;
            uc = pvisCD.getUnitConverter();
           
            sOut = "";
            sOut = sOut + "<h2>";
            sOut = sOut + "Chemical Properties";
            sOut = sOut + "</h2>";
            flReport.logMessage(sOut);
            iTableCount++;
            
            sOut = "";
            sOut = sOut + "<p>";
            sOut = sOut + "Chemical parameters are found in the systemData file 02ChemicalProperties.cvs and Table (" + iTableCount + ")";
            sOut = sOut + "</p>";
            flReport.logMessage(sOut);
            
           
            
            //the tables:  
            //variable parameter table:
         
        	flReport.logMessage("<center>");
        	flReport.logMessage("<table id=\'risk\'>");
        	sOut = "";
        	sOut = sOut + "<tr><th colspan=\'10\'>";
        	sOut = sOut + "Table " + iTableCount + " Chemical parameters from system data file 01ChemicalProperties.csv.";
        	sOut = sOut + "</th></tr>";
        	flReport.logMessage(sOut); 
         
        	
        	sOut = "<tr>";
        	sOut = sOut + addHeadingCell("Chemical");
        	sOut = sOut + addHeadingCell("Distribution");
        	sOut = sOut + addHeadingCell("Molecular Weight");
        	sOut = sOut + addHeadingCell("Solubility");
        	sOut = sOut + addHeadingCell("Density");
        	sOut = sOut + addHeadingCell("Koc");
        	sOut = sOut + addHeadingCell("Phi");
        	sOut = sOut + addHeadingCell("Diffusion in Air");
        	sOut = sOut + addHeadingCell("Diffusion in Water");
        	sOut = sOut + addHeadingCell("First Order Rate Constant");
        	sOut = sOut + "</tr>";
        	
        	flReport.logMessage(sOut);
        	boolean bWrite;
        	int iSize;
            od = new OutputDigits();
         
        	
         
        	bWrite = true;
        	 
        	String sName="";
        	bTRAlt = true;
        	
        	//Chemical chemical;
        	ArrayList<Chemical>alChemical=pvisCD.getChemicals();
        	iSize = alChemical.size();
        	
        	
        	//for (int i=0;i<iSize-1;i++)
        	for (Chemical chemical: alChemical)
        	{
        		//chemical = alChemical.get(i);
        		sName = chemical.getName();
        		
        		if (chemical.getIsChemicalUsed())
        		{	
	        		if (bWrite)
	        		{  
	    				sOut = "";
	    	        	 
	    	        	//change row formatting
	    	        	if (bTRAlt){bTRAlt = false;}
	    	        	else{bTRAlt = true;}
	    	        	 
	    	         
	    	    		if (bTRAlt){sOut = sOut + "<tr class=\'alt\'>";}
	    	    		else {sOut = sOut + "<tr>";}
	        			
	        			
	        		  //row is being written:	
	        		  //row content
	            	  sOut = sOut + addDataCell(sName);
	            	  sOut = sOut + addDataCell("");
	            	  sOut = sOut + addDataCell(od.num2String(chemical.getMolecularWeight()));
	            	  sOut = sOut + addDataCell(od.num2String(chemical.getSolubility()));
	            	  sOut = sOut + addDataCell(od.num2String(chemical.getDensity()));
	            	  sOut = sOut + addDataCell(od.num2String(chemical.getKoc()));
	            	  sOut = sOut + addDataCell(od.num2String(chemical.getPhi()));
	            	  sOut = sOut + addDataCell(od.num2String(chemical.getDa()));
	            	  sOut = sOut + addDataCell(od.num2String(chemical.getDw()));
	            	  sOut = sOut + addDataCell(od.num2String(chemical.getKw()));
	            	  sOut = sOut + "</tr>";	
	
	            	  flReport.logMessage(sOut);
	        		} 
	        		else {sOut = "";}
        		}
        		 
            }   
    	
    	flReport.logMessage("</table>");
    	flReport.logMessage("</center>");
    	
    	sOut = "<BR><BR><BR>";
    	flReport.logMessage(sOut);
    	
    	
    	
    	
 
    	
    }
    
    private String addHeadingCell(String sData){return "<th>" + " " + sData + " " + "</th>";}
    private String addDataCell(String sData){return "<td>" + " " + sData + " " + "</td>";}
    private String addDataCell(double dData){return "<td>" + " " + dData + " " + "</td>";}
    private String addListItem(String sItem){return "<li>" + sItem + "</li>";}
    
    //close out report
    public void endReport()
    {
 	 flReport.logMessage("</body>");
 	 flReport.logMessage("</html>");
    }
    
    
    //generate images and risk results
    public void generateImagesAndRisk()
    {
    	ObservableList<String> olChemical = repository.getAllChemicalChoices();
    	
    	//go through all chemicals in the model
    	for (String sChemical: olChemical)
    	{
    		//do not display graphic (first "false")
    		//save all risk results (first "true")
    		//save all graphics (second "true");
    		for (Chemical c: pvisCD.getChemicals())
    		{
    		  if (s.areTheyEqual(sChemical, c.getName()))
    		  {	  
    		    hsp.drawStatistics(c,sChemical,false,true,false);
    		    break;
    		  }
    		}
    	}
    }
    
    
    //file name
    public String getReportFileName(String sFileAndPath)
    {
 	   int iLength,iDot;
 	   String sNew;
 	   char c;
 	   
 	   iDot = 0;
 	   iLength = sFileAndPath.length();
 	   iDot = iLength;
 	   
 	   //find first dot on the right hand side
 	   for (int i=(iLength-1);i==0;i=i-1)
 	   {
 		 System.out.println("looping  i " + i);
 		 c = sFileAndPath.charAt(i);   
 		 if (c == '.')
 		 {
 			 iDot = i;
 			 break;
 		 }
 	   }
 	   
 	   
 	   sNew = "";
 	   //build output name
 	   for (int i=0;i<iDot;i++)
 	   {
 		   sNew = sNew + sFileAndPath.charAt(i); 
 	   }
 	   
 	   //add extension
 	   sNew = sNew + ".";
 	   
 	   sNew = sNew + "htm";
 	   
 	   this.sReportFileName = sNew;
 	   
 	   return sNew;
    }
    
    //sets
    public void setFileLogger(FileLogger flReport){this.flReport = flReport;}
    public void setInputFileAndPath(String sInputFileAndPath){this.sInputFileAndPath = sInputFileAndPath;}
    public void setOutputFileAndPath(String sOutputFileAndPath){this.sOutputFileAndPath = sOutputFileAndPath;}
    public void setHandlerRepository(Repository repository){this.repository = repository;}  
    public void setHandlerSitePlot(HandlerSitePlot hSitePlot){this.hSitePlot = hSitePlot;}
    
}
