package uncertaintyAnalysis;


import java.util.Arrays;
import java.util.Random;

import statistics.ResistantStatisticalMeasures;
import util.QuickSort;

public class StatisticalDistribution 
{
  //name of variable	
  private String sName;	
  //name of distribution type
  private String sDistributionType;
  //parent class name
  private String sParentName;
  //model calculation unit
  private String sCalculationUnit;
  //actual number of points defining cumulative distribution
  int nPts;
  //dimensioning of cumulative distribution array
  int nDim;
  //definition of cumulative distribution
  double[] f,c;
  //current random value of "c"
  double cValue;
  //number of simulations (used in Latin Hypercube Sampling)
  int nSimulation;
  //integer ordering to generate latin hypercube samples
  int[] iOrder;
  //current latin hypercube interval number
  int cLHSInterval;
  //total number of intervals
  int nLHSInterval;
  //increment (delta cumulative frequency)
  double dCF;
  //an interval is chosen in a LHS run
  //record the random frequency WITHIN this interval and 
  //the index in the cumulative distribution c[],f[]
  //so these can be retrieved and matched in another StatisticalDistribution
  //this is done so the same interval and random frequency within the interval
  //can be used in two StatisticalDistributions
  //The purpose is to select one random value for a parameter that requires two coefficients
  //example: the regression equation for travel time requires a slope and intercept
  //these two are used to get one cumulative frequency distribution
  //see Ada JWW-02 page 70-71.
  double chosenFrequency;
  int chosenIndex;
  //companion StatisticalDistribution
  StatisticalDistribution sdCompanion;
  //"middle value" determined from integration of the cumulative frequency curve.
  double dMiddleValue;
  //quicksorter
  QuickSort qs;
  //boolean flag to determine if this statistical distribution is actually used
  private boolean bActuallyUsed;
  //random number generator
  private RandomNumber random;
  
  public StatisticalDistribution(){initial();}
	
  public void initial()
  {
	 nPts = 0; 
	 nDim = 100;
	 f = new double[nDim];
	 c = new double[nDim];
	 cValue = 0.0;
	 //sorter
	 qs = new QuickSort();
	 //distributions are empirical unless otherwise specified
	 sDistributionType = "Empirical";
	 //distribution is used, unless otherwise noted
	 //used to turn off unused general parameters
	 bActuallyUsed = true;
  }
  
  /**
   * clear -- clear the statistical distribution
   */
  public void clear(){initial();}
  
  
  /**
   * add a point to the cumulative distribution curve
   * @param value
   * @param frequency
   */
  public void addAPoint(double value, double frequency)
  {
    c[nPts] = value;
    f[nPts] = frequency;
    nPts++;
  }
  
  /**
   * Generate normal distribution
   *   i.e., generate a series of points on a cumulative distribution curve that
   *   represent a normal distribution
   */
  public void generateNormalDistribution(double dMean, double dStandardDeviation)
  {
	  //generate points on a normal distribution give the mean and standard deviation
	  addAPoint(dMean-4.0*dStandardDeviation, 0.0000);
	  addAPoint(dMean-3.0*dStandardDeviation, 0.0013);
	  addAPoint(dMean-2.0*dStandardDeviation, 0.0227);
	  addAPoint(dMean-1.0*dStandardDeviation, 0.1587);
	  addAPoint(dMean-0.0*dStandardDeviation, 0.5000);
	  addAPoint(dMean+1.0*dStandardDeviation, 0.8413);
	  addAPoint(dMean+2.0*dStandardDeviation, 0.9773);
	  addAPoint(dMean+3.0*dStandardDeviation, 0.9987);
	  addAPoint(dMean+4.0*dStandardDeviation, 1.0000);	
	  setDistributionType("Normal");
  }
  
  /**
   * traingular distribution
   * @param dMin   minimum value of distribution
   * @param dLikely  subjective most likely value of distribution
   * @param dMax  maximum value of distribution
   * 
   * Notebook Ada jww-04 page 23-25
   */
  public void generateTriangularDistribution(double dMin, double dLikely, double dMax)
  {
	  double x;
	  
	  addAPoint(dMin, 0.0000);
	  
	  int iNpts = 20;
	  double dx = (dMax-dMin)/(double)iNpts;
	  double dDenom = (dMax-dMin)*(dLikely-dMin);
	  for (int i=1;i<iNpts;i++)
	  {
		  x = dMin + dx*(double)i;
		  if (x < dLikely){addAPoint(x,Math.pow(x-dMin,2)/dDenom);}
		  else {addAPoint(x,1.0-Math.pow(dMax-x,2)/dDenom);}
	  }
	  
	  addAPoint(dMax, 1.0000);
	  
	  //add a point at the likely value (but this will require sorting)
	  addAPoint(dLikely,Math.pow(dLikely-dMin,2)/dDenom);
	  setDistributionType("Triangular");
	  
	  //sort the distribution
	  sort();
  }
  
  /** sort the cumulative distribution by frequency
   * 
   */
  private void sort()
  {
	
	 boolean bNotSwitched;
	 
		bNotSwitched = false;
		while (bNotSwitched==false)
		{ 	
		  bNotSwitched = true;
		  for (int i=0;i<this.nPts-1;i++)
		  {	
		     
		    double dFtemp;
			double dCtemp;		    
	 
	  	    if (f[i] > f[i+1])
		    {
			  dFtemp = f[i];
			  dCtemp = c[i];
			  f[i] = f[i+1];
			  c[i] = c[i+1];
			  f[i+1] = dFtemp;
			  c[i+1] = dCtemp;
			  bNotSwitched = false;
		    }
		  }  
		}
	 
	 

  }	 

  
  /**
   * set the value from the cumulative distribution
   * for a given cumulative frequency
   * (used to set a single value -- either initial default or deterministic simulation)
   * value obtained from function getCurrentRandomValue()
   */
  public void setValue(double frequency){cValue = getValue(frequency);}
  
  /**
   * set the value from the cumulative distribution
   *  within a specified interval
   * @param interval  = interval of the distribution from 0.0 to 1.0
   * calculate value from the cumulative distribution randomly located within the specified interval
   * value obtained from function getCurrentRandomValue()
   */
  public void setValue(int interval)
  {
	//generate a frequency from the specified interval
	cValue = getValue( (iOrder[interval]*dCF)+random.getRandomNumber()*dCF );
  }
  
  /**
   * override the statistical distribution for cases where
   * certain relationships must be maintained in simulations
   * (i.e., for PVIScreen the product of the air exhange rate and residence volume
   *  must be greater than the parameter Qsoil)
   * @param cValue  value replacing internally-generated statistical value
   */
  public void overrideStatisticalValue(double cValue)
  {
	  this.cValue = cValue;
  }
  
  /** 
   * get the value from the cumulative distribution, 
   * for a specified cumulative frequency
   */
  public double getValue(double dFrequency)
  {
	double cc;
	
	cc = 0.0;
	
	for (int i=0;i<nPts-1;i++)
	{
	  if (f[i] <= dFrequency && dFrequency <= f[i+1])
	  {
		double sl;  
		sl = (c[i+1]-c[i])/(f[i+1]-f[i]);
		cc = c[i] + sl*(dFrequency-f[i]);
		cc = getSelectedValue(i,dFrequency);
		chosenFrequency = dFrequency;
		chosenIndex = i;
	  }
	}
	return cc;  
  }
  
  /**
   * 
   * @param i  chosen index for a selected interval
   * @param frequency  chosen frequence for a selected interval
   * @return  value of the distribution
   */
  public double getSelectedValue(int i, double frequency)
  {
	return c[i] + (frequency-f[i])*(c[i+1]-c[i])/(f[i+1]-f[i]);
  }
  
  
  /**
   * obtain the value of the companion satistical distribution given a chosen frequency and index
   * (these two are geneated in getValue() for a given interval in LHS 
   * @param nSimulation
   */
   public double getCompanionStatisticalDistributionValue()
   {
	 //return value of the companion statistical distribution
	 //based on the chosen distribution values in this distribution
	   
	 return sdCompanion.getSelectedValue(chosenIndex,chosenFrequency); 
   }
  
  /*
   * generate required ordering for latin hypercube sampling
   * nSimulation is the number of intervals in the Latin hypercube
   */
  public void generateOrdering(int nSimulation)
  {  
	boolean notSwitched;
	//real array of random numbers [0,1] to order integer array
	double[] order = new double[nSimulation];
	//integer array to indicate the order of intervals in latin hypercube sampling
	iOrder = new int[nSimulation];
	for(int i=0;i<nSimulation;i++)
	{
	   order[i] = random.getRandomNumber();
	   iOrder[i] = i;
	}
	
	
	
	//put the elements of "order" in order
	//using bubble sort
	System.out.println("Sorting Statistical Distribution " + this.sName);
   
	notSwitched = false;
	while (notSwitched==false)
	{ 	
	  notSwitched = true;
	  for (int i=0;i<nSimulation-1;i++)
	  {	
	    double oTemp;
	    int iTemp;
 
  	    if (order[i] > order[i+1])
	    {
		  oTemp = order[i];
		  iTemp = iOrder[i];
		  order[i] = order[i+1];
		  iOrder[i] = iOrder[i+1];
		  order[i+1] = oTemp;
		  iOrder[i+1] = iTemp;
		  notSwitched = false;
	    }
	  }  
	}
	
	//quick_sort()
  }
  
  //set up intervals for latin hypercube sampling
  public void setUpIntervals(int nLHSInterval)
  {
	  //nLHSInterval = number of intervals
	  this.nLHSInterval = nLHSInterval;
	  //divide range of 0 to 1 into intervals
	  dCF = 1.0/(double)nLHSInterval;
  }
  
  //middle value of distribution
  public double getMiddleValue()
  {
	  double dSumUp, dSumDown;
	  double dfAvg;
	  dSumUp = 0.0;
	  dSumDown = 0.0;
	  //determine the weighted average of an approximate probability density curve
	  //from the input cumulative probability curve
	  for (int i=0;i<nPts-1;i++)
	  {
		  dfAvg = 0.5*(f[i+1]-f[i]);
		  dSumUp = dSumUp + (c[i+1]-c[i])*dfAvg;
		  dSumDown = dSumDown + dfAvg;
	  }
	  dMiddleValue = dSumUp/dSumDown;
	  return dMiddleValue;
  }
  
  public void removeDuplicateEntries()
  {
	  boolean[] bKeep = new boolean[nPts];
	  double[] cTemp = new double[nPts];
	  double[] fTemp = new double[nPts];
	  int iTemp = 0;
	  for (int i=0;i<nPts;i++)
	  {
		  cTemp[i] = c[i];
		  fTemp[i] = f[i];
	  }
	  for (int j=0;j<nPts;j++){bKeep[j]=true;}
	  //eliminate duplicate entries
	  for (int i=0;i<nPts;i++)
	  {
		  if (bKeep[i])
		  {	  
		    //save entries if there are no others like them
		    for (int j=0;j<nPts;j++)
		    {
			  //for all elements j=i (element j=i is by definition a duplicate)
			  if (j!=i)
			  {	  
			    if (c[j]==c[i] && f[j]==f[i])
			    {
			    	bKeep[j] = false;
			    }
			  }    	
		    }
		  }  
	  }
	  
	  for (int i=0;i<bKeep.length;i++){if (bKeep[i]){iTemp++;}}
	  c = new double[iTemp];
	  f = new double[iTemp];
	  iTemp = 0;
	  for (int i=0;i<nPts;i++)
	  {
		  if (bKeep[i])
		  {	  
		    c[iTemp] = cTemp[i];
		    f[iTemp] = fTemp[i];
		    iTemp++;
		  }  
	  }
	  nPts = iTemp;
  }
  
  public void setAveragedParameterValue()
  {
	  removeDuplicateEntries();
	  cValue = getWeightedAverageValue();
  }
  
  //added 3-25-2020 because the average value of the degradation coefficients for a single run
  //does not match the composited average from a monte carlo run
  public void setMedianParameterValue()
  {
	  removeDuplicateEntries();
	  cValue = getMedianValue();
  }
  
//average value of distribution
  //average is weighed by the cumulative frequency
  public double getMedianValue()
  {
	
	ResistantStatisticalMeasures rsm = new ResistantStatisticalMeasures();
	double dMedianValue=0;
	
	if (nPts>1)
	{	
	  dMedianValue = rsm.calculateMedian(c);
	} 
	else if (nPts>0)
	{
		// two points only
		dMedianValue = c[0];
	}
	return dMedianValue;
  }
  
  
  //average value of distribution
  //average is weighed by the cumulative frequency
  public double getWeightedAverageValue()
  {
	double dIntervalAverage;
	double dWeightedAverage;
	
	dWeightedAverage=0.0; 
	if (nPts>0)
	{	
		if (nPts>1)
		{	
		  double dTotal = c[nPts-1]-c[0];
		  if (dTotal>0)
		  {	  
			//if these are different (as typically)  
		    for (int i=1;i<nPts;i++)
		    {	
			 dIntervalAverage = 0.5*(c[i]+c[i-1]);
			 //weight by the interval fraction
			 dWeightedAverage = dWeightedAverage + dIntervalAverage*(c[i]-c[i-1])/dTotal;
		    }
		  }  
		  else
		  {
			 //values can be the same (i.e., no difference between them)
			 //for parameters held constant
			dWeightedAverage = c[0];  
		  }	
		}
		else if (nPts==1){dWeightedAverage = c[0];}
			
	}
	 
	return dWeightedAverage;
  }
  
  //simple sets
  public void setName(String sName){this.sName = sName;}
  public void setParentName(String sParentName){this.sParentName = sParentName;}
  public void setNumberOfSimulations(int nSimulation){this.nSimulation = nSimulation;}
  public void setCompanionStatisticalDistribution(StatisticalDistribution sdCompanion){this.sdCompanion = sdCompanion;}
  public void setCalculationUnit(String sCalculationUnit){this.sCalculationUnit = sCalculationUnit;}
  //used to change the number of points defining a statistical distribution
  public void reSetNumberOfPointsDefiningStatisticalDistribution(int nPts){this.nPts = nPts;}
  //used to change a point defining a statistical distribution
  public void reSetConcentration(int nPoint, double dConcentration){this.c[nPoint] = dConcentration;}
  //used to change a frequency defining a statistical distribution
  public void reSetFrequency(int nPoint, double dFrequency){this.f[nPoint]=dFrequency;}
  //set the distribution externally (i.e., when linking a parameter)
  public void setNumberOfPointsDefiningDistribution(int nPts){this.nPts = nPts;}
  public void setConcentrationDistribution(double[] c){this.c = c;}
  public void setFrequencyDistribution(double[] f){this.f = f;}
  public void setCurrentRandomValue(double cValue){this.cValue = cValue;}
  public void setChosenFrequency(double chosenFrequency){this.chosenFrequency = chosenFrequency;}
  public void setDistributionType(String sDistributionType){this.sDistributionType = sDistributionType;}
  public void setIsThisDistributionActuallyUsed(boolean bActuallyUsed){this.bActuallyUsed = bActuallyUsed;}
  public void setRandomNumberGenerator(RandomNumber random){this.random = random;}
  
  //simple gets
  public String getName(){return sName;}
  public String getParentName(){return sParentName;}
  public String getCalculationUnit(){return sCalculationUnit;}
  public String getDistributionType(){return this.sDistributionType;}
  public double getCurrentRandomValue(){return cValue;}
  public double getChosenFrequency(){return chosenFrequency;}
  public int getChosenIndex(){return chosenIndex;}
  public int getNumberOfPointsDefiningDistribution(){return this.nPts;}
  public double getConcentration(int nPoint){return this.c[nPoint];}
  public double[] getConcentrationDistribution(){return this.c;}
  public double getFrequency(int nPoint){return this.f[nPoint];}
  public double[] getFrequencyDistribution(){return this.f;}
  public boolean getIsThisDistributionActuallyUsed(){return this.bActuallyUsed;}
 
  
  
}
