package util;

/**
 * <P>This class is for the control of the number of output digits for
 *  display, and the type of notation: either decimal or scientific.</P>
 * <P>The class uses defined ranges to decide whether or not to use
 * scientific notation. This range can be set in a constructor, or
 * the defaults can be used. The entry point is the method <code>
 * num2String()</code>.
 * <P><B>Created:</B>  19991006 </P>
 * <P><B>Revised:</B> 10-23-1999 Rounding added<BR>
 * 5-26-2000 Corrected rounding, extra "0" digit after decimal,
 * insufficient digits in mantissa<BR>
 * 6-2-2000  Original code replaced by following routines, KW<BR>
 * 6-2-2000  Changed ranges of sci. vs. dec. notation, based on client specification, KW<BR>
 * 6-7-2000  Added variable definitions, KW<BR>
 * 6-28-2000  Removed extra "}" at end of function ScientificNotation which caused error on MSIE, JWW<BR>
 * 8-25-2000  Converted from JavaScript to JAVA, JWW<BR>
 * 9-3-2013 Added simplified method for entering a double and getting the num2String result
  </P>
 * <P><B>Dependencies:</B> isNaN, Math </P>
 * <P><B>References:</B> Flanagan, D., 1997, JavaScript:
 *  The Definitive Guide, 2ed, O'Reilly and Associates  </P>
 * @author  Author of original version:<BR>
   *       Dr. Jim Weaver<BR>
   *      Regulatory Support Branch<BR>
   *      Ecosystems Research Division<BR>
   *      National Exposure Research Laboratory<BR>
   *      United States Environmental Protection Agency<BR>
   *      960 College Station Road<BR>
   *      Athens, Georgia 30605<BR>
   *      weaver.jim@epa.gov<BR>
   *      Author of current version:<BR>
   *      Kevin Weinrich<BR>
   *      OAO Corp.<BR>
   *      Contractor to US EPA<BR>
 */
public class OutputDigits {
  private String sample = "Sample";

  double value;
  int digits;
  int LowerSciExp;
  int UpperSciExp;
  int LowerDecExp;
  int UpperDecExp;

  /**
   * <P>Constructor, default, does nothing.</P>
   *<P><B>Limitations:</B>  </P>
   */
  public OutputDigits() {

  }

  /**
   * <P>Constructor, set value to format and sets defaults for the
   * cutoff values.</P>
   *<P><B>Limitations:</B>  </P>
   * @param value number to format
   */
  public OutputDigits(double value) {
    this.value = value;
    digits = 4;
    LowerSciExp = -15;
    UpperSciExp = 15;
    LowerDecExp = -7;
    UpperDecExp = 7;
  }

  /**
   * <P>Constructor, sets number and configures object with formatting
   * options.</P>
   *<P><B>Limitations:</B>  </P>
   * @param value number to convert to a string
   * @param digits number of significant digits desired in output
   * @param LowerSciExp exponent of the lower limit range for scientific notation
   * @param UpperSciExp exponent of the upper limit range for scientific notation
   * @param LowerDecExp exponent of the lower limit range for decimal notation
   * @param UpperDecExp exponent of the upper limit range for decimal notation
   */
  public OutputDigits(double value,
                      int digits,
                      int LowerSciExp,
                      int UpperSciExp,
                      int LowerDecExp,
                      int UpperDecExp) {
    this.value = value;
    this.digits = digits;
    this.LowerSciExp = LowerSciExp;
    this.UpperSciExp = UpperSciExp;
    this.LowerDecExp = LowerDecExp;
    this.UpperDecExp = UpperDecExp;
  }
  
  //method for simplified entry of a double value
  //JWW 9-3-2013
  public String num2String(double dValue)
  {
	  this.setValue(dValue);
	  return num2String();
  }

  /**
   * <P>Convert the stored double to a formatted String. </P>
   *<P><B>History:</B>
   * Function Num2String converts a number to an output string in scientific
   * or decimal notation with specified numbers of digits,
   * and ranges for each type of return (scientific or decimal)
   * Num2String is intended to preprocess a number before writing
   * to a text box.  JavaScript does not format numbers natively.
   * Language = JavaScript1.1<BR>
   * Specifications:<BR>
   * More negative than -10^7 ||| (-10^7 to -10^4) ||| (-10^4 to -10^-4) ||| (-10^-4 to -10^-7) ||| (-10^-7 to 0)<BR>
   *      "below range"       |||       sci.       |||        dec        |||        sci         |||    sci<BR>
   * (0 to 10^-7) ||| (10^-7 to 10^-4) ||| (10^-4 to 10^4) ||| (10^4 to 10^7) |||  > 10^7<BR>
   *    sci       |||       sci        |||        dec      |||      sci       ||| "above range"<BR>
   *   </P>
   *<P><B>Limitations:</B> The code should distinguish between the 0 to 10^-7
   * and 10^-7 to 10^-4 ranges, even though the notation
   * chosen would be the same.  There may be a need later to use something
   * else for the 0 to 10^-7 range.</P>
   * @return   string = representation of value, formatted according to inputs<BR>
   * Example:<BR>
   * var value = 345010;<BR>
   * var output = Num2String(value,3,-7,7,-4,4);<BR>
   * output has the value  "3.45e5"<BR>
   * var value = 345010;<BR>
   * var output = Num2String(value,5,-7,7,-4,4);<BR>
   * output has the value  "3.4501e5"<BR>
   * var value = 3450.1;<BR>
   * var output = Num2String(value,3,-7,7,-4,4);<BR>
   * output has the value  "3450."<BR>
   */
  public String num2String() {
    int iMagnitude; // iMagnitude is the lowest power of 10
                    // that the input value is less than
    String strNewVal; // Build string output
    String strTemp;
    String strMantissa; // Mantissa part of scientific notation
    String strExponent; // Exponent part of scientific notation
    boolean bNegativeNum; // Flag for negative input value
    double dThisLimit; // Some power of 10 , used as a limit to test against
    double dAbsValue; // Absolute value of input value
    double dMagnifier; // dMagnifier is the power of 10 necessary
                        // to convert "value" to a number whose units
                        //  are the rightmost significant digit before rounding.
    double dAdjustedVal; // value, after it's converted  to a number whose units
    //  are the rightmost significant digit before rounding.
    // e.g., if value = 0.04516, and digits = 3, dAdjustedVal = 451.6
    double dRoundedVal; // rounded dAdjustedVal - e.g., 452
    double dLowerDecLimit; // Some power of 10, lower limit of decimal notation
    double dUpperDecLimit; // Some power of 10, upper limit of decimal notation
    double dInnerLimit; // Some power of 10, narrow limit around 0.0
                       // for inner range of scientific notation

    //return an empty string for non numbers

    //if (isNaN(value)) {strTemp = " "; return strTemp;}

    //check for zero
    if (value == 0.) {
      strTemp = "0.";
      return strTemp;
    }

    // Check for out of range
    dThisLimit = Math.pow(10, UpperSciExp);
    if (value > dThisLimit) {
      return "above range";
    }

    dThisLimit = -Math.pow(10, UpperSciExp);
    if (value < dThisLimit) {
      return "below range";
    }

    // Check for negative # - save to be appended just before return
    if (value < 0) {
      bNegativeNum = true;
    }
    else {
      bNegativeNum = false;
    }
    dAbsValue = Math.abs(value);

    // Determine magnitude of #
    // -20 is a kludge
    for (iMagnitude = UpperSciExp; iMagnitude >= -20; iMagnitude--) {
      dThisLimit = Math.pow(10, iMagnitude);
      if (dAbsValue >= dThisLimit) {
        iMagnitude++;
        // iMagnitude is the lowest power of 10 that the input value is less than
        //  e.g., value=100, iMagnitude = 3
        //  e.g., value= 99, iMagnitude = 2
        break;
      }
    }

    dMagnifier = Math.pow(10, digits - iMagnitude);
    dAdjustedVal = dAbsValue * dMagnifier;
    dRoundedVal = Math.round(dAdjustedVal);
    dLowerDecLimit = Math.pow(10, LowerDecExp);
    dUpperDecLimit = Math.pow(10, UpperDecExp);
    dInnerLimit = Math.pow(10, LowerSciExp);

    if (dAbsValue < dInnerLimit) {
      // Narrow window around 0.0
      return scientificNotation(dRoundedVal, iMagnitude, bNegativeNum);
    }
    else {
      if (dLowerDecLimit <= dAbsValue && dAbsValue <= dUpperDecLimit) {
        // Use Regular Notation
        return regularNotation(dRoundedVal, iMagnitude, dMagnifier,
                               bNegativeNum);
      }
      else {
        // Use Scientific Notation
        return scientificNotation(dRoundedVal, iMagnitude, bNegativeNum);
      }
    }
  }

  /**
   * <P>Format the number without scientific notation. </P>
   *<P><B>Limitations:</B>  </P>
   * @param dRoundedVal rounded dAdjustedVal - e.g., 451.6 --> 452
   * @param iMagnitude lowest power of 10 that value is less than
   * @param dMagnifier the power of 10 necessary
   * to convert number to a number whose units
   * are the rightmost significant digit before rounding.
   * @param bNegativeNum flag to indicate negative number
   * @return the formatted result
   */
  private String regularNotation(double dRoundedVal,
                                 int iMagnitude,
                                 double dMagnifier,
                                 boolean bNegativeNum) {
    String strTemp;
    String strNewVal; // Build string output
    double dNewVal; // After rounding to eliminate extra digits, put back to right scale
    int iAddZeros; // Number of 0's that need to be appended to show proper # of digits
    int iPad; // Loop counter for padding 0's
    int iDecimalLoc; // Location of decimal in string
    int iNonZeroLoc; // Location of 1st non-zero (and non-.) # in string
    int iNeedDigits; // How many digits do we need, including decimal and leading 0's?
    int iELoc;

    //very annoying necessity
    String strDot;
    strDot = ".";
    char charDot = strDot.charAt(0);
    //very annoying necessity
    String strZero;
    strZero = "0";
    char charZero = strZero.charAt(0);

    dNewVal = dRoundedVal / dMagnifier;
    strNewVal = "" + dNewVal;
    iELoc = strNewVal.indexOf("e");
    if (iELoc > -1) {
      // Number already in scientific notation
      strTemp = strNewVal.substring(0, iELoc);
      if (strTemp.indexOf(".") == -1) {
        // We need to add a ".", since "/ dMagnifier" didn't.
        strTemp = strTemp + ".";
      }
      // Ensure Mantissa has right # of digits
      // 1 for "."
      iAddZeros = digits + 1 - strTemp.length();
      for (iPad = 0; iPad < iAddZeros; iPad++) {
        strTemp = strTemp + "0";
      }
      strTemp += strNewVal.substring(iELoc);
      if (bNegativeNum) {
        strTemp = "-" + strTemp;
      }
      return strTemp;
    }

    if (strNewVal.indexOf(".") == -1) {
      // We need to add a ".", since "/ dMagnifier" didn't.
      strNewVal = strNewVal + ".";
    }
    if (iMagnitude == 0) {

      if (charDot == strNewVal.charAt(0)) {
        // In Netscape, we need to prepend a "0", since "/ dMagnifier" didn't.
        strNewVal = "0" + strNewVal;
      }
    }

    for (iNonZeroLoc = 0; iNonZeroLoc < strNewVal.length(); iNonZeroLoc++) {
      if (strNewVal.charAt(iNonZeroLoc) != charZero
          && strNewVal.charAt(iNonZeroLoc) != charDot) {
        break;
      }
    }

    iDecimalLoc = strNewVal.indexOf(".");
    if (iDecimalLoc > iNonZeroLoc) {
      iNeedDigits = digits + 1;
    }
    else {
      iNeedDigits = digits;
    }

    // e.g., "0.0001" w/ 2 digits needs (5 - 6 + 2) = 1 zero appended
    iAddZeros = iNeedDigits + iNonZeroLoc - strNewVal.length();
    for (iPad = 0; iPad < iAddZeros; iPad++) {
      strNewVal = strNewVal + "0";
    }

    if (bNegativeNum) {
      strNewVal = "-" + strNewVal;
    }
    return strNewVal;
  }

  /**
   * <P>Format the number with scientific notation. </P>
   *<P><B>Limitations:</B>  </P>
   * @param dRoundedVal rounded dAdjustedVal - e.g., 451.6 --> 452
   * @param iMagnitude lowest power of 10 that value is less than
   * to convert number to a number whose units
   * are the rightmost significant digit before rounding.
   * @param bNegativeNum flag to indicate negative number
   * @return the formatted result
   */
  private String scientificNotation(double dRoundedVal,
                                    int iMagnitude,
                                    boolean bNegativeNum) {
    double dMantissa; // dRoundedVal in the scale of scientific notation 0.0 - 9.999...
    String strTemp;
    int iPad; // Loop counter for padding 0's
    String strMantissa; // Mantissa part of scientific notation
    String strExponent; // Exponent part of scientific notation

    dMantissa = dRoundedVal / Math.pow(10, digits - 1);
    if (dMantissa >= 10.0) {
      dMantissa = dRoundedVal / Math.pow(10, digits);
      iMagnitude++;
    }
    strTemp = "" + dMantissa;
    if (strTemp.indexOf(".") == -1) {
      strTemp = strTemp + ".";
    }
    // 1 for "."
    for (iPad = 0; strTemp.length() < digits + 1; iPad++) {
      // Pad it
      strTemp = strTemp + "0";
    }
    // Ensure imprecision of division doesn't tag on extra digits (like "000001")
    strMantissa = strTemp.substring(0, digits + 1);
    strExponent = "e" + (iMagnitude - 1);

    if (bNegativeNum) {
      strMantissa = "-" + strMantissa;
    }
    return strMantissa + strExponent;

  }

  //sets
  public void setSample(String newSample) {sample = newSample;}
  
  public void setValue(double value)
  {
    this.value = value;
    digits = 4;
    LowerSciExp = -15;
    UpperSciExp = 15;
    LowerDecExp = -7;
    UpperDecExp = 7;
  }

  //gets
  public String getSample() {return sample;}



}
