package util;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;

import statistics.DataFitting;
import statistics.Histogram;
import statistics.Interval;
import statistics.ResistantStatisticalMeasures;
import statistics.ResultsColumn;

/**
 * OutputProcessor   processes output from the RiverModel (adaptable to others) to produce
 *                   1)simple statistical measures:   min, max, range, mean, variance, standard deviation
 *                   2)resistant statistical measures:  min, 5th percentile, first quartile, median,
 *                                                    third quartile, median + 1.5*interquartile range,
 *                                                    95th percentile, max
 *                   3)histograms and cumulative probability distributions
 *                   
 *                   
 * @author JWEAVER
 * 
 * Jim Weaver
 * Research Hydrologist
 * US EPA
 * National Risk Management Research Laboratory
 * Occife of Research and Development
 * United States Environmental Protection Agency
 * 919 Kerr Research Drive
 * Ada, OK 74820
 * 
 * 11-28-2012
 * 
 * 
 *
 */
public class OutputProcessor 
{
	private String sName;
	private String sResultFileName;
    private FileLogger fl;
    private FileLogger flOut;
    private int iFrequencyInterval;
	private int iCount;
	private int iColumns;
	private String sColumnToFind;
	private String sHeadOne;
	private String sHeadTwo;

	private int iNumberOfResults;	
	private String[] sString;
	private double[] dValue;
	private double[] dValue2;
	private double[] dFreq;
	private Integer[] iIndex;
	private ArrayList<ResultsColumn> alResultsColumn;
	private ArrayList<ResultsColumn> alResultsColumnSubordinateLessThanCriterion;
	private ArrayList<ResultsColumn> alResultsColumnSubordinateGreaterThanCriterion;
	private String[] sHeadingExtension;
	private ResistantStatisticalMeasures rsm;
	private Histogram histogram;
	private DataFitting df;
	private String sFittingType;
	private double dFittingA, dFittingB;
	private UnitConverter uc;
	
	String[] sNameChemical = new String[50];
	double[] dRiskConcentration = new double[50];
	double[] dRiskLevel = new double[50];
	double[] dRfC = new double[50];
	int iNumberChemical = 0;
	int iNumberRisk = 0;
	int iNumberRiskLevel = 0;
	int iNumberRfC = 0;
	String sOutputRiskConcentrationUnit = "";
	private Stripper stripper;
	//flag for a column being input (true) or else (false)
	private boolean bInput;
	
	public OutputProcessor(){}
	
	public void initial()
	{
		sName = "OP";
		flOut = new FileLogger();
		stripper = new Stripper();
		sHeadingExtension = new String[2];
		bInput=true;
		
	}
	
 	
	/**
	 * readSortedResultsToArrayList   reads the sorted results file and places each column of results into a
	 *   data container (ResultsColumn)
	 *   simple statistical measures (min, max, average, variance, range) from the ResultsColumns
	 *   using generateStatistics()
	 *   
	 *   The results are written to the sorted results file by writeStatistics()
	 *   
	 *   
	 * @throws IOException
	 */
	public void readSortedResultsToArrayList() throws IOException
	{
		alResultsColumn = new ArrayList<ResultsColumn>();
		Stripper s = new Stripper();
		//open the results file
		FileReader fr = new FileReader(sResultFileName);
		BufferedReader br = new BufferedReader(fr);
		String currentLine;
		String[] lineArray = {""};
		
		//entire line of character input
		sString = new String[this.iNumberOfResults];
		iCount = 0;
		this.iNumberChemical = 0;
		this.iNumberRisk = 0;
		
		while ((currentLine = br.readLine()) != null) 
		{
		  String key, key2, key3;		  
		  key = ""; key2 = "";
		   		  
		  lineArray = currentLine.split(",",0);
		  key = lineArray[0];
		  
		  
		  /*  Unit is no longer listed in the main output file, rather is embedded in the unitconverter uc
		  //output unit for risk concentration
		  if("Control".equalsIgnoreCase(stripper.removeCharacter(' ',key)))
		  {	  
			  key2 = lineArray[1];
			  if ("OutputUnit".equalsIgnoreCase(stripper.removeCharacter(' ',key2)))
			  {
				  key3 = lineArray[2];
				  if ("RiskConcentration".equalsIgnoreCase(stripper.removeCharacter(' ',key3)))
				  {
					  this.sOutputRiskConcentrationUnit = lineArray[3];
				  }
			  }
		  }
		  */
		 
		  
		  ResultsColumn rc;
		  //the risk level lines
		  if (key.equalsIgnoreCase("Risk Level"))
		  {
			
			 key2 = lineArray[1];
			 //get chemical name
			 if (key2.equalsIgnoreCase("Chemical"))
			 {
				 //run through all elements in the line array
				 for (int i=2;i<lineArray.length;i++)
				 {
					 if (!(lineArray[i].equals("")))
					 {
						 this.sNameChemical[iNumberChemical] = lineArray[i];
						 this.iNumberChemical++;
					 }
				 }
			 }
			 
			 //get value associated with names
			 else if (key2.equalsIgnoreCase("value"))
			 {
				 //run through all elements in the line array
				 for (int i=2;i<lineArray.length;i++)
				 {
					 if (!(lineArray[i].equals("")))
					 {
						 try
						 {
                           this.dRiskConcentration[iNumberRisk] = Double.parseDouble(lineArray[i]);
						   this.iNumberRisk++;
						 }
						 catch(Exception e)
						 {
							 //risk not a valid number
							 System.out.println("problem with risk Concentration input");
						 }
					 }
				 }
				 
			 }
			 else if (key2.equalsIgnoreCase("risk level"))
			 {
				//run through all elements in the line array
				 for (int i=2;i<lineArray.length;i++)
				 {
					 if (!(lineArray[i].equals("")))
					 {
						 try
						 {
                           this.dRiskLevel[iNumberRiskLevel] = Double.parseDouble(lineArray[i]);
						   this.iNumberRiskLevel++;
						 }
						 catch(Exception e)
						 {
							 //risk not a valid number
							 System.out.println("problem with risk level input");
						 }
					 }
				 } 
			 }
			 else if (key2.equalsIgnoreCase("RfC"))
			 {
				//run through all elements in the line array
				 for (int i=2;i<lineArray.length;i++)
				 {
					 if (!(lineArray[i].equals("")))
					 {
						 try
						 {
                           this.dRfC[iNumberRfC] = Double.parseDouble(lineArray[i]);
						   this.iNumberRfC++;
						 }
						 catch(Exception e)
						 {
							 //risk not a valid number
							 System.out.println("problem with risk (RfC) input");
						 }
					 }
				 } 				 
			 }
		  }
		  
		  
		  bInput = true;
		  
		  //the sorted monte carlo results lines
		  //the function sortByMergeSortandReWrite has already been run on the data
		  if (key.equalsIgnoreCase("Sorted Result") || key.equalsIgnoreCase("Sorted Results"))    
		  {
			  
			iColumns = lineArray.length;
			if (iColumns>1)
			{	
				 key2 = lineArray[1];
				 if (key2.equalsIgnoreCase("Heading"))
				 {
					 //save the heading for later output
					 sHeadOne = currentLine;
					 for (int i=0;i<lineArray.length;i++)
					 {
						//loop over each entry in the lineArray
						//each entry belongs to a Results Column
						boolean bDataColumn=true;
						//heading
					
						if (s.areTheyEqual(lineArray[i],"Heading")
						 || s.areTheyEqual(lineArray[i], "Sorted Results")		 
						 || s.areTheyEqual(lineArray[i], "Sorted Result")
						 || s.areTheyEqual(lineArray[i], ""))
						{
					      //column does not contain data		
					      bDataColumn = false;
						}
						//(some of the columns will contain no data:  will check for these later
						//create a new ResultsColumn for each column in the output file
					    rc = new ResultsColumn();
						rc.setNumber(this.iNumberOfResults);
						rc.initializeArray();
						rc.setControlCharacter("*ValidData*");
						rc.setHeading(lineArray[i]);
						//the first columns are inputs, after the "**" marker, the columns are output
						if (s.areTheyEqual(lineArray[i],"**"))
						{
							bInput = false;
						}
						rc.setInputOrNot(bInput);
						rc.setIsThisADoubleDataColumn(bDataColumn);
					
						alResultsColumn.add(rc);
					 }
					 
				 }
				 else if (key2.equalsIgnoreCase("Unit/Count"))
				 {
					 //save the unit string for later output
					 sHeadTwo=currentLine;
					 
					 for (int i=0;i<lineArray.length;i++)
					 {
					   boolean bDataColumn = true;	 
					 
					   if (lineArray[i].equalsIgnoreCase("Heading") 
					    || lineArray[i].equalsIgnoreCase("Unit/Count"))
					   {
					     //column does not contain data		
					     bDataColumn = false;
					     rc = alResultsColumn.get(i);
					     rc.setInputOrNot(false); 
					   }				 
					     rc = alResultsColumn.get(i);
						 rc.setIsThisADoubleDataColumn(bDataColumn); 
						 rc.setUnit(lineArray[i]);
					  }
					 
				 }
				 else
				 {
					 //save entire data row
					 iCount++;
					 
					 //these are the data items
					 for (int i=0;i<lineArray.length;i++)
					 {
						 double dValue;
						 boolean bDataColumn = true;
					 
						 rc = alResultsColumn.get(i);
						 try
						 {
							 dValue = Double.parseDouble(lineArray[i]);
						     bDataColumn = true;	 
						 }
						 catch(Exception e)
						 {
							 bDataColumn = false;
							 dValue = 0.0;
						 }
						 rc.setIsThisADoubleDataColumn(bDataColumn);
						 rc.addValue(dValue,"*ValidData*",lineArray[i]);
					 }			     
				 }
		    } //if iColumns > 1	 
		  }  //if key
		}  //while
		
		
		  //validate the double data using the control character in sControlCharacter
		  System.out.println("validating data");
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  rc.validateData();
		  }

		
	  //close the file
	  fr.close();
	}
	
	/**
	 * writeStatistics   write out the simple statistical results for each column of data
	 */
	public void writeStatistics()
	{
	  int iCount = 0;
	  String sOut = "";
	  String sOut2 = "";
	  
	  this.sOutputRiskConcentrationUnit = uc.getOutputNameFromType("RiskConcentration");
	  
	  flOut.logMessage("");
	  flOut.logMessage("Simple Statistics:");	  
	  //minimum:	
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.getMin() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "Minimum,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);
	  
	  iCount = 0;
	  sOut = "";
	  //average:	
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.getAverage() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "Average,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);	  
	  
	  iCount = 0;
	  sOut = "";
	  //maximum:	
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.getMax() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "Maximum,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);
	  flOut.logMessage("");
	  
	  iCount = 0;
	  sOut = "";
	  //range:	
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.getRange() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "Range,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);	 
	  
	  iCount = 0;
	  sOut = "";
	  sOut2 = "";
	  //Variance and standard deviation
	  for (ResultsColumn rc: alResultsColumn)
	  {
		double dTemp;
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			dTemp = rc.getVariance();
			sOut = sOut + dTemp + ",";
			sOut2 = sOut2 + Math.sqrt(dTemp) + ",";
		}
		else
		{
			if (iCount == 0)
			{
				sOut = "Variance,";
			    sOut2 = "Standard Deviation,";	
			}
			else 
			{
				sOut = sOut + ",";
			    sOut2 = sOut2 + ",";	
			}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);	 	  
	  flOut.logMessage(sOut2);
	  
	  //the resistant statistical measures
	  flOut.logMessage("");
	  flOut.logMessage("Resistant Statistical Measures");
	  iCount = 0;
	  sOut = "";
	  //minimum
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.getRSMMinimum() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "Minimum,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);	 	  
	  
	  iCount = 0;
	  sOut = "";
	  //5th percentile
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.get5thPercentile() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "5th percentile,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);	
	  
	  iCount = 0;
	  sOut = "";
	  //First Quartile
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.getFirstQuartile() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "First Quartile,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);
	  
	  iCount = 0;
	  sOut = "";
	  sOut2 = "";
	  //Median
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			double dTemp = rc.getMedian();
			sOut = sOut + dTemp + ",";
			dTemp = dTemp + 1.5*rc.getInterQuartileRange();
			sOut2 = sOut2 + dTemp + ",";
		}
		else
		{
			if (iCount == 0)
			{
				sOut = "Median,";
				sOut2 = "Median + 1/2 InterQuartile Range,";
			}
			else 
			{
				sOut = sOut + ",";
				sOut2 = sOut2 + ",";
			}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);	
	  
	  iCount = 0;
	  sOut = "";
	  //Third Quartile
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.getThirdQuartile() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "Third Quartile,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);	
	  
	  //median + 1.5 InterQuartile Range
	  flOut.logMessage(sOut2);

	  iCount = 0;
	  sOut = "";
	  //95th percentile
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.get95thPercentile() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "95th percentile,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);		  
	  
	  iCount = 0;
	  sOut = "";
	  //Maximum
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.getRSMMaximum() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "Maximum,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);	  	  
	  iCount = 0;
	  
	  
	  sOut = "";
	  //InterQuartileRange
	  for (ResultsColumn rc: alResultsColumn)
	  {
		if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
		{
			sOut = sOut + rc.getInterQuartileRange() + ",";
		}
		else
		{
			if (iCount == 0){sOut = "Inter-quartile Range,";}
			else {sOut = sOut + ",";}
		}
		iCount++;
	  }
	  flOut.logMessage(sOut);	  	  	  
	}
	
	/**
	 * 	generateStatistics()  for each column of output that contains results
	 */ 
	public void generateStatistics() 
	{	  	
	  for (ResultsColumn rc:  alResultsColumn){generateStatistics(rc);}
	}
	
	
	
    /**
     * generateStatistics(ResultsColumn rc)   generate simple statistics for each column of data 
     * ( internal class: ResultsColumn  )
     * @param rc   a column of results of type ResultsColumn
     */
	private void generateStatistics(ResultsColumn rc)
	{
	  double dMax;
	  double dMin;	
	  double dAverage;
	  double dVar;
	  int iNumber;
	 
	  
	  dMin = 1.e20;
	  dMax = -1.e20;
	  
	  //generate the statistical measures for the data columns
	  if (rc.getIsThisADoubleDataColumn())
	  {	  
	  
	    //get data from the DataColumn object
	    double[] dValue = rc.getAllDoubleValues();
	    iNumber = dValue.length;
	    
	    //mean, min and max, variance
	    double dSumX = 0.0;
	    double dSumXX = 0.0;
	    //variance
        //calculation formula from
        //Moore and McCabe, 1993, Introduction to the practice of Statistics
        //2nd ed, W.H. Freeman and Company, New York, 854pp.
        //see page 48
        for (int i=0; i<iNumber; i++)
        {
    	    //find min and max
    	    if (dValue[i]<dMin){dMin = dValue[i];}
    	    if (dValue[i]>dMax){dMax = dValue[i];}
    	    //find average
    	    dSumX = dSumX + dValue[i];
    	    dSumXX = dSumXX + dValue[i]*dValue[i];
        }
        dAverage = dSumX/(double)iNumber;
        dVar = (1/(double)(iNumber-1))*(dSumXX - (1/(double)iNumber)*dSumX*dSumX);
        
        //save the results
        rc.setAverage(dAverage);
        rc.setMin(dMin);
        rc.setMax(dMax);
        rc.setRange(dMax-dMin);
        rc.setVariance(dVar);
        
        //resistant measures using the ResistantStatisticalMeasures object
        rsm.setData(dValue);
        rsm.determineMeasures();
        rc.set5thPercentile(rsm.get5thPercentile());
        rc.setFirstQuartile(rsm.getFirstQuartile());
        rc.setMedian(rsm.getMedian());
        rc.setThirdQuartile(rsm.getThirdQuartile());
        rc.set95thPercentile(rsm.get95thPercentile());
        rc.setInterQuartileRange(rsm.getInterQuartileRange());
        rc.setRSMMinimum(rsm.getMinimum());
        rc.setRSMMaximum(rsm.getMaximum());
        
        //indicate that the statistics have been calculated
        rc.setHaveTheStatisticsBeenCalculated(true);
	  } //if
	  else
	  {
		//not a data column -- no statistical calculation
		rc.setHaveTheStatisticsBeenCalculated(false);
	  } //if
	}
	
	
	
    /**
     * openOutputFile()
     * create the output file for the sorted results
     *   (key name to the result file name by adding "sorted")
     *   
     *   returns name of output file
     */
	public void openOutputFile(String sOutFileName) {flOut = new FileLogger(sOutFileName);} 
	
	
	/**
	 * processResults   sort the results according to the values selected in readResults(String sColumnToFind)
	 */
	public void sortResults()
	{
		  
	  double dValueTemp;
	  int iValueTemp;
	  String sValueTemp;
	  boolean bChanged;
		   
		 
	  bChanged = true;
	   
	   
	   
	   //now the sorting starts:
	   while (bChanged == true) 
	   {
		 bChanged = false;
	     for (int i=0;i<iCount-1;i++)
	     {
		    //sort from lowest to highest 
		    if (dValue[i]>dValue[i+1])
		    {
			  dValueTemp = this.dValue[i+1];
			  iValueTemp = this.iIndex[i+1];
			  sValueTemp = this.sString[i+1];
			  this.dValue[i+1] = this.dValue[i];
			  this.iIndex[i+1] = this.iIndex[i];
			  this.sString[i+1] = this.sString[i];
			  this.dValue[i] = dValueTemp;
			  this.iIndex[i] = iValueTemp;
			  this.sString[i] = sValueTemp;
			  bChanged = true;
		    }
	      }
	   }
	   
	   //assign cumulative probability frequencies
	   assignFrequencies();
	}
	
	/**
	 * Merge sort results read in from the readResults(String sColumnToSortOn) function
	 */
	public void sortByMergeSortAndReWrite()
	{
	  MergeSort ms = new MergeSort();  
	  ms.sortWithDependentString(dValue,sString);
	  
	  flOut.logMessage("Sorted Results");
	  flOut.logMessage("");
	  writeHeadings("Sorted");
	  //write from highest to lowest:
	  for (int i=sString.length-1;i>=0;i--)
	  {
		  flOut.logMessage("Sorted "+sString[i]);
	  }
	  flOut.logMessage(" ");
	  
	}
	
	
	
	
	
	/**
	 * sortResultsColumn    sort the results based on the specified name of the ResultsColumn object
	 *                      //column has to be all double to sort
	 * @sResultsColumnName  name of column heading to sort
	 */
	public void sortResultsColumn(String sResultsColumnName)
	{
	  double dValueTemp;
      int iCount;
	  boolean bChanged;
	  boolean bAllDouble;
	  double[] dValidated;
	  double[] dValidatedFrequency;
	  dValidated = null;
	  iCount = 0;
	  bAllDouble = false;
	  
	  MergeSort ms = new MergeSort();
	  
	  //find the chosen result column
	  for (ResultsColumn rc: alResultsColumn)
	  {
		  if (rc.getHeading().equalsIgnoreCase(sResultsColumnName))
		  {
			  //specified column has been found
			  dValidated = rc.getValidatedData();
			  break;
		  }
	  }	 
 	  
	  iCount = dValidated.length;
	  
	  //use merge sort to increase speed  jww 3-29-2016
	  dValidated = ms.sort(dValidated);
	  
	  //simple sort (commented out jww 3-29-2016)
	  /*
	   bChanged = true;
	   //now the sorting starts:
	   while (bChanged == true) 
	   {
		 bChanged = false;
	     for (int i=0;i<iCount-1;i++)
	     {
		    //sort from lowest to highest 
		    if 
		    (
		      (bAllDouble==true  &&  (dValidated[i]>dValidated[i+1]))
		    )
		    {
			  dValueTemp = dValidated[i];
			  dValidated[i] = dValidated[i+1];
			  dValidated[i+1] = dValueTemp;
			  bChanged = true;
		    }
	     }
	   }
	   */
    
	   //assign cumulative probability frequencies
	   dValidatedFrequency = assignRankings(dValidated);
 	   
	   //find the chosen result column and load the sorted results back to the column
	   for (ResultsColumn rc: alResultsColumn)
	   {
		 if (rc.getHeading().equalsIgnoreCase(sResultsColumnName))
		 {
		   //specified column has been found
		   rc.setAllDoubleValues(dValidated);
		   rc.setCumulativeDistributionFrequencies(dValidatedFrequency);
		   rc.setIsTheCumulativeDistributionSet(true);
		   break;
		  }
	    }	 	   
 	   
	}	
	
	/**
	 * assignFrequencies  assign frequencies to the cumulative distribution curve
	 */
	private double[] assignRankings(double[] dValue)
	{
	   double[] dFreq = new double[dValue.length];
	   
	   for (int i=0;i<dValue.length;i++)
	   {
		   dFreq[i] = (double)i/(double)(dValue.length-1);
	   }
	   
	   return dFreq;
	}
	
	
	
	/**
	 * processResults   sort the coupled results sort on the first entry in readCoupledResults(String sColumnToFindOne, String sColumnToFindTwo)
	 */	
	public void sortCoupledResults()
	{
		  
	  double dValueTempOne;
	  double dValueTempTwo;
	  int iValueTemp;
	  String sValueTemp;
	  boolean bChanged;
		   
		 
	   bChanged = true;
	   //now the sorting starts:
	   while (bChanged == true) 
	   {
		 bChanged = false;
	     for (int i=0;i<iCount-1;i++)
	     {
		    //sort from lowest to highest 
		    if (dValue[i]>dValue[i+1])
		    {
			  dValueTempOne = this.dValue[i+1];
			  dValueTempTwo = this.dValue2[i+1];
			  iValueTemp = this.iIndex[i+1];
			  sValueTemp = this.sString[i+1];
			  this.dValue[i+1] = this.dValue[i];
			  this.dValue2[i+1] = this.dValue2[i];
			  this.iIndex[i+1] = this.iIndex[i];
			  this.sString[i+1] = this.sString[i];
			  this.dValue[i] = dValueTempOne;
			  this.dValue2[i] = dValueTempTwo;
			  this.iIndex[i] = iValueTemp;
			  this.sString[i] = sValueTemp;
			  bChanged = true;
		    }
	      }
	   }
	   
	   //assign cumulative probability frequencies
	   assignFrequencies();
	}	
	
	
	
	
	/**
	 * assignFrequencies  assign frequencies to the cumulative distribution curve
	 */
	private void assignFrequencies()
	{
	   for (int i=0;i<dValue.length;i++)
	   {
		   dFreq[i] = (double)i/(double)(dValue.length-1);
	   }
	}
	
	
	/**
	 * fitCoupledResults   fit a 2 parameter function to the coupled results
	 *                     
	 * @param sFittingType        if sType = linear, use the linear relationship   y = a + bx
	 *                     if sType = power, use the power relationship y = ax^b
	 *                     
	 *                     
	 * 12-7-2012  (in memorandum  12-7-1941)                    
	 *                     
	 */
	public void fitCoupledResults(String sFittingType)
	{
	  this.sFittingType = sFittingType;
	  
	  //instantiate the DataFitting object	
	  df = new DataFitting();
	  //set the type to fit
	  df.setFittingType(sFittingType);
	  //set the data
	  //the primary variable is the independent variable
	  df.setXValues(dValue);
	  //the coupled variable is the dependent variable
	  df.setYValues(dValue2);
	  //set the number of values
	  df.setNumberOfValues(iCount);
	  
	  //fit the data
	  df.fitTheData();
	  
	  //get the parameter results
	  dFittingA = df.getIntercept();
	  dFittingB = df.getSlope();
	}
	
	/**
	 * writeCoupledFittingResults   write out a set of fitted data
	 */
	public void writeCoupledFittingResults() {df.writeResults(flOut);}
	
	
	/**
	 * makeCoupledHistogram   using sorted results (sortCoupledResults()) from reading coupled data (readCoupledResults(String1,String2))
	 *                        an histogram is made from the first-read result--column String1
	 *                        an histogram of the second-read column is made for each interval of the first-read histogram
	 *                        
	 */
	public void makeCoupledHistogram()
	{
		histogram = new Histogram();
		histogram.setOutputUnit(this.sOutputRiskConcentrationUnit);
		histogram.setIntervalFrequency(iFrequencyInterval);
		histogram.setCoupledHistogram(true);
		histogram.setCoupledHistogramData(this.dValue2);
		histogram.generateHistogram(dValue,iCount);
	}
	

	
 
	public void makeHistogram()
	{
		histogram = new Histogram();
		histogram.setOutputUnit(this.sOutputRiskConcentrationUnit);
		histogram.setIntervalFrequency(iFrequencyInterval);
		histogram.generateHistogram(dValue,iCount);
	}
	
    boolean bHistogramMade;
	ArrayList<Interval> alLogInterval;
	public void makeLogHistogram()
	{
       histogram  = new Histogram();
       histogram.setOutputUnit(this.sOutputRiskConcentrationUnit);
       histogram.generateLogHistogram(dValue,iCount);
	}	
	
	
	
	public void makeResultsColumnLogHistogram(String sResultsColumnName, String sHistogramChemical)
	{
		double[] dValidated = null;
		
		for (ResultsColumn rc:  alResultsColumn)
		{
			if (rc.getHeading().equalsIgnoreCase(sResultsColumnName))
			{
				dValidated = rc.getAllDoubleValues();
				break;
			}
		}
		
		histogram = new Histogram();
		histogram.setHistogramChemical(sHistogramChemical);
		histogram.setIntervalFrequency(iFrequencyInterval);
		histogram.setOutputUnit(this.sOutputRiskConcentrationUnit);
		histogram.determineLevelOfConcern(sHistogramChemical, this.dRiskConcentration, this.dRiskLevel, this.sNameChemical, this.iNumberRisk);
        histogram.determineRfC(sHistogramChemical, this.dRfC, this.sNameChemical, iNumberRfC);
		histogram.makeLogHistogram(dValidated,dValidated.length);
	}
	
	/**
	 * writeSortedData    write the sorted data set to the output file
	 */
    public void writeSortedData()
    {
    	//skip a space
		flOut.logMessage("");
		//headings saved from the original input
		flOut.logMessage(sHeadOne);
		flOut.logMessage(sHeadTwo);
	    
		//the sorted data by string (contains commas to create csv)
		for (int i=0;i<iCount;i++)
		{
			flOut.logMessage(sString[i]);
		}
		
    }
	
    
    //read the sorted results and characterize the output
    public void characterizeAndWriteData() throws IOException
    {
        String sOut = "";
        	
    	//characterize the input data in the result columns
    	for (ResultsColumn rc: this.alResultsColumn)
    	{
    		rc.characterizeData();
    	}
     
      
    	//now rewrite the results in characterized form
    	flOut.logMessage("");
    	flOut.logMessage("Characterized " + this.sHeadOne);
    	flOut.logMessage("Characterized " + this.sHeadTwo);
    	for (int i=0;i<this.iNumberOfResults;i++)
    	{
    		sOut = "Characterized,";
    		for (ResultsColumn rc: this.alResultsColumn)
    		{
    			if (rc.getInputOrNot())
    			{
    				if (rc.getIsThisADoubleDataColumn())
    				{//if an input parameter
    					sOut = sOut + rc.getCharacterizedValue(i) + ",";
    				}	
    			}
    			else
    			{
    				//not input
    				sOut = sOut + rc.getOriginalValue(i) + ",";
    			} 
    		}
    		flOut.logMessage(sOut);
    	}
	
        flOut.logMessage("");
       
       
    }
       
    


	//write out the histogram
	public void writeHistogram(String sTitle)
	{
	    //setRiskLevels(histogram);
		histogram.setHistogramTitle(sTitle);
		histogram.writeLinearHistogram(flOut);
	}
	
	//write out the histogram
	public void writeLogHistogram(String sTitle)
	{
		//setRiskLevels(histogram);
		histogram.setHistogramTitle(sTitle);
		histogram.writeLogHistogram(flOut);
	}
	
	
	//write out the coupled histograms
	public void writeCoupledHistogram(String sTitle)
	{
		histogram.setHistogramTitle(sTitle);
		histogram.writeCoupledHistgram(flOut);
	}
		
	//place risk levels into the histogram
	//the risk levels were read into the output processor in the 
	//readResultsToArrayList routine
	//they now need to be matched to the chemical of the histogram
	private void setRiskLevels(Histogram h)
	{
	    String sHChemical = h.getHistogramChemical();
	    
	    for (int i=0;i<sNameChemical.length;i++)
	    {
	    	//find a match between the chemical list and histogram chemical name
	    	if (this.sNameChemical[i].equalsIgnoreCase(sHChemical))
	    	{
	    		//match found
	    		h.setLevelOfConcern(this.dRiskConcentration[i]);
	    		//h.setRiskLevel(this.dRiskLevel[i]);
	    		break;
	    	}
	    }
	}
	
	
 	/**
	 * writeCumulativeFrequencyCurve
	 * output the cumulative frequency curve
	 */
	public void writeCumulativeFrequencyCurve()
	{
		flOut.logMessage("");
		flOut.logMessage("Cumulative Frequency Curve");
		flOut.logMessage("Cumulative Frequency, Value");
	    for (int i=0;i<dValue.length;i++)
	    {
	      flOut.logMessage(dFreq[i] + "," +  dValue[i]);	
	    }
	}
	
	/**
	 * 
	 * writeRunFileInformation    writes headings to OutputProcessor output file
	 * 
	 * @param sModelName               name of model
	 * @param sInputDataFile           main input data file name
	 * @param sOutputFileName          main output results file name (to be processed)
	 * @param sSortedOutputFileName    sorted results file name (to be produced by the OutputProcessor)
	 */
	public void writeRunFileInformation(String sModelName, String sInputDataFile,String sOutputFileName,String sSortedOutputFileName)
	{
		//write out run file information
		flOut.logMessage("Control,Statistics");
		flOut.logMessage(sModelName);
		flOut.logMessage("Post-Processed Output:  Statistics-Histograms-Cumulative Probabilities");
		flOut.logMessage("Input Data File: " + sInputDataFile);
		flOut.logMessage("Output File Name: " + sOutputFileName);
		flOut.logMessage("Sorted Output File Name (this file):  " + sSortedOutputFileName);
		flOut.logMessage("");
	}
	
	public void readResults(String sColumnToFind) throws IOException
	{
		//open the results file
		FileReader fr = new FileReader(sResultFileName);
		BufferedReader br = new BufferedReader(fr);
		String currentLine;
		int iResultColumn;
		String[] lineArray = {""};

		//prepare the storage arrays
		
		//entire line of character input
		sString = new String[this.iNumberOfResults];
		//distribution values for analysis
		dValue = new double[this.iNumberOfResults];
		//index of result line
		iIndex = new Integer[this.iNumberOfResults];
		//frequencies for cumulative frequency distribution
		//(after sorting of values)
		dFreq = new double[this.iNumberOfResults];
		
		//column to find
		this.sColumnToFind = sColumnToFind;
		
		iResultColumn = 0;
		iColumns = 0;
		iCount = 0;
		
		
		
	     
		while ((currentLine = br.readLine()) != null) 
		{
		  String key, key2;		  
		  key = ""; key2 = "";
		   		  
		  lineArray = currentLine.split(",",0);
		  key = lineArray[0];
		  
		  
		  //the results lines
		  if (key.equalsIgnoreCase("Result") || key.equalsIgnoreCase("Results")) 
		  {
			 iColumns = lineArray.length;
			 key2 = lineArray[1];
			 if (key2.equalsIgnoreCase("Heading"))
			 {
				 //save the heading for later output
				 this.sHeadOne = currentLine;
				 //find the specified column
				 for (int i=2;i<lineArray.length;i++)
				 {
					 if (lineArray[i].equalsIgnoreCase(sColumnToFind))
					 {
						 //save the column number
						 iResultColumn = i;
						 break;
					 }
				 }
			 }
			 else if (key2.equalsIgnoreCase("Unit/Count"))
			 {
				 //save the unit string for later output
				 this.sHeadTwo=currentLine;
			 }
			 else
			 {
				 //these are the data items
				 this.dValue[iCount] = Double.parseDouble(lineArray[iResultColumn]);
				 this.iIndex[iCount] = Integer.parseInt(lineArray[1]);
				 this.sString[iCount] = currentLine;
				 iCount++;
			 }

		  }
		}
		
	  //close the file
	  fr.close();
	}
	
	/**
	 * readCoupledResults read an process two coupled results
	 * @param sColumnToFindOne  first result
	 * @param sColumnToFindTwo  second of coupled results
	 * @throws IOException
	 */
	public void readCoupledResults(String sColumnToFindOne, String sColumnToFindTwo) throws IOException
	{
		//open the results file
		FileReader fr = new FileReader(sResultFileName);
		BufferedReader br = new BufferedReader(fr);
		String currentLine;
		int iResultColumnOne;
		int iResultColumnTwo;
		String[] lineArray = {""};

		//prepare the storage arrays
		
		//entire line of character input
		sString = new String[this.iNumberOfResults];
		//distribution values for analysis
		dValue = new double[this.iNumberOfResults];
		//distribution values for analysis
		dValue2 = new double[this.iNumberOfResults];
		//index of result line
		iIndex = new Integer[this.iNumberOfResults];
		//frequencies for cumulative frequency distribution
		//(after sorting of values)
		dFreq = new double[this.iNumberOfResults];
		
		//column to find
		this.sColumnToFind = sColumnToFindOne;
		
		iResultColumnOne = 0;
		iResultColumnTwo = 0;
		iColumns = 0;
		iCount = 0;
		
		
		
	     
		while ((currentLine = br.readLine()) != null) 
		{
		  String key, key2;		  
		  key = ""; key2 = "";
		   		  
		  lineArray = currentLine.split(",",0);
		  key = lineArray[0];
		  
		  
		  //the results lines
		  if (key.equalsIgnoreCase("Result") || key.equalsIgnoreCase("Results")) 
		  {
			 iColumns = lineArray.length;
			 key2 = lineArray[1];
			 if (key2.equalsIgnoreCase("Heading"))
			 {
				 //save the heading for later output
				 sHeadOne = currentLine;
				 //find the specified column
				 for (int i=2;i<lineArray.length;i++)
				 {
					 if (lineArray[i].equalsIgnoreCase(sColumnToFindOne))
					 {
						 //save the column number
						 iResultColumnOne = i;
						 
					 }
					 if (lineArray[i].equalsIgnoreCase(sColumnToFindTwo))
					 {
						 //saver the second column number
						 iResultColumnTwo = i;
					 }
				 }
			 }
			 else if (key2.equalsIgnoreCase("Unit/Count"))
			 {
				 //save the unit string for later output
				 sHeadTwo=currentLine;
			 }
			 else
			 {
				 //these are the data items
				 this.dValue[iCount] = Double.parseDouble(lineArray[iResultColumnOne]);
				 this.dValue2[iCount] = Double.parseDouble(lineArray[iResultColumnTwo]);
				 this.iIndex[iCount] = Integer.parseInt(lineArray[1]);
				 this.sString[iCount] = currentLine;
				 iCount++;
			 }

		  }
		}
		
	  //close the file
	  fr.close();
	}	
	
	public void setUpSubordinateResultsColumns(String sHeadingForSelectionCriterion,  double dSelectionCriterion)
	{
		int iSelectionCriterionColumn=0;
		int iColumnNumber=0;
		 
		ResultsColumn rc;
		
		
		//first find the column for making the selection 
		iColumnNumber = this.alResultsColumn.size();
		for (int i=0;i<iColumnNumber;i++)
		{
			rc = this.alResultsColumn.get(i);
			if (stripper.areTheyEqual(rc.getHeading(),sHeadingForSelectionCriterion))
			{
				iSelectionCriterionColumn = i;
				break;
			}
		}
		
		ResultsColumn rcSelection = alResultsColumn.get(iSelectionCriterionColumn);
		
		double[] dSelectionValues = rcSelection.getAllDoubleValues();
		
		
		//place all values beneath the value of the selection criterion in a subordinate results column
		sHeadingExtension[0] = " Values with " + rcSelection.getHeading() + " <" + Double.toString(dSelectionCriterion);
		for (ResultsColumn rcc: this.alResultsColumn)
		{
			String sSubordinateName = rcc.getHeading() + sHeadingExtension[0];
			
			rcc.addSubordinateResult(sHeadingExtension[0]);
			 	
			for (int i=0;i<dSelectionValues.length;i++)
			{
				if (dSelectionValues[i]<=dSelectionCriterion)
				{
					rcc.addToSubordinateResultColumn(sHeadingExtension[0],rcc.getValue(i));
				}
			}
			
			
			
		}
		
		
		//place all values above the value of the selection criterion in a subordinate results column
		sHeadingExtension[1] = " Values with " + rcSelection.getHeading() + " >" + Double.toString(dSelectionCriterion);
		for (ResultsColumn rcc: this.alResultsColumn)
		{
			String sSubordinateName = rcc.getHeading() + sHeadingExtension[1];
			
			rcc.addSubordinateResult(sHeadingExtension[1]);
			 	
			for (int i=0;i<dSelectionValues.length;i++)
			{
				if (dSelectionValues[i]>dSelectionCriterion)
				{
					rcc.addToSubordinateResultColumn(sHeadingExtension[1],rcc.getValue(i));
				}
			}
		}		
		
		ArrayList<ResultsColumn> alRC;
		//generate statistics on all subordinate results columns
		for (ResultsColumn rcc:  this.alResultsColumn)
		{
			alRC = rcc.getSubordinateResultsColumns();
			for (ResultsColumn rccc: alRC)
			{
				generateStatistics(rccc);
			}
		}
		
	}

	/**
	 * writeStatistics   write out the simple statistical results for each column of data
	 */
	public void writeStatisticsForSubordinateResultsColumns()
	{
	  int iCount = 0;
	  String sOut = "";
	  String sOut2 = "";
	  double dTemp = 0.;
	  ArrayList<ResultsColumn> alRCSubordinate = new ArrayList<ResultsColumn>();
	  
	 
	  
	  flOut.logMessage("");
	  flOut.logMessage("Simple Statistics:");	
	  Stripper s = new Stripper();
	  
	  for (int j=0;j<2;j++)
	  {
		  flOut.logMessage(this.sHeadingExtension[j]);
	      //minimum:	
		  for (ResultsColumn rc: alResultsColumn)
		  {
			alRCSubordinate = rc.getSubordinateResultsColumns();
			for (ResultsColumn rcc: alRCSubordinate)
			{
				if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
				{	
				  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
				  {
					sOut = sOut + rcc.getMin() + ",";
				  }
				  else
				  {
					if (iCount == 0){sOut = "Minimum,";}
					else {sOut = sOut + ",";}
				  }
				  iCount++;
				}  
			} 
		  }
		  
		  flOut.logMessage(sOut);  	
		  iCount = 0;
		  sOut = "";
		  
		  
		  //average:	
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						sOut = sOut + rcc.getAverage() + ",";
					  }
					  else
					  {
						if (iCount == 0){sOut = "Average,";}
						else {sOut = sOut + ",";}
					  }
					  iCount++;
					}  
				} 			  
			
		  }
		  flOut.logMessage(sOut);
		  
		  
		  iCount = 0;
		  sOut = "";
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						sOut = sOut + rcc.getMax() + ",";
					  }
					  else
					  {
						if (iCount == 0){sOut = "Maximum,";}
						else {sOut = sOut + ",";}
					  }
					  iCount++;
					}  
				} 			  
				
		  }	  
		  flOut.logMessage(sOut);
		  flOut.logMessage("");	  
			
		  iCount = 0;
		  sOut = "";
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						sOut = sOut + rcc.getRange() + ",";
					  }
					  else
					  {
						if (iCount == 0){sOut = "Range,";}
						else {sOut = sOut + ",";}
					  }
					  iCount++;
					}  
				} 			  				
			  }
			  flOut.logMessage(sOut);
			  
			  
	  iCount = 0;
	  sOut = "";
	  for (ResultsColumn rc: alResultsColumn)
	  {
		  alRCSubordinate = rc.getSubordinateResultsColumns();
			for (ResultsColumn rcc: alRCSubordinate)
			{
				if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
				{	
				  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
				  {
					 
					dTemp = rcc.getVariance();
					sOut = sOut + dTemp + ",";
					sOut2 = sOut2 + Math.sqrt(dTemp) + ",";
				  }
				  else
				  {
					if (iCount == 0){sOut = "Variance,"; sOut2 = "Standard Deviation,";	}
					else {sOut = sOut + ",";sOut2 = sOut2 + ",";	}
				  }
				  iCount++;
				}  
			} 			  
			
			
		  }
		  flOut.logMessage(sOut);				  
		  flOut.logMessage(sOut2);		

				
		  sOut = "";
		  sOut2 = "";
		  iCount = 0;
		  
		//the resistant statistical measures
		  flOut.logMessage("");
		  flOut.logMessage("Resistant Statistical Measures");
		  iCount = 0;
		  sOut = "";
		  //minimum		  
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						sOut = sOut + rcc.getMin() + ",";
					  }
					  else
					  {
						if (iCount == 0){sOut = "Minimum,";	}
						else {sOut = sOut + ",";}
					  }
					  iCount++;
					}  
				} 			  
				
				
	    }		  
		flOut.logMessage(sOut);
		iCount = 0;
		sOut = "";
		
		//5th percentile
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						sOut = sOut + rcc.get5thPercentile() + ",";
					  }
					  else
					  {
						if (iCount == 0){sOut = "5th Percentile,";	}
						else {sOut = sOut + ",";}
					  }
					  iCount++;
					}  
				} 			  
				
				
	    }		  
		flOut.logMessage(sOut);
		iCount = 0;
		sOut = "";		
			
		//first quartile
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						sOut = sOut + rcc.getFirstQuartile() + ",";
					  }
					  else
					  {
						if (iCount == 0){sOut = "First Quartile,";	}
						else {sOut = sOut + ",";}
					  }
					  iCount++;
					}  
				} 			  
				
				
	    }		  
		flOut.logMessage(sOut);
		iCount = 0;
		sOut = "";	
		
		//median
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						dTemp = rcc.getMedian();
						sOut = sOut + dTemp + ",";
						dTemp = dTemp + 1.5*rcc.getInterQuartileRange();
						sOut2 = sOut2 + dTemp + ",";						
					  }
					  else
					  {
						 
						if (iCount == 0)
						{
							sOut = "Median,";
							sOut2 = "Median + 1/2 InterQuartile Range,";
						}
						else 
						{
							sOut = sOut + ",";
							sOut2 = sOut2 + ",";
						}						
					  }
					  iCount++;
					}  
				} 			  
				
				
	    }		  
		flOut.logMessage(sOut);
		iCount = 0;
		sOut = "";					
		
		//third quartile
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						sOut = sOut + rcc.getThirdQuartile() + ",";
					  }
					  else
					  {
						if (iCount == 0){sOut = "Third Quartile,";	}
						else {sOut = sOut + ",";}
					  }
					  iCount++;
					}  
				} 			  
				
				
	    }		  
		flOut.logMessage(sOut);
		iCount = 0;
		sOut = "";		
		
		//median + 1.5 InterQuartile Range
		flOut.logMessage(sOut2);	
		
		
		//95th percentile
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						sOut = sOut + rcc.get95thPercentile() + ",";
					  }
					  else
					  {
						if (iCount == 0){sOut = "95th Percentile,";	}
						else {sOut = sOut + ",";}
					  }
					  iCount++;
					}  
				} 			  
				
				
	    }		  
		flOut.logMessage(sOut);
		iCount = 0;
		sOut = "";	
		
		//Maximum
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						sOut = sOut + rcc.getMax() + ",";
					  }
					  else
					  {
						if (iCount == 0){sOut = "Maximum,";	}
						else {sOut = sOut + ",";}
					  }
					  iCount++;
					}  
				} 			  
				
				
	    }		  
		flOut.logMessage(sOut);
		iCount = 0;
		sOut = "";						
		
		//IQR
		  for (ResultsColumn rc: alResultsColumn)
		  {
			  alRCSubordinate = rc.getSubordinateResultsColumns();
				for (ResultsColumn rcc: alRCSubordinate)
				{
					if (s.areTheyEqual(this.sHeadingExtension[j],rcc.getHeadingExtension()))
					{	
					  if (rc.getHaveTheStatisticsBeenSet() && rc.getIsThisADoubleDataColumn())
					  {
						sOut = sOut + rcc.getInterQuartileRange()+ ",";
					  }
					  else
					  {
						if (iCount == 0){sOut = "Inter Quartile Range,";	}
						else {sOut = sOut + ",";}
					  }
					  iCount++;
					}  
				} 			  
				
				
	    }		  
		flOut.logMessage(sOut);
	 					
		flOut.logMessage("");
		sOut = "";
		sOut2 = "";
        iCount = 0;		
	  }
	  
	 
	  
	}	  
	  
	  /**
	   * add version information to the models VersionCompilation object
	   * @param vc  VersionCompilation object
	   */
	  public void setVersionInformation(VersionCompilation vc)
	  {
		 vc.setFileData("OutputProcessor",
				 this.sName,
				 "Development Version 0.00",
				 "c:\\Documents and Settings\\jweaver\\workspace\\RiverModel\\src\\util\\OutputProcessor.java",
                 "c:\\Documents and Settings\\jweaver\\workspace\\RiverModel\\bin\\util\\OutputProcessor.class");						 
	  }
	  
	  
	public void writeHeadings(String sMessage)
	{
		flOut.logMessage(sMessage + " " + this.sHeadOne);
		flOut.logMessage(sMessage + " " + this.sHeadTwo);
	}
    
	//sets
	public void setIntervalFrequency(int iIntervalFrequency){this.iFrequencyInterval = iIntervalFrequency;}
	public void setResultsFileName(String sOutputFileName){this.sResultFileName = sOutputFileName;}
	public void setNumberOfResults(int iNumber){this.iNumberOfResults = iNumber;}
	public void setNumberOfFrequencyDistributionIntervals(int iFrequencyInterval){this.iFrequencyInterval=iFrequencyInterval;}
	public void setResistantStatisticalMeasures(ResistantStatisticalMeasures rsm){this.rsm = rsm;}
    public void setUnitConverter(UnitConverter uc){this.uc = uc;}
	
	//gets
	public int getIntervalFrequency(){return this.iFrequencyInterval;}
	public double[] getDistributionValues(){return this.dValue;}
	public FileLogger getOutputFileLogger(){return this.flOut;}
    public Histogram getHistogram(){return this.histogram;}
    public String getHeadingOne(){return this.sHeadOne;}
    public String getHeadingTwo(){return this.sHeadTwo;}
    public String getOutputRiskConcentrationUnit(){return this.sOutputRiskConcentrationUnit;}
	
     
	

}
