package util;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;

/**
 * UnitConverter--generalized unit conversions
 * @author JWEAVER
 * 8-11-2012
 */
public class UnitConverter 
{
    private FileLogger fl;
    private ArrayList<Unit> alUnit;
    private ArrayList<Unit> alOutputUnit;
    private ArrayList<UnitType> alUnitType;
    //string that holds the internal calculation unit of the model
    private String sModelUnit;
    private boolean bUnsupportedUnit;
    private Stripper stripper;

       
	public UnitConverter(){initial();}
	
	public void initial()
	{
	  alUnit = new ArrayList<Unit>();
	  alOutputUnit = new ArrayList<Unit>();
	  alUnitType = new ArrayList<UnitType>();
	  sModelUnit = "";
	  bUnsupportedUnit = false; 
	  stripper = new Stripper();
	}
	 

	//get the unit conversion factor for a specified unit type and name
	//type of unit:  i.e., length
	//name of unit:  i.e., meter
	public double getUCF(String sType, String sName)
	{
		for (Unit u: alUnit)
		{
			String sTypeT = u.getType();
			String sNameT = u.getName();
			//if (sTypeT.equalsIgnoreCase(sType) && sNameT.equalsIgnoreCase(sName))
			if (stripper.areTheyEqual(sTypeT, sType,' ') && stripper.areTheyEqual(sNameT, sName,' '))
			{
				//unit conversion found
				double ucf = u.getUCF();
				return ucf;
			}		
		}
		//unit not found
		String out = "unit conversion not found for unit: " + sName + " used for: " + sType;
		System.out.println(out);
		fl.logMessage(out);
		return 0.0;
	}
	
	//add the units from a file
	public void readUnits(String sFileName) throws IOException
	{
		FileReader fr = new FileReader(sFileName);
		BufferedReader br = new BufferedReader(fr);
		 
		String currentLine;
		String[] lineArray = {""};
		String key;
		
		while ((currentLine = br.readLine()) != null) 
		{
			lineArray = currentLine.split(",",0);

			//read each line and prepare data according to key words
			key = lineArray[0];
			//if (key.equalsIgnoreCase("Control") || key.equalsIgnoreCase("controll")) 
			if (stripper.areTheyEqual(key,"Control") || stripper.areTheyEqual(key, "Controll"))
			{
			  String key2 = lineArray[1];
			  //if (key2.equalsIgnoreCase("unit"))
			  if (stripper.areTheyEqual(key2,"unit"))
			  {
				  String sType,sName;
				  double dUCF=1.0;
				  boolean bInternalCalculationUnit=false;
				  sType = lineArray[2];
				  //key word model indicates internal calculation unit
				  //unit conversion factor is one for this unit
				  //if (lineArray[3].equalsIgnoreCase("model"))
				  if (stripper.areTheyEqual(lineArray[3],"model"))
				  {
					  bInternalCalculationUnit = true;
					  dUCF = 1.0;
				  }
				  else
				  {	  
				    //the unit conversion factor is composed of the product of the four entries:
				    dUCF = dUCF*Double.parseDouble(lineArray[4]);
				    dUCF = dUCF*Double.parseDouble(lineArray[5]);
				    dUCF = dUCF*Double.parseDouble(lineArray[6]);
				    dUCF = dUCF*Double.parseDouble(lineArray[7]);
				  }
				  
				  //allowable unit names;  item 8 through the end of entries in lineArray (variable number allowed)
				  for (int i=8;i<lineArray.length;i++)
				  {
					  sName = lineArray[i];
					  addUnit(sType,sName,dUCF,bInternalCalculationUnit);
				  }
			  }
			}
		}
		
		//close the input file
		fr.close();
		
	}
	
	
	//Read unit types for each piece of input data
	//i.e., multiple named inputs have Length as the unit--
	//DepthToWater
	//DepthToSample
	//DepthToBottom, etc all have UnitType Length
	public void readUnitTypes(String sFileName) throws IOException
	{
		UnitType ut;  
		FileReader fr = new FileReader(sFileName);
		BufferedReader br = new BufferedReader(fr);
		 
		String currentLine;
		String[] lineArray = {""};
		String key;
		
		//skip first line
		currentLine = br.readLine();
		
		while ((currentLine = br.readLine()) != null) 
		{
			lineArray = currentLine.split(",",0);
			ut = new UnitType();
			ut.setUnitType(lineArray[0], lineArray[1]);
			alUnitType.add(ut);
		}
		
		//close the input file
		fr.close();		
	}
	
	//add a new "Unit" for a unit type
	public void addUnit(String sType, String sName, double uCF, boolean bInternalCalculationUnit)
	{
	   Unit u = new Unit();
	   u.setType(sType);
	   u.setName(sName);
	   u.setUCF(uCF);
	   u.setIsUnitInternalCalculationUnit(bInternalCalculationUnit);
	   alUnit.add(u);
	}
	
	public void addOutputUnit(String sType, String sOutputName)
	{
		 Unit u = new Unit();
		 u.setType(sType);
		 u.setName(sOutputName);
		 //get the model conversion factor from the output unit to the internal calculation unit
		 double uCF = this.getUCF(sType, sOutputName);
		 //sets the inverse of the uCF:
		 //converts from the calculation unit to the desired output unit
		 //i.e., sType = length, sOutputName = feet
		 //this.uCF returns 12*25.4 which converts feet to cm
		 //on output invert this to convert cm to feet
		 //the output unit conversion is now the conversion factor from calculation unit to output unit
		 u.setUCF(1./uCF);
		 alOutputUnit.add(u);
	}
	
	
	/**
	 * getOutputUnitName   gets the user-input, output name for a specific unit type
	 * @param fl
	 */
	public String getOutputNameFromType(String sType)
	{
		String sReturn="";
		
		for (Unit u: alOutputUnit)
		{
			//if (u.getType().equalsIgnoreCase(sType))
			if (stripper.areTheyEqual(u.getType(),sType,' '))
			{
				sReturn = u.getName();
			}
		}
		return sReturn;
	}
	/**
	 * getOutputUCF   gets the user-input, output unit conversion factor for a specific unit type
	 * @param fl
	 */
	public double getOutputUCFFromType(String sType)
	{
		double dReturn=0.0;
		
		for (Unit u: alOutputUnit)
		{
			//if (u.getType().equalsIgnoreCase(sType))
			if (stripper.areTheyEqual(u.getType(),sType,' '))
			{
				dReturn = u.getUCF();
				break;
			}
		}
		return dReturn;
	}	
	
	//simple sets
	public void setFileLogger(FileLogger fl){this.fl = fl;}
	 
	
	//simple gets
	//return model unit for a specified type of unit (i.e., sType = "length")
	public String getModelUnit(String sType)
	{
	   String sModelUnit;
	   sModelUnit = "";
	   
       for (Unit u: this.alUnit)
       {
           //find the requested unit type
    	   //if (u.getType().equalsIgnoreCase(sType))
    	   if (stripper.areTheyEqual(u.getType(),sType,' '))
    	   {
    		   //find the model unit
    		   if (u.getIsUnitInternalCalculationUnit()){sModelUnit = u.getName(); break;}
    	   }
       }
       
       return sModelUnit;
		
   }
	
	//gets
	
	/**
	 * getAllUnitNamesForAType   get the names of all the units used for a unit type
	 * @param sUnitType          String unit type (i.e., "length", "hydraulicConductivity", "Concentration", etc
	 * @return
	 */
	public ArrayList<String> getAllUnitNamesForAType(String sUnitType)
	{
        ArrayList<String> alUnitTypeLocal;
        alUnitTypeLocal = new ArrayList<String>();
        String sTempUnitName;
        
	    for (Unit u: this.alUnit)
	    {
	      //find the requested unit type
	      //if (u.getType().equalsIgnoreCase(sUnitType))
	      if (stripper.areTheyEqual(u.getType(), sUnitType,' '))
	      {
	    	sTempUnitName = stripper.removeCharacter(' ', u.getName());
	    	if (!sTempUnitName.equals(""))
	    	{	
	          //store all the unit names for this type (sUnitType)
	    	  alUnitTypeLocal.add(sTempUnitName);
	    	}  
	      }
	    }		
		return alUnitTypeLocal;
	}
	
	
	public boolean getBooleanUnsupportedUnit(){return bUnsupportedUnit;}
    public String getUnitType(String sInputDataName)
    {
    	Stripper s = new Stripper();
    	String sReturn="";
    	for (UnitType ut: this.alUnitType)
    	{
    		if (s.areTheyEqual(sInputDataName,ut.getInputType(),' '))
    		{
    			sReturn = ut.getUnitType();
    			break;
    		}
    	}
    	return sReturn;
    }
	
	
	
	//internal class for holding unit definitions
	class Unit
	{
		//type of unit:  i.e., length
		private String sType;
		//name of unit:  i.e., meter
		private String sName;
		//conversion factor to internal calculation unit
		//i.e., for type="length" and name="meter":
		//i.e., if the model's internal calculation unit is meters, uCF = 1.0
		//i.e., if the model's internal calculation unit is feet, uCF = 1.0/0.3048 (equivalently uCF = 3.2808)
		//i.e., if the model's internal calculation unit is cm, uCF = 0.01
		private double uCF;
		//is this unit the internal calculation unit?  true if yes
		private boolean bInternalCalculationUnit;

		
		public Unit(){}
		
		//simple sets
		public void setName(String sName){this.sName = sName;}
		public void setType(String sType){this.sType = sType;}
		public void setUCF(double uCF){this.uCF = uCF;}
		public void setIsUnitInternalCalculationUnit(boolean bInternalCalculationUnit){this.bInternalCalculationUnit = bInternalCalculationUnit;}
		
		//simple gets
		public String getName(){return sName;}
		public String getType(){return sType;}
		public double getUCF(){return uCF;}
		public boolean getIsUnitInternalCalculationUnit(){return bInternalCalculationUnit;}

		
	}
	
	
	//internal class for holding unit types
	class UnitType
	{
		private String sInputType;
		private String sUnitType;
		
		public UnitType(){}
		
		//sets
		public void setUnitType(String sInputType, String sUnitType){this.sInputType = sInputType; this.sUnitType = sUnitType;}
		
		//gets
		public String getInputType(){return this.sInputType;}
		public String getUnitType(){return this.sUnitType;}
	}
}
