package distance

import (
	"fmt"
	"mwo"
	"parse"
)

/**
 * @algorithm
 * @Chiu, Foong Distance Calculator
 * @May 10, 2018
 * @calculator
**/
// HourDay
type HourDayDetail struct {
	dayID, hourID int
}

type HourDayKey struct {
	hourDayID int
}

var HourDay map[HourDayKey]*HourDayDetail

// County
type CountyDetail struct {
	stateID              int
	countyName           string
	altitude             string
	GPAFract             float64
	barometricPressure   float64
	barometricPressureCV float64
}

type CountyKey struct {
	countyID int
}

var County map[CountyKey]*CountyDetail

// Link
type LinkDetail struct {
	countyID        int
	zoneID          int
	roadTypeID      int
	linkLength      float64
	linkVolume      float64
	linkAvgSpeed    float64
	linkDescription string
	linkAvgGrade    float64
}

type LinkKey struct {
	linkID int
}

var Link map[LinkKey]*LinkDetail

// Link2
type Link2Detail struct {
	stateID         int
	countyID        int
	zoneID          int
	roadTypeID      int
	linkLength      float64
	linkVolume      float64
	linkAvgSpeed    float64
	linkDescription string
	linkAvgGrade    float64
}

type Link2Key struct {
	linkID int
}

var Link2 map[Link2Key]*Link2Detail

// SourceBin
type SourceBinDetail struct {
	fuelTypeID       int
	engTechID        int
	regClassID       int
	modelYearGroupID int
	engSizeID        int
	weightClassID    int
}

type SourceBinKey struct {
	sourceBinID int
}

var SourceBin map[SourceBinKey]*SourceBinDetail

// SourceBinDistribution
type SourceBinDistributionDetail struct {
	sourceBinActivityFraction   float64
	sourceBinActivityFractionCV float64
	isUserInput                 string
	sequenceNumber				int
}

type SourceBinDistributionKey struct {
	polProcessID, sourceBinID, sourceTypeModelYearID int
}

var SourceBinDistribution map[SourceBinDistributionKey]*SourceBinDistributionDetail

// SourceTypeModelYear
type SourceTypeModelYearDetail struct {
	sourceTypeModelYearID   int
	modelYearID             int
	sourceTypeID            int
	ACPenetrationFraction   float64
	ACPenetrationFractionCV float64
}

type SourceTypeModelYearKey struct {
	sourceTypeModelYearID int
}

var SourceTypeModelYear map[SourceTypeModelYearKey]*SourceTypeModelYearDetail

// SHO
type SHODetail struct {
	SHO         float64
	SHOCV       float64
	distance    float64
	isUserInput string
}

type SHOKey struct {
	ageID, hourDayID, linkID, monthID, sourceTypeID, yearID int
}

var SHO map[SHOKey]*SHODetail

// SBD2
type SBD2Detail struct {
	fuelTypeActivityFraction float64
}

type SBD2Key struct {
	sourceTypeModelYearID, regClassID, fuelTypeID int
}

// DistFracts
type DistFractsDetail struct {
	fuelTypeActivityFraction                      float64
	sourceTypeModelYearID, regClassID, fuelTypeID int
}

type DistFractsKey struct {
	sourceTypeID, modelYearID int
}

// SHO2
type SHO2Detail struct {
	distance float64
}

type SHO2Key struct {
	yearID, monthID, dayID, hourID, modelYearID, linkID, sourceTypeID int
}

var SHO2 map[SHO2Key]*SHO2Detail

// SHO3
type SHO3Detail struct {
	distance float64
}

type SHO3Key struct {
	yearID, monthID, dayID, hourID, modelYearID, linkID, sourceTypeID, stateID, countyID, zoneID, roadTypeID int
}

// Initialize package-level variables, creating empty lookup tables.
func init() {
	HourDay = make(map[HourDayKey]*HourDayDetail)
	County = make(map[CountyKey]*CountyDetail)
	Link = make(map[LinkKey]*LinkDetail)
	Link2 = make(map[Link2Key]*Link2Detail)
	SourceBin = make(map[SourceBinKey]*SourceBinDetail)
	SourceBinDistribution = make(map[SourceBinDistributionKey]*SourceBinDistributionDetail)
	SourceTypeModelYear = make(map[SourceTypeModelYearKey]*SourceTypeModelYearDetail)
	SHO = make(map[SHOKey]*SHODetail)
	SHO2 = make(map[SHO2Key]*SHO2Detail)
}

func contains(s []int, e int) bool {
	for _, a := range s {
		if a == e {
			return true
		}
	}
	return false
}

func StartSetup() {
	// County
	parse.ReadAndParseFile("county", func(parts []string) {
		if len(parts) < 7 {
			return
		}

		k := CountyKey{
			parse.GetInt(parts[0]),
		}

		v := new(CountyDetail)
		v.stateID = parse.GetInt(parts[1])
		v.countyName = parse.GetString(parts[2])
		v.altitude = parse.GetString(parts[3])
		v.GPAFract = parse.GetFloat(parts[4])
		v.barometricPressure = parse.GetFloat(parts[5])
		v.barometricPressureCV = parse.GetFloat(parts[6])
		County[k] = v
	})

	fmt.Println("County: ", len(County))

	// HourDay
	parse.ReadAndParseFile("hourday", func(parts []string) {
		if len(parts) < 3 {
			return
		}

		k := HourDayKey{
			parse.GetInt(parts[0]),
		}

		v := new(HourDayDetail)
		v.dayID = parse.GetInt(parts[1])
		v.hourID = parse.GetInt(parts[2])

		HourDay[k] = v
	})

	fmt.Println("HourDay: ", len(HourDay))

	// Link
	parse.ReadAndParseFile("link", func(parts []string) {
		if len(parts) < 9 {
			return
		}

		k := LinkKey{
			parse.GetInt(parts[0]),
		}

		v := new(LinkDetail)
		v.countyID = parse.GetInt(parts[1])
		v.zoneID = parse.GetInt(parts[2])
		v.roadTypeID = parse.GetInt(parts[3])
		v.linkLength = parse.GetFloat(parts[4])
		v.linkVolume = parse.GetFloat(parts[5])
		v.linkAvgSpeed = parse.GetFloat(parts[6])
		v.linkDescription = parse.GetString(parts[7])
		v.linkAvgGrade = parse.GetFloat(parts[8])

		Link[k] = v
	})

	fmt.Println("Link: ", len(Link))

	// SHO
	parse.ReadAndParseFile("sho", func(parts []string) {
		if len(parts) < 10 {
			return
		}

		k := SHOKey{
			parse.GetInt(parts[3]),
			parse.GetInt(parts[0]),
			parse.GetInt(parts[4]),
			parse.GetInt(parts[1]),
			parse.GetInt(parts[5]),
			parse.GetInt(parts[2]),
		}

		v := new(SHODetail)
		v.SHO = parse.GetFloat(parts[6])
		v.SHOCV = parse.GetFloat(parts[7])
		v.distance = parse.GetFloat(parts[8])
		v.isUserInput = parse.GetString(parts[9])

		SHO[k] = v
	})

	fmt.Println("SHO: ", len(SHO))

	// SourceBin
	parse.ReadAndParseFile("sourcebin", func(parts []string) {
		if len(parts) < 7 {
			return
		}

		k := SourceBinKey{
			parse.GetInt(parts[0]),
		}

		v := new(SourceBinDetail)
		v.fuelTypeID = parse.GetInt(parts[1])
		v.engTechID = parse.GetInt(parts[2])
		v.regClassID = parse.GetInt(parts[3])
		v.modelYearGroupID = parse.GetInt(parts[4])
		v.engSizeID = parse.GetInt(parts[5])
		v.weightClassID = parse.GetInt(parts[6])
		SourceBin[k] = v
	})

	fmt.Println("SourceBin: ", len(SourceBin))

	// SourceBinDistribution
	parse.ReadAndParseFile("sourcebindistribution", func(parts []string) {

		if len(parts) < 6 {
			return
		}

		k := SourceBinDistributionKey{
			parse.GetInt(parts[1]),
			parse.GetInt(parts[2]),
			parse.GetInt(parts[0]),
		}

		v := new(SourceBinDistributionDetail)
		v.sourceBinActivityFraction = parse.GetFloat(parts[3])
		v.sourceBinActivityFractionCV = parse.GetFloat(parts[4])
		v.isUserInput = parse.GetString(parts[5])
		v.sequenceNumber = length(SourceBinDistribution)

		SourceBinDistribution[k] = v
	})

	fmt.Println("SourceBinDistribution: ", len(SourceBinDistribution))

	// SourceTypeModelYear
	parse.ReadAndParseFile("sourcetypemodelyear", func(parts []string) {

		if len(parts) < 5 {
			return
		}

		k := SourceTypeModelYearKey{
			parse.GetInt(parts[0]),
		}

		v := new(SourceTypeModelYearDetail)
		v.modelYearID = parse.GetInt(parts[1])
		v.sourceTypeID = parse.GetInt(parts[2])
		v.ACPenetrationFraction = parse.GetFloat(parts[3])
		v.ACPenetrationFractionCV = parse.GetFloat(parts[4])

		SourceTypeModelYear[k] = v
	})

	fmt.Println("SourceTypeModelYear: ", len(SourceTypeModelYear))

}

// Complete any pending asychronous operations initiated during StartSetup().
func FinishSetup() {
	// There are no asynchronous reads, so nothing to do here.

	// Done
	fmt.Println("DistanceCalculator finished reading setup files.")
}

// Launch computation threads. Each thread reads from the inputBlocks channel and writes to the outputBlocks channel.
func StartCalculating(howManyThreads int, inputActivityBlocks chan *mwo.MWOActivityBlock, outputActivityBlocks chan *mwo.MWOActivityBlock) {
	// Do one-time things that don't write to a channel

	// Create SHO2
	for shoK, shoV := range SHO {
		hourDayV := HourDay[HourDayKey{shoK.hourDayID}]

		if hourDayV != nil {
			var k SHO2Key
			k.yearID = shoK.yearID
			k.monthID = shoK.monthID
			k.dayID = hourDayV.dayID
			k.hourID = hourDayV.hourID
			k.modelYearID = (shoK.yearID - shoK.ageID)
			k.linkID = shoK.linkID
			k.sourceTypeID = shoK.sourceTypeID

			ov := SHO2[k]
			if ov != nil {
				fmt.Println("---- has issue in creating SHO2")
			} else {
				SHO2[k] = &SHO2Detail{distance: shoV.distance}
			}
		}
	}

	// Create Link2
	for lK, lV := range Link {

		countyV := County[CountyKey{countyID: lV.countyID}]
		if countyV != nil {
			k := Link2Key{linkID: lK.linkID}

			ov := Link2[k]

			if ov != nil {
				fmt.Println("---- has issue in creating Link2")
			} else {
				Link2[k] = &Link2Detail{stateID: countyV.stateID, countyID: lV.countyID, zoneID: lV.zoneID, roadTypeID: lV.roadTypeID, linkLength: lV.linkLength, linkVolume: lV.linkVolume, linkAvgSpeed: lV.linkAvgSpeed, linkDescription: lV.linkDescription, linkAvgGrade: lV.linkAvgGrade}
			}
		}
	}

	// Do parallel things that do write to a channel
	for i:=0;i<howManyThreads;i++ {
		go calculate(i,howManyThreads,outputActivityBlocks)
	}
}

func calculate(myThreadIndex, howManyThreads int, outputActivityBlocks chan *mwo.MWOActivityBlock) {
	//var SBD2 map[SBD2Key]*SBD2Detail
	SBD2 := make(map[SBD2Key]*SBD2Detail)
	//var DistFracts map[DistFractsKey][]*DistFractsDetail
	DistFracts := make(map[DistFractsKey][]*DistFractsDetail)
	//var SHO3 map[SHO3Key]*SHO3Detail
	SHO3 := make(map[SHO3Key]*SHO3Detail)

	// Create SBD2
	for sbdK, sbdV := range SourceBinDistribution {
		if sbdV.sequenceNumber % howManyThreads == myThreadIndex {
			var k SBD2Key

			sbV := SourceBin[SourceBinKey{sourceBinID: sbdK.sourceBinID}]

			k.sourceTypeModelYearID = sbdK.sourceTypeModelYearID
			k.fuelTypeID = sbV.fuelTypeID
			k.regClassID = sbV.regClassID

			sbd2V := SBD2[k]

			if sbd2V != nil {
				sbd2V.fuelTypeActivityFraction += sbdV.sourceBinActivityFraction
			} else {
				SBD2[k] = &SBD2Detail{fuelTypeActivityFraction: sbdV.sourceBinActivityFraction}
			}
		}
	}

	// Create DistFracts
	for sbd2K, sbd2V := range SBD2 {
		stmyV := SourceTypeModelYear[SourceTypeModelYearKey{sourceTypeModelYearID: sbd2K.sourceTypeModelYearID}]

		if stmyV != nil {
			v := new(DistFractsDetail)
			v.fuelTypeActivityFraction = sbd2V.fuelTypeActivityFraction
			v.fuelTypeID = sbd2K.fuelTypeID
			v.regClassID = sbd2K.regClassID
			v.sourceTypeModelYearID = sbd2K.sourceTypeModelYearID

			var k DistFractsKey
			k.modelYearID = stmyV.modelYearID
			k.sourceTypeID = stmyV.sourceTypeID

			ov := DistFracts[k]
			if ov == nil {
				DistFracts[k] = []*DistFractsDetail{v}
			} else {
				DistFracts[k] = append(DistFracts[k], v)
			}
		}
	}

	// Create SHO3
	for sho2K, sho2V := range SHO2 {

		l2V := Link2[Link2Key{linkID: sho2K.linkID}]

		if l2V != nil {

			var sho3K SHO3Key
			sho3K.yearID = sho2K.yearID
			sho3K.countyID = l2V.countyID
			sho3K.dayID = sho2K.dayID
			sho3K.hourID = sho2K.hourID
			sho3K.modelYearID = sho2K.modelYearID
			sho3K.monthID = sho2K.monthID
			sho3K.roadTypeID = l2V.roadTypeID
			sho3K.sourceTypeID = sho2K.sourceTypeID
			sho3K.stateID = l2V.stateID
			sho3K.zoneID = l2V.zoneID
			sho3K.linkID = sho2K.linkID

			ov := SHO3[sho3K]

			if ov != nil {
				fmt.Println("---- has issue in creating SHO3")
			} else {
				SHO3[sho3K] = &SHO3Detail{sho2V.distance}
			}
		}
	}

	ab := mwo.NewActivityBlock()

	activityFuelBlocks := make(map[mwo.MWOActivityKey]*mwo.ActivityFuelBlock)

	for sho3k, sho3v := range SHO3 {

		var dfk DistFractsKey

		dfk.modelYearID = sho3k.modelYearID
		dfk.sourceTypeID = sho3k.sourceTypeID

		dfvs := DistFracts[dfk]

		for _, dfv := range dfvs {
			if dfv != nil {

				var key mwo.MWOActivityKey

				key.YearID = sho3k.yearID
				key.MonthID = sho3k.monthID
				key.DayID = sho3k.dayID
				key.HourID = sho3k.hourID
				key.StateID = sho3k.stateID
				key.CountyID = sho3k.countyID
				key.ZoneID = sho3k.zoneID
				key.LinkID = sho3k.linkID
				key.RoadTypeID = sho3k.roadTypeID
				key.SourceTypeID = sho3k.sourceTypeID
				key.ModelYearID = sho3k.modelYearID
				key.ActivityTypeID = 1
				key.RegClassID = dfv.regClassID
				key.FuelTypeID = dfv.fuelTypeID

				afb := activityFuelBlocks[key]

				a := new(mwo.MWOActivity)
				a.Activity = sho3v.distance * dfv.fuelTypeActivityFraction

				if afb == nil {
					nafb := mwo.NewActivityFuelBlock(nil)
					nafb.Key = key
					nafb.AddActivity(a)
					activityFuelBlocks[key] = nafb

				} else {
					afb.AddActivity(a)
				}
			}
		}
	}

	fmt.Println("---activityFuelBlocks: ", len(activityFuelBlocks))
	for _, afb := range activityFuelBlocks {
		ab.AddActivityFuelBlock(afb)
	}

	outputActivityBlocks <- ab
}
