package no

import (
	"fmt"
	"mwo"
	"parse"
)

/**
 * @algorithm
 * @owner NO Calculator
 * @calculator
**/

// NOCopyOfSourceUseType
var NOCopyOfSourceUseType map[int]int

// NOCopyOfPPA
type NOCopyOfPPADetail struct {
	polProcessID int
	processID    int
	pollutantID  int
}

type NOCopyOfPPAKey struct {
	polProcessID int
}

var NOCopyOfPPA map[NOCopyOfPPAKey]*NOCopyOfPPADetail

// NOCopyOfPPMY
type NOCopyOfPPMYDetail struct {
	modelYearID   int
	fuelMYGroupID int
}

type NOCopyOfPPMYKey struct {
	polProcessID     int
	modelYearGroupID int
}

var NOCopyOfPPMY map[NOCopyOfPPMYKey][]*NOCopyOfPPMYDetail

// NOCopyOfFuelType
var NOCopyOfFuelType map[int]int

// NOCopyOfNONO2Ratio
type NOCopyOfNONO2RatioDetail struct {
	polProcessID     int
	sourceTypeID     int
	fuelTypeID       int
	modelYearGroupID int
	NOxRatio         float64
	NOxRatioCV       float64
	dataSourceId     int
}

var NOCopyOfNONO2Ratio []*NOCopyOfNONO2RatioDetail

// NOCalculation1
type NOCalculation1Detail struct {
	polProcessID int
	processID    int
	pollutantID  int
	NOxRatio     float64
}

type NOCalculation1Key struct {
	sourceTypeID int
	fuelTypeID   int
	modelYearID  int
}

var NOCalculation1 map[NOCalculation1Key][]*NOCalculation1Detail

// Initialize package-level variables, creating empty lookup tables.
func init() {
	NOCopyOfSourceUseType = make(map[int]int)
	NOCopyOfPPA = make(map[NOCopyOfPPAKey]*NOCopyOfPPADetail)
	NOCopyOfPPMY = make(map[NOCopyOfPPMYKey][]*NOCopyOfPPMYDetail)
	NOCopyOfFuelType = make(map[int]int)
	NOCalculation1 = make(map[NOCalculation1Key][]*NOCalculation1Detail)
}

func contains(s []int, e int) bool {
	for _, a := range s {
		if a == e {
			return true
		}
	}
	return false
}

func StartSetup() {
	// NOCopyOfSourceUseType
	parse.ReadAndParseFile("nocopyofsourceusetype", func(parts []string) {
		// sourceTypeID

		if len(parts) < 1 {
			return
		}

		NOCopyOfSourceUseType[parse.GetInt(parts[0])] = parse.GetInt(parts[0])
	})

	// NOCopyOfPPA
	parse.ReadAndParseFile("nocopyofppa", func(parts []string) {
		// polProcessID	int,
		// processID		smallint(6),
		// pollutantID		smallint(6)

		// key: processID, fuelTypeID, sourceTypeID, monthID, modelYearID
		if len(parts) < 3 {
			return
		}

		k := NOCopyOfPPAKey{
			parse.GetInt(parts[0]),
		}

		v := new(NOCopyOfPPADetail)
		v.processID = parse.GetInt(parts[1])
		v.pollutantID = parse.GetInt(parts[2])

		NOCopyOfPPA[k] = v
	})

	// NOCopyOfPPMY
	parse.ReadAndParseFile("nocopyofppmy", func(parts []string) {
		// polProcessID		int,
		// modelYearID			smallint(6),
		// modelYearGroupID	int(11),
		// fuelMYGroupID		int(11)

		if len(parts) < 4 {
			return
		}

		k := NOCopyOfPPMYKey{
			parse.GetInt(parts[0]),
			parse.GetInt(parts[2]),
		}

		v := new(NOCopyOfPPMYDetail)
		v.modelYearID = parse.GetInt(parts[1])
		v.fuelMYGroupID = parse.GetInt(parts[3])

		NOCopyOfPPMY[k] = append(NOCopyOfPPMY[k], v)
	})

	// NOCopyOfFuelType
	parse.ReadAndParseFile("nocopyoffueltype", func(parts []string) {
		// fuelTypeID

		if len(parts) < 1 {
			return
		}

		NOCopyOfFuelType[parse.GetInt(parts[0])] = parse.GetInt(parts[0])
	})

	// NOCopyOfNONO2Ratio
	parse.ReadAndParseFile("nocopyofnono2ratio", func(parts []string) {
		// polProcessID		int,
		// sourceTypeID		smallint(6),
		// fuelTypeID			smallint(6),
		// modelYearGroupID	int(11),
		// NOxRatio 			float,
		// NOxRatioCV			float,
		// dataSourceId		smallint(6)

		if len(parts) < 7 {
			return
		}

		v := new(NOCopyOfNONO2RatioDetail)
		v.polProcessID = parse.GetInt(parts[0])
		v.sourceTypeID = parse.GetInt(parts[1])
		v.fuelTypeID = parse.GetInt(parts[2])
		v.modelYearGroupID = parse.GetInt(parts[3])
		v.NOxRatio = parse.GetFloat(parts[4])
		v.NOxRatioCV = parse.GetFloat(parts[5])
		v.dataSourceId = parse.GetInt(parts[6])

		NOCopyOfNONO2Ratio = append(NOCopyOfNONO2Ratio, v)
	})

	InitNOCalculation1()
}

// Complete any pending asychronous operations initiated during StartSetup().
func FinishSetup() {
	// There are no asynchronous reads, so nothing to do here.

	// Done
	fmt.Println("NOCalculator finished reading setup files.")
}

// Launch computation threads. Each thread reads from the inputBlocks channel and writes to the outputBlocks channel.
func StartCalculating(howManyThreads int, inputBlocks chan *mwo.MWOBlock, outputBlocks chan *mwo.MWOBlock) {
	for i := 0; i < howManyThreads; i++ {
		go calculate(inputBlocks, outputBlocks)
	}
}

func InitNOCalculation1() {
	for _, nnr := range NOCopyOfNONO2Ratio {

		var k NOCalculation1Key
		k.fuelTypeID = nnr.fuelTypeID

		var ppaK NOCopyOfPPAKey
		ppaK.polProcessID = nnr.polProcessID

		ppa := NOCopyOfPPA[ppaK]

		ns := NOCopyOfSourceUseType[nnr.sourceTypeID]

		var ppmyKey NOCopyOfPPMYKey
		ppmyKey.modelYearGroupID = nnr.modelYearGroupID
		ppmyKey.polProcessID = nnr.polProcessID

		ppmys := NOCopyOfPPMY[ppmyKey]

		if ppmys != nil && ppa != nil && ns != 0 {
			for _, ppmy := range ppmys {
				var nc1K NOCalculation1Key
				nc1Detail := new(NOCalculation1Detail)

				nc1K.fuelTypeID = nnr.fuelTypeID
				nc1K.modelYearID = ppmy.modelYearID
				nc1K.sourceTypeID = nnr.sourceTypeID

				nc1Detail.NOxRatio = nnr.NOxRatio
				nc1Detail.polProcessID = nnr.polProcessID
				nc1Detail.pollutantID = ppa.pollutantID
				nc1Detail.processID = ppa.processID

				NOCalculation1[nc1K] = append(NOCalculation1[nc1K], nc1Detail)
			}
		}
	}
}

func calculate(inputBlocks chan *mwo.MWOBlock, outputBlocks chan *mwo.MWOBlock) {
	for {
		b := <-inputBlocks

		var newFuelBlocks []*mwo.FuelBlock // created on demand in order to save memory

		for _, fb := range b.FuelBlocks {

			innerFuelBlocks := make(map[int]*mwo.FuelBlock)

			for _, e := range fb.Emissions { // For every fuel formulation's emissions...
				ff := mwo.FuelFormulations[e.FuelFormulationID]
				if ff == nil {
					fmt.Println("Unable to find fuel formulation ", e.FuelFormulationID)
					continue
				}

				emissions := make(map[int]*mwo.MWOEmission)
				ppid := 0

				// Calculate for 301
				ppid = 3*100 + fb.Key.ProcessID

				if mwo.NeededPolProcessIDs[ppid] && fb.Key.PollutantID == 3 {

					var nocK NOCalculation1Key
					nocK.fuelTypeID = fb.Key.FuelTypeID
					nocK.modelYearID = fb.Key.ModelYearID
					nocK.sourceTypeID = fb.Key.SourceTypeID

					nocs := NOCalculation1[nocK]

					if nocs != nil {
						for _, noc := range nocs {
							emissions[32] = mwo.NewEmissionScaled(e, noc.NOxRatio)
							emissions[34] = mwo.NewEmissionScaled(e, noc.NOxRatio)
											
						}
					}
				}

				for pollutantID, e := range emissions {
					if e == nil {
						continue
					}

					// Make a FuelBlock for the pollutant if one doesn't already exist.
					// Copy everything in the key and change the key's pollutantID.
					nfb := innerFuelBlocks[pollutantID]
					if nfb == nil {
						nfb = mwo.NewFuelBlock(fb)
						nfb.Key.PollutantID = pollutantID
						nfb.Key.PolProcessID = pollutantID*100 + nfb.Key.ProcessID
						innerFuelBlocks[pollutantID] = nfb
					}

					// Add the emissions to the fuel block.
					nfb.AddEmission(e)
				}
			}

			// Add new values to the list of new fuel blocks
			for _, nfb := range innerFuelBlocks {
				if nfb.NeedsGFRE {
					// Apply any GeneralFuelRatio[Expression] at the fuel formulation level
					// No GFRE is used by the HCSpeciation calculator.
				}
				if newFuelBlocks == nil {
					newFuelBlocks = make([]*mwo.FuelBlock, 0, cap(b.FuelBlocks))
				}
				newFuelBlocks = append(newFuelBlocks, nfb)
			}
		}

		// Add new fuel blocks to the main block
		if newFuelBlocks != nil {
			for _, nfb := range newFuelBlocks {
				b.AddFuelBlock(nfb)
			}
		}

		outputBlocks <- b
	}
}
