package no2

import (
	"fmt"
	"mwo"
	"parse"
)

/**
 * @algorithm
 * @owner NO2 Calculator
 * @calculator
**/

// NO2CopyOfSourceUseType
var NO2CopyOfSourceUseType map[int]int

// NO2CopyOfPPA
type NO2CopyOfPPADetail struct {
	polProcessID int
	processID    int
	pollutantID  int
}

type NO2CopyOfPPAKey struct {
	polProcessID int
}

var NO2CopyOfPPA map[NO2CopyOfPPAKey]*NO2CopyOfPPADetail

// NO2CopyOfPPMY
type NO2CopyOfPPMYDetail struct {
	modelYearID   int
	fuelMYGroupID int
}

type NO2CopyOfPPMYKey struct {
	polProcessID     int
	modelYearGroupID int
}

var NO2CopyOfPPMY map[NO2CopyOfPPMYKey][]*NO2CopyOfPPMYDetail

// NO2CopyOfFuelType
var NO2CopyOfFuelType map[int]int

// NO2CopyOfNONO2Ratio
type NO2CopyOfNONO2RatioDetail struct {
	polProcessID     int
	sourceTypeID     int
	fuelTypeID       int
	modelYearGroupID int
	NOxRatio         float64
	NOxRatioCV       float64
	dataSourceId     int
}

var NO2CopyOfNONO2Ratio []*NO2CopyOfNONO2RatioDetail

// NO2Calculation1
type NO2Calculation1Detail struct {
	polProcessID int
	processID    int
	pollutantID  int
	NOxRatio     float64
}

type NO2Calculation1Key struct {
	sourceTypeID int
	fuelTypeID   int
	modelYearID  int
}

var NO2Calculation1 map[NO2Calculation1Key][]*NO2Calculation1Detail

// Initialize package-level variables, creating empty lookup tables.
func init() {
	NO2CopyOfSourceUseType = make(map[int]int)
	NO2CopyOfPPA = make(map[NO2CopyOfPPAKey]*NO2CopyOfPPADetail)
	NO2CopyOfPPMY = make(map[NO2CopyOfPPMYKey][]*NO2CopyOfPPMYDetail)
	NO2CopyOfFuelType = make(map[int]int)
	NO2Calculation1 = make(map[NO2Calculation1Key][]*NO2Calculation1Detail)
}

func contains(s []int, e int) bool {
	for _, a := range s {
		if a == e {
			return true
		}
	}
	return false
}

func StartSetup() {
	// NO2CopyOfSourceUseType
	parse.ReadAndParseFile("no2copyofsourceusetype", func(parts []string) {
		// sourceTypeID

		if len(parts) < 1 {
			return
		}

		NO2CopyOfSourceUseType[parse.GetInt(parts[0])] = parse.GetInt(parts[0])
	})

	// NO2CopyOfPPA
	parse.ReadAndParseFile("no2copyofppa", func(parts []string) {
		// polProcessID	int,
		// processID		smallint(6),
		// pollutantID		smallint(6)

		// key: processID, fuelTypeID, sourceTypeID, monthID, modelYearID
		if len(parts) < 3 {
			return
		}

		k := NO2CopyOfPPAKey{
			parse.GetInt(parts[0]),
		}

		v := new(NO2CopyOfPPADetail)
		v.processID = parse.GetInt(parts[1])
		v.pollutantID = parse.GetInt(parts[2])

		NO2CopyOfPPA[k] = v
	})

	// NO2CopyOfPPMY
	parse.ReadAndParseFile("no2copyofppmy", func(parts []string) {
		// polProcessID		int,
		// modelYearID			smallint(6),
		// modelYearGroupID	int(11),
		// fuelMYGroupID		int(11)

		if len(parts) < 4 {
			return
		}

		k := NO2CopyOfPPMYKey{
			parse.GetInt(parts[0]),
			parse.GetInt(parts[2]),
		}

		v := new(NO2CopyOfPPMYDetail)
		v.modelYearID = parse.GetInt(parts[1])
		v.fuelMYGroupID = parse.GetInt(parts[3])

		NO2CopyOfPPMY[k] = append(NO2CopyOfPPMY[k], v)
	})

	// NO2CopyOfFuelType
	parse.ReadAndParseFile("no2copyoffueltype", func(parts []string) {
		// fuelTypeID

		if len(parts) < 1 {
			return
		}

		NO2CopyOfFuelType[parse.GetInt(parts[0])] = parse.GetInt(parts[0])
	})

	// NO2CopyOfNONO2Ratio
	parse.ReadAndParseFile("no2copyofnono2ratio", func(parts []string) {
		// polProcessID		int,
		// sourceTypeID		smallint(6),
		// fuelTypeID			smallint(6),
		// modelYearGroupID	int(11),
		// NOxRatio 			float,
		// NOxRatioCV			float,
		// dataSourceId		smallint(6)

		if len(parts) < 7 {
			return
		}

		v := new(NO2CopyOfNONO2RatioDetail)
		v.polProcessID = parse.GetInt(parts[0])
		v.sourceTypeID = parse.GetInt(parts[1])
		v.fuelTypeID = parse.GetInt(parts[2])
		v.modelYearGroupID = parse.GetInt(parts[3])
		v.NOxRatio = parse.GetFloat(parts[4])
		v.NOxRatioCV = parse.GetFloat(parts[5])
		v.dataSourceId = parse.GetInt(parts[6])

		NO2CopyOfNONO2Ratio = append(NO2CopyOfNONO2Ratio, v)
	})

	InitNOCalculation1()
}

// Complete any pending asychronous operations initiated during StartSetup().
func FinishSetup() {
	// There are no asynchronous reads, so nothing to do here.

	// Done
	fmt.Println("NOCalculator finished reading setup files.")
}

// Launch computation threads. Each thread reads from the inputBlocks channel and writes to the outputBlocks channel.
func StartCalculating(howManyThreads int, inputBlocks chan *mwo.MWOBlock, outputBlocks chan *mwo.MWOBlock) {
	for i := 0; i < howManyThreads; i++ {
		go calculate(inputBlocks, outputBlocks)
	}
}

func InitNOCalculation1() {
	for _, nnr := range NO2CopyOfNONO2Ratio {

		var k NO2Calculation1Key
		k.fuelTypeID = nnr.fuelTypeID

		var ppaK NO2CopyOfPPAKey
		ppaK.polProcessID = nnr.polProcessID

		ppa := NO2CopyOfPPA[ppaK]

		ns := NO2CopyOfSourceUseType[nnr.sourceTypeID]

		var ppmyKey NO2CopyOfPPMYKey
		ppmyKey.modelYearGroupID = nnr.modelYearGroupID
		ppmyKey.polProcessID = nnr.polProcessID

		ppmys := NO2CopyOfPPMY[ppmyKey]

		if ppmys != nil && ppa != nil && ns != 0 {
			for _, ppmy := range ppmys {
				var nc1K NO2Calculation1Key
				nc1Detail := new(NO2Calculation1Detail)

				nc1K.fuelTypeID = nnr.fuelTypeID
				nc1K.modelYearID = ppmy.modelYearID
				nc1K.sourceTypeID = nnr.sourceTypeID

				nc1Detail.NOxRatio = nnr.NOxRatio
				nc1Detail.polProcessID = nnr.polProcessID
				nc1Detail.pollutantID = ppa.pollutantID
				nc1Detail.processID = ppa.processID

				NO2Calculation1[nc1K] = append(NO2Calculation1[nc1K], nc1Detail)
			}
		}
	}
}

func calculate(inputBlocks chan *mwo.MWOBlock, outputBlocks chan *mwo.MWOBlock) {
	for {
		b := <-inputBlocks

		var newFuelBlocks []*mwo.FuelBlock // created on demand in order to save memory

		for _, fb := range b.FuelBlocks {

			innerFuelBlocks := make(map[int]*mwo.FuelBlock)

			for _, e := range fb.Emissions { // For every fuel formulation's emissions...
				ff := mwo.FuelFormulations[e.FuelFormulationID]
				if ff == nil {
					fmt.Println("Unable to find fuel formulation ", e.FuelFormulationID)
					continue
				}

				emissions := make(map[int]*mwo.MWOEmission)
				ppid := 0

				// Calculate for 301
				ppid = 3*100 + fb.Key.ProcessID

				if mwo.NeededPolProcessIDs[ppid] && fb.Key.PollutantID == 3 {

					var nocK NO2Calculation1Key
					nocK.fuelTypeID = fb.Key.FuelTypeID
					nocK.modelYearID = fb.Key.ModelYearID
					nocK.sourceTypeID = fb.Key.SourceTypeID

					nocs := NO2Calculation1[nocK]

					if nocs != nil {
						for _, noc := range nocs {
							emissions[33] = mwo.NewEmissionScaled(e, noc.NOxRatio)
						}
					}
				}

				for pollutantID, e := range emissions {
					if e == nil {
						continue
					}

					// Make a FuelBlock for the pollutant if one doesn't already exist.
					// Copy everything in the key and change the key's pollutantID.
					nfb := innerFuelBlocks[pollutantID]
					if nfb == nil {
						nfb = mwo.NewFuelBlock(fb)
						nfb.Key.PollutantID = pollutantID
						nfb.Key.PolProcessID = pollutantID*100 + nfb.Key.ProcessID
						innerFuelBlocks[pollutantID] = nfb
					}

					// Add the emissions to the fuel block.
					nfb.AddEmission(e)
				}
			}

			// Add new values to the list of new fuel blocks
			for _, nfb := range innerFuelBlocks {
				if nfb.NeedsGFRE {
					// Apply any GeneralFuelRatio[Expression] at the fuel formulation level
					// No GFRE is used by the HCSpeciation calculator.
				}
				if newFuelBlocks == nil {
					newFuelBlocks = make([]*mwo.FuelBlock, 0, cap(b.FuelBlocks))
				}
				newFuelBlocks = append(newFuelBlocks, nfb)
			}
		}

		// Add new fuel blocks to the main block
		if newFuelBlocks != nil {
			for _, nfb := range newFuelBlocks {
				b.AddFuelBlock(nfb)
			}
		}

		outputBlocks <- b
	}
}
