package pm10

import (
	"fmt"
	"mwo"
	"parse"
)

/**
 * @algorithm
 * @owner PM 10 Calculator
 * @calculator
**/

type PM10EmissionRatioKey struct {
	SourceTypeID int
	FuelTypeID   int
}

type PM10EmissionRatioDetail struct {
	PM10PM25Ratio   float64
	PM10PM25RatioCV float64

	PolProcessID int

	MinModelYearID int
	MaxModelYearID int
}

var PM10EmissionRatio map[PM10EmissionRatioKey][]*PM10EmissionRatioDetail

type PM10PollutantProcessAssocDetail struct {
	isAffectedByExhaustIM string
	isAffectedByEvapIM    string
	PollutantID           int
}

type PM10PollutantProcessAssocKey struct {
	PolProcessID int
	ProcessID    int
}

var PM10PollutantProcessAssoc map[PM10PollutantProcessAssocKey][]*PM10PollutantProcessAssocDetail

// Initialize package-level variables, creating empty lookup tables.
func init() {
	PM10EmissionRatio = make(map[PM10EmissionRatioKey][]*PM10EmissionRatioDetail)
	PM10PollutantProcessAssoc = make(map[PM10PollutantProcessAssocKey][]*PM10PollutantProcessAssocDetail)
}

func StartSetup() {
	// PM10EmissionRatio
	parse.ReadAndParseFile("pm10emissionratio", func(parts []string) {

		// `polProcessID` int(11) NOT NULL,
		// `sourceTypeID` smallint(6) NOT NULL,
		// `fuelTypeID` smallint(6) NOT NULL,
		// `PM10PM25Ratio` float NOT NULL,
		// `PM10PM25RatioCV` float DEFAULT NULL,
		// `minModelYearID` smallint(6) NOT NULL DEFAULT '1940',
		// `maxModelYearID` smallint(6) NOT NULL DEFAULT '2050',

		if len(parts) < 7 {
			return
		}

		k := PM10EmissionRatioKey{
			parse.GetInt(parts[1]),
			parse.GetInt(parts[2]),
		}

		v := new(PM10EmissionRatioDetail)
		v.PolProcessID = parse.GetInt(parts[0])
		v.PM10PM25Ratio = parse.GetFloat(parts[3])
		v.PM10PM25RatioCV = parse.GetFloat(parts[4])
		v.MinModelYearID = parse.GetInt(parts[5])
		v.MaxModelYearID = parse.GetInt(parts[6])

		PM10EmissionRatio[k] = append(PM10EmissionRatio[k], v)
	})

	// PM10PollutantProcessAssoc
	parse.ReadAndParseFile("pm10pollutantprocessassoc", func(parts []string) {

		if len(parts) < 5 {
			return
		}

		k := PM10PollutantProcessAssocKey{
			parse.GetInt(parts[0]),
			parse.GetInt(parts[1]),
		}

		v := new(PM10PollutantProcessAssocDetail)
		v.isAffectedByEvapIM = parse.GetString(parts[3])
		v.isAffectedByExhaustIM = parse.GetString(parts[4])
		v.PollutantID = parse.GetInt(parts[2])
		PM10PollutantProcessAssoc[k] = append(PM10PollutantProcessAssoc[k], v)
	})
}

// Complete any pending asychronous operations initiated during StartSetup().
func FinishSetup() {
	// There are no asynchronous reads, so nothing to do here.

	// Done
	fmt.Println("PM10 finished reading setup files.")
}

func contains(s []int, e int) bool {
	for _, a := range s {
		if a == e {
			return true
		}
	}
	return false
}

// Launch computation threads. Each thread reads from the inputBlocks channel and writes to the outputBlocks channel.
func StartCalculating(howManyThreads int, inputBlocks chan *mwo.MWOBlock, outputBlocks chan *mwo.MWOBlock) {
	for i := 0; i < howManyThreads; i++ {
		go calculate(inputBlocks, outputBlocks)
	}
}

func calculate(inputBlocks chan *mwo.MWOBlock, outputBlocks chan *mwo.MWOBlock) {
	for {
		b := <-inputBlocks

		var newFuelBlocks []*mwo.FuelBlock // created on demand in order to save memory

		for _, fb := range b.FuelBlocks {
			// Skip any block that doesn't have a required input pollutant
			// if fb.Key.PollutantID != 116 {
			// 	continue
			// }

			innerFuelBlocks := make(map[int]*mwo.FuelBlock)

			for _, e := range fb.Emissions { // For every fuel formulation's emissions...
				ff := mwo.FuelFormulations[e.FuelFormulationID]
				if ff == nil {
					fmt.Println("Unable to find fuel formulation ", e.FuelFormulationID)
					continue
				}

				emissions := make(map[int]*mwo.MWOEmission)
				ppid := 0

				ppid = fb.Key.PollutantID*100 + fb.Key.ProcessID

				if mwo.NeededPolProcessIDs[ppid] && fb.Key.PollutantID == 116 {
					emissions[106] = mwo.NewEmissionScaled(e, 1)
				}

				var rK PM10EmissionRatioKey
				rK.FuelTypeID = fb.Key.FuelTypeID
				rK.SourceTypeID = fb.Key.SourceTypeID

				rs := PM10EmissionRatio[rK]

				if rs != nil {
					for _, r := range rs {
						if r.MinModelYearID <= fb.Key.ModelYearID && r.MaxModelYearID >= fb.Key.ModelYearID {
							var ppaK PM10PollutantProcessAssocKey
							ppaK.PolProcessID = r.PolProcessID
							ppaK.ProcessID = fb.Key.ProcessID

							ppas := PM10PollutantProcessAssoc[ppaK]

							if ppas != nil {
								for _, ppa := range ppas {
									if ppa.PollutantID == 100 && fb.Key.PollutantID == 110 {
										emissions[100] = mwo.NewEmissionScaled(e, r.PM10PM25Ratio)
									} else if ppa.PollutantID == 106 && fb.Key.PollutantID == 116 {
										emissions[106] = mwo.NewEmissionScaled(e, r.PM10PM25Ratio)
									} else if ppa.PollutantID == 107 && fb.Key.PollutantID == 117 {
										emissions[107] = mwo.NewEmissionScaled(e, r.PM10PM25Ratio)
									} else if ppa.PollutantID == 102 && fb.Key.PollutantID == 112 {
										emissions[102] = mwo.NewEmissionScaled(e, r.PM10PM25Ratio)
									} else if ppa.PollutantID == 101 && fb.Key.PollutantID == 111 {
										emissions[101] = mwo.NewEmissionScaled(e, r.PM10PM25Ratio)
									} else if ppa.PollutantID == 105 && fb.Key.PollutantID == 115 && contains([]int{1, 2, 90, 91}, fb.Key.ProcessID) {
										emissions[105] = mwo.NewEmissionScaled(e, r.PM10PM25Ratio)
									}
								}
							}
						}
					}
				}

				for pollutantID, e := range emissions {
					if e == nil {
						continue
					}

					// Make a FuelBlock for the pollutant if one doesn't already exist.
					// Copy everything in the key and change the key's pollutantID.
					nfb := innerFuelBlocks[pollutantID]
					if nfb == nil {
						nfb = mwo.NewFuelBlock(fb)
						nfb.Key.PollutantID = pollutantID
						nfb.Key.PolProcessID = pollutantID*100 + nfb.Key.ProcessID
						innerFuelBlocks[pollutantID] = nfb
					}

					// Add the emissions to the fuel block.
					nfb.AddEmission(e)
				}
			}

			// Add new values to the list of new fuel blocks
			for _, nfb := range innerFuelBlocks {
				if nfb.NeedsGFRE {
					// Apply any GeneralFuelRatio[Expression] at the fuel formulation level
					// No GFRE is used by the HCSpeciation calculator.
				}
				if newFuelBlocks == nil {
					newFuelBlocks = make([]*mwo.FuelBlock, 0, cap(b.FuelBlocks))
				}
				newFuelBlocks = append(newFuelBlocks, nfb)
			}
		}

		// Add new fuel blocks to the main block
		if newFuelBlocks != nil {
			for _, nfb := range newFuelBlocks {
				b.AddFuelBlock(nfb)

				// ppid := nfb.Key.PollutantID*100 + 15

				// if mwo.NeededPolProcessIDs[ppid] {
				// 	nfb15 := mwo.NewFuelBlock(nfb)
				// 	nfb15.Key.ProcessID = 15
				// 	nfb15.Key.PolProcessID = nfb.Key.PollutantID*100 + 15
				// 	b.AddFuelBlock(nfb15)
				// }
			}
		}

		outputBlocks <- b
	}
}
