package sulfatepm

import (
	"fmt"
	"math"
	"mwo"
	"parse"
)

/**
 * @algorithm
 * @owner Sulfate PM Calculator
 * @calculator
**/

// SPMOneCountyYearGeneralFuelRatio
type SPMOneCountyYearGeneralFuelRatioDetail struct {
	processID       int
	fuelEffectRatio float64
}

type SPMOneCountyYearGeneralFuelRatioKey struct {
	fuelTypeID, sourceTypeID, monthID, pollutantID, modelYearID, yearID int
}

var SPMOneCountyYearGeneralFuelRatio map[SPMOneCountyYearGeneralFuelRatioKey]*SPMOneCountyYearGeneralFuelRatioDetail

// OneCountyYearSulfateFractions
type OneCountyYearSulfateFractionsDetail struct {
	SulfateNonECPMFraction, H2ONonECPMFraction, UnadjustedSulfatenonECPMFraction, UnadjustedH2ONonECPMFraction float64
}

type OneCountyYearSulfateFractionsKey struct {
	processID, fuelTypeID, sourceTypeID, monthID, modelYearID int
}

var OneCountyYearSulfateFractions map[OneCountyYearSulfateFractionsKey]*OneCountyYearSulfateFractionsDetail

// PMSpeciation
type PMSpeciationDetail struct {
	pmSpeciationFraction                              float64
	minModelYearID, maxModelYearID, outputPollutantID int
}

type PMSpeciationKey struct {
	processID, inputPollutantID, sourceTypeID, fuelTypeID int
}

var PMSpeciation map[PMSpeciationKey][]*PMSpeciationDetail

// OneZoneYearTemperatureFactor
type OneZoneYearTemperatureFactorDetail struct {
	correctionFactor               float64
	minModelYearID, maxModelYearID int
}

type OneZoneYearTemperatureFactorKey struct {
	zoneID, monthID, hourID, processID, pollutantID, fuelTypeID, sourceTypeID int
}

var OneZoneYearTemperatureFactor map[OneZoneYearTemperatureFactorKey][]*OneZoneYearTemperatureFactorDetail

// CrankCaseSplit
type CrankCaseSplitDetail struct {
	crankcaseRatio                            float64
	processID, minModelYearID, maxModelYearID int
}

type CrankCaseSplitKey struct {
	pollutantID, sourceTypeID, fuelTypeID int
}

var CrankCaseSplit map[CrankCaseSplitKey][]*CrankCaseSplitDetail

// Initialize package-level variables, creating empty lookup tables.
func init() {
	SPMOneCountyYearGeneralFuelRatio = make(map[SPMOneCountyYearGeneralFuelRatioKey]*SPMOneCountyYearGeneralFuelRatioDetail)
	OneCountyYearSulfateFractions = make(map[OneCountyYearSulfateFractionsKey]*OneCountyYearSulfateFractionsDetail)
	PMSpeciation = make(map[PMSpeciationKey][]*PMSpeciationDetail)
	OneZoneYearTemperatureFactor = make(map[OneZoneYearTemperatureFactorKey][]*OneZoneYearTemperatureFactorDetail)
	CrankCaseSplit = make(map[CrankCaseSplitKey][]*CrankCaseSplitDetail)
}

func contains(s []int, e int) bool {
	for _, a := range s {
		if a == e {
			return true
		}
	}
	return false
}

func StartSetup() {
	// SPMOneCountyYearGeneralFuelRatio
	parse.ReadAndParseFile("spmonecountyyeargeneralfuelratio", func(parts []string) {
		// fuelTypeID
		// sourceTypeID
		// monthID
		// pollutantID
		// processID
		// modelYearID
		// yearID
		// fuelEffectRatio

		// key: fuelTypeID, sourceTypeID, monthID, pollutantID, modelYearID, yearID

		if len(parts) < 8 {
			return
		}

		k := SPMOneCountyYearGeneralFuelRatioKey{
			parse.GetInt(parts[0]),
			parse.GetInt(parts[1]),
			parse.GetInt(parts[2]),
			parse.GetInt(parts[3]),
			parse.GetInt(parts[5]),
			parse.GetInt(parts[6]),
		}

		v := new(SPMOneCountyYearGeneralFuelRatioDetail)
		v.processID = parse.GetInt(parts[4])
		v.fuelEffectRatio = parse.GetFloat(parts[7])
		SPMOneCountyYearGeneralFuelRatio[k] = v
	})

	// OneCountyYearSulfateFractions
	parse.ReadAndParseFile("onecountyyearsulfatefractions", func(parts []string) {
		// processID,
		// fuelTypeID,
		// sourceTypeID,
		// monthID,
		// modelYearID,
		// SulfateNonECPMFraction,
		// H2ONonECPMFraction,
		// UnadjustedSulfatenonECPMFraction,
		// UnadjustedH2ONonECPMFraction,

		// key: processID, fuelTypeID, sourceTypeID, monthID, modelYearID
		if len(parts) < 9 {
			return
		}

		k := OneCountyYearSulfateFractionsKey{
			parse.GetInt(parts[0]),
			parse.GetInt(parts[1]),
			parse.GetInt(parts[2]),
			parse.GetInt(parts[3]),
			parse.GetInt(parts[4]),
		}

		v := new(OneCountyYearSulfateFractionsDetail)
		v.SulfateNonECPMFraction = parse.GetFloat(parts[5])
		v.H2ONonECPMFraction = parse.GetFloat(parts[6])
		v.UnadjustedSulfatenonECPMFraction = parse.GetFloat(parts[7])
		v.UnadjustedH2ONonECPMFraction = parse.GetFloat(parts[8])

		OneCountyYearSulfateFractions[k] = v
	})

	// PMSpeciation
	parse.ReadAndParseFile("pmspeciation", func(parts []string) {
		// processID,
		// inputPollutantID,
		// sourceTypeID,
		// fuelTypeID,
		// minModelYearID,
		// maxModelYearID,
		// outputPollutantID,
		// pmSpeciationFraction

		// key: processID, inputPollutantID, sourceTypeID, fuelTypeID, minModelYearID, maxModelYearID, outputPollutantID

		if len(parts) < 8 {
			return
		}

		k := PMSpeciationKey{
			parse.GetInt(parts[0]),
			parse.GetInt(parts[1]),
			parse.GetInt(parts[2]),
			parse.GetInt(parts[3]),
		}

		v := new(PMSpeciationDetail)
		v.minModelYearID = parse.GetInt(parts[4])
		v.maxModelYearID = parse.GetInt(parts[5])
		v.outputPollutantID = parse.GetInt(parts[6])
		v.pmSpeciationFraction = parse.GetFloat(parts[7])

		PMSpeciation[k] = append(PMSpeciation[k], v)
	})

	// OneZoneYearTemperatureFactor
	parse.ReadAndParseFile("onezoneyeartemperaturefactor", func(parts []string) {
		// zoneID
		// monthID
		// hourID
		// processID
		// pollutantID
		// fuelTypeID
		// sourceTypeID
		// minModelYearID
		// maxModelYearID
		// correctionFactor

		// key: zoneID, monthID, hourID, processID, pollutantID, fuelTypeID, sourceTypeID, minModelYearID, maxModelYearID
		if len(parts) < 10 {
			return
		}

		k := OneZoneYearTemperatureFactorKey{
			parse.GetInt(parts[0]),
			parse.GetInt(parts[1]),
			parse.GetInt(parts[2]),
			parse.GetInt(parts[3]),
			parse.GetInt(parts[4]),
			parse.GetInt(parts[5]),
			parse.GetInt(parts[6]),
		}

		v := new(OneZoneYearTemperatureFactorDetail)
		v.minModelYearID = parse.GetInt(parts[7])
		v.maxModelYearID = parse.GetInt(parts[8])
		v.correctionFactor = parse.GetFloat(parts[9])

		OneZoneYearTemperatureFactor[k] = append(OneZoneYearTemperatureFactor[k], v)
	})

	// CrankCaseSplit
	parse.ReadAndParseFile("crankcasesplit", func(parts []string) {
		// processID,
		// pollutantID,
		// sourceTypeID,
		// fuelTypeID,
		// minModelYearID,
		// maxModelYearID,
		// crankcaseRatio double not null,

		// key: processID, pollutantID, sourceTypeID, fuelTypeID, minModelYearID, maxModelYearID
		if len(parts) < 7 {
			return
		}

		k := CrankCaseSplitKey{
			parse.GetInt(parts[1]),
			parse.GetInt(parts[2]),
			parse.GetInt(parts[3]),
		}

		v := new(CrankCaseSplitDetail)
		v.processID = parse.GetInt(parts[0])
		v.minModelYearID = parse.GetInt(parts[4])
		v.maxModelYearID = parse.GetInt(parts[5])
		v.crankcaseRatio = parse.GetFloat(parts[6])

		CrankCaseSplit[k] = append(CrankCaseSplit[k], v)
	})

	//InitSpmSplit1()
}

// Complete any pending asychronous operations initiated during StartSetup().
func FinishSetup() {
	// There are no asynchronous reads, so nothing to do here.

	// Done
	fmt.Println("SulfatePM finished reading setup files.")
}

// Launch computation threads. Each thread reads from the inputBlocks channel and writes to the outputBlocks channel.
func StartCalculating(howManyThreads int, inputBlocks chan *mwo.MWOBlock, outputBlocks chan *mwo.MWOBlock) {
	for i := 0; i < howManyThreads; i++ {
		go calculate(inputBlocks, outputBlocks)
	}
}

func calculate(inputBlocks chan *mwo.MWOBlock, outputBlocks chan *mwo.MWOBlock) {
	for {
		b := <-inputBlocks

		nb := new(mwo.MWOBlock)

		var newFuelBlocks []*mwo.FuelBlock // created on demand in order to save memory

		var crankCaseSplits []*CrankCaseSplitDetail
		var spmSplit120, spmSplit115, spmSplit119 *OneCountyYearSulfateFractionsDetail
		var feRatio120, feRatio115, feRatio119 *SPMOneCountyYearGeneralFuelRatioDetail
		var correctionFactors120, correctionFactors115, correctionFactors119 []*OneZoneYearTemperatureFactorDetail
		var crankCaseSplits120, crankCaseSplits115, crankCaseSplits119 []*CrankCaseSplitDetail

		for _, fb := range b.FuelBlocks {

			// Skip any block that doesn't have a required input pollutant
			if fb.Key.PollutantID != 118 && fb.Key.PollutantID != 112 {
				nb.AddFuelBlock(fb)
				continue
			}

			innerFuelBlocks := make(map[int]*mwo.FuelBlock)

			if fb.Key.PollutantID == 118 {
				spmSplit120, feRatio120, correctionFactors120, crankCaseSplits120 = getFactors(&fb.Key, 120)
				spmSplit115, feRatio115, correctionFactors115, crankCaseSplits115 = getFactors(&fb.Key, 115)
				spmSplit119, feRatio119, correctionFactors119, crankCaseSplits119 = getFactors(&fb.Key, 119)
			} else if fb.Key.PollutantID == 112 {
				var k CrankCaseSplitKey
				k.fuelTypeID = fb.Key.FuelTypeID
				k.sourceTypeID = fb.Key.SourceTypeID
				k.pollutantID = fb.Key.PollutantID
				crankCaseSplits = CrankCaseSplit[k]
			}

			for _, e := range fb.Emissions { // For every fuel formulation's emissions...
				ff := mwo.FuelFormulations[e.FuelFormulationID]
				if ff == nil {
					fmt.Println("Unable to find fuel formulation ", e.FuelFormulationID)
					continue
				}

				emissions := make(map[int]*mwo.MWOEmission)

				ppid112 := fb.Key.PollutantID*100 + fb.Key.ProcessID

				if mwo.NeededPolProcessIDs[ppid112] && fb.Key.PollutantID == 112 {
					if crankCaseSplits != nil {
						for _, crankCaseSplit := range crankCaseSplits {
							if fb.Key.ModelYearID >= crankCaseSplit.minModelYearID &&
								fb.Key.ModelYearID <= crankCaseSplit.maxModelYearID {

								// 112
								ccPpid112 := 112*100 + crankCaseSplit.processID
								if mwo.NeededPolProcessIDs[ccPpid112] {
									emissions[ccPpid112] = mwo.NewEmissionScaled(e, crankCaseSplit.crankcaseRatio)
								}

								// 110
								ppid110 := 110*100 + crankCaseSplit.processID
								if mwo.NeededPolProcessIDs[ppid110] {
									emissions[ppid110] = mwo.NewEmissionScaled(emissions[ccPpid112], 1)
								}
							}
						}
					}
				}

				ppid118 := fb.Key.PollutantID*100 + fb.Key.ProcessID
				ppid115 := 115*100 + fb.Key.ProcessID

				if mwo.NeededPolProcessIDs[ppid118] && fb.Key.PollutantID == 118 {
					if spmSplit115 != nil {
						emissions[ppid115] = mwo.NewEmissionScaled(e, spmSplit115.SulfateNonECPMFraction)
					}

					if feRatio115 != nil && feRatio115.processID == fb.Key.ProcessID {
						if emissions[ppid115] == nil {
							emissions[ppid115] = mwo.NewEmissionScaled(e, feRatio115.fuelEffectRatio)
						} else {
							emissions[ppid115] = mwo.NewEmissionScaled(emissions[ppid115], feRatio115.fuelEffectRatio)
						}
					}

					if correctionFactors115 != nil {
						for _, correctionFactor := range correctionFactors115 {
							if fb.Key.ModelYearID >= correctionFactor.minModelYearID &&
								fb.Key.ModelYearID <= correctionFactor.maxModelYearID {

								if emissions[ppid115] == nil {
									emissions[ppid115] = mwo.NewEmissionScaled(e, correctionFactor.correctionFactor)
								} else {
									emissions[ppid115] = mwo.NewEmissionScaled(emissions[ppid115], correctionFactor.correctionFactor)
								}
							}
						}
					}

					if crankCaseSplits115 != nil {
						for _, crankCaseSplit := range crankCaseSplits115 {
							if fb.Key.ModelYearID >= crankCaseSplit.minModelYearID &&
								fb.Key.ModelYearID <= crankCaseSplit.maxModelYearID {

								ccPpid115 := 115*100 + crankCaseSplit.processID

								if emissions[ccPpid115] == nil {
									if emissions[ppid115] == nil {
										emissions[ccPpid115] = mwo.NewEmissionScaled(e, crankCaseSplit.crankcaseRatio)
									} else {
										emissions[ccPpid115] = mwo.NewEmissionScaled(emissions[ppid115], crankCaseSplit.crankcaseRatio)
									}

								} else {
									emissions[ccPpid115] = mwo.NewEmissionScaled(emissions[ccPpid115], crankCaseSplit.crankcaseRatio)
								}
							}
						}
					}

					ppid119 := 119*100 + fb.Key.ProcessID

					if spmSplit119 != nil {
						emissions[ppid119] = mwo.NewEmissionScaled(e, spmSplit119.H2ONonECPMFraction)
					}

					if feRatio119 != nil && feRatio119.processID == fb.Key.ProcessID {
						if emissions[ppid119] != nil {
							emissions[ppid119] = mwo.NewEmissionScaled(emissions[ppid119], feRatio119.fuelEffectRatio)

						} else {
							emissions[ppid119] = mwo.NewEmissionScaled(e, feRatio119.fuelEffectRatio)
						}
					}

					if correctionFactors119 != nil {
						for _, correctionFactor := range correctionFactors119 {
							if fb.Key.ModelYearID >= correctionFactor.minModelYearID &&
								fb.Key.ModelYearID <= correctionFactor.maxModelYearID {

								if emissions[ppid119] == nil {
									emissions[ppid119] = mwo.NewEmissionScaled(e, correctionFactor.correctionFactor)
								} else {
									emissions[ppid119] = mwo.NewEmissionScaled(emissions[ppid119], correctionFactor.correctionFactor)
								}
							}
						}
					}

					if crankCaseSplits119 != nil {
						for _, crankCaseSplit := range crankCaseSplits119 {
							if fb.Key.ModelYearID >= crankCaseSplit.minModelYearID &&
								fb.Key.ModelYearID <= crankCaseSplit.maxModelYearID {

								ccPpid119 := 119*100 + crankCaseSplit.processID

								if emissions[ccPpid119] == nil {
									if emissions[ppid119] == nil {
										emissions[ccPpid119] = mwo.NewEmissionScaled(e, crankCaseSplit.crankcaseRatio)
									} else {
										emissions[ccPpid119] = mwo.NewEmissionScaled(emissions[ppid119], crankCaseSplit.crankcaseRatio)
									}

								} else {
									emissions[ccPpid119] = mwo.NewEmissionScaled(emissions[ccPpid119], crankCaseSplit.crankcaseRatio)
								}
							}
						}
					}

					ppid120 := 120*100 + fb.Key.ProcessID

					if spmSplit120 != nil {
						emissions[ppid120] = mwo.NewEmissionScaled(e, math.Max(1-spmSplit120.UnadjustedH2ONonECPMFraction-spmSplit120.UnadjustedSulfatenonECPMFraction, 0))
					}

					if feRatio120 != nil && feRatio120.processID == fb.Key.ProcessID {
						if emissions[ppid120] == nil {
							emissions[ppid120] = mwo.NewEmissionScaled(e, feRatio120.fuelEffectRatio)
						} else {
							emissions[ppid120] = mwo.NewEmissionScaled(emissions[ppid120], feRatio120.fuelEffectRatio)
						}
					}

					if correctionFactors120 != nil {
						for _, correctionFactor := range correctionFactors120 {
							if fb.Key.ModelYearID >= correctionFactor.minModelYearID &&
								fb.Key.ModelYearID <= correctionFactor.maxModelYearID {

								if emissions[ppid120] == nil {
									emissions[ppid120] = mwo.NewEmissionScaled(e, correctionFactor.correctionFactor)
								} else {
									emissions[ppid120] = mwo.NewEmissionScaled(emissions[ppid120], correctionFactor.correctionFactor)
								}
							}
						}
					}

					if crankCaseSplits120 != nil {

						for _, crankCaseSplit := range crankCaseSplits120 {
							if fb.Key.ModelYearID >= crankCaseSplit.minModelYearID &&
								fb.Key.ModelYearID <= crankCaseSplit.maxModelYearID {

								ccPpid120 := 120*100 + crankCaseSplit.processID

								if emissions[ccPpid120] == nil {
									if emissions[ppid120] == nil {
										emissions[ccPpid120] = mwo.NewEmissionScaled(e, crankCaseSplit.crankcaseRatio)
									} else {
										emissions[ccPpid120] = mwo.NewEmissionScaled(emissions[ppid120], crankCaseSplit.crankcaseRatio)
									}
								} else {
									emissions[ccPpid120] = mwo.NewEmissionScaled(emissions[ccPpid120], crankCaseSplit.crankcaseRatio)
								}
							}
						}
					}

					for _, processID := range []int{1, 2, 90, 91, 15, 16, 17} {
						ppid115 := 115*100 + processID
						ppid119 := 119*100 + processID
						ppid120 := 120*100 + processID
						ppid118 := 118*100 + processID
						ppid110 := 110*100 + processID

						e115 := emissions[ppid115]
						e119 := emissions[ppid119]
						e120 := emissions[ppid120]

						if mwo.NeededPolProcessIDs[ppid118] {
							emissions[ppid118] = mwo.NewEmissionSum(e115, e119)
							emissions[ppid118] = mwo.NewEmissionSum(emissions[ppid118], e120)

						}

						if mwo.NeededPolProcessIDs[ppid110] {
							emissions[ppid110] = mwo.NewEmissionSum(e115, e119)
							emissions[ppid110] = mwo.NewEmissionSum(emissions[ppid110], e120)
						}
					}

					for _, processID := range []int{1, 2, 90, 91, 15, 16, 17} {
						// PM speciation
						var ks PMSpeciationKey
						ks.processID = processID
						ks.inputPollutantID = 120
						ks.sourceTypeID = fb.Key.SourceTypeID
						ks.fuelTypeID = fb.Key.FuelTypeID

						pmSpecs := PMSpeciation[ks]

						for _, pmSpec := range pmSpecs {

							if fb.Key.ModelYearID >= pmSpec.minModelYearID &&
								fb.Key.ModelYearID <= pmSpec.maxModelYearID {
								ppid := 120*100 + processID
								ppidOut := pmSpec.outputPollutantID*100 + processID

								if mwo.NeededPolProcessIDs[ppidOut] {
									emissions[ppidOut] = mwo.NewEmissionScaled(emissions[ppid], pmSpec.pmSpeciationFraction)
								}
							}
						}
					}
				}

				for ppid, e := range emissions {
					pollutantID := ppid / 100
					processID := ppid % 100

					if e == nil || pollutantID == 120 {
						continue
					}

					// Make a FuelBlock for the pollutant if one doesn't already exist.
					// Copy everything in the key and change the key's pollutantID.
					nfb := innerFuelBlocks[ppid]
					if nfb == nil && mwo.NeededPolProcessIDs[ppid] {
						nfb = mwo.NewFuelBlock(fb)

						nfb.Key.PollutantID = pollutantID
						nfb.Key.PolProcessID = ppid
						nfb.Key.ProcessID = processID
						innerFuelBlocks[ppid] = nfb
					}

					// Add the emissions to the fuel block.
					nfb.AddEmission(e)
				}
			}

			// Add new values to the list of new fuel blocks
			for _, nfb := range innerFuelBlocks {
				if nfb.NeedsGFRE {
					// Apply any GeneralFuelRatio[Expression] at the fuel formulation level
					// No GFRE is used by the HCSpeciation calculator.
				}
				if newFuelBlocks == nil {
					newFuelBlocks = make([]*mwo.FuelBlock, 0, cap(b.FuelBlocks))
				}
				newFuelBlocks = append(newFuelBlocks, nfb)
			}
		}

		// Add new fuel blocks to the main block
		if newFuelBlocks != nil {
			for _, nfb := range newFuelBlocks {
				nb.AddFuelBlock(nfb)
			}
		}

		outputBlocks <- nb
	}
}

func getFactors(fbKey *mwo.MWOKey, pollutantID int) (spmSplit *OneCountyYearSulfateFractionsDetail, feRatio *SPMOneCountyYearGeneralFuelRatioDetail, correctionFactors []*OneZoneYearTemperatureFactorDetail, crankCaseSplits []*CrankCaseSplitDetail) {
	var k1 OneCountyYearSulfateFractionsKey
	k1.processID = fbKey.ProcessID
	k1.fuelTypeID = fbKey.FuelTypeID
	k1.sourceTypeID = fbKey.SourceTypeID
	k1.monthID = fbKey.MonthID
	k1.modelYearID = fbKey.ModelYearID
	spmSplit = OneCountyYearSulfateFractions[k1]

	var k2 SPMOneCountyYearGeneralFuelRatioKey
	k2.fuelTypeID = fbKey.FuelTypeID
	k2.modelYearID = fbKey.ModelYearID
	k2.monthID = fbKey.MonthID
	k2.sourceTypeID = fbKey.SourceTypeID
	k2.yearID = fbKey.YearID
	k2.pollutantID = pollutantID

	feRatio = SPMOneCountyYearGeneralFuelRatio[k2]

	var k3 OneZoneYearTemperatureFactorKey
	k3.zoneID = fbKey.ZoneID
	k3.monthID = fbKey.MonthID
	k3.hourID = fbKey.HourID
	k3.processID = fbKey.ProcessID
	k3.fuelTypeID = fbKey.FuelTypeID
	k3.sourceTypeID = fbKey.SourceTypeID
	k3.pollutantID = pollutantID

	correctionFactors = OneZoneYearTemperatureFactor[k3]

	var k4 CrankCaseSplitKey
	k4.fuelTypeID = fbKey.FuelTypeID
	k4.sourceTypeID = fbKey.SourceTypeID
	k4.pollutantID = pollutantID

	crankCaseSplits = CrankCaseSplit[k4]

	return spmSplit, feRatio, correctionFactors, crankCaseSplits
}

func getFactorsFor112(fbKey *mwo.MWOKey) (*SPMOneCountyYearGeneralFuelRatioDetail, []*OneZoneYearTemperatureFactorDetail, []*CrankCaseSplitDetail, []*PMSpeciationDetail) {
	var k1 SPMOneCountyYearGeneralFuelRatioKey
	k1.fuelTypeID = fbKey.FuelTypeID
	k1.modelYearID = fbKey.ModelYearID
	k1.monthID = fbKey.MonthID
	k1.pollutantID = fbKey.PollutantID
	k1.sourceTypeID = fbKey.SourceTypeID
	k1.yearID = fbKey.YearID

	feRatio := SPMOneCountyYearGeneralFuelRatio[k1]

	var k2 OneZoneYearTemperatureFactorKey
	k2.zoneID = fbKey.ZoneID
	k2.monthID = fbKey.MonthID
	k2.hourID = fbKey.HourID
	k2.processID = fbKey.ProcessID
	k2.pollutantID = fbKey.PollutantID
	k2.fuelTypeID = fbKey.FuelTypeID
	k2.sourceTypeID = fbKey.SourceTypeID

	correctionFactors := OneZoneYearTemperatureFactor[k2]

	var k3 CrankCaseSplitKey
	k3.fuelTypeID = fbKey.FuelTypeID
	k3.sourceTypeID = fbKey.SourceTypeID
	k3.pollutantID = fbKey.PollutantID

	crankCaseSplits := CrankCaseSplit[k3]

	var k4 PMSpeciationKey
	k4.processID = fbKey.ProcessID
	k4.inputPollutantID = fbKey.PollutantID
	k4.sourceTypeID = fbKey.SourceTypeID
	k4.fuelTypeID = fbKey.FuelTypeID

	pmSpecs := PMSpeciation[k4]

	return feRatio, correctionFactors, crankCaseSplits, pmSpecs
}
