// Package configuration establishes the runtime environment, validates
// and completes the input parameters, opens the execution database.
package configuration

import (
	"flag"
	"fmt"
  "strings"

	_ "mysql"
	"database/sql"
)

//type Configuration struct creates a structure encompassing the
//startup for the current run.
type Configuration struct {
	IsReady bool
	MaximumMemory int64
	ClassName string
	ExecutionDatabase string
	DBUser, DBPassword string
	RawParameters string
	Parameters []string
	IsProject bool
	SaveROMD bool
}

// The one and only Configuration object in existence
var Singleton Configuration

// init performs package-level initialization.
// Currently, there is nothing to initialize.
func init() {
	// Nothing to do here
}

// Setup reads the command line parameters, storing their values into
// an existing Configuration object.
func (c *Configuration) Setup() {
	fmt.Println("Reading the global configuration values...")
	c.IsReady = false
	c.MaximumMemory = 1024 * 1024 * 1024 // 1GB
	c.MaximumMemory *= 4 // 4GB

	classArg := flag.String("class", "classname", "Name of module to be run")
	databaseArg := flag.String("db", "executiondbname", "Name of MOVES Execution MySQL database")
	dbUserArg := flag.String("dbuser", "username", "MySQL user name")
	dbPasswordArg := flag.String("dbpwd", "password", "MySQL password")
	paramsArg := flag.String("params", "parameters", "Optional context parameters")
	projectArg := flag.String("isproject", "isproject", "Optional project flag")
	saveROMDArg := flag.String("saveromd", "saveROMD", "Optional flag to save ROMD data")

	flag.Parse()
	c.ClassName = *classArg
	c.ExecutionDatabase = *databaseArg
	c.DBUser = *dbUserArg
	c.DBPassword = *dbPasswordArg
	c.RawParameters = *paramsArg
	c.Parameters = strings.Split(c.RawParameters,",")
	if c.Parameters != nil && len(c.Parameters) > 0 {
		for i:=0;i<len(c.Parameters);i++ {
			c.Parameters[i] = strings.TrimSpace(c.Parameters[i])
		}
	}
	c.IsProject = *projectArg == "1"
	c.SaveROMD = *saveROMDArg == "1"

	fmt.Println("Class name to execute:",c.ClassName)
	fmt.Println("Execution parameters:",c.RawParameters)
	fmt.Println("Execution database:",c.ExecutionDatabase)
	fmt.Println("Using MySQL user:",c.DBUser)
	// Don't print the password
	fmt.Println("IsProject:",c.IsProject)
	fmt.Println("SaveROMD:",c.SaveROMD)

	fmt.Println("Done reading the global configuration values.")
}

// OpenInputDatabase opens the execution database and confirms it is accessible.
func OpenExecutionDatabase() *sql.DB {
	fmt.Println("Opening execution database connection...")
	//connectionString := configuration.Singleton.DBUser + ":" + configuration.Singleton.DBPassword + "@/" + configuration.Singleton.InputDatabase + "?charset=utf8"
	connectionString := Singleton.DBUser + ":" + Singleton.DBPassword + "@/" + Singleton.ExecutionDatabase
	db, err := sql.Open("mysql", connectionString)
	CheckErr(err)
	fmt.Println("Doing database ping to verify execution database connection...")
	err = db.Ping()
	CheckErr(err)
	return db
}

// CheckErr terminates the program upon an error
func CheckErr(err error) {
	if err != nil {
		panic(err)
	}
}

// TestInputDatabase opens the input database and tests reading rows from the nrAge table.
// This function also serves as a template for database reading logic.
func TestExecutionDatabase() {
	// https://astaxie.gitbooks.io/build-web-application-with-golang/content/en/05.2.html
	db := OpenExecutionDatabase()
	defer db.Close()

	fmt.Println("Running query in execution database...")
	rows, err := db.Query("select ageID, ageGroupID from ageCategory")
	CheckErr(err)

    defer rows.Close()
    for rows.Next() {
		var ageID int
		var ageGroupID int
        err = rows.Scan(&ageID, &ageGroupID)
		CheckErr(err)
        //fmt.Println(ageID)
    }
	fmt.Println("Execution database query succeeded.")
}
