/**************************************************************************************************
 * @(#)GraphUtilities.java
 *
 *************************************************************************************************/
package gov.epa.otaq.moves.common.graph;

import java.util.*;

/**
 * Utility routines for finding resources within node inputs and outputs.
 * 
 * @author		Wesley Faler
 * @version		2010-05-24
**/
public class GraphUtilities {
	/**
	 * Check for overlap in the resources generated by a producer and those
	 * required by a consumer.
	 * @param producer node that creates resources
	 * @param consumer node that consumes resources
	 * @return true if any resource required by the consumer is directly
	 * generated by the producer
	**/
	public static boolean areRelated(INode producer, INode consumer) {
		for(Iterator<IResource> i=producer.getOutputs();i!=null && i.hasNext();) {
			IResource r = i.next();
			if(isInInputs(r,consumer)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Scan a node's inputs for a resource.  A resource may be fully
	 * or partially specified as noted in the IResource documentation.
	 * @param r resource to be found
	 * @param n node to be scanned
	 * @return true if the resource is an input needed by the node.
	**/
	public static boolean isInInputs(IResource r, INode n) {
		return isInList(r,n.getInputs());
	}

	/**
	 * Scan a node's outputs for a resource.  A resource may be fully
	 * or partially specified as noted in the IResource documentation.
	 * @param r resource to be found
	 * @param n node to be scanned
	 * @return true if the resource is an output populated by the node.
	**/
	public static boolean isInOutputs(IResource r, INode n) {
		return isInList(r,n.getOutputs());
	}

	/**
	 * Scan a list of resources for a given resource.  Resources may be
	 * fully or partially specified as noted in the IResource documentation.
	 * @param r resource to be found
	 * @param resources set of resources to be scanned
	 * @return true if the resource is found in the set of scanned resources
	**/
	public static boolean isInList(IResource r, Iterator<IResource> resources) {
		if(resources == null || r == null) {
			return false;
		}
		// true if anything in resources startsWith(r)
		// true if r startsWith(anything in resources)
		String rPath = r.getResourcePath().toLowerCase();
		for(;resources.hasNext();) {
			IResource candidate = resources.next();
			String candidatePath = candidate.getResourcePath().toLowerCase();
			if(candidatePath.startsWith(rPath) || rPath.startsWith(candidatePath)) {
				return true;
			}
		}
		return false;
	}
}
