/**************************************************************************************************
 * @(#)SystemConfiguration.java
 *
 *************************************************************************************************/
package gov.epa.otaq.moves.master.framework;

import gov.epa.otaq.moves.common.*;
import java.io.*;
import java.util.*;
import java.net.*;

/**
 * Manages various system configuration attributes such as database locations and local
 * and shared path information.  This doesn't include RunSpec information which is maintained
 * elsewhere. It contains the databases, and information about them, such as
 * name, user, password, the Distributed Master ID, the directories and file names
 * used by the application, such as Shared Distributed Folder path, GREET
 * directory, GREETWellToPump and GREETManufacture applications.
 *
 * @author		Wesley Faler
 * @author		Sarah Luo
 * @author		Tim Hull
 * @version		2016-03-14
**/
public class SystemConfiguration {
	/** Name of the file holding the prior master's ID file name and path **/
	static public String priorIDFileName = "PriorMasterID.txt";

	/**
	 * Connection params for each of the database types, index is by
	 * MOVESDatabaseType.getIndex().
	**/
	public DatabaseSelection[] databaseSelections = new DatabaseSelection[MOVESDatabaseType.NUM_TYPES];

	/** The name of the MOVES configuration file **/
	public static String MOVES_CONFIGURATION_FILE_NAME = "MOVESConfiguration.txt";

	/**
	 * Value for the location of the create tables sql script, used to generate
	 * an output database.
	**/
	public static final String CREATE_OUTPUT_SQL_FILE_PATH = "database/CreateOutputDB.sql";

	/** reference to the single SystemConfiguration() instance **/
	static public SystemConfiguration theSystemConfiguration = new SystemConfiguration();

	/**
	 * Deprecated: The GREET directory path.
	**/
	public File GREETDirectory;

	/**
	 * Deprecated: The GREET WTP application.
	**/
	public File GREETWTPApplication;

	/**
	 * Deprecated: The GREET Manufacture application
	**/
	public File GREETManufactureApplication;

	/**
	 * The folder used for distributed computing coordination. The master system queues
	 * work files here, the workers detect them, process them, and store result files here.
	**/
	public File sharedDistributedFolderPath;

	/** The ID to use within the distributed processing system. **/
	public String distributedMasterID;

	/** Computer ID from the MOVESConfiguration.txt file **/
	public String computerID;

	/** Path to file containing the computer ID assignment line **/
	public String computerIDPath;

	/** Full path and name of the NONROAD EXE external program. **/
	public File nonroadExePath = null;

	/**
	 * This master uses this folder to store temporary internal files. Temporary
	 * sub-directories are created on startup to assure that there is no potential confusion
	 * with files not belonging to this master.
	**/
	public File masterFolderPath;

	/**
	 * Optional location to store TODO files generated by this master.  If null, TODO files
	 * are not saved.
	**/
	public File saveTODOPath;

	/** Full path and name of the external generator program, if any. **/	
	public File generatorExePath = null;


	/** The name of the MOVES Worker configuration file **/
	public static String MOVES_WORKER_CONFIGURATION_FILE_NAME = "WorkerConfiguration.txt";

	/** The name of the manyworkers configuration file **/
	public static String MANY_WORKERS_CONFIGURATION_FILE_NAME = "manyworkers.txt";

	/** The name of the maketodo configuration file **/
	public static String MAKE_TODO_CONFIGURATION_FILE_NAME = "maketodo.txt";

	/** access method to return the singleton **/
	public static SystemConfiguration getTheSystemConfiguration() {
		return theSystemConfiguration;
	}

	/** Indicates if the configuration file was successfully loaded. **/
	public boolean didLoad = false;

	/** Constructor
	 *  Creates NUM_TYPES databaseSelections objects. The value of NUM_TYPES
	 *  is defined in MOVESDatabaseType class. Loads the System Configuration
	 *  file, and obtains an ID from the DistributedIDBroker class.
	**/
	public SystemConfiguration() {
		Logger.log(LogMessageCategory.INFO,"Loading system configuration...");
		if(!CompilationFlags.USE_NONROAD) {
			databaseSelections = new DatabaseSelection[1+MOVESDatabaseType.NUM_TYPES];
		}
		Logger.log(LogMessageCategory.INFO,"System configuration setting up the instance counter...");
		InstanceCounter.setup();
		Logger.log(LogMessageCategory.INFO,"System configuration clearing prior IDs...");
		clearPriorIDs();
		Logger.log(LogMessageCategory.INFO,"System configuration reading computer ID...");
		initComputerIDAndPath();

		for(int i = 0; i < databaseSelections.length; i++) {
			databaseSelections[i] = new DatabaseSelection();

			// Ensure defaults on these params
			databaseSelections[i].userName = DatabaseSelection.SERVER_USER_NAME;
			databaseSelections[i].password = DatabaseSelection.SERVER_PASSWORD;
			databaseSelections[i].databaseName = "";
		}

		databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].serverName = "localhost";
		databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].databaseName = "MOVESExecution";

		try {
			String alternateName = System.getProperty("MOVES_CONFIGURATION_FILE_NAME","");
			if(alternateName != null && alternateName.length() > 0) {
				MOVES_CONFIGURATION_FILE_NAME = alternateName;
				Logger.log(LogMessageCategory.INFO,"Using alternate master configuration file: " + alternateName);
			}
			Logger.log(LogMessageCategory.INFO,"Using master configuration file: " + MOVES_CONFIGURATION_FILE_NAME);
			Logger.log(LogMessageCategory.INFO,"System configuration loading configuration files...");
			loadConfigurationData();
			Logger.log(LogMessageCategory.INFO,"System configuration setting up temporary files...");
			TemporaryFileManager.setup(masterFolderPath);
		} catch(Exception e) {
			// SystemConfiguration currently is only concerned with IOExceptions, if any
			// database info is invalid, then the DatabaseConnectionManager will complain
			/**
			 * @explain An error occurred while reading the MOVESConfiguration.txt file.
			 * The data in this file is required.
			**/
			Logger.logError(e, "Unable to load System Configuration Data.");

			//System.out.println("Unable to load System Configuration Data: " + e.getMessage());
			//e.printStackTrace();
		}

		try {
			Logger.log(LogMessageCategory.INFO,"System configuration acquiring distributed master ID...");
			distributedMasterID = DistributedIDBroker.acquireID(
						DistributedIDBroker.PREFIX_MASTER_ID, computerID,
						sharedDistributedFolderPath);

			rememberCurrentID(DistributedIDBroker.getIDFilePath(
						DistributedIDBroker.PREFIX_MASTER_ID, distributedMasterID,
						computerID, sharedDistributedFolderPath));
		} catch (IOException exception) {
			/**
			 * @explain An ID file could not be created to identify this computer
			 * as a MOVES master.  Check the configuration file, ensuring that the
			 * shared work folder exists and you have rights to create, read, and
			 * delete files within it.
			**/
			Logger.logError(exception, "Unable to acquire the Master ID.");
			distributedMasterID = "0";
		}
		Logger.log(LogMessageCategory.INFO,"Done loading system configuration.");
	}

	/**
	 * Saves the system configuration to the text file.
	 * @throws IOException if can't open the file
	**/
	public void saveConfigurationData() throws IOException {
		File configurationPath = new File(MOVES_CONFIGURATION_FILE_NAME);

		final String eol = System.getProperty("line.separator");
		Writer configWriter = new BufferedWriter(
				new OutputStreamWriter(new FileOutputStream(configurationPath)));

		configWriter.write("defaultServerName" + " = " +
				databaseSelections[MOVESDatabaseType.DEFAULT.getIndex()].serverName + eol);
		configWriter.write("defaultDatabaseName" + " = " +
				databaseSelections[MOVESDatabaseType.DEFAULT.getIndex()].databaseName + eol);
		//configWriter.write("defaultUserName" + " = " +
		//		databaseSelections[MOVESDatabaseType.DEFAULT.getIndex()].userName + eol);
		//configWriter.write("defaultPassword" + " = " +
		//		databaseSelections[MOVESDatabaseType.DEFAULT.getIndex()].password + eol);

		/*
		configWriter.write("nrdefaultServerName" + " = " +
				databaseSelections[MOVESDatabaseType.NRDEFAULT.getIndex()].serverName + eol);
		configWriter.write("nrdefaultDatabaseName" + " = " +
				databaseSelections[MOVESDatabaseType.NRDEFAULT.getIndex()].databaseName + eol);
		//configWriter.write("nrdefaultUserName" + " = " +
		//		databaseSelections[MOVESDatabaseType.NRDEFAULT.getIndex()].userName + eol);
		//configWriter.write("nrdefaultPassword" + " = " +
		//		databaseSelections[MOVESDatabaseType.NRDEFAULT.getIndex()].password + eol);
		*/

		configWriter.write("executionServerName" + " = " +
				databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].serverName + eol);
		configWriter.write("executionDatabaseName" + " = " +
				databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].databaseName + eol);
		//configWriter.write("executionUserName" + " = " +
		//		databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].userName + eol);
		//configWriter.write("executionPassword" + " = " +
		//		databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].password + eol);

		configWriter.write("outputServerName" + " = " +
				databaseSelections[MOVESDatabaseType.OUTPUT.getIndex()].serverName + eol);
		configWriter.write("outputDatabaseName" + " = " +
				databaseSelections[MOVESDatabaseType.OUTPUT.getIndex()].databaseName + eol);
		//configWriter.write("outputUserName" + " = " +
		//		databaseSelections[MOVESDatabaseType.OUTPUT.getIndex()].userName + eol);
		//configWriter.write("outputPassword" + " = " +
		//		databaseSelections[MOVESDatabaseType.OUTPUT.getIndex()].password + eol);

		if(nonroadExePath != null) {
			configWriter.write("nonroadExePath" + " = " + nonroadExePath.getPath() + eol);
		}

		configWriter.write("sharedDistributedFolderPath" + " = " +
				sharedDistributedFolderPath.getPath() + eol);
				
		/*
		configWriter.write("GREETWTPApplication" + " = " +
				GREETWTPApplication.getPath() + eol);
		configWriter.write("GREETManufactureApplication" + " = " +
				GREETManufactureApplication.getPath() + eol);
		configWriter.write("GREETDirectory" + " = " +
				GREETDirectory.getPath() + eol);
		*/
		configWriter.write("computerIDPath" + " = " +
				computerIDPath + eol);
		configWriter.write("masterFolderPath" + " = " +
				masterFolderPath.getPath() + eol);
		configWriter.write("saveTODOPath" + " = " +
				(saveTODOPath == null? "" : saveTODOPath.getPath()) + eol);

		configWriter.write("mysqlUserName" + " = " + DatabaseSelection.userProvidedUserName + eol);
		configWriter.write("mysqlPassword" + " = "
				+ PasswordChecker.encode(DatabaseSelection.userProvidedUserName,DatabaseSelection.userProvidedPassword) + eol);

		if(generatorExePath != null) {
			configWriter.write("generatorExePath" + " = " + generatorExePath.getPath() + eol);
		}

		configWriter.close();
		
		readAndResetWorkerConfigurationFile(MOVES_WORKER_CONFIGURATION_FILE_NAME);
		readAndResetWorkerConfigurationFile(MANY_WORKERS_CONFIGURATION_FILE_NAME);
		readAndResetWorkerConfigurationFile(MAKE_TODO_CONFIGURATION_FILE_NAME);

		if(computerIDPath != null && computerIDPath.length() > 0) {
			configWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(computerIDPath)));
			configWriter.write("ComputerID" + " = " +
					computerID + eol);
			configWriter.close();
		}
	}
	
	/**
	 * Change the shared work directory in another configuration file, if it exists.
	 * Also changes MySQL login credentials so all uses of MOVES have the same security settings.
	 * @param fileName name of the other file
	**/
	void readAndResetWorkerConfigurationFile(String fileName)
			throws IOException {	
		File configFile = new File(fileName);
		try {	
			boolean fileExists = FileUtilities.fileExists(configFile);
			if(!fileExists) {
				// Return without complaining if the file does not exist. It may not be used.
				return;
			}
			ArrayList<String> lines = FileUtilities.readLines(configFile);
			if(lines == null || lines.size() <= 0) {
				// Return without complaining that the file could not be read.
				// If the user actually uses the file, they will get an error message then.
				return;
			}
			//prepare for rewriting the worker configuration file
			File configurationPath = new File(fileName);

			final String eol = System.getProperty("line.separator");
			Writer configWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(configurationPath)));
			
			//reading the existing worker configuration file
			String iterLine;
			for(Iterator<String> li=lines.iterator();li.hasNext();) {
				iterLine = (String)li.next();
				iterLine = iterLine.trim();

				if((iterLine.length() == 0) || iterLine.startsWith("//")) {
					configWriter.write(iterLine + eol);
					continue;
				}

				StringTokenizer iterTokenizer = new StringTokenizer(iterLine, "=");

				if(iterTokenizer.countTokens() > 2) {
					configWriter.write(iterLine + eol);
					continue;
				}
				String name = "";
				String value = "";

				if(iterTokenizer.countTokens() == 1) { //blank entry
					name = iterTokenizer.nextToken().trim();
				} else {
					name = iterTokenizer.nextToken().trim();
					value = iterTokenizer.nextToken().trim();
				}
				
				//reset the value of sharedDistributedFolderPath and MySQL login credentials
				if(name.equalsIgnoreCase("sharedDistributedFolderPath")) {
					value = sharedDistributedFolderPath.getPath();
				} else if(name.equalsIgnoreCase("mysqlUserName")) {
					value = DatabaseSelection.userProvidedUserName;
				} else if(name.equalsIgnoreCase("mysqlPassword")) {
					value = PasswordChecker.encode(DatabaseSelection.userProvidedUserName,DatabaseSelection.userProvidedPassword);
				}

				//write to the file
				configWriter.write(name + " = " + value + eol);		
			}
			configWriter.close();
		} catch(FileNotFoundException e) {
			throw new IOException("File not found : " + fileName);
		}
	}
	
	/**
	 * Read a configuration file, recursing into other configuration files as needed.
	 * @param fileNamesAlreadyRead set of names already read, used to prevent infinite recursion.
	 * @param fileName full name and path of the file to be read.
	 * @throws IOException If a file I/O error occured.
	**/
	void readConfigurationFile(TreeSetIgnoreCase fileNamesAlreadyRead, String fileName)
			throws IOException {
		File configFile = new File(fileName);
		String mysqlUserName = null;
		String mysqlPassword = null;
		try {
			String fullPath = configFile.getCanonicalPath();
			if(fileNamesAlreadyRead.contains(fullPath)) {
				return;
			}
			fileNamesAlreadyRead.add(fullPath);

			ArrayList<String> lines = null;
			boolean fileExists = FileUtilities.fileExists(configFile);
			if(fileExists) {
				lines = FileUtilities.readLines(configFile);
				if(lines == null || lines.size() <= 0) {
					fileExists = false;
				}
			}
			if(fileExists) {
				String iterLine;
				for(Iterator<String> li=lines.iterator();li.hasNext();) {
					iterLine = (String)li.next();
					iterLine = iterLine.trim();

					if((iterLine.length() == 0) || iterLine.startsWith("//")) {
						continue;
					}

					StringTokenizer iterTokenizer = new StringTokenizer(iterLine, "=");

					if(iterTokenizer.countTokens() > 2) {
						/**
						 * @explain The MOVESConfiguration.txt file contains an entry
						 * that is not recognized.  Verify that all entries in the file
						 * are correct.
						**/
						Logger.log(LogMessageCategory.WARNING, "Invalid configuration entry: " + iterLine);
						continue;
					}
					String name = "";
					String value = "";

					if(iterTokenizer.countTokens() == 1) { //blank entry
						name = iterTokenizer.nextToken().trim();
					} else {
						name = iterTokenizer.nextToken().trim();
						value = iterTokenizer.nextToken().trim();
					}

					if(name.compareToIgnoreCase("defaultServerName") == 0) {
						databaseSelections[MOVESDatabaseType.DEFAULT.getIndex()].serverName =
							new String(value);
					} else if(name.compareToIgnoreCase("defaultDatabaseName") == 0) {
						databaseSelections[MOVESDatabaseType.DEFAULT.getIndex()].databaseName =
							new String(value);
					} else if(name.compareToIgnoreCase("defaultUserName") == 0) {
						//databaseSelections[MOVESDatabaseType.DEFAULT.getIndex()].userName =
						//	new String(value);
					} else if(name.compareToIgnoreCase("defaultPassword") == 0) {
						//databaseSelections[MOVESDatabaseType.DEFAULT.getIndex()].password =
						//	new String(value);
					/*
					} else if(name.compareToIgnoreCase("nrdefaultServerName") == 0) {
						databaseSelections[MOVESDatabaseType.NRDEFAULT.getIndex()].serverName =
							new String(value);
					} else if(name.compareToIgnoreCase("nrdefaultDatabaseName") == 0) {
						databaseSelections[MOVESDatabaseType.NRDEFAULT.getIndex()].databaseName =
							new String(value);
					} else if(name.compareToIgnoreCase("nrdefaultUserName") == 0) {
						//databaseSelections[MOVESDatabaseType.NRDEFAULT.getIndex()].userName =
						//	new String(value);
					} else if(name.compareToIgnoreCase("nrdefaultPassword") == 0) {
						//databaseSelections[MOVESDatabaseType.NRDEFAULT.getIndex()].password =
						//	new String(value);
					*/
					} else if(name.compareToIgnoreCase("outputServerName") == 0) {
						databaseSelections[MOVESDatabaseType.OUTPUT.getIndex()].serverName =
								new String(value);
					} else if(name.compareToIgnoreCase("outputDatabaseName") == 0) {
						databaseSelections[MOVESDatabaseType.OUTPUT.getIndex()].databaseName =
								new String(value);
					} else if(name.compareToIgnoreCase("outputUserName") == 0) {
						//databaseSelections[MOVESDatabaseType.OUTPUT.getIndex()].userName =
						//	new String(value);
					} else if(name.compareToIgnoreCase("outputPassword") == 0) {
						//databaseSelections[MOVESDatabaseType.OUTPUT.getIndex()].password =
						//	new String(value);
					} else if(name.compareToIgnoreCase("executionServerName") == 0) {
						databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].serverName =
								new String(value);
					} else if(name.compareToIgnoreCase("executionDatabaseName") == 0) {
						databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].databaseName =
								new String(value);
					} else if(name.compareToIgnoreCase("executionUserName") == 0) {
						//databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].userName =
						//	new String(value);
					} else if(name.compareToIgnoreCase("executionPassword") == 0) {
						//databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].password =
						//	new String(value);
					} else if(name.compareToIgnoreCase("sharedDistributedFolderPath") == 0) {
						if(value != null && value.length() > 0) {
							sharedDistributedFolderPath = new File(value);
						}
					} else if (name.compareToIgnoreCase("nonroadExePath") == 0) {
						if (value != null && value.length() > 0) {
							nonroadExePath = new File(value);
						}
					} else if(name.compareToIgnoreCase("masterFolderPath") == 0) {
						if(value != null && value.length() > 0) {
							masterFolderPath = new File(value);
						}
					} else if(name.compareToIgnoreCase("GREETWTPApplication") == 0) {
			 			// Nothing to do here, GREET is no longer used, just ignore the configuration entry
			 			// GREETWTPApplication = new File(value);
					} else if(name.compareToIgnoreCase("GREETManufactureApplication") == 0) {
			 			// Nothing to do here, GREET is no longer used, just ignore the configuration entry
						// GREETManufactureApplication = new File(value);
					} else if(name.compareToIgnoreCase("GREETDirectory") == 0) {
			 			// Nothing to do here, GREET is no longer used, just ignore the configuration entry
						// GREETDirectory = new File(value);
					} else if(name.compareToIgnoreCase("ComputerID") == 0) {
						computerID = value;
						if(computerID.length() == 0) {
							computerID = "NoComputerID";
						} else if(computerID.length() > 20) {
							computerID = computerID.substring(0,20).trim();
							if(computerID.length() == 0) {
								computerID = "NoComputerID";
							}
						}
					} else if(name.compareToIgnoreCase("computerIDPath") == 0) {
						computerIDPath = value;
						if(value != null && value.length() > 0 && new File(value).exists()){
							readConfigurationFile(fileNamesAlreadyRead,value);
						}
					} else if(name.compareToIgnoreCase("saveTODOPath") == 0) {
						if(value != null && value.length() > 0) {
							saveTODOPath = new File(value);
						}
					} else if(name.compareToIgnoreCase("mysqlUserName") == 0) {
						if(value != null) {
							mysqlUserName = value.trim();
						}
					} else if(name.compareToIgnoreCase("mysqlPassword") == 0) {
						if(value != null && value.length() > 0) {
							mysqlPassword = value.trim();
						}
					} else if (name.compareToIgnoreCase("generatorExePath") == 0) {
						if (value != null && value.length() > 0) {
							generatorExePath = new File(value);
						}
					} else {
						/**
						 * @explain The MOVESConfiguration.txt file contains an entry
						 * that is not recognized.  Verify that all entries in the file
						 * are correct.
						**/
						Logger.log(LogMessageCategory.INFO, "Unknown configuration name: " + iterLine);
					}
				}
			} else {
				/**
				 * @explain The MOVESConfiguration.txt file is required but couldn't be located.
				**/
				Logger.log(LogMessageCategory.WARNING, "Couldn't find configuration file: " + fileName);
			}
		} catch(FileNotFoundException e) {
			throw new IOException("File not found : " + fileName);
		} finally {
			if(mysqlUserName != null && mysqlPassword != null) {
				String realPassword = PasswordChecker.decode(mysqlUserName,mysqlPassword);
				if(realPassword == null) {
					Logger.log(LogMessageCategory.ERROR,"Corrupt MySQL username/password provided.");
				} else {
					DatabaseSelection.userProvidedUserName = mysqlUserName;
					DatabaseSelection.userProvidedPassword = realPassword;
				}
			}
		}
	}

	/**
	 * Loads the system configuration from the text file.
	 * @throws IOException if can't open the file
	**/
	public void loadConfigurationData() throws IOException {
		File configurationPath = new File(MOVES_CONFIGURATION_FILE_NAME);

		boolean fileExists = FileUtilities.fileExists(configurationPath);
		if(fileExists && !configurationPath.canWrite()) {
			/**
			 * @explain The MOVESConfiguration.txt file is required and cannot be Read-Only.
			 * Close MOVES, access the properties of the file and remove the Read-Only flag,
			 * then restart MOVES.
			**/
			Logger.log(LogMessageCategory.ERROR, "Configuration file error:  " +
					MOVES_CONFIGURATION_FILE_NAME + " is read only.");
			return;
		}
		if(fileExists) {
			TreeSetIgnoreCase fileNames = new TreeSetIgnoreCase();
			readConfigurationFile(fileNames,MOVES_CONFIGURATION_FILE_NAME);
			if(masterFolderPath != null && !masterFolderPath.isDirectory()) {
				throw new FileNotFoundException("masterFolderPath doesn't exist");
			}
			if(masterFolderPath == null) {
				masterFolderPath = new File(".");
			}
		} else {
			/**
			 * @explain The MOVESConfiguration.txt file is required but couldn't be located.
			**/
			Logger.log(LogMessageCategory.WARNING, "Couldn't find configuration file. " +
				"Using defaults and generating default configuration file.");

			writeDefaultConfigurationFile();
		}

		String executionDatabaseName = databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].databaseName;
		if(executionDatabaseName == null || executionDatabaseName.length() <= 0
				|| executionDatabaseName.equalsIgnoreCase("*")) {
			// Create a unique but repeatable execution database name
			executionDatabaseName = InstanceCounter.getDB("MOVESExecution");
			databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].databaseName = executionDatabaseName;
		}
		didLoad = true;
	}

	/**
	 * Creates a default configuration text file.
	 * @throws IOException if can't open the file
	**/
	public void writeDefaultConfigurationFile() throws IOException {
		sharedDistributedFolderPath = new File("SharedWork");
		databaseSelections[MOVESDatabaseType.DEFAULT.getIndex()].serverName = "localhost";
		databaseSelections[MOVESDatabaseType.DEFAULT.getIndex()].databaseName = "MOVESDefault";
		//databaseSelections[MOVESDatabaseType.NRDEFAULT.getIndex()].serverName = "localhost";
		//databaseSelections[MOVESDatabaseType.NRDEFAULT.getIndex()].databaseName = "MOVESNRDefault";
		databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].serverName = "localhost";
		databaseSelections[MOVESDatabaseType.EXECUTION.getIndex()].databaseName = "MOVESExecution";
		databaseSelections[MOVESDatabaseType.OUTPUT.getIndex()].serverName = "localhost";
		databaseSelections[MOVESDatabaseType.OUTPUT.getIndex()].databaseName = "MOVESOutput";
		masterFolderPath = new File("testing.txt").getParentFile();
		saveTODOPath = null;

		// MOVES no longer uses GREET but must continue to instantiate these objects so older code won't get null references
		GREETDirectory = new File("GREET");
		GREETWTPApplication = new File("GREETWTPStub.exe");
		GREETManufactureApplication = new File("GREETMfgStub.exe");

		nonroadExePath = new File("NONROAD.EXE");

		initComputerIDAndPath();
		saveConfigurationData();
	}

	/**
	 * Initializes the computer ID and path to default values based on system settings.
	**/
	void initComputerIDAndPath() {
		try {
			computerID = InetAddress.getLocalHost().getHostName();
		} catch (UnknownHostException e) {
			computerID = "NoComputerID";
		}
		computerIDPath = "";
	}

	/**
	 * Note the current ID file in case it needs to be removed later.
	 * @param idFile name of the current ID file
	**/
	static void rememberCurrentID(File idFile) {
		try {
			File priorIDFile = new File(priorIDFileName);
			String fullPath = idFile.getCanonicalPath();
			FileUtilities.appendLine(priorIDFile,fullPath);
		} catch(Exception e) {
			/**
			 * @explain An attempt to record the master's ID has failed.  This will
			 * prevent automatic removal of the master ID file if an unexpected
			 * shutdown occurs.
			**/
			Logger.logError(e,"Unable to store current master ID");
		}
	}

	/** Remove the prior master's ID file, if present. **/
	public static void clearPriorIDs() {
		try {
			File priorIDFile = new File(priorIDFileName);
			if(priorIDFile.exists()) {
				ArrayList lines = FileUtilities.readLines(priorIDFile);
				if(lines != null) {
					for(int i=0;i<lines.size();i++) {
						String fileName = (String)lines.get(i);
						fileName = StringUtilities.safeGetString(fileName).trim();
						if(fileName.length() > 0) {
							File f = new File(fileName);
							FileUtilities.deleteFileWithRetry(f);
						}
					}
				}
				FileUtilities.deleteFileWithRetry(priorIDFile);
			}
		} catch(Exception e) {
			/**
			 * @explain An error occurred while removing ID files from prior runs.
			 * Check the shared work folder for extraneous ID files.
			**/
			Logger.logError(e,"Unable to clear prior master ID files");
		}
	}
}
