/**************************************************************************************************
 * @(#)MeteorologyGenerator.java
 *
 *
 *
 *************************************************************************************************/
package gov.epa.otaq.moves.master.implementation.general;

import gov.epa.otaq.moves.master.framework.*;

import gov.epa.otaq.moves.common.*;
import java.util.Iterator;
import java.sql.SQLException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

/**
 * This builds the meteorology database tables for the ELDB.
 * ELDB is the Execution Location Database explained in TotalActivityGenerator
 *
 * @author		Wes Faler
 * @author		Ed Glover - EPA, minor revs by Mitch C.
 * @author		Gwo Shyu - EPA, minor revs for Task 216
 * @author      William Aikman - EPA, changed specific Humidity to .000000011702
 * @version		2013-11-13
**/
public class MeteorologyGenerator extends Generator {
	/**
	 * @algorithm
	 * @owner Meteorology Generator
	 * @generator
	**/

	/** Flag for whether the data tables have been cleared/setup **/
	boolean isFirst = true;
	/** Database connection used by all functions.  Setup by executeLoop and cleanDataLoop. **/
	Connection db;
	/** String sql contains active SQL update statement code **/
	String sql;
	/** milliseconds spent during one time operations **/
	long setupTime = 0;
	/** milliseconds spent during non-one-time operations **/
	long totalTime = 0;

	/** Standard Constructor **/
	public MeteorologyGenerator() {
	}

	/**
	 * Requests that this object subscribe to the given loop at desired looping points.
	 * Objects can assume that all necessary MasterLoopable objects have been instantiated.
	 * @param targetLoop The loop to subscribe to.
	**/
	public void subscribeToMe(MasterLoop targetLoop) {
		String[] processNames = {
			"Running Exhaust",
			"Extended Idle Exhaust",
			"Start Exhaust",
			"Auxiliary Power Exhaust",
			"Tirewear",
			"Brakewear"
		};
		for(int i=0;i<processNames.length;i++) {
			EmissionProcess p = EmissionProcess.findByName(processNames[i]);
			if(p != null) {
				targetLoop.subscribe(this, p, MasterLoopGranularity.PROCESS, MasterLoopPriority.GENERATOR);
			}
		}
	}

	/**
	 * Called during each relevant iteration of the MasterLoop.
	 * @param context The current context of the loop.
	**/
	public void executeLoop(MasterLoopContext context) {
		try {
			db = DatabaseConnectionManager.checkOutConnection(MOVESDatabaseType.EXECUTION);
			long start;
			if (isFirst) {
				start = System.currentTimeMillis();
				doHeatIndex(db);
				isFirst=false;
				setupTime += System.currentTimeMillis() - start;
			}
		} catch (Exception e) {
			Logger.logError(e,"Unable to calculate HeatIndex.");
		} finally {
			DatabaseConnectionManager.checkInConnection(MOVESDatabaseType.EXECUTION, db);
			db = null;
		}

		Logger.log(LogMessageCategory.INFO,"MetGen setupTime=" + setupTime + " bundleTime=" + totalTime);
	}

	/**
	 * This method in MeteorologyGenerator is empty.
	 * Typically it is used to clean
	 * up the data generated by this class during execution.
	 * It is invoked only after all the calculators are done with their executions.
	 * @param context The MasterLoopContext that applies to this execution.
	**/
	public void cleanDataLoop(MasterLoopContext context) {
	}

	/**
	 * This method in MeteorologyGenerator implements the standard heat index
	 * calculation used by the Weather Service.  The method fills the
	 * ZoneMonthHour table of Connection db in one loop.
	 * @param db The database connection
	**/
	void doHeatIndex(Connection db) {
		try {
			// Provide default barometric pressures for counties that don't list them.
			// These are taken as the average pressures of the counties in each altitude group.

			/**
			 * @step 010
			 * @algorithm Provide default barometric pressures for counties that don't list them.
			 * Counties with 'H' (High) altitude are assigned 24.69 inHg. Others are assigned 28.94.
			 * @condition Counties with barometricPressure of null or <= 0.
			 * @output County
			**/
			sql="update County set barometricPressure=("
					+ " case"
					+ "	when altitude='H' then 24.59"
					+ "	else 28.94"
					+ " end"
					+ " )"
					+ " where (barometricPressure is null) or barometricPressure<=0";
			SQLRunner.executeSQL(db,sql);

			/**
			 * @step 010
			 * @algorithm Provide default altitude setting for counties that do not provide an altitude.
			 * Counties with barometricPressure >= 25.8403 inHg are 'L' (Low) altitude, all others are 'H' (High).
			 * @condition Counties where altitude is null or altitude is neither H nor L.
			 * @output County
			**/
			sql = "update County set altitude=("
					+ " case when barometricPressure >= 25.8403 then 'L'"
					+ " else 'H'"
					+ " end"
					+ " )"
					+ " where barometricPressure is not null"
					+ " and (altitude is null or altitude not in ('H','L'))";
			SQLRunner.executeSQL(db,sql);

			/**
			 * @step 010
			 * @algorithm heatIndex = temperature.
			 * @condition temperature < 78F.
			 * @output ZoneMonthHour
			**/
			sql="UPDATE ZoneMonthHour SET heatIndex = temperature " +
					" WHERE temperature < 78 ";
			SQLRunner.executeSQL(db,sql);

			/**
			 * @step 010
			 * @algorithm heatIndex = least(
			 * -42.379 + 2.04901523*temperature + 10.14333127*relHumidity +
			 * - 0.22475541*temperature*relHumidity +
			 * -0.00683783*temperature*temperature +
			 * -0.05481717 * relHumidity * relHumidity +
			 * 0.00122874*temperature*temperature*relHumidity +
			 * 0.00085282*temperature*relHumidity*relHumidity +
			 * -0.00000199*temperature*temperature*relHumidity*relHumidity, 120).
			 * @condition temperature >= 78F.
			 * @output ZoneMonthHour
			**/
			sql="UPDATE ZoneMonthHour SET heatIndex = least(" +
					"-42.379 + 2.04901523*temperature + 10.14333127*relHumidity + " +
					"- 0.22475541*temperature*relHumidity + " +
					"-0.00683783*temperature*temperature + " +
					"-0.05481717 * relHumidity * relHumidity + " +
					"0.00122874*temperature*temperature*relHumidity + " +
					"0.00085282*temperature*relHumidity*relHumidity + " +
					"-0.00000199*temperature*temperature*relHumidity*relHumidity, 120) " +
					" WHERE temperature >= 78.0 ";
			SQLRunner.executeSQL(db,sql);

			sql = "DROP TABLE IF EXISTS TKT0";
			SQLRunner.executeSQL(db,sql);

			/**
			 * @step 010
			 * @algorithm TK = 0.56*(temperature-32)+273 AS TK.
			 * T0 = 374.27-0.56*(temperature-32).
			 * @input ZoneMonthHour
			 * @output TKT0
			**/
			sql = "CREATE TABLE TKT0 SELECT monthID, zoneID, hourID, relHumidity, " +
					"0.56*(temperature-32)+273 AS TK, 374.27-0.56*(temperature-32) AS T0 " +
					"FROM ZoneMonthHour";
			SQLRunner.executeSQL(db,sql);

			sql = "DROP TABLE IF EXISTS PV";
			SQLRunner.executeSQL(db,sql);

			/**
			 * @step 010
			 * @algorithm PB = barometricPressure.
			 * PV = (relHumidity/100)*6527.557*POW(10,(-T0/TK)*((3.2437+0.00588*T0+0.000000011702*POW(T0,3))/(1+0.00219*T0))).
			 * @input Zone
			 * @input County
			 * @input TKT0
			 * @output PV
			**/
			sql = "CREATE TABLE PV SELECT monthID, z.zoneID, hourID, barometricPressure AS PB, " +
					"(relHumidity/100)*6527.557*POW(10,(-T0/TK)*((3.2437+0.00588*T0+ " +
					"0.000000011702*POW(T0,3))/(1+0.00219*T0))) AS PV FROM TKT0 " +
					"INNER JOIN Zone z USING (zoneID) INNER JOIN County USING (countyID)";
			SQLRunner.executeSQL(db,sql);

			sql = "CREATE UNIQUE INDEX XPKPV ON PV"
					+ "(monthID ASC, zoneID ASC, hourID ASC)";
			SQLRunner.executeSQL(db,sql);

			sql = "DROP TABLE TKT0";
			SQLRunner.executeSQL(db,sql);

			/**
			 * @step 010
			 * @algorithm specificHumidity=4347.8*PV/(PB-PV).
			 * @input PV
			 * @output ZoneMonthHour
			**/
			sql = "UPDATE ZoneMonthHour, PV SET ZoneMonthHour.specificHumidity=4347.8*PV/(PB-PV) "+
					"WHERE ZoneMonthHour.monthID = PV.monthID AND ZoneMonthHour.zoneID = " +
					"PV.zoneID AND ZoneMonthHour.hourID = PV.hourID";
			SQLRunner.executeSQL(db,sql);

			sql = "DROP TABLE PV";
			SQLRunner.executeSQL(db,sql);
		} catch (SQLException e) {
			Logger.logSqlError(e, "SQL error in heat index.calculation", sql);
		}
	}
}
