/**************************************************************************************************
 * @(#)RefuelingLossCalculator.java
 *
 *
 *
 *************************************************************************************************/
package gov.epa.otaq.moves.master.implementation.ghg;

import gov.epa.otaq.moves.common.*;
import gov.epa.otaq.moves.master.framework.*;

import java.util.*;
import java.sql.*;

/**
 * Refueling Displacement Vapor Loss and Refueling Spillage Loss calculator.
 * Calculations are chained to the Total Energy Consumption on all road types for
 * Running Exhaust, Start Exhaust, and Extended Idle Exhaust.
 *
 * @author		Wesley Faler
 * @version		2011-12-12
**/
public class RefuelingLossCalculator extends EmissionCalculator {
	/**
	 * @algorithm
	 * @owner Refueling Loss Calculator
	 * @calculator
	**/

	/** Pollutants generated by the refueling processes **/
	static final int[] pollutantIDs = { 1 };
	/** Refueling processes **/
	static final int[] processIDs = { 18, 19 };
	/** Calculator script section names for the refueling processes **/
	static final String[] processSectionNames = {
		"RefuelingDisplacementVaporLoss",
		"RefuelingSpillageLoss"
	};
	/** Processes that supply data needed for refueling calculations **/
	static final int[] sourceProcessIDs = { 1, 2, 90, 91 };
	/** Script-variable names of the pollutant ID lists for each process **/
	static final String[] pollutantIDSetNames = {
		"##refuelingDisplacement.pollutantIDs##",
		"##refuelingSpillage.pollutantIDs##"
	};

	/** true if refuelingVaporProcess and refuelingSpillageProcess have been setup **/
	boolean didSetupProcesses = false;
	/** The Refueling Vapor Loss process, or null if it is not used by the run **/
	EmissionProcess refuelingVaporProcess = null;
	/** The Refueling Spillage Loss process, or null if it is not used by the run **/
	EmissionProcess refuelingSpillageProcess = null;

	/**
	 * Constructor, including registration of potential processes and pollutants
	 * handled by this calculator.  Such registration facilitates calculator
	 * chaining.
	**/
	public RefuelingLossCalculator() {
		for(int i=0;i<pollutantIDs.length;i++) {
			Pollutant pollutant = Pollutant.findByID(pollutantIDs[i]);
			if(pollutant != null) {
				for(int j=0;j<processIDs.length;j++) {
					EmissionProcess process = EmissionProcess.findByID(processIDs[j]);
					if(process != null) {
						EmissionCalculatorRegistration.register(pollutant,process,this);
					}
				}
			}
		}
	}

	/**
	 * MasterLoopable override that performs loop registration.
	 * @param targetLoop The loop to subscribe to.
	**/
	public void subscribeToMe(MasterLoop targetLoop) {
		// This is a chained calculator, so don't subscribe to the MasterLoop.  Instead,
		// find calculators that produce data this one needs.
		Pollutant totalEnergy = Pollutant.findByID(91);
		if(totalEnergy == null) {
			return;
		}
		LinkedList<EmissionCalculator> calculators = new LinkedList<EmissionCalculator>();
		for(int i=0;i<sourceProcessIDs.length;i++) {
			EmissionProcess process = EmissionProcess.findByID(sourceProcessIDs[i]);
			if(process != null) {
				LinkedList<EmissionCalculator> t =
						EmissionCalculatorRegistration.find(totalEnergy,process);
				EmissionCalculatorRegistration.merge(calculators,t);
			}
		}

		for(Iterator<EmissionCalculator> i=calculators.iterator();i.hasNext();) {
			EmissionCalculator c = (EmissionCalculator)i.next();
			if(c != this) {
				c.chainCalculator(this);
			}
		}
	}

	/**
	 * Builds SQL statements for a distributed worker to execute. This is called by
	 * EnergyConsumptionCalculator.executeLoop. Implementations of this method
	 * should contain uncertainty logic when UncertaintyParameters specifies that
	 * this mode is enabled.
	 * @param context The MasterLoopContext that applies to this execution.
	 * @return The resulting sql lists as an SQLForWorker object.
	**/
	public SQLForWorker doExecute(MasterLoopContext context) {
		if(!didSetupProcesses) {
			EmissionProcess p = EmissionProcess.findByID(18);
			if(p != null && ExecutionRunSpec.theExecutionRunSpec.doesHavePollutantAndProcess(
					null,p.processName)) {
				refuelingVaporProcess = p;
			}
			p = EmissionProcess.findByID(19);
			if(p != null && ExecutionRunSpec.theExecutionRunSpec.doesHavePollutantAndProcess(
					null,p.processName)) {
				refuelingSpillageProcess = p;
			}
		}

		String[] pollutantIDSets = new String[processIDs.length];
		for(int i=0;i<pollutantIDSets.length;i++) {
			pollutantIDSets[i] = "";
		}

		// Determine which pollutant(s) should be calculated
		boolean foundPollutant = false;

		for(int i=0;i<processIDs.length;i++) {
			EmissionProcess process = EmissionProcess.findByID(processIDs[i]);
			if(process == null) {
				continue;
			}
			for(int j=0;j<pollutantIDs.length;j++) {
				Pollutant pollutant = Pollutant.findByID(pollutantIDs[j]);
				if(pollutant == null) {
					continue;
				}
				if(ExecutionRunSpec.theExecutionRunSpec.doesHavePollutantAndProcess(
						pollutant.pollutantName,process.processName)) {
					foundPollutant = true;
					if(pollutantIDSets[i].length() > 0) {
						pollutantIDSets[i] += ",";
					}
					pollutantIDSets[i] += "" + pollutantIDs[j];
				}
			}
		}

		if(!foundPollutant) {
			return null;
		}

		SQLForWorker sqlForWorker = new SQLForWorker();

		TreeMapIgnoreCase replacements = new TreeMapIgnoreCase();
		TreeSetIgnoreCase enabledSectionNames = new TreeSetIgnoreCase();

		String processIDsText = "";

		for(int i=0;i<processIDs.length;i++) {
			if(pollutantIDSets[i].length() > 0) {
				if(processIDsText.length() > 0) {
					processIDsText += ",";
				}
				processIDsText += "" + processIDs[i];
				enabledSectionNames.add(processSectionNames[i]);
				replacements.put(pollutantIDSetNames[i],pollutantIDSets[i]);
			}
		}

		replacements.put("##refuelingProcessIDs##",processIDsText);

		boolean isOK = readAndHandleScriptedCalculations(context,replacements,
				"database/RefuelingLossCalculator.sql",enabledSectionNames,
				sqlForWorker);

		if(isOK) {
			context.addChainedProcess(refuelingVaporProcess);
			context.addChainedProcess(refuelingSpillageProcess);
			return sqlForWorker;
		} else {
			return null;
		}
	}
}
