//AQUATOX SOURCE CODE Copyright (c) 2005-2017 Eco Modeling and Warren Pinnacle Consulting, Inc.
//Code Use and Redistribution is Subject to Licensing, SEE AQUATOX_License.txt
// 
{ ************************************************************ }
{ *                                                          * }
{ *   NEW ESTUARY CODE,  AS SPECIFIED IN OCTOBER 2002 QAPP   * }
{ *                                                          * }
{ ************************************************************ }

Function TVolume.FreshwaterHead: Double;
Begin
  FreshWaterHead := ResidFlow / Location.Locale.SurfArea;
 {m on a given day}  {m3/d}                        {m2}
End;

Function TVolume.FracUpper: Double;
Var FWH: Double;
Begin
   FWH := FreshWaterHead;
              {m}
   If (TidalAmplitude(AllStates.TPresent) + FWH) = 0
     then FracUpper := 0.99 {avoid a crash}
     else FracUpper := 1.5 * (FWH / (TidalAmplitude(AllStates.TPresent) + FWH));
         {unitless}           {m}        {m}                              {m}

   If Result  > 1 then FracUpper := 0.99;
   If Result <= 0 then FracUpper := 0.01;
End;

Function    TVolume.SaltWaterInflow: Double;
Var PSal: TSalinity;
Begin
  PSal := GetStatePointer(Salinity, StV,WaterCol);

  If PSal.SalinityUpper=0 then Raise EAQUATOXError.Create('Upper Salinity must always be greater than zero for calculations in "salt balance" approach.');

  With PSal do begin
    if SalinityUpper > (0.99* SalinityLower)
      then SaltWaterInflow := 99*ResidFlow else
    SaltWaterInflow := ResidFlow / ((SalinityLower/SalinityUpper) - 1 );
    end;
End;

Function TVolume.UpperOutflow: Double;
Var PSal: TSalinity;
Begin
  PSal := GetStatePointer(Salinity, StV,WaterCol);
  With PSal do
    Begin
      If SalinityUpper > (0.99* SalinityLower)
        then UpperOutflow := 100*ResidFlow
        else UpperOutflow := ResidFlow / (1-(SalinityUpper/SalinityLower));
    End;
End;

Function TStateVariable.EstuaryEntrainment: Double;
Var HypState: Double;
    PV: TVolume;
Begin
  EstuaryEntrainment := 0;
  If Not AllStates.EstuarySegment then exit;

  With AllStates do
    Begin
      If VSeg=Hypolimnion then HypState := State
                          else HypState := HypoSegment.GetState(NState,SVType,Layer);

      PV := GetStatePointer(Volume,StV,WaterCol);                    
      EstuaryEntrainment := HypState *  PV.SaltWaterInflow / SegVol;
      {unit / L}         {unit/L hyp}          {m3}            {m3}

      If VSeg=Hypolimnion then Result := -Result;  {entrainment is leaving the hypolimnion}
    End;
End;

{----------------------------------------------------------------------------------------------------------}

Function TVolume.TidalAmplitude(TimeIndex:TDateTime): Double;
Const NUM_CONSTITUENTS = 8;
Var Height: Double;
    n  : integer;
    ThisYr,ThisMo,ThisDay: Word;
    amplitude,k : Array[1..NUM_CONSTITUENTS] of Double;
    THrs, TStart : Double;
    HeightMax, HeightMin: Double;

Type YearArray = Array [1970..2037] of Double;
     ParmArray = Array[1..NUM_CONSTITUENTS] of YearArray;
Const
  cspeed :Array[1..NUM_CONSTITUENTS] of Double =
                                 ({M2} 28.9841042,  {Speeds of each constituent in degrees per hour}
                                  {S2} 30.0000000,
                                  {N2} 28.4397295,
                                  {K1} 15.0410686,
                                  {O1} 13.9430356,
                                 {SSA}  0.0821373,
                                  {SA}  0.0410686,
                                  {p1} 14.9589314);

                              {node factor for each constituent for each year}
  nodefactor : ParmArray=({M2}(0.9665,0.9734,0.9833,0.9953,1.0078,1.0195,1.0291,1.0355,1.0378,1.0359,
                               1.0299,1.0205,1.0089,0.9964,0.9844,0.9742,0.9670,0.9635,0.9641,0.9687,
                               0.9769,0.9878,1.0001,1.0125,1.0236,1.0320,1.0369,1.0376,1.0340,1.0266,
                               1.0162,1.0041,0.9916,0.9802,0.9710,0.9651,0.9632,0.9654,0.9715,0.9809,
                               0.9925,1.0050,1.0170,1.0272,1.0344,1.0377,1.0367,1.0315,1.0229,1.0117,
                               0.9992,0.9870,0.9763,0.9683,0.9639,0.9636,0.9674,0.9748,0.9852,0.9973,
                               1.0098,1.0212,1.0304,1.0361,1.0378,1.0352,1.0286,1.0188),
                         {S2} (1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
                               1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1),
                         {N2} (0.9665,0.9734,0.9833,0.9953,1.0078,1.0195,1.0291,1.0355,1.0378,1.0359,
                               1.0299,1.0205,1.0089,0.9964,0.9844,0.9742,0.9670,0.9635,0.9641,0.9687,
                               0.9769,0.9878,1.0001,1.0125,1.0236,1.0320,1.0369,1.0376,1.0340,1.0266,
                               1.0162,1.0041,0.9916,0.9802,0.9710,0.9651,0.9632,0.9654,0.9715,0.9809,
                               0.9925,1.0050,1.0170,1.0272,1.0344,1.0377,1.0367,1.0315,1.0229,1.0117,
                               0.9992,0.9870,0.9763,0.9683,0.9639,0.9636,0.9674,0.9748,0.9852,0.9973,
                               1.0098,1.0212,1.0304,1.0361,1.0378,1.0352,1.0286,1.0188),
                         {K1} (1.1052,1.0884,1.0627,1.0294,0.9910,0.9514,0.9161,0.8912,0.8817,0.8897,
                               0.9133,0.9479,0.9873,1.0260,1.0600,1.0864,1.1040,1.1122,1.1108,1.0998,
                               1.0795,1.0507,1.0150,0.9755,0.9369,0.9050,0.8855,0.8827,0.8972,0.9257,
                               0.9629,1.0027,1.0399,1.0712,1.0943,1.1083,1.1128,1.1077,1.0930,1.0693,
                               1.0375,0.9999,0.9602,0.9234,0.8957,0.8824,0.8864,0.9068,0.9394,0.9782,
                               1.0176,1.0529,1.0812,1.1008,1.1112,1.1119,1.1031,1.0849,1.0578,1.0235,
                               0.9845,0.9453,0.9113,0.8886,0.8818,0.8925,0.9183,0.9541),
                         {O1} (1.1702,1.1428,1.1010,1.0470,0.9849,0.9207,0.8629,0.8216,0.8057,0.8190,
                               0.8582,0.9150,0.9789,1.0415,1.0965,1.1395,1.1683,1.1818,1.1794,1.1613,
                               1.1282,1.0814,1.0237,0.9598,0.8970,0.8444,0.8121,0.8074,0.8315,0.8786,
                               0.9394,1.0038,1.0640,1.1146,1.1524,1.1754,1.1828,1.1743,1.1503,1.1115,
                               1.0601,0.9994,0.9349,0.8748,0.8290,0.8068,0.8135,0.8475,0.9011,0.9643,
                               1.0279,1.0850,1.1309,1.1631,1.1801,1.1813,1.1668,1.1370,1.0930,1.0374,
                               0.9745,0.9107,0.8549,0.8171,0.8059,0.8237,0.8665,0.9250),
                        {SSA} (1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
                               1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1),
                         {SA} (1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
                               1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1),
                         {P1} (1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
                               1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1));

            { equilibrium argument for each constituent for each year in degrees for the meridian of Greenwich}
  equilarg : ParmArray = ({M2}(165.43,266.78,7.96,84.54,185.26,285.75,26.03,101.78,201.83,301.88,
                               42.01,117.89,218.36,319.06,60.00,136.77,238.11,339.56,81.04,158.08,
                               259.37,0.47,101.33,177.58,277.98,18.20,118.30,193.96,294.03,34.20,
                               134.53,210.71,311.50,52.53,153.76,230.77,332.24,73.70,175.08,251.92,
                               352.94,93.71,194.25,270.19,10.35,110.42,210.46,286.18,26.41,126.83,
                               227.47,303.98,45.09,146.39,247.82,324.92,66.36,167.68,268.83,345.37,
                               86.05,186.50,286.76,2.49,102.54,202.60,302.74,18.65),
                          {S2}(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
                               0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
                          {N2}(270.11,282.73,295.19,269.98,281.98,293.74,305.31,279.27,290.60,301.93,
                               313.33,287.43,299.17,311.15,323.36,298.35,310.97,323.69,336.45,311.70,
                               324.27,336.65,348.79,323.25,334.93,346.43,357.80,331.67,343.02,354.47,
                               6.08,340.47,352.54,4.84,17.36,352.58,5.32,18.07,30.72,5.78,
                               18.07,30.12,41.94,16.09,27.53,38.87,50.20,24.13,35.63,47.33,
                               59.25,33.97,46.36,58.94,71.64,46.95,59.67,72.28,84.70,59.45,
                               71.41,83.14,94.67,68.62,79.95,91.28,102.70,76.83),
                          {K1}(13.44,15.49,17.10,19.05,19.16,18.22,16.17,14.11,10.46,6.78,
                               3.66,2.49,1.46,1.47,2.36,4.91,6.93,9.21,11.55,14.76,
                               16.66,18.05,18.71,19.41,18.04,15.58,12.24,9.48,5.96,3.18,
                               1.46,1.83,2.21,3.38,5.15,8.28,10.60,12.92,15.04,17.76,
                               18.89,19.20,18.51,17.70,14.86,11.30,7.56,5.26,2.87,1.58,
                               1.37,3.07,4.49,6.43,8.66,12.00,14.26,16.25,17.78,19.60,
                               19.55,18.43,16.21,14.02,10.32,6.69,3.71,2.72),
                          {O1}(150.70,249.04,347.76,61.72,161.97,263.47,6.50,85.70,191.28,296.92,
                               41.60,119.41,221.05,321.40,60.80,134.19,232.56,330.72,68.82,141.65,
                               240.11,339.03,78.68,154.01,256.08,359.72,104.77,185.15,290.51,34.64,
                               137.21,212.94,312.88,52.01,150.59,223.48,321.60,59.73,158.00,231.24,
                               330.41,70.43,171.62,248.92,353.16,98.58,204.31,283.91,27.45,129.41,
                               230.02,304.24,43.13,141.57,239.75,312.48,50.65,149.04,247.83,321.92,
                               62.35,164.09,267.37,346.78,92.45,197.99,302.46,20.01),
                        {SSA} (200.47,199.99,199.52,201.01,200.53,200.05,199.58,201.07,200.59,200.12,
                               199.64,201.13,200.65,200.18,199.70,201.19,200.72,200.24,199.76,201.26,
                               200.78,200.30,199.82,201.32,200.84,200.36,199.88,201.38,200.90,200.42,
                               199.95,201.44,200.96,200.48,200.01,201.50,201.02,200.55,200.07,201.56,
                               201.09,200.61,200.13,201.62,201.15,200.67,200.19,201.69,201.21,200.73,
                               200.25,201.75,201.27,200.79,200.32,201.81,201.33,200.85,200.38,201.87,
                               201.39,200.92,200.44,201.93,201.45,200.98,200.50,201.99),
                        {SA}  (280.24,280.00,279.76,280.50,280.27,280.03,279.79,280.54,280.30,280.06,
                               279.82,280.57,280.33,280.09,279.85,280.60,280.36,280.12,279.88,280.63,
                               280.39,280.15,279.91,280.66,280.42,280.18,279.94,280.69,280.45,280.21,
                               279.97,280.72,280.48,280.24,280.00,280.75,280.51,280.27,280.03,280.78,
                               280.54,280.30,280.07,280.81,280.57,280.33,280.10,280.84,280.60,280.37,
                               280.13,280.87,280.64,280.40,280.16,280.90,280.67,280.43,280.19,280.94,
                               280.70,280.46,280.22,280.97,280.73,280.49,280.25,281.00),
                         {P1} (349.76,350.00,350.24,349.50,349.73,349.97,350.21,349.46,349.70,349.94,
                               350.18,349.43,349.67,349.91,350.15,349.40,349.64,349.88,350.12,349.37,
                               349.61,349.85,350.09,349.34,349.58,349.82,350.06,349.31,349.55,349.79,
                               350.03,349.28,349.52,349.76,350.00,349.25,349.49,349.73,349.97,349.22,
                               349.46,349.70,349.93,349.19,349.43,349.67,349.90,349.16,349.40,349.63,
                               349.87,349.13,349.36,349.60,349.84,349.10,349.33,349.57,349.81,349.06,
                               349.30,349.54,349.78,349.03,349.27,349.51,349.75,349.00));

Begin
  {optimization}
  If Trunc(TimeIndex) = LastTimeTA then {calculate only once per day}
    Begin
      TidalAmplitude := LastCalcTA;
      Exit;
    End;

  DecodeDate(TimeIndex,ThisYr,ThisMo,ThisDay);

  If (ThisYr<1970) or (ThisYr>2037) then
     Raise EAQUATOXError.create('TidalAmplitude only works in a date range of 1970-2037');

  TStart := (JulianDate(TimeIndex)-1) * 24;
  Thrs := TStart;

  With Location.Locale do
    Begin
  {m2} amplitude[1] := amplitude1; k[1]:= k1;
  {s2} amplitude[2] := amplitude2; k[2]:= k2;
  {n2} amplitude[3] := amplitude3; k[3]:= k3;
  {k1} amplitude[4] := amplitude4; k[4]:= k4;
  {o1} amplitude[5] := amplitude5; k[5]:= k5;
 {SSA} amplitude[6] := amplitude6; k[6]:= k6;
  {SA} amplitude[7] := amplitude7; k[7]:= k7;
  {P1} amplitude[8] := amplitude8; k[8]:= k8;
   End;

  HeightMax := -9e99;
  HeightMin :=  9e99;
  Repeat
  {Sum each factor}
    Height := 0;  {Datum irrelevant for amplitude calculation}
    For n := 1 to NUM_CONSTITUENTS do
        Height := Height + (amplitude[n] * nodefactor[n,ThisYr]* Cos(DegToRad((cspeed[n]*Thrs)+equilarg[n,ThisYr]-k[n])));
    Thrs := Thrs + 0.1;

    If Height < HeightMin then HeightMin := Height;
    If Height > HeightMax then HeightMax := Height;

  Until Thrs > TStart + 24;

  TidalAmplitude := (HeightMax-HeightMin)/ 2;
      {m}

  LastTimeTA := TimeIndex;
  LastCalcTA := Result;
End;
