//AQUATOX SOURCE CODE Copyright (c) 2005-2017 Eco Modeling and Warren Pinnacle Consulting, Inc.
//Code Use and Redistribution is Subject to Licensing, SEE AQUATOX_License.txt
//
unit SV_IO;

{ PRIMARY INTERFACE BETWEEN DATABASES AND STATE VARIABLES:

Comments:  1. Don't change order of elements within
              DBase Files... Correct Loading depends on the order.
           2. Working with Dbase Files requires the name *.DB,
              so some file-name manipulation is required...
           3. These functions return false if the transfer is not
              successful, but do not raise an error unless a file transfer
              error occurs, or the appropriate entry is not found.  JonC }

{ NOTE:  THE PROCESS TO CHANGE DATABASES / DATA ASSOCIATED WITH
         STATE VARIABLES REQUIRES 8 STEPS:

o     1. make a note of the size of the record data structure in GLOBAL.PAS
o     2. changing the record data structure in GLOBAL.PAS
o     3. altering the databases accordingly (Database AND Program directory)
o     4. changing the sv_io interface between database and data structure _ TEXT OUTPUT (9)
o     5. updating the store/load procedures for new studies, may require
         new version num in global or update of all relevant studies.
o     6. creating the portion of the entry screen with the correct units, tab order,
         and entry boxes, setting DATAFIELD within DBEDIT components
o     7. getting the appropriate default values for the databases
o     8. modifying process code to use the new data, testing
o     9. modifying text output
o     10. Add to uncertainty, sensitivity parameters (DISTRIBS.PAS)
x     11. Tech. Doc. modifications  (Add Process Eqns to Tech doc or placeholder
           Add parameters to Appendix B in Tech Doc or placeholder }

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, FireDAC.Comp.Client, DB, Global, TCollect,
  FireDAC.Stan.Intf, FireDAC.Stan.Option, FireDAC.Stan.Param,
   FireDAC.Stan.Error, FireDAC.DatS, FireDAC.Phys.Intf, FireDAC.DApt.Intf,
  FireDAC.Stan.Async, FireDAC.DApt, FireDAC.UI.Intf, FireDAC.Stan.Def,
  FireDAC.Stan.Pool, FireDAC.Phys, FireDAC.Phys.SQLite, FireDAC.Phys.SQLiteDef,
  FireDAC.Stan.ExprFuncs, FireDAC.VCLUI.Wait, FireDAC.Comp.DataSet;


type
  EAquatoxError=Exception;

{ ***** Interaction of Records and Databases ***** }
Procedure EmptyTable(Var TT: TFDTable);

Function DBase_To_ChemRecord2  (DBName,TName,LookupN: AnsiString; Num: Integer; Var CR: Chemicalrecord):Boolean;
Function ChemRecord_To_Dbase2  (DBName,TName,LookupN: AnsiString; CR: Chemicalrecord; Prompt:Boolean):Boolean;
Procedure ChemRec_To_Text     (Var LF: TextFile; CR: Chemicalrecord);

Function DBase_To_AnimalRecord2(DBName,TName,LookupN: AnsiString; Num: Integer; Var AR: ZooRecord):Boolean;
Function AnimalRecord_To_Dbase2(DBName,TName,LookupN: AnsiString; AR: ZooRecord; Prompt:Boolean):Boolean;
Procedure AnimRec_To_Text     (Var LF: TextFile; AR:ZooRecord);

Function DBase_To_SiteRecord2  (DBName,TName,LookupN: AnsiString; Var SR: SiteRecord):Boolean;
Function SiteRecord_To_Dbase2  (DBName,TName,LookupN: AnsiString; SR: SiteRecord; Prompt:Boolean):Boolean;
Procedure SiteRec_To_Text     (Var LF: TextFile; SR: SiteRecord);

Function DBase_To_ReminRecord2 (DBName,TName,LookupN: AnsiString; Var RR: ReminRecord):Boolean;
Function ReminRecord_To_Dbase2 (DBName,TName,LookupN: AnsiString; RR: ReminRecord; Prompt:Boolean):Boolean;
Procedure ReminRec_To_Text    (Var LF: TextFile; RR: ReminRecord);

Function DBase_To_PlantRecord2 (DBName,TName,LookupN: AnsiString; Num: Integer; Var PR: PlantRecord):Boolean;
Function PlantRecord_To_Dbase2 (DBName,TName,LookupN: AnsiString; PR: PlantRecord; Prompt:Boolean):Boolean;
Procedure PlantRec_To_Text    (Var LF: TextFile; PR: PlantRecord);
Procedure ResetFile;

Function AQTRenameFile(Name1,Name2: AnsiString): Boolean;

Var SV_IO_Table: TFDTable;
    SV_IO_Connect: TFDConnection;

implementation

Uses System.UITypes;

Procedure EmptyTable(Var TT: TFDTable);
Var WasActive:Boolean;
    i: Integer;
Begin
    WasActive := TT.Active;
    TT.Active := True;
    TT.Connection.FetchOptions.RecordCountMode := cmTotal;
    TT.Refresh;
    TT.CachedUpdates := True;
    TT.DisableControls;
    TT.First;
    for i := 0 to TT.recordcount-1 do
     Begin
       TT.Delete;
       TT.Next;
     End;
    TT.EnableControls;
    TT.ApplyUpdates();
    TT.CommitUpdates;
    TT.CachedUpdates := False;
    TT.Active := WasActive;

End;

Function AQTRenameFile(Name1,Name2: AnsiString):Boolean;
{Rename a file in Dir from Name1 to Name2 }
Var F: File;

Begin
   Try
     CleanDashes(Name1);
     CleanDashes(Name2);

     AssignFile(f,Name1);
     Rename(f,Name2);
     AQTRenameFile:=True;
   Except
     Raise EAquatoxError.Create('File Rename Error');
     AQTRenameFile:=False;
   End;
End;

Function GetEntryByNum(DBName,TName: AnsiString; Num: Integer):Boolean;
Var  i: Integer;

Begin
   SV_IO_Connect.Params.Database:=DBName;
   GetEntryByNum:=True;
   Try
   With SV_IO_Table do begin
      Active:=False;
      {Find the Appropriate Entry}
      Tablename:=TName;
      SV_IO_Connect.Params.Database:=DBName;
      Active:=True;

      SV_IO_Table.First;
      For i := 2 to Num do
        SV_IO_Table.Next;

   End; {With}
   Except
     GetEntryByNum:=False;
   End; {Try Except}
End;


Function GetEntry(DBName,TName,LookupN: AnsiString; AddIt, Prompt: Boolean):Boolean;
{Ready the database file for reading or writing by finding the relevant Entry}
Var FoundEntry :Boolean;
Begin

   GetEntry:=True;
   Try

   With SV_IO_Table do
     begin
      Active:=False;
      {Find the Appropriate Entry}
      Tablename:=TName;
      SV_IO_Connect.Params.Database:=DBName;

      Active:=True;
      FoundEntry := False;
      SV_IO_Table.Refresh;

      if SV_IO_Table.Active and (SV_IO_Table.RecordCount > 0) then
        Begin
          SV_IO_Table.DisableControls;
          SV_IO_Table.First;

          while (not SV_IO_Table.EOF) and (not FoundEntry) do
             begin
                FoundEntry := LookupN = SV_IO_Table.Fields[0].AsAnsiString;
                If not FoundEntry then SV_IO_Table.Next;
             end;
          SV_IO_Table.EnableControls;
        End;

      {Raise Error if Entry is not Found}
      If Not FoundEntry then
         Begin
           If AddIt
             then
               with SV_IO_Table do
                 Begin
                   Insert;
                   Fields[0].AsAnsiString := LookupN;
                   Post;
                 End
             else
               begin  MessageDlg('DBase Error, ['+TName+':'+LookupN+'] not found.',mterror,[mbOK],0);
                                  Active:=False; GetEntry:=False;
               end;
         End
        Else If Prompt
          then If MessageDlg('The entry "'+Fields[0].AsAnsiString+'" already exists in that Database.  Overwrite?',MtConfirmation,[mbYes,mbNo],0) = mrno
           then Result := False
           else with SV_IO_Table do
                 Begin
                   Edit;
                   Delete;
                 End;
    End; {With}

   Except
   GetEntry:=False;
   End; {Try Except}
End;

Procedure ResetFile;
{Reset the Dbase File to its original state after working with it}

Begin
  SV_IO_Table.Active:=False;
  SV_IO_Table.Disconnect;
  SV_IO_Connect.ReleaseClients(rmOffline)
End;

{*************************************************
****** Interaction of Records and Databases ******
**************************************************}

Function DBase_To_ChemRecord2(DBName,TName,LookupN: AnsiString; Num: Integer; Var CR: Chemicalrecord):Boolean;
Begin
   Try

   If LookupN = ''
      then DBase_To_ChemRecord2:=GetEntryByNum(DbName,TName,Num)
      else DBase_To_ChemRecord2:=GetEntry(DBName,TName,LookupN,False,False);

   With SV_IO_Table do begin
   With Cr do begin
      ChemName             := Fieldbyname('ChemName').AsAnsiString;
      MolWt                := Fields[1].asfloat;
      Solubility           := Fields[2].asfloat;
      XSolubility          := Fields[3].asAnsiString;
      Henry                := Fields[4].asfloat;
      XHenry               := Fields[5].asAnsiString;
      Xpka                 := Fields[6].AsAnsiString;
      pka                  := Fields[7].asfloat;
      CASRegNo             := Fields[8].AsAnsiString;
      VPress               := Fields[9].asfloat;
      XVPress              := Fields[10].asAnsiString;
      LogKow               := Fields[11].asfloat;
      XLogKow              := Fields[12].asAnsiString;
      En                   := Fields[13].asfloat;
      XEn                  := Fields[14].asAnsiString;
      KMDegrAnaerobic      := Fields[15].asfloat;
      XKMDegrAnaerobic     := Fields[16].asAnsiString;
      KMDegrdn             := Fields[17].asfloat;
      XKMDegrdn            := Fields[18].asAnsiString;
      KUnCat               := Fields[19].asfloat;
      XKUncat              := Fields[20].asAnsiString;
      KAcid                := Fields[21].asfloat;
      XKAcid               := Fields[22].asAnsiString;
      KBase                := Fields[23].asfloat;
      XKBase               := Fields[24].asAnsiString;
      PhotoLysisRate       := Fields[25].asfloat;
      XPhotoLysisRate      := Fields[26].asAnsiString;
      OxRateConst          := Fields[27].asfloat;
      XOxRateConst         := Fields[28].asAnsiString;

      KPSed            := Fields[29].asfloat;
      XKpSed           := Fields[30].asAnsiString;
      Weibull_Shape    := Fields[31].asfloat;
      XWeibull_Shape   := Fields[32].asAnsiString;
      ChemIsBase       := Fields[33].asBoolean;
      CalcKPSed        := Fields[34].AsBoolean;

      CohesivesK1          := Fields[35].asfloat;
      CohesivesK2          := Fields[36].asfloat;
      CohesivesKp          := Fields[37].asfloat;
      CohesivesRef         := Fields[38].asAnsiString;
      NonCohK1             := Fields[39].asfloat;
      NonCohK2             := Fields[40].asfloat;
      NonCohKp             := Fields[41].asfloat;
      NonCohRef            := Fields[42].asAnsiString;
      NonCoh2K1            := Fields[43].asfloat;
      NonCoh2K2            := Fields[44].asfloat;
      NonCoh2Kp            := Fields[45].asfloat;
      NonCoh2Ref           := Fields[46].asAnsiString;

      IsPFA                := Fields[47].asBoolean;
      PFAType              := Fields[48].asAnsiString;
      PFAChainLength       := Fields[49].asFloat;
      XPFAChainLength      := Fields[50].asAnsiString;
      PFASedKom            := Fields[51].asFloat;
      XPFASedKom           := Fields[52].asAnsiString;
      PFAAlgBCF            := Fields[53].asFloat;
      XPFAAlgBCF           := Fields[54].asAnsiString;
      PFAMacroBCF          := Fields[55].asFloat;
      XPFAMacroBCF         := Fields[56].asAnsiString;

      WeibullSlopeFactor   := Fields[57].asFloat;
      XWeibullSlopeFactor  := Fields[58].asAnsiString;

      CalcKOMRefrDOM       := Fields[59].asBoolean;
      KOMRefrDOM           := Fields[60].asFloat;
      XKOMRefrDOM          := Fields[61].asAnsiString;

      K1Detritus           := Fields[62].asFloat;
      XK1Detritus          := Fields[63].asAnsiString;

      BCFUptake            := Fields[64].asBoolean;

   end; {with CR}
   end; {with SV_IO_Form.SV_IO_Table}

   ResetFile;
   Except

   DBase_To_ChemRecord2:=False;
   ResetFile;
   End; {try Except}
End;    {DBase_To_ChemRecord}

{-----------------------------------------------------------------}

Function ChemRecord_To_Dbase2(DBName,TName,LookupN: AnsiString; CR: Chemicalrecord; Prompt:Boolean):Boolean;
Begin

   Try
      ChemRecord_To_DBase2:=GetEntry(DBName,TName,LookupN,True,Prompt);
      If not Result then Exit;

      SV_IO_Table.Active := True;

  With SV_IO_Table do begin

      Edit;

  With Cr do begin
      Fieldbyname('ChemName').AsAnsiString:=ChemName;
      Fields[1].asfloat:=MolWt;
      Fields[2].asfloat:=Solubility;
      Fields[3].asAnsiString:=XSolubility;
      Fields[4].asfloat:=Henry;
      Fields[5].asAnsiString:=XHenry;
      Fields[6].AsAnsiString :=XPkA;
      Fields[7].asfloat:=pka;
      Fields[8].AsAnsiString:=CASRegNo;
      Fields[9].asfloat:=VPress;
      Fields[10].asAnsiString:=XVPress;
      Fields[11].asfloat:=LogKow;
      Fields[12].asAnsiString:=XLogKow;
      Fields[13].asfloat:=En;
      Fields[14].asAnsiString:=XEn;
      Fields[15].asfloat:=KMDegrAnaerobic;
      Fields[16].asAnsiString:=XKMDegrAnaerobic;
      Fields[17].asfloat:=KMDegrdn;
      Fields[18].asAnsiString:=XKMDegrdn;
      Fields[19].asfloat:=KUnCat;
      Fields[20].asAnsiString:=XKUncat;
      Fields[21].asfloat:=KAcid;
      Fields[22].asAnsiString:=XKAcid;
      Fields[23].asfloat:=KBase;
      Fields[24].asAnsiString:=XKBase;
      Fields[25].asfloat:=PhotoLysisRate;
      Fields[26].asAnsiString:=XPhotoLysisRate;
      Fields[27].asfloat:=OxRateConst;
      Fields[28].asAnsiString:=XOxRateConst;
      Fields[29].asfloat  :=KPSed;
      Fields[30].asAnsiString :=XKpSed;
      Fields[31].asfloat  :=Weibull_Shape;
      Fields[32].asAnsiString :=XWeibull_Shape;
      Fields[33].asBoolean:=ChemIsBase;
      Fields[34].AsBoolean:=CalcKPSed;

      Fields[35].asfloat:=CohesivesK1;         
      Fields[36].asfloat:=CohesivesK2;
      Fields[37].asfloat:=CohesivesKp;
      Fields[38].asAnsiString:=CohesivesRef;
      Fields[39].asfloat:=NonCohK1;
      Fields[40].asfloat:=NonCohK2;
      Fields[41].asfloat:=NonCohKp;
      Fields[42].asAnsiString:=NonCohRef;
      Fields[43].asfloat:=NonCoh2K1;
      Fields[44].asfloat:=NonCoh2K2;
      Fields[45].asfloat:=NonCoh2Kp;
      Fields[46].asAnsiString:=NonCoh2Ref;

      Fields[47].asBoolean := IsPFA;
      Fields[48].asAnsiString  := PFAType;
      Fields[49].asFloat   := PFAChainLength;
      Fields[50].asAnsiString  := XPFAChainLength;
      Fields[51].asFloat   := PFASedKom;
      Fields[52].asAnsiString  := XPFASedKom;
      Fields[53].asFloat   := PFAAlgBCF;
      Fields[54].asAnsiString  := XPFAAlgBCF;
      Fields[55].asFloat   := PFAMacroBCF;
      Fields[56].asAnsiString  := XPFAMacroBCF;

      Fields[57].asFloat   := WeibullSlopeFactor;
      Fields[58].asAnsiString  := XWeibullSlopeFactor;

      Fields[59].asBoolean := CalcKOMRefrDOM;
      Fields[60].asFloat   := KOMRefrDOM;
      Fields[61].asAnsiString  := XKOMRefrDOM;

      Fields[62].asFloat   := K1Detritus;
      Fields[63].asAnsiString  := XK1Detritus;

      Fields[64].asBoolean := BCFUptake;

   end; {with CR}
   Post;
   end; {with SV_IO_Form.SV_IO_Table}

   ResetFile;

   Except

   ChemRecord_To_DBase2:=False;
   ResetFile;

   End; {try Except}
End;    {ChemRecord_To_Dbase}

{-----------------------------------------------------------------}

Procedure ChemRec_To_Text;
Begin
  With CR do
    Begin
     Writeln(LF,'ChemName: '+ChemName);
     Writeln(LF,'CASRegNo: '+CASRegNo);
     Writeln(LF,'MolWt: '+FloatToStrF(MolWt,ffgeneral,5,5));
     Writeln(LF,'Solubility: '+FloatToStrF(Solubility,ffgeneral,5,5)+' -- '+XSolubility);
     Writeln(LF,'Henry: '+FloatToStrF(Henry,ffgeneral,5,5)+' -- '+XHenry);
     If BCFUptake then Writeln(LF,' USE BCF FOR UPTAKE');
     Writeln(LF,'pka: '+FloatToStrF(pka,ffgeneral,5,5)+' -- '+Xpka);
     Writeln(LF,'VPress: '+FloatToStrF(VPress,ffgeneral,5,5)+' -- '+XVPress);
     Writeln(LF,'LogP: '+FloatToStrF(LogKow,ffgeneral,5,5)+' -- '+XLogKow);
     Writeln(LF,'En: '+FloatToStrF(En,ffgeneral,5,5)+' -- '+XEn);
     Writeln(LF,'KMDegrdn: '+FloatToStrF(KMDegrdn,ffgeneral,5,5)+' -- '+XKMDegrdn);
     Writeln(LF,'KMDegrAnaerobic: '+FloatToStrF(KMDegrAnaerobic,ffgeneral,5,5)+' -- '+XKMDegrAnaerobic);
     Writeln(LF,'KUnCat: '+FloatToStrF(KUnCat,ffgeneral,5,5)+' -- '+XKUnCat);
     Writeln(LF,'KAcid: '+FloatToStrF(KAcid,ffgeneral,5,5)+' -- '+XKAcid);
     Writeln(LF,'KBase: '+FloatToStrF(KBase,ffgeneral,5,5)+' -- '+XKBase);
     Writeln(LF,'PhotolysisRate: '+FloatToStrF(PhotolysisRate,ffgeneral,5,5)+' -- '+XPhotolysisRate);
     Writeln(LF,'OxRateConst: '+FloatToStrF(OxRateConst,ffgeneral,5,5)+' -- '+XOxRateConst);
     Write(LF,'CalcKPSed: '); If CalcKPSed then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     If Not CalcKPSed then Writeln(LF,'KPSed: '+FloatToStrF(KPSed,ffgeneral,5,5)+' -- '+XKPSed);
     Write(LF,'CalcKOMRefrDOM: '); If CalcKOMRefrDOM then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     If Not CalcKOMRefrDOM then Writeln(LF,'KOMRefrDOM: '+FloatToStrF(KOMRefrDOM,ffgeneral,5,5)+' -- '+XKOMRefrDOM);
     Writeln(LF,'Weibull_Shape: '+FloatToStrF(Weibull_Shape,ffgeneral,5,5)+' -- '+XWeibull_Shape);
     Write(LF,'ChemIsBase: '); If ChemIsBase then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     Writeln(LF,'K1Detritus: '+FloatToStrF(K1Detritus,ffgeneral,5,5) +' -- ' + XK1Detritus);
     Writeln(LF,'CohesivesK1: '+FloatToStrF(CohesivesK1,ffgeneral,5,5));
     Writeln(LF,'CohesivesK2: '+FloatToStrF(CohesivesK2,ffgeneral,5,5));
     Writeln(LF,'CohesivesKp: '+FloatToStrF(CohesivesKp,ffgeneral,5,5));
     Writeln(LF,'CohesivesRef: '+CohesivesRef);
     Writeln(LF,'NonCohK1: '+FloatToStrF(NonCohK1,ffgeneral,5,5));
     Writeln(LF,'NonCohK2: '+FloatToStrF(NonCohK2,ffgeneral,5,5));
     Writeln(LF,'NonCohKp: '+FloatToStrF(NonCohKp,ffgeneral,5,5));
     Writeln(LF,'NonCohRef: '+NonCohRef);
     Writeln(LF,'NonCoh2K1: '+FloatToStrF(NonCoh2K1,ffgeneral,5,5));
     Writeln(LF,'NonCoh2K2: '+FloatToStrF(NonCoh2K2,ffgeneral,5,5));
     Writeln(LF,'NonCoh2Kp: '+FloatToStrF(NonCoh2Kp,ffgeneral,5,5));
     Writeln(LF,'NonCoh2Ref: '+NonCoh2Ref);
     Write(LF,'IsPFA: '); If IsPFA then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     Writeln(LF,'PFAType: '+PFAType);
     Writeln(LF,'PFAChainLength: '+FloatToStrF(PFAChainLength,ffgeneral,5,5)+' -- '+XPFAChainLength);
     Writeln(LF,'PFASedKom: '+FloatToStrF(PFASedKom,ffgeneral,5,5)+' -- '+XPFASedKom);
     Writeln(LF,'PFAAlgBCF: '+FloatToStrF(PFAAlgBCF,ffgeneral,5,5)+' -- '+XPFAAlgBCF);
     Writeln(LF,'PFAMacroBCF: '+FloatToStrF(PFAMacroBCF,ffgeneral,5,5)+' -- '+XPFAMacroBCF);
     Writeln(LF,'WeibullSlopeFactor: '+FloatToStrF(WeibullSlopeFactor,ffgeneral,5,5)+' -- '+XWeibullSlopeFactor);
    End; {with}
End; {proc}


{-----------------------------------------------------------------}

Function DBase_To_AnimalRecord2(DBName,TName,LookupN: AnsiString; Num: Integer; Var AR: ZooRecord):Boolean;
Begin

   Try
   If LookupN = ''
      then DBase_To_AnimalRecord2:=GetEntryByNum(DbName,TName,Num)
      else DBase_To_AnimalRecord2:=GetEntry(DBName,TName,LookupN,False,False);

   With SV_IO_Table do begin
   With AR do begin
      AnimalName        := Fieldbyname('AnimalName').AsAnsiString;
      FHalfSat          := Fields[1].AsFloat;
      XFHalfSat         := Fields[2].AsAnsiString;
      CMax              := Fields[3].AsFloat;
      XCMax             := Fields[4].AsAnsiString;
      BMin              := Fields[5].AsFloat;
      XBMin             := Fields[6].AsAnsiString;
      Q10               := Fields[7].AsFloat;
      XQ10              := Fields[8].AsAnsiString;
      TOpt              := Fields[9].AsFloat;
      XTOpt             := Fields[10].AsAnsiString;
      TMax              := Fields[11].AsFloat;
      XTMax             := Fields[12].AsAnsiString;
      TRef              := Fields[13].AsFloat;
      XTRef             := Fields[14].AsAnsiString;
      EndogResp         := Fields[15].AsFloat;
      XEndogResp        := Fields[16].AsAnsiString;
      KResp             := Fields[17].AsFloat;
      XKResp            := Fields[18].AsAnsiString;
      KExcr             := Fields[19].AsFloat;
      XKExcr            := Fields[20].AsAnsiString;
      PctGamete         := Fields[21].AsFloat;
      XPctGamete        := Fields[22].AsAnsiString;
      GMort             := Fields[23].AsFloat;
      XGMort            := Fields[24].AsAnsiString;
      KMort             := Fields[25].AsFloat;
      XKMort            := Fields[26].AsAnsiString;
      KCap              := Fields[27].AsFloat;
      XKCap             := Fields[28].AsAnsiString;
      MeanWeight        := Fields[29].AsFloat;
      XMeanWeight       := Fields[30].AsAnsiString;
      FishFracLipid     := Fields[31].AsFloat;
      XFishFracLipid    := Fields[32].AsAnsiString;
      ToxicityRecord    := Fields[33].AsAnsiString;
      LifeSpan          := Fields[34].AsFloat;
      XLifeSpan         := Fields[35].AsAnsiString;
      Animal_Type       := Fields[36].AsAnsiString;
      AveDrift          := Fields[37].asfloat;
      XAveDrift         := Fields[38].asAnsiString;
      AutoSpawn         := Fields[39].asBoolean;
      SpawnDate1        := Fields[40].asDateTime;
      SpawnDate2        := Fields[41].asDateTime;
      SpawnDate3        := Fields[42].asDateTime;
      XSpawnDate        := Fields[43].asAnsiString;
      UnlimitedSpawning := Fields[44].asBoolean;
      SpawnLimit        := Fields[45].asInteger;
      UseAllom_C        := Fields[46].asBoolean;
      CA                := Fields[47].asFloat;
      CB                := Fields[48].asFloat;
      UseAllom_R        := Fields[49].asBoolean;
      RA                := Fields[50].asFloat;
      RB                := Fields[51].asFloat;
      UseSet1           := Fields[52].asBoolean;
      RQ                := Fields[53].asFloat;
      RTO               := Fields[54].asFloat;
      RTM               := Fields[55].asFloat;
      RTL               := Fields[56].asFloat;
      RK1               := Fields[57].asFloat;
      RK4               := Fields[58].asFloat;
      ACT               := Fields[59].asFloat;
      BACT              := Fields[60].asFloat;
      FracInWaterCol    := Fields[61].asFloat;
      XFracInWaterCol   := Fields[62].asAnsiString;
      Guild_Taxa        := Fields[63].asAnsiString;

      PrefRiffle        := Fields[64].asFloat;
      XPrefRiffle       := Fields[65].asAnsiString;
      PrefPool          := Fields[66].asFloat;
      XPrefPool         := Fields[67].asAnsiString;
      VelMax            := Fields[68].asFloat;
      XVelMax           := Fields[69].asAnsiString;
      XAllomConsumpt    := Fields[70].asAnsiString;
      XAllomResp        := Fields[71].asAnsiString;
      Salmin_Ing        := Fields[72].asFloat;
      SalMax_Ing        := Fields[73].asFloat;
      Salcoeff1_Ing     := Fields[74].asFloat;
      Salcoeff2_Ing     := Fields[75].asFloat;
      XSalinity_Ing     := Fields[76].asAnsiString;
      Salmin_Gam        := Fields[77].asFloat;
      SalMax_Gam        := Fields[78].asFloat;
      Salcoeff1_Gam     := Fields[79].asFloat;
      Salcoeff2_Gam     := Fields[80].asFloat;
      XSalinity_Gam     := Fields[81].asAnsiString;
      Salmin_Rsp        := Fields[82].asFloat;
      SalMax_Rsp        := Fields[83].asFloat;
      Salcoeff1_Rsp     := Fields[84].asFloat;
      Salcoeff2_Rsp     := Fields[85].asFloat;
      XSalinity_Rsp     := Fields[86].asAnsiString;
      Salmin_Mort       := Fields[87].asFloat;
      SalMax_Mort       := Fields[88].asFloat;
      Salcoeff1_Mort    := Fields[89].asFloat;
      Salcoeff2_Mort    := Fields[90].asFloat;   //10/17/2012 Fix
      XSalinity_Mort    := Fields[91].asAnsiString;
      Fishing_Frac      := Fields[92].asFloat;
      XFishing_Frac     := Fields[93].asAnsiString;
{       Pct_Habitat       := Fields[94].asFloat;
      XPct_Habitat      := Fields[95].asAnsiString; }

      P2Org             := Fields[96].asFloat;
      XP2Org            := Fields[97].asAnsiString;
      N2Org             := Fields[98].asFloat;
      XN2Org            := Fields[99].asAnsiString;
      Wet2Dry           := Fields[100].AsFloat;
      XWet2Dry          := Fields[101].AsAnsiString;

      O2_LethalConc     := Fields[102].AsFloat;
      O2_LethalPct      := Fields[103].AsFloat;
      O2_LCRef          := Fields[104].AsAnsiString;
      O2_EC50growth     := Fields[105].AsFloat;
      XO2_EC50growth    := Fields[106].AsAnsiString;
      O2_EC50repro      := Fields[107].AsFloat;
      XO2_EC50repro     := Fields[108].AsAnsiString;

      Ammonia_LC50          := Fields[109].AsFloat;
      XAmmonia_LC50         := Fields[110].AsAnsiString;

      Sorting              := Fields[111].AsFloat;    {3.46, SABS}
      XSorting             := Fields[112].AsAnsiString;
      SuspSedFeeding       := Fields[113].AsBoolean;
      XSuspSedFeeding      := Fields[114].AsAnsiString;
      SlopeSSFeed          := Fields[115].AsFloat;
      XSlopeSSFeed         := Fields[116].AsAnsiString;
      InterceptSSFeed      := Fields[117].AsFloat;
      XInterceptSSFeed     := Fields[118].AsAnsiString;
      SenstoSediment       := Fields[119].AsAnsiString;
      XSensToSediment      := Fields[120].AsAnsiString;
      Trigger              := Fields[121].AsFloat;
      XTrigger             := Fields[122].AsAnsiString;

      SenstoPctEmbed     := Fields[123].AsBoolean;
      PctEmbedThreshold  := Fields[124].AsFloat  ;
      XPctEmbedThreshold := Fields[125].AsAnsiString ;
      BenthicDesignation := Fields[126].AsAnsiString ;
      ScientificName     := Fields[127].AsAnsiString ;

      CanSeekRefuge      := Fields[128].AsBoolean;
      Visual_Feeder      := Fields[129].AsBoolean;
      Burrow_Index       := Fields[130].AsFloat;
      XBurrow_Index      := Fields[131].AsString ;

   end; {with AR}
   end; {with SV_IO_Form.SV_IO_Table}

   ResetFile;
   Except

   DBase_To_animalRecord2:=False;
   ResetFile;

   End; {try Except}
End;    {DBase_To_animalRecord}

{-----------------------------------------------------------------}

Function AnimalRecord_to_DBase2(DBName,TName,LookupN: AnsiString; AR: ZooRecord;Prompt:Boolean):Boolean;

Begin

   Try

      AnimalRecord_to_Dbase2:=GetEntry(DBName,TName,LookupN,True,Prompt);
      If not Result then Exit;

   With SV_IO_Table do Begin
   SV_IO_Table.Active := True;
   SV_IO_Table.Edit;

   With AR do begin
      Fieldbyname('AnimalName').AsAnsiString:=AnimalName;
      Fields[1].AsFloat   := FHalfSat;
      Fields[2].AsAnsiString  := XFHalfSat;
      Fields[3].AsFloat   := CMax;
      Fields[4].AsAnsiString  := XCMax;
      Fields[5].AsFloat   := BMin;
      Fields[6].AsAnsiString  := XBMin;
      Fields[7].AsFloat   := Q10;
      Fields[8].AsAnsiString  := XQ10;
      Fields[9].AsFloat   := TOpt;
      Fields[10].AsAnsiString := XTOpt;
      Fields[11].AsFloat  := TMax;
      Fields[12].AsAnsiString := XTMax;
      Fields[13].AsFloat  := TRef;
      Fields[14].AsAnsiString := XTRef;
      Fields[15].AsFloat  := EndogResp;
      Fields[16].AsAnsiString := XEndogResp;
      Fields[17].AsFloat  := KResp;
      Fields[18].AsAnsiString := XKResp;
      Fields[19].AsFloat  := KExcr;
      Fields[20].AsAnsiString := XKExcr;
      Fields[21].AsFloat  := PctGamete;
      Fields[22].AsAnsiString := XPctGamete;
      Fields[23].AsFloat  := GMort;
      Fields[24].AsAnsiString := XGMort;
      Fields[25].AsFloat  := KMort;
      Fields[26].AsAnsiString := XKMort;
      Fields[27].AsFloat  := KCap;
      Fields[28].AsAnsiString := XKCap;
      Fields[29].AsFloat  := MeanWeight;
      Fields[30].AsAnsiString := XMeanWeight;
      Fields[31].AsFloat  := FishFracLipid;
      Fields[32].AsAnsiString := XFishFracLipid;
      Fields[33].AsAnsiString := ToxicityRecord;
      Fields[34].AsFloat  := LifeSpan;
      Fields[35].AsAnsiString := XLifeSpan;
      Fields[36].AsAnsiString := Animal_Type;
      Fields[37].asFloat  := AveDrift;
      Fields[38].asAnsiString := XAveDrift;
      Fields[39].asBoolean:= AutoSpawn;
      Fields[40].asDateTime := SpawnDate1;
      Fields[41].asDateTime := SpawnDate2;
      Fields[42].asDateTime := SpawnDate3;
      Fields[43].asAnsiString := XSpawnDate;
      Fields[44].asBoolean:= UnlimitedSpawning;
      Fields[45].asInteger:= SpawnLimit;
      Fields[46].asBoolean:= UseAllom_C;
      Fields[47].asFloat  := CA;
      Fields[48].asFloat  := CB;
      Fields[49].asBoolean:= UseAllom_R;
      Fields[50].asFloat  := RA;
      Fields[51].asFloat  := RB;
      Fields[52].asBoolean:= UseSet1;
      Fields[53].asFloat  := RQ;
      Fields[54].asFloat  := RTO;
      Fields[55].asFloat  := RTM;
      Fields[56].asFloat  := RTL;
      Fields[57].asFloat  := RK1;
      Fields[58].asFloat  := RK4;
      Fields[59].asFloat  := ACT;
      Fields[60].asFloat  := BACT;
      Fields[61].asFloat  := FracInWaterCol;
      Fields[62].asAnsiString := XFracInWaterCol;
      Fields[63].asAnsiString := Guild_Taxa;

      Fields[64].asFloat :=PrefRiffle;
      Fields[65].asAnsiString :=XPrefRiffle;
      Fields[66].asFloat :=PrefPool;
      Fields[67].asAnsiString :=XPrefPool;
      Fields[68].asFloat :=VelMax;
      Fields[69].asAnsiString :=XVelMax;
      Fields[70].asAnsiString :=XAllomConsumpt;
      Fields[71].asAnsiString :=XAllomResp;
      Fields[72].asFloat :=Salmin_Ing;
      Fields[73].asFloat :=SalMax_Ing;
      Fields[74].asFloat :=Salcoeff1_Ing;
      Fields[75].asFloat :=Salcoeff2_Ing;
      Fields[76].asAnsiString :=XSalinity_Ing;
      Fields[77].asFloat :=Salmin_Gam;
      Fields[78].asFloat :=SalMax_Gam;
      Fields[79].asFloat :=Salcoeff1_Gam;
      Fields[80].asFloat :=Salcoeff2_Gam;
      Fields[81].asAnsiString :=XSalinity_Gam;
      Fields[82].asFloat :=Salmin_Rsp;
      Fields[83].asFloat :=SalMax_Rsp;
      Fields[84].asFloat :=Salcoeff1_Rsp;
      Fields[85].asFloat :=Salcoeff2_Rsp;
      Fields[86].asAnsiString :=XSalinity_Rsp;
      Fields[87].asFloat :=Salmin_Mort;
      Fields[88].asFloat :=SalMax_Mort;
      Fields[89].asFloat :=Salcoeff1_Mort;
      Fields[90].asFloat :=Salcoeff2_Mort;   //10/17/2012 Fix
      Fields[91].asAnsiString :=XSalinity_Mort;
      Fields[92].asFloat :=Fishing_Frac;
      Fields[93].asAnsiString :=XFishing_Frac;
{      Fields[94].asFloat :=Pct_Habitat;
      Fields[95].asAnsiString :=XPct_Habitat; }

      Fields[96].asFloat := P2Org;
      Fields[97].asAnsiString := XP2Org;
      Fields[98].asFloat := N2Org;
      Fields[99].asAnsiString := XN2Org;
      Fields[100].AsFloat := Wet2Dry;
      Fields[101].AsAnsiString := XWet2Dry;

      Fields[102].AsFloat := O2_LethalConc;
      Fields[103].AsFloat := O2_LethalPct;
      Fields[104].AsAnsiString := O2_LCRef;
      Fields[105].AsFloat := O2_EC50growth;
      Fields[106].AsAnsiString := XO2_EC50growth;
      Fields[107].AsFloat := O2_EC50repro;
      Fields[108].AsAnsiString := XO2_EC50repro;

      Fields[109].AsFloat := Ammonia_LC50;
      Fields[110].AsAnsiString := XAmmonia_LC50;

      Fields[111].AsFloat :=Sorting;
      Fields[112].AsAnsiString:=XSorting;
      Fields[113].AsBoolean:=SuspSedFeeding;
      Fields[114].AsAnsiString:=XSuspSedFeeding;
      Fields[115].AsFloat:=SlopeSSFeed;
      Fields[116].AsAnsiString:=XSlopeSSFeed;
      Fields[117].AsFloat:=InterceptSSFeed;
      Fields[118].AsAnsiString:=XInterceptSSFeed;
      Fields[119].AsAnsiString:=SenstoSediment;
      Fields[120].AsAnsiString:=XSensToSediment;
      Fields[121].AsFloat:=Trigger;
      Fields[122].AsAnsiString:=XTrigger;

      Fields[123].AsBoolean := SenstoPctEmbed;
      Fields[124].AsFloat   := PctEmbedThreshold;
      Fields[125].AsAnsiString  := XPctEmbedThreshold;
      Fields[126].AsAnsiString  := BenthicDesignation;
      Fields[127].AsAnsiString  := ScientificName;

      Fields[128].AsBoolean := CanSeekRefuge;
      Fields[129].AsBoolean := Visual_Feeder;
      Fields[130].AsFloat := Burrow_Index;
      Fields[131].AsString := XBurrow_Index ;

   end; {with AR}
   Post;
   end; {with SV_IO_Form.SV_IO_Table}

   ResetFile;
   Except

   AnimalRecord_to_Dbase2:=False;
   ResetFile;
   End; {try Except}
End;    {AnimalRecord_to_Dbase}

{-----------------------------------------------------------------}


Procedure AnimRec_To_Text;
Begin
  With AR do
    Begin
     Writeln(LF,'AnimalName: '+AnimalName);
     Writeln(LF,'ScientificName: '+ScientificName);
     Writeln(LF,'ToxicityRecord: '+ToxicityRecord);
     Writeln(LF,'FHalfSat: '+FloatToStrF(FHalfSat,ffgeneral,5,5)+' -- '+XFHalfSat);
     Writeln(LF,'CMax: '+FloatToStrF(CMax,ffgeneral,5,5)+' -- '+XCMax);
     Writeln(LF,'BMin: '+FloatToStrF(BMin,ffgeneral,5,5)+' -- '+XBMin);
     Writeln(LF,'Q10: '+FloatToStrF(Q10,ffgeneral,5,5)+' -- '+XQ10);
     Writeln(LF,'TOpt: '+FloatToStrF(TOpt,ffgeneral,5,5)+' -- '+XTOpt);
     Writeln(LF,'TMax: '+FloatToStrF(TMax,ffgeneral,5,5)+' -- '+XTMax);
     Writeln(LF,'TRef: '+FloatToStrF(TRef,ffgeneral,5,5)+' -- '+XTRef);
     Writeln(LF,'EndogResp: '+FloatToStrF(EndogResp,ffgeneral,5,5)+' -- '+XEndogResp);
     Writeln(LF,'KResp: '+FloatToStrF(KResp,ffgeneral,5,5)+' -- '+XKResp);
     Writeln(LF,'KExcr: '+FloatToStrF(KExcr,ffgeneral,5,5)+' -- '+XKExcr);
     Writeln(LF,'PctGamete: '+FloatToStrF(PctGamete,ffgeneral,5,5)+' -- '+XPctGamete);
     Writeln(LF,'GMort: '+FloatToStrF(GMort,ffgeneral,5,5)+' -- '+XGMort);
     Writeln(LF,'KMort: '+FloatToStrF(KMort,ffgeneral,5,5)+' -- '+XKMort);
     Writeln(LF,'Placeholder: '+FloatToStrF(Placeholder,ffgeneral,5,5)+' -- '+XPlaceholder);
     Writeln(LF,'KCap: '+FloatToStrF(KCap,ffgeneral,5,5)+' -- '+XKCap);
     Writeln(LF,'MeanWeight: '+FloatToStrF(MeanWeight,ffgeneral,5,5)+' -- '+XMeanWeight);
     Writeln(LF,'FishFracLipid: '+FloatToStrF(FishFracLipid,ffgeneral,5,5)+' -- '+XFishFracLipid);
     Writeln(LF,'LifeSpan: '+FloatToStrF(LifeSpan,ffgeneral,5,5)+' -- '+XLifeSpan);
     Writeln(LF,'Animal_Type: '+Animal_Type);
     If BenthicDesignation <> '' then Writeln(LF,'Benthic Designation: '+BenthicDesignation);

     Writeln(LF,'AveDrift: '+FloatToStrF(AveDrift,ffgeneral,5,5)+' -- '+XAveDrift);
     Write(LF,'AutoSpawn: '); If AutoSpawn then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     If not AutoSpawn then
       Begin
         Writeln(LF,'SpawnDate1: '+DateToStr(SpawnDate1));
         Writeln(LF,'SpawnDate2: '+DateToStr(SpawnDate2));
         Writeln(LF,'SpawnDate3: '+DateToStr(SpawnDate3));
       End;
     Writeln(LF,'XSpawnDate: '+XSpawnDate);
     Write(LF,'UnlimitedSpawning: '); If UnlimitedSpawning then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     Writeln(LF,'SpawnLimit: '+IntToStr(SpawnLimit));
     Write(LF,'UseAllom_C: '); If UseAllom_C then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     Writeln(LF,'CA: '+FloatToStrF(CA,ffgeneral,5,5));
     Writeln(LF,'CB: '+FloatToStrF(CB,ffgeneral,5,5));
     Write(LF,'UseAllom_R: '); If UseAllom_R then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     Writeln(LF,'RA: '+FloatToStrF(RA,ffgeneral,5,5));
     Writeln(LF,'RB: '+FloatToStrF(RB,ffgeneral,5,5));
     Write(LF,'UseSet1: '); If UseSet1 then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     Writeln(LF,'RK1,: '+FloatToStrF(RK1,ffgeneral,5,5));
     Writeln(LF,'FracInWaterCol: '+FloatToStrF(FracInWaterCol,ffgeneral,5,5)+' -- '+XFracInWaterCol);
     Writeln(LF,'Guild_Taxa: '+Guild_Taxa);
     Writeln(LF,'PrefRiffle: '+FloatToStrF(PrefRiffle,ffgeneral,5,5)+' -- '+XPrefRiffle);
     Writeln(LF,'PrefPool: '+FloatToStrF(PrefPool,ffgeneral,5,5)+' -- '+XPrefPool);
     Writeln(LF,'VelMax: '+FloatToStrF(VelMax,ffgeneral,5,5)+' -- '+XVelMax);
     Writeln(LF,'XAllomConsumpt: '+XAllomConsumpt);
     Writeln(LF,'XAllomResp: '+XAllomResp);
     Writeln(LF,'Salcoeff2_Ing: '+FloatToStrF(Salcoeff2_Ing,ffgeneral,5,5));
     Writeln(LF,'XSalinity_Ing: '+XSalinity_Ing);
     Writeln(LF,'Salcoeff2_Gam: '+FloatToStrF(Salcoeff2_Gam,ffgeneral,5,5));
     Writeln(LF,'XSalinity_Gam: '+XSalinity_Gam);
     Writeln(LF,'Salcoeff2_Rsp: '+FloatToStrF(Salcoeff2_Rsp,ffgeneral,5,5));
     Writeln(LF,'XSalinity_Rsp: '+XSalinity_Rsp);
     Writeln(LF,'Salcoeff2_Mort: '+FloatToStrF(Salcoeff2_Mort,ffgeneral,5,5));
     Writeln(LF,'XSalinity_Mort: '+XSalinity_Mort);
     Writeln(LF,'Fishing_Frac: '+FloatToStrF(Fishing_Frac,ffgeneral,5,5)+' -- '+XFishing_Frac);
{     Writeln(LF,'Pct_Habitat: '+FloatToStrF(Pct_Habitat,ffgeneral,5,5)+' -- '+XPct_Habitat); }
     Writeln(LF,'P2Org: '+FloatToStrF(P2Org,ffgeneral,5,5)+' -- '+XP2Org);
     Writeln(LF,'N2Org: '+FloatToStrF(N2Org,ffgeneral,5,5)+' -- '+XN2Org);
     Writeln(LF,'Wet2Dry: '+FloatToStrF(Wet2Dry,ffgeneral,5,5)+' -- '+XWet2Dry);
     Writeln(LF,'LC'+FloatToStrF(O2_LethalPct,ffgeneral,5,5)+
                     FloatToStrF(O2_LethalConc,ffgeneral,5,5)+' -- '+O2_LCRef);
     Writeln(LF,'EC50 Growth: '+FloatToStrF(O2_EC50growth,ffgeneral,5,5)+' -- '+XO2_EC50growth);
     Writeln(LF,'EC50 Repro: '+FloatToStrF(O2_EC50repro,ffgeneral,5,5)+' -- '+XO2_EC50repro);
     Writeln(LF,'Ammonia LC50: '+FloatToStrF(Ammonia_LC50,ffgeneral,5,5)+' -- '+XAmmonia_LC50);

     Writeln(LF,'Sorting: '+FloatToStrF(Sorting,ffgeneral,5,5)+' -- '+XSorting);
     If SuspSedFeeding then
       Begin
         Writeln(LF,'Animal IS a Visual Feeder'+' -- '+XSuspSedFeeding);
         Writeln(LF,'SlopeSSFeed: '+FloatToStrF(SlopeSSFeed,ffgeneral,5,5)+' -- '+XSlopeSSFeed);
         Writeln(LF,'InterceptSSFeed: '+FloatToStrF(InterceptSSFeed,ffgeneral,5,5)+' -- '+XInterceptSSFeed);
       End;
     Writeln(LF,'SensToSediment: '+SensToSediment+' -- '+XSensToSediment);
     Writeln(LF,'Drift Trigger: '+FloatToStrF(Trigger,ffgeneral,5,5)+' -- '+XTrigger);
     If Not SenstoPctEmbed
       then Writeln(LF,'Animal not sensitive to percent embeddedness')
       else Writeln(LF,'Pct. Embedded Threshhold: '+FloatToSTrF(PctEmbedThreshold,ffgeneral,5,5) +' -- '+XPctEmbedThreshold);

     If CanSeekRefuge then Writeln(LF,'Can Seek Refuge.');
     If Visual_Feeder then Writeln(LF,'Is Visual Feeder.');
     Writeln(LF,'Burrow_Index : ' +FloatToStrF(Burrow_Index,ffgeneral,5,5) +' -- '+XBurrow_Index);
    End;
End;

{-----------------------------------------------------------------}

Function DBase_To_SiteRecord2(DBName,TName,LookupN: AnsiString; Var SR: SiteRecord):Boolean;

Begin

   Try
      DBase_To_siteRecord2:=GetEntry(DBName,TName,LookupN,False,False);

   If Result then
   With SV_IO_Table do begin
   With SR do begin
      siteName             := Fieldbyname('SiteName').AsAnsiString;
      SiteLength           := Fields[1].AsFloat;
      XLength              := Fields[2].AsAnsiString;
      StaticVolume         := Fields[3].AsFloat;
      XVolume              := Fields[4].AsAnsiString;
      SurfArea             := Fields[5].AsFloat;
      XSurfArea            := Fields[6].AsAnsiString;
      ICZMean              := Fields[7].AsFloat;
      XZMean               := Fields[8].AsAnsiString;
      ZMax                 := Fields[9].AsFloat;
      XZMax                := Fields[10].AsAnsiString;
      TempMean[Epilimnion] := Fields[11].AsFloat;
      XTempMean[Epilimnion]:= Fields[12].AsAnsiString;
      TempMean[Hypolimnion]:= Fields[13].AsFloat;
      XTempMean[Hypolimnion]:= Fields[14].AsAnsiString;
      TempRange[Epilimnion] := Fields[15].AsFloat;
      XTempRange[Epilimnion]:= Fields[16].AsAnsiString;
      TempRange[Hypolimnion]:= Fields[17].AsFloat;
      XTempRange[Hypolimnion]:=Fields[18].AsAnsiString;
      Latitude             := Fields[19].AsFloat;
      XLatitude            := Fields[20].AsAnsiString;
      LightMean            := Fields[21].AsFloat;
      XLightMean           := Fields[22].AsAnsiString;
      LightRange           := Fields[23].AsFloat;
      XLightRange          := Fields[24].AsAnsiString;
      AlkCaCO3             := Fields[25].AsFloat;
      XAlkCaCO3            := Fields[26].AsAnsiString;
      HardCaCO3            := Fields[27].AsFloat;
      XHardCaCO3           := Fields[28].AsAnsiString;
      SiteComment          := Fields[29].AsAnsiString;
      SO4Conc              := Fields[30].AsFloat;
      XSO4Conc             := Fields[31].AsAnsiString;
      TotalDissSolids      := Fields[32].AsFloat;
      XTotalDissSolids     := Fields[33].AsAnsiString;
      StreamType           := Fields[34].AsAnsiString;
      Channel_Slope        := Fields[35].AsFloat;
      XChannel_Slope       := Fields[36].AsAnsiString;
      Max_Chan_Depth       := Fields[37].AsFloat;
      XMax_Chan_Depth      := Fields[38].AsAnsiString;
      SedDepth             := Fields[39].AsFloat;
      XSedDepth            := Fields[40].AsAnsiString;
      EnclWallArea        := Fields[41].AsFloat;
      XEnclWallArea       := Fields[42].AsAnsiString;
      MeanEvap             := Fields[43].AsFloat;   {inches / year}
      XMeanEvap            := Fields[44].AsAnsiString;
      UseEnteredManning    := Fields[45].AsBoolean;
      EnteredManning       := Fields[46].AsFloat;
      UseBathymetry        := Fields[47].AsBoolean;
      ECoeffWater          := Fields[48].AsFloat;
      XECoeffWater         := Fields[49].AsAnsiString;

      PctRiffle            := Fields[50].AsFloat;
      XPctRiffle           := Fields[51].AsAnsiString;
      PctPool              := Fields[52].AsFloat;
      XPctPool             := Fields[53].AsAnsiString;
      ts_clay              := Fields[54].AsFloat;
      Xts_clay             := Fields[55].AsAnsiString;
      ts_silt              := Fields[56].AsFloat;
      Xts_silt             := Fields[57].AsAnsiString;
      tdep_clay            := Fields[58].AsFloat;
      Xtdep_clay           := Fields[59].AsAnsiString;
      tdep_silt            := Fields[60].AsFloat;
      Xtdep_silt           := Fields[61].AsAnsiString;
      FallVel_clay         := Fields[62].AsFloat;
      XFallVel_clay        := Fields[63].AsAnsiString;
      FallVel_silt         := Fields[64].AsFloat;
      XFallVel_silt        := Fields[65].AsAnsiString;
      SiteWidth            := Fields[66].AsFloat;
      XSiteWidth           := Fields[67].AsAnsiString;
 {m2} amplitude1           := Fields[68].AsFloat;
      k1                   := Fields[69].AsFloat;
      ConstRef1            := Fields[70].AsAnsiString;
 {s2} amplitude2           := Fields[71].AsFloat;
      k2                   := Fields[72].AsFloat;
      ConstRef2            := Fields[73].AsAnsiString;
 {n2} amplitude3           := Fields[74].AsFloat;
      k3                   := Fields[75].AsFloat;
      ConstRef3            := Fields[76].AsAnsiString;
 {k1} amplitude4           := Fields[77].AsFloat;
      k4                   := Fields[78].AsFloat;
      ConstRef4            := Fields[79].AsAnsiString;
 {o1} amplitude5           := Fields[80].AsFloat;
      k5                   := Fields[81].AsFloat;
      ConstRef5            := Fields[82].AsAnsiString;
{SSA} amplitude6           := Fields[83].AsFloat;
      k6                   := Fields[84].AsFloat;
      ConstRef6            := Fields[85].AsAnsiString;
 {SA} amplitude7           := Fields[86].AsFloat;
      k7                   := Fields[87].AsFloat;
      ConstRef7            := Fields[88].AsAnsiString;
 {P1} amplitude8           := Fields[89].AsFloat;
      k8                   := Fields[90].AsFloat;
      ConstRef8            := Fields[91].AsAnsiString;
      Min_Vol_Frac         := Fields[92].AsFloat;
      XMin_Vol_Frac        := Fields[93].AsAnsiString;

      WaterShedArea        := Fields[94].AsFloat;
      XWaterShedArea       := Fields[95].AsAnsiString;
      EnterTotalLength     := Fields[96].AsBoolean;
      TotalLength          := Fields[97].AsFloat;
      XTotalLength         := Fields[98].AsAnsiString;

      ECoeffSED            := Fields[99].AsFloat;
      XECoeffSED           := Fields[100].AsAnsiString;
      ECoeffDOM            := Fields[101].AsFloat;
      XECoeffDOM           := Fields[102].AsAnsiString;
      ECoeffPOM            := Fields[103].AsFloat;
      XECoeffPOM           := Fields[104].AsAnsiString;

      UseCovar             := Fields[105].AsBoolean;
      EnteredKReaer        := Fields[106].AsFloat;
      XEnteredKReaer       := Fields[107].AsAnsiString;

      UsePhytoRetention    := Fields[108].AsBoolean;

      BasePercentEmbed     := Fields[109].AsFloat;
      XBasePercentEmbed    := Fields[110].AsAnsiString;

      Altitude             := Fields[111].AsFloat;
      XAltitude            := Fields[112].AsAnsiString;

      FractalD             := Fields[113].AsFloat;
      XFractalD            := Fields[114].AsString;
      FD_Refuge_Coeff      := Fields[115].AsFloat;
      XFD_Refuge_Coeff     := Fields[116].AsString;
      HalfSatOysterRefuge  := Fields[117].AsFloat;
      XHalfSatOysterRefuge := Fields[118].AsString;

 end; {with SR}
 end; {with SV_IO_Form.SV_IO_Table}

   ResetFile;

   Except

   DBase_To_siteRecord2:=False;
   ResetFile;
   End; {try Except}
End;    {DBase_To_siteRecord}
{-----------------------------------------------------------------}

Function SiteRecord_To_DBase2(DBName,TName,LookupN: AnsiString; SR: SiteRecord; Prompt:Boolean):Boolean;

Begin

   Try
      SiteRecord_to_DBase2:=GetEntry(DBName,TName,LookupN,True,Prompt);
      if Not Result then Exit;

   SV_IO_Table.Active := True;
   SV_IO_Table.Edit;
   With SV_IO_Table do Begin

   With SR do begin
      Fieldbyname('SiteName').AsAnsiString:=siteName;
      Fields[1].AsFloat:=SiteLength;
      Fields[2].AsAnsiString:=XLength;
      Fields[3].AsFloat:=StaticVolume;
      Fields[4].AsAnsiString:=XVolume;
      Fields[5].AsFloat:=SurfArea;
      Fields[6].AsAnsiString:=XSurfArea;
      Fields[7].AsFloat:=ICZMean;
      Fields[8].AsAnsiString:=XZMean;
      Fields[9].AsFloat:=ZMax;
      Fields[10].AsAnsiString:=XZMax;
      Fields[11].AsFloat:=TempMean[Epilimnion];
      Fields[12].AsAnsiString:=XTempMean[Epilimnion];
      Fields[13].AsFloat:=TempMean[Hypolimnion];
      Fields[14].AsAnsiString:=XTempMean[Hypolimnion];
      Fields[15].AsFloat:=TempRange[Epilimnion];
      Fields[16].AsAnsiString:=XTempRange[Epilimnion];
      Fields[17].AsFloat :=TempRange[Hypolimnion];
      Fields[18].AsAnsiString:=XTempRange[Hypolimnion];
      Fields[19].AsFloat :=Latitude;
      Fields[20].AsAnsiString:=XLatitude;
      Fields[21].AsFloat :=LightMean;
      Fields[22].AsAnsiString:=XLightMean;
      Fields[23].AsFloat :=LightRange;
      Fields[24].AsAnsiString:=XLightRange;
      Fields[25].AsFloat :=AlkCaCO3;
      Fields[26].AsAnsiString:=XAlkCaCO3;
      Fields[27].AsFloat :=HardCaCO3;
      Fields[28].AsAnsiString:=XHardCaCO3;
      Fields[29].AsAnsiString:=SiteComment;
      Fields[30].AsFloat := SO4Conc;
      Fields[31].AsAnsiString:=XSO4Conc;
      Fields[32].AsFloat :=TotalDissSolids;
      Fields[33].AsAnsiString:=XTotalDissSolids;
      Fields[34].AsAnsiString:=StreamType;
      Fields[35].AsFloat :=Channel_Slope;
      Fields[36].AsAnsiString:=XChannel_Slope;
      Fields[37].AsFloat :=Max_Chan_Depth;
      Fields[38].AsAnsiString:=XMax_Chan_Depth;
      Fields[39].AsFloat :=SedDepth;
      Fields[40].AsAnsiString:=XSedDepth;
      Fields[41].AsFloat :=EnclWallArea;
      Fields[42].AsAnsiString:=XEnclWallArea;
      Fields[43].AsFloat :=MeanEvap;
      Fields[44].AsAnsiString:=XMeanEvap;
      Fields[45].AsBoolean:=UseEnteredManning;
      Fields[46].AsFloat  :=EnteredManning;
      Fields[47].AsBoolean :=UseBathymetry;
      Fields[48].AsFloat  := ECoeffWater;
      Fields[49].AsAnsiString := XECoeffWater;

      Fields[50].AsFloat :=PctRiffle;
      Fields[51].AsAnsiString :=XPctRiffle;
      Fields[52].AsFloat :=PctPool;
      Fields[53].AsAnsiString :=XPctPool;
      Fields[54].AsFloat :=ts_clay;
      Fields[55].AsAnsiString :=Xts_clay;
      Fields[56].AsFloat :=ts_silt;
      Fields[57].AsAnsiString :=Xts_silt;
      Fields[58].AsFloat :=tdep_clay;
      Fields[59].AsAnsiString :=Xtdep_clay;
      Fields[60].AsFloat :=tdep_silt;
      Fields[61].AsAnsiString :=Xtdep_silt;
      Fields[62].AsFloat :=FallVel_clay;
      Fields[63].AsAnsiString :=XFallVel_clay;
      Fields[64].AsFloat :=FallVel_silt;
      Fields[65].AsAnsiString :=XFallVel_silt;
      Fields[66].AsFloat :=SiteWidth;
      Fields[67].AsAnsiString :=XSiteWidth;
      Fields[68].AsFloat :={m2} amplitude1;
      Fields[69].AsFloat :=k1;
      Fields[70].AsAnsiString :=ConstRef1;
      Fields[71].AsFloat :={s2} amplitude2;
      Fields[72].AsFloat :=k2;
      Fields[73].AsAnsiString :=ConstRef2;
      Fields[74].AsFloat :={n2} amplitude3;
      Fields[75].AsFloat :=k3;
      Fields[76].AsAnsiString :=ConstRef3;
      Fields[77].AsFloat :={k1} amplitude4;
      Fields[78].AsFloat :=k4;
      Fields[79].AsAnsiString :=ConstRef4;
      Fields[80].AsFloat :={o1} amplitude5;
      Fields[81].AsFloat :=k5;
      Fields[82].AsAnsiString :=ConstRef5;
      Fields[83].AsFloat :={SSA} amplitude6;
      Fields[84].AsFloat :=k6;
      Fields[85].AsAnsiString :=ConstRef6;
      Fields[86].AsFloat :={SA} amplitude7;
      Fields[87].AsFloat :=k7;
      Fields[88].AsAnsiString :=ConstRef7;
      Fields[89].AsFloat :={P1} amplitude8;
      Fields[90].AsFloat := k8;
      Fields[91].AsAnsiString:= ConstRef8;
      Fields[92].AsFloat := Min_Vol_Frac;
      Fields[93].AsAnsiString:= XMin_Vol_Frac;

      Fields[94].AsFloat := WaterShedArea;
      Fields[95].AsAnsiString := XWaterShedArea;
      Fields[96].AsBoolean := EnterTotalLength;
      Fields[97].AsFloat := TotalLength;
      Fields[98].AsAnsiString := XTotalLength;

      Fields[99].AsFloat := ECoeffSED ;
      Fields[100].AsAnsiString := XECoeffSED;
      Fields[101].AsFloat := ECoeffDOM ;
      Fields[102].AsAnsiString := XECoeffDOM;
      Fields[103].AsFloat := ECoeffPOM ;
      Fields[104].AsAnsiString := XECoeffPOM;

      Fields[105].AsBoolean :=UseCovar;
      Fields[106].AsFloat   :=EnteredKReaer;
      Fields[107].AsAnsiString  :=XEnteredKReaer;
      Fields[108].AsBoolean := UsePhytoRetention;

      Fields[109].AsFloat   := BasePercentEmbed;
      Fields[110].AsAnsiString  := XBasePercentEmbed ;

      Fields[111].AsFloat   := Altitude;
      Fields[112].AsAnsiString  := XAltitude ;

      Fields[113].AsFloat := FractalD;
      Fields[114].AsString := XFractalD;
      Fields[115].AsFloat := FD_Refuge_Coeff;
      Fields[116].AsString := XFD_Refuge_Coeff;
      Fields[117].AsFloat := HalfSatOysterRefuge;
      Fields[118].AsString := XHalfSatOysterRefuge;
    end; {with SR}
 Post;
 end; {with SV_IO_Form.SV_IO_Table}

   ResetFile;
   Except

   SiteRecord_to_DBase2:=False;
   ResetFile;
   End; {try Except}
End;    {SiteRecord_to_Dbase}
{-----------------------------------------------------------------}

Procedure SiteRec_To_Text;
Begin
 With SR do
  Begin
     Writeln(LF,'SiteName: '+SiteName);
     Writeln(LF,'ECoeffWater: '+FloatToStrF(ECoeffWater,ffgeneral,5,5));
     Writeln(LF,'SiteLength: '+FloatToStrF(SiteLength,ffgeneral,5,5));
     Writeln(LF,'XLength: '+XLength);
     Writeln(LF,'StaticVolume: '+FloatToStrF(StaticVolume,ffgeneral,5,5));
     Writeln(LF,'XVolume: '+XVolume);
     Writeln(LF,'SurfArea: '+FloatToStrF(SurfArea,ffgeneral,5,5)+' -- '+XSurfArea);
     Writeln(LF,'ICZMean: '+FloatToStrF(ICZMean,ffgeneral,5,5));
     Writeln(LF,'XZMean: '+XZMean);
     Writeln(LF,'ZMax: '+FloatToStrF(ZMax,ffgeneral,5,5)+' -- '+XZMax);
     Writeln(LF,'TempMean[Epilimnion]: '+FloatToStrF(TempMean[Epilimnion],ffgeneral,5,5)+' -- '+XTempMean[Epilimnion]);
     Writeln(LF,'TempRange[Epilimnion]: '+FloatToStrF(TempRange[Epilimnion],ffgeneral,5,5)+' -- '+XTempRange[Epilimnion]);
     Writeln(LF,'TempMean[Hypolimnion]: '+FloatToStrF(TempMean[Hypolimnion],ffgeneral,5,5)+' -- '+XTempMean[Hypolimnion]);
     Writeln(LF,'TempRange[Hypolimnion]: '+FloatToStrF(TempRange[Hypolimnion],ffgeneral,5,5)+' -- '+XTempRange[Hypolimnion]);
     Writeln(LF,'Latitude: '+FloatToStrF(Latitude,ffgeneral,5,5)+' -- '+XLatitude);
     Writeln(LF,'Altitude: '+FloatToStrF(Altitude,ffgeneral,5,5)+' -- '+XAltitude);
     Writeln(LF,'LightMean: '+FloatToStrF(LightMean,ffgeneral,5,5)+' -- '+XLightMean);
     Writeln(LF,'LightRange: '+FloatToStrF(LightRange,ffgeneral,5,5)+' -- '+XLightRange);
     Writeln(LF,'AlkCaCO3: '+FloatToStrF(AlkCaCO3,ffgeneral,5,5)+' -- '+XAlkCaCO3);
     Writeln(LF,'HardCaCO3: '+FloatToStrF(HardCaCO3,ffgeneral,5,5)+' -- '+XHardCaCO3);
     Writeln(LF,'SiteComment: '+SiteComment);
     Writeln(LF,'SO4ConC: '+FloatToStrF(SO4ConC,ffgeneral,5,5)+' -- '+XSO4ConC);
     Writeln(LF,'TotalDissSolids: '+FloatToStrF(TotalDissSolids,ffgeneral,5,5)+' -- '+XTotalDissSolids);
     Writeln(LF,'StreamType: '+StreamType);
     Writeln(LF,'Channel_Slope: '+FloatToStrF(Channel_Slope,ffgeneral,5,5)+' -- '+XChannel_Slope);
     Writeln(LF,'Max_Chan_Depth: '+FloatToStrF(Max_Chan_Depth,ffgeneral,5,5)+' -- '+XMax_Chan_Depth);
     Writeln(LF,'SedDepth: '+FloatToStrF(SedDepth,ffgeneral,5,5)+' -- '+XSedDepth);
     Writeln(LF,'EnclWallArea: '+FloatToStrF(EnclWallArea,ffgeneral,5,5)+' -- '+XEnclWallArea);
     Writeln(LF,'MeanEvap: '+FloatToStrF(MeanEvap,ffgeneral,5,5)+' -- '+XMeanEvap);
     Write(LF,'UseEnteredManning: '); If UseEnteredManning then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     Writeln(LF,'EnteredManning: '+FloatToStrF(EnteredManning,ffgeneral,5,5));
     Writeln(LF,'XECoeffWater: '+XECoeffWater);
     Writeln(LF,'PctRiffle: '+FloatToStrF(PctRiffle,ffgeneral,5,5)+' -- '+XPctRiffle);
     Writeln(LF,'PctPool: '+FloatToStrF(PctPool,ffgeneral,5,5)+' -- '+XPctPool);
     Write(LF,'UseBathymetry: '); If UseBathymetry then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
     Writeln(LF,'ts_clay: '+FloatToStrF(ts_clay,ffgeneral,5,5)+' -- '+Xts_clay);
     Writeln(LF,'ts_silt: '+FloatToStrF(ts_silt,ffgeneral,5,5)+' -- '+Xts_silt);
     Writeln(LF,'tdep_clay: '+FloatToStrF(tdep_clay,ffgeneral,5,5)+' -- '+Xtdep_clay);
     Writeln(LF,'tdep_silt: '+FloatToStrF(tdep_silt,ffgeneral,5,5)+' -- '+Xtdep_silt);
     Writeln(LF,'FallVel_clay: '+FloatToStrF(FallVel_clay,ffgeneral,5,5)+' -- '+XFallVel_clay);
     Writeln(LF,'FallVel_silt: '+FloatToStrF(FallVel_silt,ffgeneral,5,5)+' -- '+XFallVel_silt);
     Writeln(LF,'SiteWidth: '+FloatToStrF(SiteWidth,ffgeneral,5,5)+' -- '+XSiteWidth);
     Writeln(LF,'{m2}: amplitude '+FloatToStrF(amplitude1 ,ffgeneral,5,5)+'; k '+FloatToStrF( k1 ,ffgeneral,5,5));
     Writeln(LF,'{s2}: amplitude '+FloatToStrF(amplitude2 ,ffgeneral,5,5)+'; k '+FloatToStrF( k2 ,ffgeneral,5,5));
     Writeln(LF,'{n2}: amplitude '+FloatToStrF(amplitude3 ,ffgeneral,5,5)+'; k '+FloatToStrF( k3 ,ffgeneral,5,5));
     Writeln(LF,'{k1}: amplitude '+FloatToStrF(amplitude4 ,ffgeneral,5,5)+'; k '+FloatToStrF( k4 ,ffgeneral,5,5));
     Writeln(LF,'{o1}: amplitude '+FloatToStrF(amplitude5 ,ffgeneral,5,5)+'; k '+FloatToStrF( k5 ,ffgeneral,5,5));
     Writeln(LF,'{SSA}:amplitude '+FloatToStrF(amplitude6 ,ffgeneral,5,5)+'; k '+FloatToStrF( k6 ,ffgeneral,5,5));
     Writeln(LF,'{SA}: amplitude '+FloatToStrF(amplitude7 ,ffgeneral,5,5)+'; k '+FloatToStrF( k7 ,ffgeneral,5,5));
     Writeln(LF,'{P1}: amplitude '+FloatToStrF(amplitude8 ,ffgeneral,5,5)+'; k '+FloatToStrF( k8 ,ffgeneral,5,5));
     Writeln(LF,'Min_Vol_Frac: '+FloatToStrF(Min_Vol_Frac,ffgeneral,5,5)+' -- '+XMin_Vol_Frac);
     Writeln(LF,'WaterShedArea: '+FloatToStrF(WaterShedArea,ffgeneral,5,5)+' -- '+XWaterShedArea);
     If UsePhytoRetention then
       Begin
         Writeln(LF,'Use enhanced phytoplankton / zooplankton retention algorithm:');
         Write  (LF,'   EnterTotalLength: '); If EnterTotalLength then Writeln(LF,'TRUE') else Writeln(LF,'FALSE');
         Writeln(LF,'   TotalLength: '+FloatToStrF(TotalLength,ffgeneral,5,5)+' -- '+XTotalLength);
       End;

     Writeln(LF,'ECoeffSed: '+FloatToStrF(ECoeffSed,ffgeneral,5,5)+' -- '+XECoeffSed);
     Writeln(LF,'ECoeffDOM: '+FloatToStrF(ECoeffDOM,ffgeneral,5,5)+' -- '+XECoeffDOM);
     Writeln(LF,'ECoeffPOM: '+FloatToStrF(ECoeffPOM,ffgeneral,5,5)+' -- '+XECoeffPOM);

     If UseCovar then Writeln(LF,'Using default reaeration (Covar)')
                 else Writeln(LF,'Entered KReaer '+FloatToStrF(EnteredKReaer,ffgeneral,5,5)+' -- '+XEnteredKReaer);

     Writeln(LF,'Baseline Percent Embedded: '+FloatToStrF(BasePercentEmbed,ffgeneral,5,5)+' -- '+XBasePercentEmbed);

     Writeln(LF,'Fracal Dimension for site: '+FloatToStrF(FractalD,ffgeneral,5,5)+' -- '+XFractalD);
     Writeln(LF,'Fractal Dimension Refuge Coeff: '+FloatToStrF(FD_Refuge_Coeff,ffgeneral,5,5)+' -- '+XFD_Refuge_Coeff);
     Writeln(LF,'Half Sat Oyster Refuge: '+FloatToStrF(HalfSatOysterRefuge  ,ffgeneral,5,5)+' -- '+XHalfSatOysterRefuge  );

  End;
End;

{-----------------------------------------------------------------}

Function DBase_To_ReminRecord2(DBName,TName,LookupN: AnsiString; Var RR: ReminRecord):Boolean;

Begin

  Try
      DBase_To_ReminRecord2:=GetEntry(DBName,TName,LookupN,False,False);

   With SV_IO_Table do begin

   With RR do begin
      RemRecName    := Fields[0].AsAnsiString;
      DecayMax_Lab  := Fields[1].AsFloat;
      XDecayMax_Lab := Fields[2].AsAnsiString;
      Q10_NotUsed   := Fields[3].AsFloat;
      XQ10          := Fields[4].AsAnsiString;
      TOpt          := Fields[5].AsFloat;
      XTOpt         := Fields[6].AsAnsiString;
      TMax          := Fields[7].AsFloat;
      XTMax         := Fields[8].AsAnsiString;
      TRef_NotUsed  := Fields[09].AsFloat;
      XTRef         := Fields[10].AsAnsiString;
      pHMin         := Fields[11].AsFloat;
      XpHMin        := Fields[12].AsAnsiString;
      pHMax         := Fields[13].AsFloat;
      XpHMax        := Fields[14].AsAnsiString;
      P2OrgLab      := Fields[15].AsFloat;
      XP2OrgLab     := Fields[16].AsAnsiString;
      N2OrgLab      := Fields[17].AsFloat;
      XN2OrgLab     := Fields[18].AsAnsiString;
      O2Biomass     := Fields[19].AsFloat;
      XO2Biomass    := Fields[20].AsAnsiString;
      O2N           := Fields[21].AsFloat;
      XO2N          := Fields[22].AsAnsiString;
      KSed          := Fields[23].AsFloat;
      XKsed         := Fields[24].AsAnsiString;
      PSedRelease_NotUsed   := Fields[25].AsFloat;
      XPSedrelease  := Fields[26].AsAnsiString;
      NSedRelease_NotUsed   := Fields[27].AsFloat;
      XNSedRelease  := Fields[28].AsAnsiString;
      DecayMax_Refr := Fields[29].AsFloat;
      XDecayMax_Refr:= Fields[30].AsAnsiString;

      KSedTemp      := Fields[31].AsFloat;
      XKSedTemp     := Fields[32].AsAnsiString;
      KSedSalinity  := Fields[33].AsFloat;
      XKSedSalinity := Fields[34].AsAnsiString;

      P2Org_Refr     := Fields[35].AsFloat;
      XP2Org_Refr    := Fields[36].AsAnsiString;
      N2Org_Refr     := Fields[37].AsFloat;
      XN2Org_Refr    := Fields[38].AsAnsiString;
      Wet2DryPRefr  := Fields[39].AsFloat;
      XWet2DryPRefr := Fields[40].AsAnsiString;
      Wet2DryPLab   := Fields[41].AsFloat;
      Xet2DryPLab   := Fields[42].AsAnsiString;
      Wet2DrySRefr  := Fields[43].AsFloat;
      XWet2DrySRefr := Fields[44].AsAnsiString;
      Wet2DrySLab   := Fields[45].AsFloat;
      XWet2DrySLab  := Fields[46].AsAnsiString;
      N2OrgDissLab :=  Fields[47].AsFloat;
      XN2OrgDissLab := Fields[48].AsAnsiString;
      P2OrgDissLab  := Fields[49].AsFloat;
      XP2OrgDissLab := Fields[50].AsAnsiString;
      N2OrgDissRefr := Fields[51].AsFloat;
      XN2OrgDissRefr:= Fields[52].AsAnsiString;
      P2OrgDissRefr := Fields[53].AsFloat;
      XP2OrgDissRefr:= Fields[54].AsAnsiString;
      KD_P_Calcite  := Fields[55].AsFloat;
      XKD_P_Calcite := Fields[56].AsAnsiString;

      NotUsed  := Fields[57].AsFloat;
      XNotUsed := Fields[58].AsAnsiString;

      KNitri        := Fields[59].AsFloat;
      XKNitri       := Fields[60].AsAnsiString;
      KDenitri_Bot  := Fields[61].AsFloat;
      XKDenitri_Bot := Fields[62].AsAnsiString;
      KDenitri_Wat  := Fields[63].AsFloat;
      XKDenitri_Wat := Fields[64].AsAnsiString;

 end; {with RR}
 end; {with SV_IO_Form.SV_IO_Table}


   ResetFile;
   Except

   DBase_To_ReminRecord2:=False;
   ResetFile;
   End; {try Except}
End;    {DBase_To_ReminRecord}
{-----------------------------------------------------------------}

Function ReminRecord_To_DBase2(DBName,TName,LookupN: AnsiString; RR: ReminRecord; Prompt:Boolean):Boolean;

Begin

   Try
      ReminRecord_to_DBase2:=GetEntry(DBName,TName,LookupN,True,Prompt);
      if Not Result then Exit;

   SV_IO_Table.Active := True;
   SV_IO_Table.Edit;
   With SV_IO_Table do Begin

   With RR do begin
      Fields[0].AsAnsiString  := RemRecName;
      Fields[1].AsFloat   := DecayMax_Lab;
      Fields[2].AsAnsiString  := XDecayMax_Lab;
      Fields[3].AsFloat   := Q10_NotUsed;
      Fields[4].AsAnsiString  := XQ10;
      Fields[5].AsFloat   := TOpt;
      Fields[6].AsAnsiString  := XTOpt;
      Fields[7].AsFloat   := TMax;
      Fields[8].AsAnsiString  := XTMax;
      Fields[09].AsFloat  := TRef_NotUsed;
      Fields[10].AsAnsiString := XTRef;
      Fields[11].AsFloat  := pHMin;
      Fields[12].AsAnsiString := XpHMin;
      Fields[13].AsFloat  := pHMax;
      Fields[14].AsAnsiString := XpHMax;
      Fields[15].AsFloat  := P2OrgLab;
      Fields[16].AsAnsiString := XP2OrgLab;
      Fields[17].AsFloat  := N2OrgLab;
      Fields[18].AsAnsiString := XN2OrgLab;
      Fields[19].AsFloat  := O2Biomass;
      Fields[20].AsAnsiString := XO2Biomass;
      Fields[21].AsFloat  := O2N;
      Fields[22].AsAnsiString := XO2N;
      Fields[23].AsFloat  := KSed;
      Fields[24].AsAnsiString := XKsed;
      Fields[25].AsFloat  := PSedRelease_NotUsed;
      Fields[26].AsAnsiString := XPSedrelease;
      Fields[27].AsFloat  := NSedRelease_NotUsed;
      Fields[28].AsAnsiString := XNSedRelease;
      Fields[29].AsFloat  := DecayMax_Refr;
      Fields[30].AsAnsiString := XDecayMax_Refr;

      Fields[31].AsFloat  :=KSedTemp;
      Fields[32].AsAnsiString :=XKSedTemp;
      Fields[33].AsFloat  :=KSedSalinity;
      Fields[34].AsAnsiString :=XKSedSalinity;

      Fields[35].AsFloat  := P2Org_Refr;
      Fields[36].AsAnsiString := XP2Org_Refr;
      Fields[37].AsFloat  := N2Org_Refr;
      Fields[38].AsAnsiString := XN2Org_Refr;
      Fields[39].AsFloat  :=Wet2DryPRefr;
      Fields[40].AsAnsiString :=XWet2DryPRefr;
      Fields[41].AsFloat  :=Wet2DryPLab;
      Fields[42].AsAnsiString :=Xet2DryPLab;
      Fields[43].AsFloat  :=Wet2DrySRefr;
      Fields[44].AsAnsiString :=XWet2DrySRefr;
      Fields[45].AsFloat  :=Wet2DrySLab;
      Fields[46].AsAnsiString :=XWet2DrySLab;
      Fields[47].AsFloat :=N2OrgDissLab;
      Fields[48].AsAnsiString :=XN2OrgDissLab;
      Fields[49].AsFloat :=P2OrgDissLab;
      Fields[50].AsAnsiString :=XP2OrgDissLab;
      Fields[51].AsFloat :=N2OrgDissRefr;
      Fields[52].AsAnsiString :=XN2OrgDissRefr;
      Fields[53].AsFloat :=P2OrgDissRefr;
      Fields[54].AsAnsiString :=XP2OrgDissRefr;
      Fields[55].AsFloat := KD_P_Calcite;
      Fields[56].AsAnsiString :=XKD_P_Calcite;

      Fields[57].AsFloat  := NotUsed ;
      Fields[58].AsAnsiString := XNotUsed;

      Fields[59].AsFloat  := KNitri    ;
      Fields[60].AsAnsiString := XKNitri   ;
      Fields[61].AsFloat  := KDenitri_Bot;
      Fields[62].AsAnsiString := XKDenitri_Bot;
      Fields[63].AsFloat  := KDenitri_Wat;
      Fields[64].AsAnsiString := XKDenitri_Wat;

 end; {with RR}
  Post;
 end; {with SV_IO_Form.SV_IO_Table}


   ResetFile;
   Except

   ReminRecord_to_DBase2:=False;
   ResetFile;
   End; {try Except}
End;    {ReminRecord_to_Dbase}
{-----------------------------------------------------------------}
Procedure ReminRec_To_Text    (Var LF: TextFile; RR: ReminRecord);
Begin
 With RR do
  Begin
     Writeln(LF,'RemRecName: '+RemRecName);
     Writeln(LF,'DecayMax_Lab: '+FloatToStrF(DecayMax_Lab,ffgeneral,5,5)+' -- '+XDecayMax_Lab);
     Writeln(LF,'TOpt: '+FloatToStrF(TOpt,ffgeneral,5,5)+' -- '+XTOpt);
     Writeln(LF,'TMax: '+FloatToStrF(TMax,ffgeneral,5,5)+' -- '+XTMax);
     Writeln(LF,'pHMin: '+FloatToStrF(pHMin,ffgeneral,5,5)+' -- '+XpHMin);
     Writeln(LF,'pHMax: '+FloatToStrF(pHMax,ffgeneral,5,5)+' -- '+XpHMax);
     Writeln(LF,'P2OrgLab: '+FloatToStrF(P2OrgLab,ffgeneral,5,5));
     Writeln(LF,'N2OrgLab: '+FloatToStrF(N2OrgLab,ffgeneral,5,5));
     Writeln(LF,'PlaceHolder: '+FloatToStrF(PlaceHolder,ffgeneral,5,5));
     Writeln(LF,'XP2OrgLab: '+XP2OrgLab);
     Writeln(LF,'XN2OrgLab: '+XN2OrgLab);
     Writeln(LF,'O2Biomass: '+FloatToStrF(O2Biomass,ffgeneral,5,5)+' -- '+XO2Biomass);
     Writeln(LF,'O2N: '+FloatToStrF(O2N,ffgeneral,5,5)+' -- '+XO2N);
     Writeln(LF,'KSed: '+FloatToStrF(KSed,ffgeneral,5,5)+' -- '+XKSed);
     Writeln(LF,'DecayMax_Refr: '+FloatToStrF(DecayMax_Refr,ffgeneral,5,5)+' -- '+XDecayMax_Refr);
     Writeln(LF,'KSedTemp: '+FloatToStrF(KSedTemp,ffgeneral,5,5)+' -- '+XKSedTemp);
     Writeln(LF,'KSedSalinity: '+FloatToStrF(KSedSalinity,ffgeneral,5,5)+' -- '+XKSedSalinity);
     Writeln(LF,'P2Org_Refr: '+FloatToStrF(P2Org_Refr,ffgeneral,5,5)+' -- '+XP2Org_Refr);
     Writeln(LF,'N2Org_Refr: '+FloatToStrF(N2Org_Refr,ffgeneral,5,5)+' -- '+XN2Org_Refr);
     Writeln(LF,'Wet2DryPRefr: '+FloatToStrF(Wet2DryPRefr,ffgeneral,5,5)+' -- '+XWet2DryPRefr);
     Writeln(LF,'Wet2DryPLab: '+FloatToStrF(Wet2DryPLab,ffgeneral,5,5));
     Writeln(LF,'Xet2DryPLab: '+Xet2DryPLab);
     Writeln(LF,'Wet2DrySRefr: '+FloatToStrF(Wet2DrySRefr,ffgeneral,5,5)+' -- '+XWet2DrySRefr);
     Writeln(LF,'Wet2DrySLab: '+FloatToStrF(Wet2DrySLab,ffgeneral,5,5)+' -- '+XWet2DrySLab);
     Writeln(LF,'N2OrgDissLab: '+FloatToStrF(N2OrgDissLab,ffgeneral,5,5)+' -- '+XN2OrgDissLab);
     Writeln(LF,'P2OrgDissLab: '+FloatToStrF(P2OrgDissLab,ffgeneral,5,5)+' -- '+XP2OrgDissLab);
     Writeln(LF,'N2OrgDissRefr: '+FloatToStrF(N2OrgDissRefr,ffgeneral,5,5)+' -- '+XN2OrgDissRefr);
     Writeln(LF,'P2OrgDissRefr: '+FloatToStrF(P2OrgDissRefr,ffgeneral,5,5)+' -- '+XP2OrgDissRefr);
     Writeln(LF,'KD_P_Calcite: '+FloatToStrF(KD_P_Calcite,ffgeneral,5,5)+' -- '+XKD_P_Calcite);
     Writeln(LF,'BOD5_CBODu Conv. Factor no longer used');
     Writeln(LF,'KNitri: '+FloatToStrF(KNitri,ffgeneral,5,5)+' -- '+XKNitri);
     Writeln(LF,'KDenitri_Bot: '+FloatToStrF(KDenitri_Bot,ffgeneral,5,5)+' -- '+XKDenitri_Bot);
     Writeln(LF,'KDenitri_Wat: '+FloatToStrF(KDenitri_Wat,ffgeneral,5,5)+' -- '+XKDenitri_Wat);
  End;
End;
{-----------------------------------------------------------------}


Function DBase_To_PlantRecord2(DBName,TName,LookupN: AnsiString; Num: Integer; Var PR: PlantRecord):Boolean;

Begin

   Try
   If LookupN = ''
      then DBase_To_PlantRecord2:=GetEntryByNum(DbName,TName,Num)
      else DBase_To_PlantRecord2:=GetEntry(DBName,TName,LookupN,False,False);


   With SV_IO_Table do begin
     With PR do begin
      PlantName           :=FieldbyName('PlantName').AsAnsiString;
      EnteredLightSat     :=Fields[1].AsFloat;
      XLightSat           :=Fields[2].AsAnsiString;
      KPO4                :=Fields[3].AsFloat;
      XKPO4               :=Fields[4].AsAnsiString;
      KN                  :=Fields[5].AsFloat;  {n half Sat}
      XKN                 :=Fields[6].AsAnsiString;
      KCarbon             :=Fields[7].AsFloat;
      XKCarbon            :=Fields[8].AsAnsiString;
      Q10                 :=Fields[9].AsFloat;
      XQ10                :=Fields[10].AsAnsiString;
      TOpt                :=Fields[11].AsFloat;
      XTOpt               :=Fields[12].AsAnsiString;
      TMax                :=Fields[13].AsFloat;
      XTMax               :=Fields[14].AsAnsiString;
      TRef                :=Fields[15].AsFloat;
      XTRef               :=Fields[16].AsAnsiString;
      PMax                :=Fields[17].AsFloat;
      XPMax               :=Fields[18].AsAnsiString;
      KResp               :=Fields[19].AsFloat;
      XKResp              :=Fields[20].AsAnsiString;
      KMort               :=Fields[21].AsFloat;
      XKMort              :=Fields[22].AsAnsiString;
      EMort               :=Fields[23].AsFloat;
      XEMort              :=Fields[24].AsAnsiString;
      KSed                :=Fields[25].AsFloat;
      XKSed               :=Fields[26].AsAnsiString;
      ESed                :=Fields[27].AsFloat;
      XESed               :=Fields[28].AsAnsiString;
      P2OrgInit           :=Fields[29].AsFloat;
      XP2Org              :=Fields[30].AsAnsiString;
      N2OrgInit           :=Fields[31].AsFloat;
      XN2Org              :=Fields[32].AsAnsiString;
      ECoeffPhyto         :=Fields[33].AsFloat;
      XECoeffPhyto        :=Fields[34].AsAnsiString;
      CarryCapac          :=Fields[35].AsFloat;
      XCarryCapac         :=Fields[36].AsAnsiString;
      Red_Still_Water     :=Fields[37].AsFloat;
      XRed_Still_Water    :=Fields[38].AsAnsiString;
      PlantType           :=Fields[39].AsAnsiString;
      ToxicityRecord      :=Fields[40].AsAnsiString;
      Macrophyte_Type     :=Fields[41].AsAnsiString;
      Macro_Drift         :=Fields[42].AsFloat;
      Taxonomic_Type      :=Fields[43].AsAnsiString;

      PrefRiffle          := Fields[44].AsFloat;
      XPrefRiffle         := Fields[45].AsAnsiString;
      PrefPool            := Fields[46].AsFloat;
      XPrefPool           := Fields[47].AsAnsiString;
      FCrit               := Fields[48].AsFloat;
      XFCrit              := Fields[49].AsAnsiString;
      Macro_VelMax        := Fields[50].AsFloat;
      XVelMax             := Fields[51].AsAnsiString;
      KSedTemp            := Fields[52].AsFloat;
      XKSedTemp           := Fields[53].AsAnsiString;
      KSedSalinity        := Fields[54].AsFloat;
      XKSedSalinity       := Fields[55].AsAnsiString;
      Salmin_Phot         := Fields[56].AsFloat;
      SalMax_Phot         := Fields[57].AsFloat;
      Salcoeff1_Phot      := Fields[58].AsFloat;
      Salcoeff2_Phot      := Fields[59].AsFloat;
      XSalinity_Phot      := Fields[60].AsAnsiString;
      Salmin_Mort         := Fields[61].AsFloat;
      SalMax_Mort         := Fields[62].AsFloat;
      Salcoeff1_Mort      := Fields[63].AsFloat;
      Salcoeff2_Mort      := Fields[64].AsFloat;
      XSalinity_Mort      := Fields[65].AsAnsiString;

      Wet2Dry             :=Fields[66].AsFloat;
      XWet2Dry            :=Fields[67].AsAnsiString;
      Resp20              :=Fields[68].AsFloat;
      XResp20             :=Fields[69].AsAnsiString;
      PctSloughed         :=Fields[70].AsFloat;
      XPctSloughed        :=Fields[71].AsAnsiString;
      UseAdaptiveLight    :=Fields[72].AsBoolean;
      MaxLightSat         :=Fields[73].AsFloat;
      XMaxLightSat        :=Fields[74].AsAnsiString;
      MinLightSat         :=Fields[75].AsFloat;
      XMinLightSat        :=Fields[76].AsAnsiString;
      ScientificName      :=Fields[77].AsAnsiString;
      PlantFracLipid      := Fields[78].AsFloat ;
      XPlantFracLipid     := Fields[79].AsAnsiString ;

      SurfaceFloating     := Fields[80].AsBoolean ;

      NHalfSatInternal    := Fields[81].AsFloat ;
      XNHalfSatInternal   := Fields[82].AsAnsiString ;
      PHalfSatInternal    := Fields[83].AsFloat ;
      XPHalfSatInternal   := Fields[84].AsAnsiString ;
      MaxNUptake          := Fields[85].AsFloat ;
      XMaxNUptake         := Fields[86].AsAnsiString ;
      MaxPUptake          := Fields[87].AsFloat ;
      XMaxPUptake         := Fields[88].AsAnsiString ;
      Min_N_Ratio         := Fields[89].AsFloat ;
      XMin_N_Ratio        := Fields[90].AsAnsiString ;
      Min_P_Ratio         := Fields[91].AsFloat ;
      XMin_P_Ratio        := Fields[92].AsAnsiString ;

      Plant_to_Chla       := Fields[93].AsFloat;
      XPlant_to_Chla      := Fields[94].AsString;

 end; {with PR}
 end; {with SV_IO_Form.SV_IO_Table}

   ResetFile;
   Except

   DBase_To_PlantRecord2:=False;
   ResetFile;
   End; {try Except}
End;    {DBase_To_PlantRecord}
{-----------------------------------------------------------------}

Function PlantRecord_To_DBase2(DBName,TName,LookupN: AnsiString; PR: PlantRecord; Prompt: Boolean):Boolean;

Begin

   Try
      PlantRecord_to_DBase2:=GetEntry(DBName,TName,LookupN,True,Prompt);
      if Not Result then Exit;

   SV_IO_Table.Active := True;
   SV_IO_Table.Edit;
   With SV_IO_Table do Begin

   With PR do
    begin
      FieldbyName('PlantName').AsAnsiString:=PlantName;
      Fields[1].AsFloat:=EnteredLightSat;
      Fields[2].AsAnsiString:=XLightSat;
      Fields[3].AsFloat:=KPO4;
      Fields[4].AsAnsiString:=XKPO4;
      Fields[5].AsFloat:=KN;
      Fields[6].AsAnsiString:=XKN;
      Fields[7].AsFloat:=KCarbon;
      Fields[8].AsAnsiString:=XKCarbon;
      Fields[9].AsFloat:=Q10;
      Fields[10].AsAnsiString:=XQ10;
      Fields[11].AsFloat:=TOpt;
      Fields[12].AsAnsiString:=XTOpt;
      Fields[13].AsFloat:=TMax;
      Fields[14].AsAnsiString:=XTMax;
      Fields[15].AsFloat:=TRef;
      Fields[16].AsAnsiString:=XTRef;
      Fields[17].AsFloat:=PMax;
      Fields[18].AsAnsiString:=XPMax;
      Fields[19].AsFloat:=KResp;
      Fields[20].AsAnsiString:=XKResp;
      Fields[21].AsFloat:= KMort;
      Fields[22].AsAnsiString:=XKMort;
      Fields[23].AsFloat:=EMort;
      Fields[24].AsAnsiString:=XEMort;
      Fields[25].AsFloat:=KSed;
      Fields[26].AsAnsiString:=XKSed;
      Fields[27].AsFloat:=ESed;
      Fields[28].AsAnsiString:=XESed;
      Fields[29].AsFloat:=P2OrgInit;
      Fields[30].AsAnsiString:=XP2Org;
      Fields[31].AsFloat:=N2OrgInit;
      Fields[32].AsAnsiString:=XN2Org;
      Fields[33].AsFloat:=ECoeffPhyto;
      Fields[34].AsAnsiString:=XECoeffPhyto;
      Fields[35].AsFloat:= CarryCapac;
      Fields[36].AsAnsiString:=XCarryCapac;
      Fields[37].AsFloat:=Red_Still_Water;
      Fields[38].AsAnsiString:=XRed_Still_Water;
      Fields[39].AsAnsiString:=PlantType;
      Fields[40].AsAnsiString:=ToxicityRecord;
      Fields[41].AsAnsiString :=Macrophyte_Type;
      Fields[42].AsFloat :=Macro_Drift;
      Fields[43].AsAnsiString :=Taxonomic_Type;

      Fields[44].AsFloat :=PrefRiffle;
      Fields[45].AsAnsiString :=XPrefRiffle;
      Fields[46].AsFloat :=PrefPool;
      Fields[47].AsAnsiString :=XPrefPool;
      Fields[48].AsFloat :=FCrit;
      Fields[49].AsAnsiString :=XFCrit;
      Fields[50].AsFloat :=Macro_VelMax;
      Fields[51].AsAnsiString :=XVelMax;
      Fields[52].AsFloat :=KSedTemp;
      Fields[53].AsAnsiString :=XKSedTemp;
      Fields[54].AsFloat :=KSedSalinity;
      Fields[55].AsAnsiString :=XKSedSalinity;
      Fields[56].AsFloat :=Salmin_Phot;
      Fields[57].AsFloat :=SalMax_Phot;
      Fields[58].AsFloat :=Salcoeff1_Phot;
      Fields[59].AsFloat :=Salcoeff2_Phot;
      Fields[60].AsAnsiString :=XSalinity_Phot;
      Fields[61].AsFloat :=Salmin_Mort;
      Fields[62].AsFloat :=SalMax_Mort;
      Fields[63].AsFloat :=Salcoeff1_Mort;
      Fields[64].AsFloat :=Salcoeff2_Mort;
      Fields[65].AsAnsiString :=XSalinity_Mort;

      Fields[66].AsFloat  := Wet2Dry;
      Fields[67].AsAnsiString := XWet2Dry;
      Fields[68].AsFloat  := Resp20;
      Fields[69].AsAnsiString := XResp20;
      Fields[70].AsFloat  := PctSloughed;
      Fields[71].AsAnsiString  := XPctSloughed;
      Fields[72].AsBoolean := UseAdaptiveLight;
      Fields[73].AsFloat  := MaxLightSat;
      Fields[74].AsAnsiString := XMaxLightSat;
      Fields[75].AsFloat  := MinLightSat;
      Fields[76].AsAnsiString := XMinLightSat;
      Fields[77].AsAnsiString := ScientificName;
      Fields[78].AsFloat   := PlantFracLipid;
      Fields[79].AsAnsiString  := XPlantFracLipid;
      Fields[80].AsBoolean := SurfaceFloating;

      Fields[81].AsFloat   := NHalfSatInternal;
      Fields[82].AsAnsiString  := XNHalfSatInternal;
      Fields[83].AsFloat   := PHalfSatInternal;
      Fields[84].AsAnsiString  := XPHalfSatInternal;
      Fields[85].AsFloat   := MaxNUptake;
      Fields[86].AsAnsiString  := XMaxNUptake;
      Fields[87].AsFloat   := MaxPUptake;
      Fields[88].AsAnsiString  := XMaxPUptake;
      Fields[89].AsFloat   := Min_N_Ratio;
      Fields[90].AsAnsiString  := XMin_N_Ratio;
      Fields[91].AsFloat   := Min_P_Ratio;
      Fields[92].AsAnsiString  := XMin_P_Ratio;

      Fields[93].AsFloat := Plant_to_Chla;
      Fields[94].AsString := XPlant_to_Chla;
    end; {with PR}
  Post;
 end;{with SV_IO_Form.SV_IO_Table}


   ResetFile;
   Except

   PlantRecord_to_DBase2:=False;
   ResetFile;
   End; {try Except}
End;    {PlantRecord_to_Dbase}

{-----------------------------------------------------------------}
Procedure PlantRec_To_Text(Var LF: TextFile; PR: PlantRecord);
Begin
 With PR do
  Begin
     Writeln(LF,'PlantName: '+PlantName);
     Writeln(LF,'ScientificName: '+ScientificName);
     Writeln(LF,'PlantType: '+PlantType);
     If SurfaceFloating then Writeln(LF,' -- Surface Floating');
     Writeln(LF,'ToxicityRecord: '+ToxicityRecord);
     Writeln(LF,'EnteredLightSat: '+FloatToStrF(EnteredLightSat,ffgeneral,5,5)+' -- '+XLightSat);
     Writeln(LF,'KPO4: '+FloatToStrF(KPO4,ffgeneral,5,5)+' -- '+XKPO4);
     Writeln(LF,'KN: '+FloatToStrF(KN,ffgeneral,5,5)+' -- '+XKN);
     Writeln(LF,'KCarbon: '+FloatToStrF(KCarbon,ffgeneral,5,5)+' -- '+XKCarbon);
     Writeln(LF,'Q10: '+FloatToStrF(Q10,ffgeneral,5,5)+' -- '+XQ10);
     Writeln(LF,'TOpt: '+FloatToStrF(TOpt,ffgeneral,5,5)+' -- '+XTOpt);
     Writeln(LF,'TMax: '+FloatToStrF(TMax,ffgeneral,5,5)+' -- '+XTMax);
     Writeln(LF,'TRef: '+FloatToStrF(TRef,ffgeneral,5,5)+' -- '+XTRef);
     Writeln(LF,'PMax: '+FloatToStrF(PMax,ffgeneral,5,5)+' -- '+XPMax);
     Writeln(LF,'KResp: '+FloatToStrF(KResp,ffgeneral,5,5)+' -- '+XKResp);
     Writeln(LF,'KMort: '+FloatToStrF(KMort,ffgeneral,5,5)+' -- '+XKMort);
     Writeln(LF,'EMort: '+FloatToStrF(EMort,ffgeneral,5,5)+' -- '+XEMort);
     Writeln(LF,'KSed: '+FloatToStrF(KSed,ffgeneral,5,5)+' -- '+XKSed);
     Writeln(LF,'ESed: '+FloatToStrF(ESed,ffgeneral,5,5)+' -- '+XESed);
     Writeln(LF,'P2OrgInit: '+FloatToStrF(P2OrgInit,ffgeneral,5,5)+' -- '+XP2Org);
     Writeln(LF,'N2OrgInit: '+FloatToStrF(N2OrgInit,ffgeneral,5,5)+' -- '+XN2Org);
     Writeln(LF,'ECoeffPhyto: '+FloatToStrF(ECoeffPhyto,ffgeneral,5,5)+' -- '+XECoeffPhyto);
     Writeln(LF,'CarryCapac: '+FloatToStrF(CarryCapac,ffgeneral,5,5)+' -- '+XCarryCapac);
     Writeln(LF,'Red_Still_Water: '+FloatToStrF(Red_Still_Water,ffgeneral,5,5)+' -- '+XRed_Still_Water);
     Writeln(LF,'Macrophyte_Type: '+Macrophyte_Type);
     Writeln(LF,'Macro_Drift: '+FloatToStrF(Macro_Drift,ffgeneral,5,5));
     Writeln(LF,'Taxonomic_Type: '+Taxonomic_Type);
     Writeln(LF,'PrefRiffle: '+FloatToStrF(PrefRiffle,ffgeneral,5,5)+' -- '+XPrefRiffle);
     Writeln(LF,'PrefPool: '+FloatToStrF(PrefPool,ffgeneral,5,5)+' -- '+XPrefPool);
     Writeln(LF,'FCrit: '+FloatToStrF(FCrit,ffgeneral,5,5)+' -- '+XFCrit);
     Writeln(LF,'Macro_VelMax: '+FloatToStrF(Macro_VelMax,ffgeneral,5,5));
     Writeln(LF,'XVelMax: '+XVelMax);
     Writeln(LF,'KSedTemp: '+FloatToStrF(KSedTemp,ffgeneral,5,5)+' -- '+XKSedTemp);
     Writeln(LF,'KSedSalinity: '+FloatToStrF(KSedSalinity,ffgeneral,5,5)+' -- '+XKSedSalinity);
     Writeln(LF,'Salcoeff2_Phot: '+FloatToStrF(Salcoeff2_Phot,ffgeneral,5,5));
     Writeln(LF,'XSalinity_Phot: '+XSalinity_Phot);
     Writeln(LF,'Salcoeff2_Mort: '+FloatToStrF(Salcoeff2_Mort,ffgeneral,5,5));
     Writeln(LF,'XSalinity_Mort: '+XSalinity_Mort);
     Writeln(LF,'Wet2Dry: '+FloatToStrF(Wet2Dry,ffgeneral,5,5)+' -- '+XWet2Dry);
     Writeln(LF,'Frac Lipid (wet): '+FloatToStrF(PlantFracLipid,ffgeneral,5,5)+' -- '+XPlantFracLipid);
     Writeln(LF,'Resp20: '+FloatToStrF(Resp20,ffgeneral,5,5)+' -- '+XResp20);
     Writeln(LF,'PctSloughed: '+FloatToStrF(PctSloughed,ffgeneral,5,5)+' -- '+XPctSloughed);
     If UseAdaptiveLight then Writeln(LF,' -- Use Adaptive Light for this Plant -- ');
     Writeln(LF,'MaxLightSat: '+FloatToStrF(MaxLightSat,ffgeneral,5,5)+' -- '+XMaxLightSat);
     Writeln(LF,'MinLightSat: '+FloatToStrF(MinLightSat,ffgeneral,5,5)+' -- '+XMinLightSat);

     Writeln(LF,'NHalfSatInternal: '+FloatToStrF(NHalfSatInternal,ffgeneral,5,5)+' -- '+XNHalfSatInternal);
     Writeln(LF,'PHalfSatInternal: '+FloatToStrF(PHalfSatInternal,ffgeneral,5,5)+' -- '+XPHalfSatInternal);
     Writeln(LF,'MaxNUptake: '+FloatToStrF(MaxNUptake,ffgeneral,5,5)+' -- '+XMaxNUptake);
     Writeln(LF,'MaxPUptake: '+FloatToStrF(MaxPUptake,ffgeneral,5,5)+' -- '+XMaxPUptake);
     Writeln(LF,'Min_N_Ratio: '+FloatToStrF(Min_N_Ratio,ffgeneral,5,5)+' -- '+XMin_N_Ratio);
     Writeln(LF,'Min_P_Ratio: '+FloatToStrF(Min_P_Ratio,ffgeneral,5,5)+' -- '+XMin_P_Ratio);

     Writeln(LF,'Plant to Chla: '+FloatToStrF(Plant_to_Chla,ffgeneral,5,5)+' -- '+XPlant_to_Chla);
  End;
End;


Begin
   SV_IO_Table := TFDTable.Create(nil);
   SV_IO_Connect := TFDConnection.Create(nil);
   SV_IO_Table.Connection := SV_IO_Connect;
   SV_IO_Connect.Params.DriverID:='SQLite';
   SV_IO_Connect.Params.Database:='ProgramDBs.sdb';
   SV_IO_Table.Connection.Params.Add('LockingMode=Normal');
   SV_IO_Table.Connection.Params.Add('SharedCache=False');
end.

