classdef class_REVS_CVM_control < class_REVS_control
    %UNTITLED Summary of this class goes here
    %   Detailed explanation goes here
    
    properties
        
        start_stop_enable = false;                                  % Enable engine start-stop operation 
        start_stop_off_delay_secs = 0.1;                            % Delay before start-stop engine shutdown
        start_stop_warmup_condition = '@engine.runtime_sec >= 100'; % Condition to enable/disable start-stop based upon model parameters
        hot_soak_warmup_start_condition = '@cycle_pos_secs > 0 ';
        
        pedal_map_type = enum_engine_pedal_map_type.lookup_table;
        pedal_map_engine_torque_Nm;
        
        shift_inertia_est_kgm2 = 0;
        
    end
    
    properties ( Hidden)        
        engine_max_torque_Nm;
        engine_min_torque_Nm;
        engine_torque_range_speed_radps;
        engine_idle_speed_radps;
    end
    
    methods
        function obj = class_REVS_CVM_control(engine)
            %UNTITLED Construct an instance of this class
            %   Detailed explanation goes here

            
            obj.pedal_map_engine_torque_Nm = class_REVS_dynamic_lookup();
            
            if nargin > 0
                [max_trq_spd_radps, max_trq_Nm] = engine.get_max_torque_curve();
                [min_trq_spd_radps, min_trq_Nm] = engine.get_min_torque_curve();

                obj.engine_torque_range_speed_radps = unique( [max_trq_spd_radps(:);min_trq_spd_radps(:)])';
                
                obj.engine_max_torque_Nm = interp1(max_trq_spd_radps, max_trq_Nm, obj.engine_torque_range_speed_radps, 'linear','extrap');
                obj.engine_min_torque_Nm = interp1(min_trq_spd_radps, min_trq_Nm, obj.engine_torque_range_speed_radps, 'linear','extrap');
                
                obj.engine_max_torque_Nm = max(obj.engine_max_torque_Nm, obj.engine_min_torque_Nm);
                                
                obj.engine_idle_speed_radps = engine.nominal_idle_speed_radps;
            end
            
        end
        
        function obj = update_powertrain_info(obj, engine, transmission, accessories)
                [max_trq_spd_radps, max_trq_Nm] = engine.get_max_torque_curve();
                [min_trq_spd_radps, min_trq_Nm] = engine.get_min_torque_curve();

                obj.engine_torque_range_speed_radps = unique( [max_trq_spd_radps(:);min_trq_spd_radps(:)])';
                
                obj.engine_max_torque_Nm = interp1(max_trq_spd_radps, max_trq_Nm, obj.engine_torque_range_speed_radps, 'linear','extrap');
                obj.engine_min_torque_Nm = interp1(min_trq_spd_radps, min_trq_Nm, obj.engine_torque_range_speed_radps, 'linear','extrap');
                
                obj.engine_max_torque_Nm = max(obj.engine_max_torque_Nm, obj.engine_min_torque_Nm);
                                
                obj.engine_idle_speed_radps = engine.nominal_idle_speed_radps;
                
                obj.shift_inertia_est_kgm2 =	engine.inertia_kgm2 + ...
                                    accessories.air_conditioner.inertia_kgm2 + ...
                                    accessories.fan.inertia_kgm2 + ...
                                    accessories.power_steering.inertia_kgm2 + ...
                                    accessories.generic_loss.inertia_kgm2;

                if isa(transmission, 'class_REVS_AT_transmission')								
                    obj.shift_inertia_est_kgm2 = obj.shift_inertia_est_kgm2 + transmission.torque_converter.pump_inertia_kgm2;
                end

        end
               
        function val = get.pedal_map_engine_torque_Nm(obj)
            
            if  obj.pedal_map_type == enum_engine_pedal_map_type.lookup_table
               
                val = obj.pedal_map_engine_torque_Nm;
                return
                
            end
            
            if  isempty( obj.engine_torque_range_speed_radps) || isempty( obj.engine_max_torque_Nm) || isempty( obj.engine_min_torque_Nm)                
                error('Misssing engine torque information, specify engine when constructing class')
            end    

            % Slightly less than lower torque curve to ensure 
            pedal_min_torque_Nm = obj.engine_min_torque_Nm*1.01;
            
           switch( obj.pedal_map_type)

               case  enum_engine_pedal_map_type.linear_WOT
                   pedal_max_torque_Nm = obj.engine_max_torque_Nm;

                case enum_engine_pedal_map_type.linear_max_engine_torque
                    pedal_max_torque_Nm = max(obj.engine_max_torque_Nm) * ones( size(obj.engine_max_torque_Nm));

                case enum_engine_pedal_map_type.max_engine_power
                    max_power_W =  max( obj.engine_max_torque_Nm .* obj.engine_torque_range_speed_radps);
                    pedal_max_torque_Nm =  max_power_W ./ max(obj.engine_torque_range_speed_radps, obj.engine_idle_speed_radps);

                case enum_engine_pedal_map_type.linear_enhanced_WOT
                    pedal_max_torque_Nm = 1.1 * obj.engine_max_torque_Nm;

                case enum_engine_pedal_map_type.linear_enhanced_max_engine_torque
                    pedal_max_torque_Nm = 1.1* max(obj.engine_max_torque_Nm) * ones( size(obj.engine_max_torque_Nm));

                otherwise
                    error('Unknown pedal map type : %s', obj.pedal_map_type);
           end
            
            val = class_REVS_dynamic_lookup();
            val.axis_1.signal = 'driver_acl_norm';
            val.axis_1.breakpoints = [0,1];
            val.axis_2.signal = 'eng_spd_radps';
            val.axis_2.breakpoints = obj.engine_torque_range_speed_radps;
            val.table = [pedal_min_torque_Nm;pedal_max_torque_Nm];
               
        end
            
    end
end

