classdef class_REVS_CVM_result < class_REVS_VM_result
	%class_REVS_VM_result
	%   Definition of class_REVS_VM_result class
        
    properties ( Dependent= true, Transient = true )
        
        fuel_consumed_g;                    % Fuel mass consumed [g]
        fuel_consumed_vol_gallons;          % Fuel volume consumed [gallons]
        fuel_consumed_CAFE_gallons;         % Fuel volume consumed in CAFE equivalent [gallons]
        fuel_consumption_gpmi;
        
        gCO2;                               % CO2 emissions [g]
        gCO2pmi;                            % CO2 emissions [g/mi]
        
        FE_vol_mpg;                         % Volumetric Fuel Economy [mpg]
        FE_CAFE_mpg;                        % CAFE Fuel Economy [mpg]
        
        fuel_Jpm;                           % Fuel Energy Consumption [J/m]
        engine_Jpm;                         % Engine Output Energy [J/m]
                                    
        engine_efficiency_norm;             % Engine Efficiency
        transmission_efficiency_norm ;      % Transmission Efficiency
        total_efficiency_norm;              % Total Vehicle Efficiency 
    end
     
    
    methods
        %% Constructor
        function obj = class_REVS_CVM_result( root, cycle_or_phase_name, varargin )
            
            obj@class_REVS_VM_result( root, cycle_or_phase_name, varargin{:});
            
        end
        
        %% Print Result - to screen or file
        function print( obj, fid )
            
            if nargin < 2 || isempty(fid)
                fid = 1;
            end
            
            for p = 1:length(obj.fuel_consumed_g)
                
                if isprop(obj, 'cycle_name')
                    name = obj.cycle_name;
                elseif isprop(obj, 'phase_name')
                    name = obj.phase_name(p);
                else
                    name = '';
                end
                
                fprintf(fid,'   %s: %s\n',                                          name, repmat('-', 1,40-length(name)));
                if isprop(obj, 'driver') && isprop(obj.driver, 'error_time_secs') && ~contains(name,'EPA_combined')
                    fprintf(fid,'   Percent Time Missed by 2mph = %6.2f %%\n',          obj.driver.error_time_secs(p) ./ obj.time_secs(p) * 100);
                end
                fprintf(fid,'       Distance                    = %6.3f mi\n',			obj.distance_mi(p));
                fprintf(fid,'       Fuel Consumption            = %6.4f grams\n',		obj.fuel_consumed_g(p));
                fprintf(fid,'       Fuel Consumption            = %6.4f gallons\n',		obj.fuel_consumed_vol_gallons(p));
                fprintf(fid,'       Fuel Economy (Volumetric)   = %6.3f mpg\n',			obj.FE_vol_mpg(p));
                fprintf(fid,'       Fuel Economy (CAFE)         = %6.3f mpg\n',			obj.FE_CAFE_mpg(p));
                fprintf(fid,'       Fuel Consumption            = %6.3f g/mile\n',		obj.fuel_consumption_gpmi(p));
                if ~isempty(obj.gCO2pmi)
                    fprintf(fid,'       CO2 Emission                = %6.2f g/mile\n',		obj.gCO2pmi(p));
                end
                fprintf(fid,'\n');
            end
        end
        

        %% Fuel
        function val = get.fuel_consumed_g(obj)            
            if ~isempty(obj.aggregator)
                val = obj.aggregator.sum('fuel_consumed_g');
            elseif isprop(obj, 'engine') && isprop(obj.engine, 'fuel_consumed_g')
                val = obj.engine.fuel_consumed_g .* obj.map_fuel.energy_density_MJpkg ./ obj.output_fuel.energy_density_MJpkg; % Conversion @ R = 1
            else
                val = nan(size(obj.distance_m));
            end
        end
        
        function set.fuel_consumed_g(obj, val)
            if ~isempty(obj.aggregator)
                 error('Fuel properties not modifyable for aggregated results.')
            else
                obj.engine.fuel_consumed_g = val .* obj.output_fuel.energy_density_MJpkg ./ obj.map_fuel.energy_density_MJpkg;   % for setting cold-corrected values based on measured values...
            end
        end
        
        function val = get.fuel_consumed_vol_gallons(obj)
            val = obj.fuel_consumed_g ./ 1000 ./ obj.output_fuel.density_kgpL_15C .* unit_convert.lit2gal;
        end
        
        function val = get.fuel_consumed_CAFE_gallons(obj)
            val = obj.distance_mi ./ obj.FE_CAFE_mpg;
        end
        
        function val = get.fuel_consumption_gpmi(obj)
           val =   1 ./ obj.FE_vol_mpg .* unit_convert.gal2lit .* obj.output_fuel.density_kgpL_15C .* 1000;
        end
        
        %% Fuel Economy
        function val = get.FE_vol_mpg(obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.dist_per_quant('FE_vol_mpg');
            else            
                val = obj.distance_mi ./ obj.fuel_consumed_vol_gallons;
            end
        end
        
        function val = get.FE_CAFE_mpg(obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.dist_per_quant('FE_CAFE_mpg');
            else            
                val = obj.distance_mi .* 5174e4 .* obj.output_fuel.specific_gravity  ./ ( obj.fuel_consumed_g .* ( 0.6 * obj.output_fuel.specific_gravity * obj.output_fuel.energy_density_MJpkg * unit_convert.MJpkg2BTUplbm + 5471));
            end            
        end
        
        %% CO2
        function val = get.gCO2(obj)
            val = obj.fuel_consumed_vol_gallons .* obj.output_fuel.gCO2pgal; % was obj.fuel_consumed_g .* obj.output_fuel.carbon_weight_fraction * (44.0095/12.0107);
        end
        
        function val = get.gCO2pmi(obj)
            val     = obj.output_fuel.gCO2pgal ./ obj.FE_vol_mpg;
        end
        
        %% Energy Rate Calcs
        function val = get.fuel_Jpm(obj)
            val = 1./(obj.FE_CAFE_mpg * unit_convert.mi2mtr * unit_convert.lit2gal / obj.output_fuel.density_kgpL_15C / obj.output_fuel.energy_density_MJpkg / 1e6);
        end
                
        function val = get.engine_Jpm  (obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.quant_per_dist('engine_Jpm');
            else
                val = obj.engine.crankshaft_pos_kJ * 1000 ./  obj.distance_m;
            end
        end
              
        
        %% Efficiency Metrics
        function val = get.engine_efficiency_norm(obj)
               val = obj.engine_Jpm ./ obj.fuel_Jpm;
        end
        
        function val = get.transmission_efficiency_norm(obj)
                val = obj.roadload_Jpm ./ obj.engine_Jpm;
        end
        
        function val = get.total_efficiency_norm(obj)     
                val = obj.unadjusted_roadload_Jpm ./ obj.fuel_Jpm ;
        end
        
        
    end
    
    methods ( Access = protected)
		
        function val = map_fuel(obj)
            val = obj.root.map_fuel;
        end
        
         function val = output_fuel(obj)
            val = obj.root.map_fuel;
        end
        
		%Select properties for display
		function grp = getPropertyGroups(obj)		

            grp = matlab.mixin.util.PropertyGroup({'cycle_name','phase_name'});                                   
            grp(end+1) = matlab.mixin.util.PropertyGroup({'fuel_consumed_g', 'fuel_consumed_vol_gallons', 'fuel_consumed_CAFE_gallons','fuel_consumption_gpmi','FE_vol_mpg', 'FE_CAFE_mpg','gCO2', 'gCO2pmi'}, 'Fuel Consumption & CO2 Emissions');
            grp(end+1) = matlab.mixin.util.PropertyGroup({'fuel_Jpm','engine_Jpm','roadload_Jpm','unadjusted_roadload_Jpm','engine_efficiency_norm','transmission_efficiency_norm','total_efficiency_norm'}, 'Energy Rates & Efficiency');
            grp(end+1) = matlab.mixin.util.PropertyGroup({'drive_quality','drive_quality_unadjusted','avg_speed_mps', 'avg_speed_mph', 'shifts_per_mi'}, 'Drive Metrics');                       
            grp(end+1) = matlab.mixin.util.PropertyGroup({'time_secs', 'distance_m', 'distance_mi'}, 'Simulation Data');  
            
            props = properties(obj);
            for g = 1:length( grp)
                props = setdiff(props, grp(g).PropertyList);
            end
            		
            grp(end).PropertyList = [grp(end).PropertyList, props'];
            		
        end
		
	end
    
end
