classdef class_REVS_DCT_transmission
	%class_REVS_DCT_transmission
	%   Definition of DCT transmission class

    properties ( Constant = true )
		type = enum_transmission_type.DCT;
    end

	properties
        variant = 'dct transmission system';

	    matrix_vintage = enum_matrix_vintage.present;
        
		name = '';
		
        rated_torque_Nm = 0;                                            % transmission rated input torque in Newton-meters

		% Base Properties
		dual_clutch = class_REVS_dual_clutch;							% Dual Clutch Module Definition
		gear;															% Generic Gearbox Definition - Common accross transmissions
		gear1 = class_REVS_gearbox;										% Odd Gearbox Details
		gear2 = class_REVS_gearbox;										% Even Gearbox Details
		control = class_REVS_DCT_control;								% DCT Control Parameters
		thermal ;														% Thermal Definintion
		
		gear_strategy;													% Gear selection strategy structure
				
	end
	
	methods
		
		% Constructor
		function obj = class_REVS_DCT_transmission(engine)
			
			obj.gear = struct('number',[],'ratio',[],'inertia_kgm2',[],'shift_duration_secs',0.3,'efficiency_norm',[],'common_input_inertia_kgm2',[],'common_output_inertia_kgm2',[]);
			obj.gear1.type = enum_transmission_type.DCT;
			obj.gear2.type = enum_transmission_type.DCT;
			
			% Default lossless gearbox 1
			obj.gear1.input_torque_loss_Nm.axis_1.signal = 'powertrain.trans.core.gearbox1.gb_spd_in_radps';
			obj.gear1.input_torque_loss_Nm.axis_1.breakpoints = [0,1e6];
			obj.gear1.input_torque_loss_Nm.table = [0,0];
			
			obj.gear1.output_torque_loss_Nm.axis_1.signal = 'powertrain.trans.core.gearbox1.gb_spd_out_radps';
			obj.gear1.output_torque_loss_Nm.axis_1.breakpoints = [0,1e6];
			obj.gear1.output_torque_loss_Nm.table = [0,0];
			
			obj.gear1.efficiency_norm.axis_1.signal = 'powertrain.trans.core.gearbox1.gb_cmd_gear';
			obj.gear1.efficiency_norm.axis_1.breakpoints = [0,1,100];
			obj.gear1.efficiency_norm.table = [0,1,1];
			
			% Default lossless gearbox 2			
			obj.gear2.input_torque_loss_Nm.axis_1.signal = 'powertrain.trans.core.gearbox2.gb_spd_in_radps';
			obj.gear2.input_torque_loss_Nm.axis_1.breakpoints = [0,1e6];
			obj.gear2.input_torque_loss_Nm.table = [0,0];
			
			obj.gear2.output_torque_loss_Nm.axis_1.signal = 'powertrain.trans.core.gearbox2.gb_spd_out_radps';
			obj.gear2.output_torque_loss_Nm.axis_1.breakpoints = [0,1e6];
			obj.gear2.output_torque_loss_Nm.table = [0,0];
			
			obj.gear2.efficiency_norm.axis_1.signal = 'powertrain.trans.core.gearbox2.gb_cmd_gear';
			obj.gear2.efficiency_norm.axis_1.breakpoints = [0,1,100];
			obj.gear2.efficiency_norm.table = [0,1,1];
			
			if nargin >= 1
				% Auto configuration of items using engine information
				obj.control = class_REVS_DCT_control(engine);
				obj.dual_clutch = class_REVS_dual_clutch(engine);
			end
			
		end
		
		
		% Gear setter - allocates inputs to two gearboxes
		function obj = set.gear( obj, val)
			bad_fields = setdiff( fieldnames(val), {'number','ratio','inertia_kgm2','shift_duration_secs','efficiency_norm','common_input_inertia_kgm2','common_output_inertia_kgm2'});
			
			if ~isempty(bad_fields)
				error('No appropriate method, property, or field %s for class class_REVS_DCT_transmission.gear',bad_fields{1});
			elseif ~isnumeric( val.number)
				error('For class_REVS_DCT_transmission gear.number must be a numeric vector');
			elseif ~isnumeric( val.ratio) || (  ~isempty(val.ratio) && numel( val.number) ~= numel(val.ratio))
				error('Automatic configuration of DCT gear ratios requires 1 numeric value per gear');
			elseif ~isnumeric( val.inertia_kgm2) || (  ~isempty(val.inertia_kgm2) && numel( val.number) ~= numel(val.inertia_kgm2))
				error('Automatic configuration of DCT gear inertias requires 1 numeric value per gear');
			elseif 	~isnumeric( val.efficiency_norm) || (  ~isempty(val.efficiency_norm) && numel( val.number) ~= numel(val.efficiency_norm))
				error('Automatic configuration of DCT gear efficiencies requires 1 numeric value per gear');
			elseif 	~isnumeric( val.common_input_inertia_kgm2) || numel(val.common_input_inertia_kgm2) > 1
				error('Automatic configuration of DCT gear input inertia requires a scalar value');
			elseif 	~isnumeric( val.common_output_inertia_kgm2) || numel(val.common_output_inertia_kgm2) > 1
				error('Automatic configuration of DCT gear output inertia requires a scalar value');
			end
			
			
			obj.gear = val;
			
			odd = mod( obj.gear.number, 2) | obj.gear.number == 0;
			even = ~(mod( obj.gear.number, 2)) | obj.gear.number == 0;
			
			obj.gear1.number = obj.gear.number(odd);
			obj.gear2.number = obj.gear.number(even);
			
			
			if  numel( obj.gear.number) == numel(val.ratio)
				obj.gear1.ratio = obj.gear.ratio(odd);
				obj.gear2.ratio = obj.gear.ratio(even);
			end
			
			if  numel( obj.gear.number) == numel(val.inertia_kgm2)
				obj.gear1.inertia_kgm2 = obj.gear.inertia_kgm2(odd);
				obj.gear2.inertia_kgm2 = obj.gear.inertia_kgm2(even);
			end
			
			if numel( obj.gear.number) == numel(val.efficiency_norm)
				
				obj.gear1.efficiency_norm.axis_1.signal = 'powertrain.trans.core.gearbox1.gb_cmd_gear';
				obj.gear1.efficiency_norm.axis_1.breakpoints = obj.gear.number(odd);
				obj.gear1.efficiency_norm.table	= obj.gear.efficiency_norm(odd);
				
				obj.gear2.efficiency_norm.axis_1.signal = 'powertrain.trans.core.gearbox2.gb_cmd_gear';
				obj.gear2.efficiency_norm.axis_1.breakpoints = obj.gear.number(even);
				obj.gear2.efficiency_norm.table = obj.gear.efficiency_norm(even);
			end
			
			
			if ~isempty( val.common_input_inertia_kgm2)
				obj.gear1.common_input_inertia_kgm2 = val.common_input_inertia_kgm2/2;
				obj.gear2.common_input_inertia_kgm2 = val.common_input_inertia_kgm2/2;
			end
			
			if ~isempty( val.common_output_inertia_kgm2)
				obj.gear1.common_output_inertia_kgm2 = val.common_output_inertia_kgm2/2;
				obj.gear2.common_output_inertia_kgm2 = val.common_output_inertia_kgm2/2;
			end
			
			if ~isempty( val.shift_duration_secs)
				obj.gear1.shift_duration_secs = val.shift_duration_secs;
				obj.gear2.shift_duration_secs = val.shift_duration_secs;
			end
			
		end
		
		
	end
	
end

