classdef class_REVS_HVM_result < class_REVS_VM_result
	%class_REVS_HVM_result
	%   Definition of class_REVS_HVM_result class
    
    properties
        % Required Parameters
        s_factor                                    % J1711 factor for adjusting for net stored energy change
    end
    
    properties ( Dependent= true , Transient = true)
        fuel_consumed_g;                            % Fuel mass consumed [g]
        fuel_consumed_vol_gallons;                  % Fuel volume consumed [gallons]
        
        % EV stats for plugins...        
        energy_consumption_kWhp100mi                % Energy [kWh per 100 miles]
        energy_consumed_Wh                          % Energy [Wh]

        % EV mode metrics
        energy_consumed_including_charging_Wh       % Energy [Wh] - Accounting for charging efficiency
        energy_consumed_e_gallons                   % Fuel consumed [e gallons]
        MPGe; 
        
        % SAE J1711 NEC (net energy change) corrected fuel values
        energy_consumption_fuel_pct;                % [%] Wh/Wh
        energy_consumed_equivalent_fuel_g;          % Fuel consumed [g]

        
        NEC_fuel_consumed_g;                        % Fuel mass consumed adjusted for stored energy
        NEC_fuel_consumed_vol_gallons;              % Fuel volume consumed adjusted for stored energy [gallons]
        NEC_fuel_consumed_CAFE_gallons;             % Fuel consumed CAFE equivalent adjusted for stored energy [gallons]
        NEC_fuel_consumption_gpmi;                  % Fuel consumption [g/mile]
        
        FE_vol_mpg;                                 % Volumetric Fuel Economy adjusted for stored energy [mpg]
        FE_raw_vol_mpg;                             % Volumetric Fuel Economy not adjusted for stored energy [mpg]
        FE_CAFE_mpg;                                % CAFE Fuel Economy adjusted for fuel properties and stored energy [mpg]
        FE_raw_CAFE_mpg;                            % CAFE Fuel Economy adjusted not for fuel properties and stored energy [mpg]

        gCO2;                                       % CO2 emissions adjusted for stored energy [g]
        gCO2pmi;                                    % CO2 emissions adjusted for stored energy [g/mile]

        gCO2_raw;                                   % CO2 emissions [g]
        gCO2pmi_raw;                                % CO2 emissions [g/mile]

        fuel_Jpm;                                   % Fuel Energy Consumption [J/m]            
        engine_Jpm;                                 % Engine Output Energy [J/m]
        battery_Jpm;                                
        emachine_Jpm;
        regen_captured_Jpm;
        regen_available_Jpm;
                
        engine_efficiency_norm; 
        emachine_efficiency_norm;
        transmission_efficiency_norm ;        
        total_efficiency_norm;
        regen_efficiency_norm;
        
    end
    
    methods
        %% Constructor
        function obj = class_REVS_HVM_result( root, cycle_or_phase_name, varargin )            
            obj@class_REVS_VM_result( root, cycle_or_phase_name, varargin{:});                        
        end
        
        %% Print Result - to screen or file
        function print( obj, fid )
            
            if nargin < 2 || isempty(fid)
                fid = 1;
            end
            
            for p = 1:length(obj.time_secs)
                
                if isprop(obj, 'cycle_name')
                    name = obj.cycle_name;
                elseif isprop(obj, 'phase_name')
                    name = obj.phase_name(p);
                else
                    name = '';
                end
                                
                fprintf(fid,'   %s: %s\n',                                          name, repmat('-', 1,25-length(name)));
                if isprop(obj, 'driver') && isprop(obj.driver, 'error_time_secs')
                    fprintf(fid,'   Percent Time Missed by 2mph = %6.2f %%\n',          obj.driver.error_time_secs(p) ./ obj.time_secs(p) * 100);
                end
                fprintf(fid,'   Distance                    = %6.3f mi\n',			obj.distance_mi(p));
%                 if obj.fuel_consumed_g(p) > 0
                    fprintf(fid,'   Fuel Consumption            = %6.4f grams\n',		obj.NEC_fuel_consumed_g(p));
                    fprintf(fid,'   Fuel Consumption            = %6.4f gallons\n',		obj.NEC_fuel_consumed_vol_gallons(p));
                    fprintf(fid,'   Fuel Economy (Volumetric)   = %6.3f mpg\n',			obj.FE_vol_mpg(p));
%                     fprintf(fid,'   Fuel Economy (CFR)          = %6.3f mpg\n',			obj.FE_CAFE_mpg(p));
                    fprintf(fid,'   Fuel Consumption            = %6.3f g/mile\n',		obj.NEC_fuel_consumption_gpmi(p) );
                    fprintf(fid,'   CO2 Emission                = %6.2f g/mile\n',		obj.gCO2pmi(p));
                    fprintf(fid,'\n');
%                 end
                fprintf(fid,'   Energy Economy              = %6.2f MPGe\n',      obj.MPGe(p));
                fprintf(fid,'   Energy Efficiency           = %6.2f kWh/100mi\n', obj.energy_consumption_kWhp100mi(p));
                fprintf(fid,'   Energy Consumption          = %6.2f Wh\n',        obj.energy_consumed_Wh(p));
                fprintf(fid,'   Energy Efficiency           = %6.0f Wh/mi\n',     obj.energy_consumption_kWhp100mi(p) * 10);
                fprintf(fid,'\n');
            end
        end
        
        %% Fuel
        
        function val = get.fuel_consumed_g(obj)            
            if ~isempty(obj.aggregator)
                val = obj.aggregator.sum('fuel_consumed_g');
            elseif isprop(obj, 'engine') && isprop(obj.engine, 'fuel_consumed_g')
                val = obj.engine.fuel_consumed_g .* obj.map_fuel.energy_density_MJpkg ./ obj.output_fuel.energy_density_MJpkg; % Conversion @ R = 1
            else
                val = NaN * obj.distance_mi;
            end
        end
        
        function set.fuel_consumed_g(obj, val)
            if ~isempty(obj.aggregator)
                 error('Fuel properties not modifyable for aggregated results.')
            else
                obj.engine.fuel_consumed_g = val .* obj.output_fuel.energy_density_MJpkg ./ obj.map_fuel.energy_density_MJpkg;   % for setting cold-corrected values based on measured values...
            end
        end
        
        function val = get.fuel_consumed_vol_gallons(obj)
            val = obj.fuel_consumed_g ./ 1000 ./ obj.output_fuel.density_kgpL_15C .* unit_convert.lit2gal;
        end
        
        
        
        %% Stored Energy
        
        function val = get.energy_consumed_Wh(obj)           
            % this one is measured at the battery terminals:
                      
            if ~isempty(obj.aggregator)
                val = obj.aggregator.sum('energy_consumed_Wh');
            elseif isproperty(obj, 'electric.propulsion_battery.output_tot_kJ')
                val = obj.electric.propulsion_battery.output_tot_kJ * 1000 / 3600;
            else
                val = NaN .* obj.distance_mi;
            end
            
        end
                
        function val = get.energy_consumption_kWhp100mi(obj)
            val = obj.energy_consumed_Wh / 1000 ./ obj.distance_mi * 100;
        end        
        
        function val = get.energy_consumed_including_charging_Wh(obj)
            val = obj.energy_consumed_Wh ./ obj.charging_efficiency; %https://ieeexplore.ieee.org/document/7046253
        end
               
        function val = get.energy_consumed_e_gallons(obj)
            val = obj.energy_consumed_including_charging_Wh / 33705;
            
            % MPGe & e gallons only valid if fully electric
            val(obj.fuel_consumed_g ~= 0.0) = nan;
        end        
        
        function val = get.MPGe(obj)                       
            val = obj.distance_mi ./ obj.energy_consumed_e_gallons;            
        end        
        
        %% J1711 Combined & Adjusted 
        function val = get.s_factor(obj)
           
            val = nan(size( obj.distance_m));
            
            if ~isempty(obj.aggregator)  
                
                phase_data = obj.aggregator.get_data('s_factor');
                
                if all(phase_data == phase_data(1))
                    val = phase_data(1);
                end
                
            else                  
                for p = 1:numel(obj.phase_name)                    
                    if startsWith(obj.phase_name{p},{'EPA_FTP4BAG_1','EPA_FTP4BAG_2','EPA_UDDS_1','EPA_UDDS_2'})
                        val(p) = 3.1;
                    else
                        val(p) = 2.7;    
                    end
                end                
            end
        end
        
        
        function val = get.energy_consumption_fuel_pct(obj) 
            val = (obj.energy_consumed_Wh .* 3600) ./ ( (obj.fuel_consumed_g) .* obj.output_fuel.energy_density_MJpkg .* 1000).*100;
        end
        
        function val = get.energy_consumed_equivalent_fuel_g(obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.sum('energy_consumed_equivalent_fuel_g');
            else
                val = (obj.energy_consumed_Wh .* 3600 .* obj.s_factor) ./ (obj.output_fuel.energy_density_MJpkg .* 1000 );
            end
        end        
                
        function val = get.NEC_fuel_consumed_g(obj)
            if isprop(obj, 'initial_charge_sustain_state')
                val = [];
                for p = 1:length(obj.initial_charge_sustain_state)
                    if obj.initial_charge_sustain_state(p)  % report NEC value if phase is charge sustaining
                        val(end+1) = obj.fuel_consumed_g(p) + obj.energy_consumed_equivalent_fuel_g(p);
                    else % report NaN
                        val(end+1) = NaN;
                    end
                end
            else
                val = obj.fuel_consumed_g + obj.energy_consumed_equivalent_fuel_g;
            end               
        end
        
        %% Volumetric FE
        
        function val = get.NEC_fuel_consumed_vol_gallons(obj)            
            val = obj.NEC_fuel_consumed_g ./ 1000 ./ obj.output_fuel.density_kgpL_15C .* unit_convert.lit2gal;
        end
        
        function val = get.FE_vol_mpg(obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.dist_per_quant('FE_vol_mpg');
            else            
                val = obj.distance_mi ./ obj.NEC_fuel_consumed_vol_gallons;
            end            
        end
        
        function val = get.NEC_fuel_consumption_gpmi(obj)
           val =   1 ./ obj.FE_vol_mpg .* unit_convert.gal2lit .* obj.output_fuel.density_kgpL_15C .* 1000;
        end

        function val = get.FE_raw_vol_mpg(obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.dist_per_quant('FE_raw_vol_mpg');
            else            
                val = obj.distance_mi ./ obj.fuel_consumed_vol_gallons;
            end                        
        end
        
        
        %% CO2
        
        function val = get.gCO2(obj)
            if isprop(obj, 'initial_charge_sustain_state')
                val = [];
                for p = 1:length(obj.initial_charge_sustain_state)
                    if obj.initial_charge_sustain_state(p)  % report NEC value if phase is charge sustaining
                        val(end+1) = obj.NEC_fuel_consumed_vol_gallons(p) .* obj.output_fuel.gCO2pgal;
                    else % report NaN
                        val(end+1) = NaN;
                    end
                end
            else
                val = obj.NEC_fuel_consumed_vol_gallons .* obj.output_fuel.gCO2pgal; % was obj.fuel_consumed_g .* obj.output_fuel.carbon_weight_fraction * (44.0095/12.0107);
            end            
        end
        
        function val = get.gCO2pmi(obj)
            val = obj.gCO2 ./ obj.distance_mi;
        end

        function val = get.gCO2_raw(obj)
            val = obj.fuel_consumed_vol_gallons .* obj.output_fuel.gCO2pgal; % was obj.fuel_consumed_g .* obj.output_fuel.carbon_weight_fraction * (44.0095/12.0107);
        end
        
        function val = get.gCO2pmi_raw(obj)
            val = obj.gCO2_raw ./ obj.distance_mi;
        end

        %% CAFE FE 
        
        function val = get.FE_CAFE_mpg(obj)
            if obj.output_fuel.alcohol_pct_vol > 1
                val = NaN * obj.distance_mi;
            elseif ~isempty(obj.aggregator)
                val = obj.aggregator.dist_per_quant('FE_CAFE_mpg');
            else
                val = obj.distance_mi ./ obj.NEC_fuel_consumed_CAFE_gallons;
            end
        end
        
        function val = get.NEC_fuel_consumed_CAFE_gallons(obj)                        
            val = ( obj.NEC_fuel_consumed_g .* ( 0.6 * obj.output_fuel.specific_gravity * obj.output_fuel.energy_density_MJpkg * unit_convert.MJpkg2BTUplbm + 5471)) ./ ( 5174e4 .* obj.output_fuel.specific_gravity);            
            if obj.output_fuel.alcohol_pct_vol > 1
                val = nan * obj.distance_mi;                % CAFE calculation not valid for alcohol fuels
            end
        end
         
        function val = get.FE_raw_CAFE_mpg(obj)
            if obj.output_fuel.alcohol_pct_vol > 1
                val = NaN * obj.distance_mi;
            elseif ~isempty(obj.aggregator)
                val = obj.aggregator.dist_per_quant('FE_raw_CAFE_mpg');
            else
                val = obj.distance_mi ./  (( obj.fuel_consumed_g .* ( 0.6 * obj.output_fuel.specific_gravity * obj.output_fuel.energy_density_MJpkg * unit_convert.MJpkg2BTUplbm + 5471)) ./ ( 5174e4 .* obj.output_fuel.specific_gravity));
            end
        end
        
        %% Energy Rate Calcs
        function val = get.fuel_Jpm(obj)
            val = obj.fuel_consumed_g .* obj.output_fuel.energy_density_MJpkg * 1000 ./ obj.distance_m;
        end
                
        function val = get.engine_Jpm  (obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.quant_per_dist('engine_Jpm');
                return
            end
            
            if obj.isproperty('engine.crankshaft_pos_kJ')
                val = obj.engine.crankshaft_pos_kJ * 1000 ./  obj.distance_m;
            else
                val = nan(size(obj.distance_m));
            end
        end

        
        function val = get.battery_Jpm(obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.quant_per_dist('battery_Jpm');
                return
            end
                
            if obj.isproperty('electric.propulsion_battery.output_tot_kJ')
                val = obj.electric.propulsion_battery.output_tot_kJ * 1000 ./  obj.distance_m;
            else
                val = nan(size(obj.distance_m)); 
            end
        end
        
        function val = get.emachine_Jpm(obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.quant_per_dist('emachine_Jpm');
                return 
            end
                        
            if obj.isproperty( 'transmission.MG1.output_tot_kJ') && obj.isproperty('transmission.MG2.output_pos_kJ')               
                val = ( obj.transmission.MG1.output_tot_kJ + obj.transmission.MG2.output_pos_kJ) * 1000 ./  obj.distance_m;
            elseif obj.isproperty('ttransmission.P0_MG.output_pos_kJ')
                val = obj.transmission.P0_MG.output_pos_kJ * 1000 ./  obj.distance_m;
            elseif obj.isproperty('transmission.P2_MG.output_pos_kJ')
                val = obj.transmission.P2_MG.output_pos_kJ * 1000 ./  obj.distance_m;
            else 
                val = nan(size(obj.distance_m));          
            end
            
        end
        
        
        function val = get.regen_captured_Jpm(obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.quant_per_dist('regen_captured_Jpm');
                return
            end
            
            if obj.isproperty('transmission.P2_MG.regen_neg_kJ')
                val = obj.transmission.P2_MG.regen_neg_kJ * 1000 ./  obj.distance_m;
            elseif obj.isproperty('ttransmission.P0_MG.regen_neg_kJ')
                val = obj.transmission.P0_MG.regen_neg_kJ * 1000 ./  obj.distance_m;
            elseif obj.isproperty('transmission.MG2.regen_neg_kJ')
                val = obj.transmission.MG2.regen_neg_kJ * 1000 ./  obj.distance_m;
            else
                val = nan(size(obj.distance_m)); 
            end
                        
        end
        
        
        function val = get.regen_available_Jpm(obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.quant_per_dist('regen_available_Jpm');
                return
            end
            
            if obj.isproperty('vehicle.decel_KE_kJ')
                val = -obj.vehicle.decel_KE_kJ *1000 ./  obj.distance_m;
            else
                val = nan(size(obj.distance_m)); 
            end
            
        end
        
        
        %% Efficiency Metrics
        function val = get.engine_efficiency_norm(obj)
                val = obj.engine_Jpm ./ obj.fuel_Jpm;
        end
        
        function val = get.emachine_efficiency_norm(obj)
            val = obj.emachine_Jpm ./ obj.battery_Jpm;
        end
        
        function val = get.transmission_efficiency_norm(obj)
                val = obj.roadload_Jpm ./ (obj.engine_Jpm + obj.emachine_Jpm);
        end
        
        function val = get.total_efficiency_norm(obj)
                val = obj.unadjusted_roadload_Jpm ./ (obj.fuel_Jpm + obj.battery_Jpm) ;
        end
               
        function val = get.regen_efficiency_norm(obj)          
            val = obj.regen_captured_Jpm ./ (obj.regen_available_Jpm) ;            
        end

    end
    
     methods ( Access = protected)
		
        function val = map_fuel(obj)
            val = obj.root.map_fuel;
        end
        
         function val = output_fuel(obj)
            val = obj.root.map_fuel;
         end
         
         function val = charging_efficiency(obj)
             val = obj.root.charging_efficiency;
         end
         
         
         %Select properties for display
		function grp = getPropertyGroups(obj)		

            if ~isempty(obj.aggregator)
                grp = matlab.mixin.util.PropertyGroup({'cycle_name','map_fuel','output_fuel'});
            else
                grp = matlab.mixin.util.PropertyGroup({'phase_name','map_fuel','output_fuel'});
            end
                                    
            grp(end+1) = matlab.mixin.util.PropertyGroup({'fuel_consumed_g', 'fuel_consumed_vol_gallons'}, 'Fuel Consumption');            
            grp(end+1) = matlab.mixin.util.PropertyGroup({'energy_consumed_Wh', 'energy_consumption_kWhp100mi', 'energy_consumed_including_charging_Wh','energy_consumed_e_gallons','MPGe'}, 'Stored Energy Consumption'); 
            grp(end+1) = matlab.mixin.util.PropertyGroup({'s_factor', 'energy_consumption_fuel_pct', 'energy_consumed_equivalent_fuel_g', 'NEC_fuel_consumed_g', 'NEC_fuel_consumption_gpmi', 'NEC_fuel_consumed_vol_gallons', 'NEC_fuel_consumed_CAFE_gallons','NEC_fuel_consumption_gpmi','FE_vol_mpg', 'FE_CAFE_mpg','gCO2', 'gCO2pmi'}, 'Net Energy Change Corrected Fuel Consumption & CO2 Emissions');
            grp(end+1) = matlab.mixin.util.PropertyGroup({'fuel_Jpm','engine_Jpm','battery_Jpm','emachine_Jpm','roadload_Jpm','unadjusted_roadload_Jpm','engine_efficiency_norm','emachine_efficiency','transmission_efficiency_norm','total_efficiency_norm', 'regen_efficiency_norm'}, 'Energy Rates & Efficiency');
            grp(end+1) = matlab.mixin.util.PropertyGroup({'drive_quality','drive_quality_unadjusted','avg_speed_mps', 'avg_speed_mph', 'shifts_per_mi'}, 'Drive Metrics');                       
            grp(end+1) = matlab.mixin.util.PropertyGroup({'time_secs', 'distance_m', 'distance_mi'}, 'Simulation Data');  
            
            props = properties(obj);
            for g = 1:length( grp)
                props = setdiff(props, grp(g).PropertyList);
            end
            		
            grp(end).PropertyList = [grp(end).PropertyList, props'];
            		
        end
   
     end
    
end

