classdef class_REVS_P2_control < class_REVS_control
    %class_REVS_P2_control
    %   For mild hybrid P2 controls
    
    properties
        stop_delay_secs
        start_stop_enable
        start_stop_off_delay_secs
        
        start_stop_warmup_condition
        cold_start_enable 
        hot_soak_start_enable 
        hot_soak_warmup_start_condition 

        pedal_map_type = enum_engine_pedal_map_type.lookup_table
        pedal_map_engine_torque_Nm

        motor_torque_limit_soc_norm
        motor_torque_limit_norm

        regen_torque_limit_soc_norm
        regen_torque_limit_norm

        regen_speed_limit_mps
        regen_speed_limit_norm

        motor_creep_speed_mps
        motor_creep_torque_norm

        engine_min_bsfc_speed_radps
        engine_min_bsfc_torque_Nm

        emachine_saturation_speed_mps
        emachine_saturation_torque_Nm

        battery_min_soc_norm
        battery_hyst_soc_norm
        battery_loadlevel_soc_norm

        max_EV_speed_mps

        engine_min_on_time_secs
        engine_off_delay_secs
        engine_min_off_time_secs
        engine_torque_rate
        
        disconnect_clutch class_REVS_clutch_control % disconnect clutch control
    end
    
    properties ( Hidden)        
        engine_max_torque_Nm;
        engine_min_torque_Nm;
        engine_torque_range_speed_radps;
        engine_idle_speed_radps;        
    end
    
    methods %New methods added
        
        function obj = class_REVS_P2_control(varargin)
            
            % obj = obj@class_REVS_control(varargin{:});
            
            obj.stop_delay_secs         = 0.1;
            obj.start_stop_enable       = true;
            obj.start_stop_warmup_condition = '(@cycle_pos_secs >= 100)';
                        
            obj.pedal_map_engine_torque_Nm = class_REVS_dynamic_lookup();
            
            obj.engine_torque_rate = 0.025;
            
            if nargin > 0
                [max_trq_spd_radps, max_trq_Nm] = engine.get_max_torque_curve();
                [min_trq_spd_radps, min_trq_Nm] = engine.get_min_torque_curve();

                obj.engine_torque_range_speed_radps = unique( [max_trq_spd_radps(:);min_trq_spd_radps(:)])';
                
                obj.engine_max_torque_Nm = interp1(max_trq_spd_radps, max_trq_Nm, obj.engine_torque_range_speed_radps, 'linear','extrap');
                obj.engine_min_torque_Nm = interp1(min_trq_spd_radps, min_trq_Nm, obj.engine_torque_range_speed_radps, 'linear','extrap');
                
                obj.engine_max_torque_Nm = max(obj.engine_max_torque_Nm, obj.engine_min_torque_Nm);
                                
                obj.engine_idle_speed_radps = engine.nominal_idle_speed_radps;
            end
            
        end
        
        function obj = update_powertrain_info(obj, engine, transmission, electric, vehicle, varargin)
                [max_trq_spd_radps, max_trq_Nm] = engine.get_max_torque_curve();
                [min_trq_spd_radps, min_trq_Nm] = engine.get_min_torque_curve();

                obj.engine_torque_range_speed_radps = unique( [max_trq_spd_radps(:);min_trq_spd_radps(:)])';
                
                obj.engine_max_torque_Nm = interp1(max_trq_spd_radps, max_trq_Nm, obj.engine_torque_range_speed_radps, 'linear','extrap');
                obj.engine_min_torque_Nm = interp1(min_trq_spd_radps, min_trq_Nm, obj.engine_torque_range_speed_radps, 'linear','extrap');
                
                obj.engine_max_torque_Nm = max(obj.engine_max_torque_Nm, obj.engine_min_torque_Nm);
                                
                obj.engine_idle_speed_radps = engine.nominal_idle_speed_radps;
                                
                min_bsfc_curve = engine.min_bsfc_curves();
                obj.engine_min_bsfc_speed_radps = min_bsfc_curve.speed_radps;
                obj.engine_min_bsfc_torque_Nm = min_bsfc_curve.torque_Nm;
        end
               
        function val = get.pedal_map_engine_torque_Nm(obj)
            
            if  obj.pedal_map_type == enum_engine_pedal_map_type.lookup_table
               
                val = obj.pedal_map_engine_torque_Nm;
                return
                
            end
            
            if  isempty( obj.engine_torque_range_speed_radps) || isempty( obj.engine_max_torque_Nm) || isempty( obj.engine_min_torque_Nm)                
                error('Misssing engine torque information, specify engine when constructing class')
            end    
                
            
                 
            % Slightly less than lower torque curve to ensure 
            pedal_min_torque_Nm = obj.engine_min_torque_Nm*1.01;
            
           switch( obj.pedal_map_type)

               case  enum_engine_pedal_map_type.linear_WOT
                   pedal_max_torque_Nm = obj.engine_max_torque_Nm;

                case enum_engine_pedal_map_type.linear_max_engine_torque
                    pedal_max_torque_Nm = max(obj.engine_max_torque_Nm) * ones( size(obj.engine_max_torque_Nm));

                case enum_engine_pedal_map_type.max_engine_power
                    max_power_W =  max( obj.engine_max_torque_Nm .* obj.engine_torque_range_speed_radps);
                    pedal_max_torque_Nm =  max_power_W ./ max(obj.engine_torque_range_speed_radps, obj.engine_idle_speed_radps);

                case enum_engine_pedal_map_type.linear_enhanced_WOT
                    pedal_max_torque_Nm = 1.1 * obj.engine_max_torque_Nm;

                case enum_engine_pedal_map_type.linear_enhanced_max_engine_torque
                    pedal_max_torque_Nm = 1.1* max(obj.engine_max_torque_Nm) * ones( size(obj.engine_max_torque_Nm));

                otherwise
                    error('Unknown pedal map type : %s', obj.pedal_map_type);
           end
            
            val = class_REVS_dynamic_lookup();
            val.axis_1.signal = 'driver_acl_norm';
            val.axis_1.breakpoints = [0,1];
            val.axis_2.signal = 'eng_spd_radps';
            val.axis_2.breakpoints = obj.engine_torque_range_speed_radps;
            val.table = [pedal_min_torque_Nm;pedal_max_torque_Nm];
               
        end    
                        
        end
        
end
    
