classdef class_REVS_VM_result < class_REVS_logging_object
	%class_REVS_VM_result
	%   Definition of class_REVS_VM_result class
        
    properties
        % Required Parameters       
        drive_quality;                  % structure of SAEJ2951 drive quality metrics, from run_cycle_postproc
        drive_quality_unadjusted;       % structure of SAEJ2951 drive quality metrics, from run_cycle_postproc using only unadjusted ABC roadload
        
        time_secs;
    end
    
    properties ( Dependent= true , Transient = true)
                
        distance_m;     % Distance [m]       
        distance_mi;    % Distance [mi]
        
        avg_speed_mps;  % Avg speed [mps]
        avg_speed_mph;  % Avg speed [mph]
        
        shifts_per_mi;  
        
        roadload_Jpm;
        unadjusted_roadload_Jpm;
        
    end
    
    properties( Hidden)
       aggregator; 
    end
    
    methods
        
        %% Constructor
        function obj = class_REVS_VM_result(root, cycle_or_phase_name,  aggregator)
            
            obj@class_REVS_logging_object(root);
            
            if nargin < 3
                % No aggregator - phase results
                obj.addprop('phase_name');
                obj.phase_name = string(cycle_or_phase_name);                
            elseif isa( aggregator, 'class_REVS_cycle_aggregator')
               obj.aggregator =  aggregator;
               obj.addprop('cycle_name');
               obj.cycle_name = cycle_or_phase_name; 
            else
                error('Unknown result aggregator interface, but be a superclass of class_REVS_cycle aggregator');
            end

            
        end
        
        %% Direct Values from Model Output
        function val = get.distance_m(obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.sum('distance_m');
            else
                val = obj.vehicle.distance_m;
            end
        end
        
        %% Distance
        function val = get.distance_mi(obj)
            val = obj.distance_m .* unit_convert.mtr2mi;
        end
        
        %% Time
        function val = get.time_secs(obj)
           if ~isempty(obj.aggregator)
               val = obj.aggregator.sum('time_secs');
           else
                val = obj.time_secs;
           end
        end
        
        %% Average Speed - when moving
        function val = get.avg_speed_mps(obj)
            if isprop(obj.vehicle,'stopped_time_secs')
                val = obj.distance_m ./max(0, obj.time_secs - obj.vehicle.stopped_time_secs);
            else
                val = nan(size(obj.distance_m));
            end
        end
        
        function val = get.avg_speed_mph(obj)
            val = obj.avg_speed_mps * unit_convert.mps2mph;
        end

        %% Shifts
        
        function val = get.shifts_per_mi(obj)
            if isprop(obj,'transmission') && isprop(obj.transmission,'num_shifts')
                val = obj.transmission.num_shifts ./ obj.distance_mi;
            else
                 val = nan(size(obj.distance_m));
            end
        end
        
        %% Powertrain Energy
        
        function val = get.roadload_Jpm (obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.quant_per_dist('roadload_Jpm');
            elseif ~isempty(obj.drive_quality)
                val = obj.drive_quality.CEd_J ./ obj.drive_quality.Dd_m;
            else
                val = nan(size(obj.distance_m));
            end
        end
        
        function val = get.unadjusted_roadload_Jpm (obj)
            if ~isempty(obj.aggregator)
                val = obj.aggregator.quant_per_dist('unadjusted_roadload_Jpm');
            elseif ~isempty(obj.drive_quality)
                val = obj.drive_quality_unadjusted.CEd_J ./ obj.drive_quality_unadjusted.Dd_m;
            else
                val = nan(size(obj.distance_m));
            end
        end
        
        
        %% Drive Quality
        function add_SAE_J2951(obj, vehicle, time, phase, driven_speed_mps, cycle_speed_mps)
                       
           data = struct('time',time,'phase',phase,'driven_spd_mps',driven_speed_mps,'cycle_spd_mps',cycle_speed_mps,'vehicle',vehicle);
           
           obj.drive_quality = create_drive_quality_entry(data);
           obj.drive_quality_unadjusted = create_drive_quality_entry(data, 'use_unadjusted_ABCs');
                      
            function val = create_drive_quality_entry(data, varargin)
                
                dq = REVS_SAEJ2951(data, varargin{:} );
                val = class_REVS_logging_object(obj);
                                
                for prop = {'time_secs', 'CEt_J', 'CEt_dist_Jpm', 'CEd_J', 'CEd_dist_Jpm', 'ASCt_mps2', 'ASCd_mps2', 'Dt_m', 'Dd_m', 'RMSSE_mph'}
                    val.addprop(prop{1});
                    val.(prop{1}) = dq.(prop{1});
                end
                
                meta = val.addprop('ER_pct');
                meta.Dependent = 1;
                meta.GetMethod = @(o) (o.CEd_J - o.CEt_J) ./ o.CEt_J * 100;
                
                meta = val.addprop('EER_pct');
                meta.Dependent = 1;
                meta.GetMethod = @(o) (1-( ((o.DR_pct/100)+1) ./ ((o.ER_pct/100)+1)) ) * 100;
                
                meta = val.addprop('DR_pct');
                meta.Dependent = 1;
                meta.GetMethod = @(o) (o.Dd_m - o.Dt_m) ./ o.Dt_m * 100;
                
                meta = val.addprop('ASCR_pct');
                meta.Dependent = 1;
                meta.GetMethod = @(o) (o.ASCd_mps2 - o.ASCt_mps2) ./ o.ASCt_mps2 * 100;
                
            end
                      
        end
        
        function print_drive_quality(obj, unadjusted, output_fid)
           
            if nargin < 3
                output_fid = 1;
            end
            
            if nargin < 2 || ~unadjusted
                drive_quality_stats = obj.drive_quality;
            else
                drive_quality_stats = obj.drive_quality_unadjusted;
            end
                        
            if isempty(drive_quality_stats)
                fprintf(output_fid, 'SAE J2951 Drive Quality Metrics Not Available\n');
            else    
                fprintf(output_fid, 'SAE J2951 Drive Quality Metrics:\n');
                fprintf(output_fid, 'Time secs         %f\n',   drive_quality_stats.time_secs(p));
                fprintf(output_fid, 'CEt MJ            %f\n',   (drive_quality_stats.CEt_J(p)/10^6) );
                fprintf(output_fid, 'CEt_dist J/m      %f\n',   (drive_quality_stats.CEt_dist_Jpm(p)) );
                fprintf(output_fid, 'CEd MJ            %f\n',   (drive_quality_stats.CEd_J(p)/10^6) );
                fprintf(output_fid, 'CEd_dist J/m      %f\n',   (drive_quality_stats.CEd_dist_Jpm(p)) );
                fprintf(output_fid, 'ER %%             %2.2f\n',(drive_quality_stats.ER_pct(p)));
                fprintf(output_fid, 'DR %%             %2.2f\n',(drive_quality_stats.DR_pct(p)) );
                fprintf(output_fid, 'EER %%            %2.2f\n',(drive_quality_stats.EER_pct(p)));
                fprintf(output_fid, 'ASCt              %f\n',   (drive_quality_stats.ASCt_mps2(p) / 1000));
                fprintf(output_fid, 'ASCd              %f\n',   (drive_quality_stats.ASCd_mps2(p) / 1000));
                fprintf(output_fid, 'ASCR %%           %2.2f\n',(drive_quality_stats.ASCR_pct(p)));
                fprintf(output_fid, 'Dt mi             %f\n',   (drive_quality_stats.Dt_m(p)/1609.344));
                fprintf(output_fid, 'Dt m              %f\n',   drive_quality_stats.Dt_m(p));
                fprintf(output_fid, 'Dd mi             %f\n',   (drive_quality_stats.Dd_m(p)/1609.344));
                fprintf(output_fid, 'Dd m              %f\n',   drive_quality_stats.Dd_m(p));
                fprintf(output_fid, 'Distance Error mi %f\n',   (drive_quality_stats.Dt_m(p) - drive_quality_stats.Dd_m(p))/1609.344 );
                fprintf(output_fid, 'RMSSE_mph         %f\n',   (drive_quality_stats.RMSSE_mph(p)));
                fprintf(output_fid,'\n');
            end          
            
        end
        
    end
    
end
