classdef class_REVS_detailed_engine < class_REVS_engine_base
	%class_REVS_engine
	%   Definition of class_REVS_engine class
	
	properties
		
		% Torque Limits
		maximum_torque_Nm class_REVS_dynamic_lookup 	% Maximum torque map vs speed, temp, etc. [Nm] - Dynamic Lookup 
		minimum_torque_Nm class_REVS_dynamic_lookup 	% Minimum torque map vs speed, temp, etc. [Nm] - Dynamic Lookup 
		
		% Torque response characteristics
		rising_torque_response_secs = class_REVS_dynamic_lookup;
		falling_torque_response_secs = class_REVS_dynamic_lookup;
		
        % Run state
        run_state_activation_speed_radps = 1;				% Speed above which engine control activates [rad/sec]
        run_state_activation_delay_secs = 0.5;              % Time above activation speed before control begins [sec]
        run_state_deactivation_speed_radps = 0;				% Speed below which enginecontrol deactivates [rad/sec]
       
        % Idle properties
        idle_target_speed_radps class_REVS_dynamic_lookup;      % Target idle speed Dynamic Lookup - can be set to scalar value [rad/sec]
        idle_control_Kp = 25;									% Idle Speed Control PID Proportional Term
        idle_control_Ki = 65;									% Idle Speed Control PID Integral Term
        idle_control_Kd = 0;									% Idle Speed Control PID Derivative Term
        idle_control_ramp_radps = 100 * unit_convert.rpm2radps; % Idle Speed Control Idle Ramp Speed [rad/s]
        idle_control_ramp_time_secs = 1;                        % Idle Speed Control Idle Ramp Time [sec]
        idle_control_torque_reserve_Nm;							% Idle Speed Control Fast response torque reserve [Nm]
        idle_control_slow_est_time_constant_sec = 0.2;
        
        
		fuel_consumption_gps class_REVS_dynamic_lookup;			% Fuel consumption map [g/sec]
		     
		deac_strategy;											% Structure / Class defining type and calibration for cylinder deactivation logic
		deac_num_cylinders = 0;									% Number of cylinders that have deactivation capability
		
		deac_transition_on_duration_secs = 0.99;					% Cylinder deactivation turn on fuel map blending
		deac_transition_off_duration_secs = 0.11;					% Cylinder deactivation turn off fuel map blending
		deac_transition_off_fuel_multiplier = [1.0 1.0];			% Cylinder deactivation turn off penalty fuel multiplier profile
		deac_transition_off_fuel_multiplier_time_secs = [0.0 0.1];	% Cylinder deactivation turn off penalty fuel multiplier profile time [sec]
		deac_transition_off_fuel_multiplier_limit_gps = inf;		% Cylinder deactivation turn off penalty maximum additional fueling [g/sec]
		
		% Enrichment or other fuel adjustments for temperature		
		thermal_management_activation_normps class_REVS_dynamic_lookup;
		thermal_management_norm class_REVS_dynamic_lookup;
		thermal_management_ramp_duration_secs = 0.1;
		
		% Fast Torque Fueling Adjustment
		fast_torque_fuel_adjust_norm;								% Portion of fuel cut associated with fast torque reduction [0 - 1] generally 0 for gasoline 1 for diesel
				
		DFCO_enable_condition = '@veh_spd_mps>5';					% Conditions to allow decel fuel cutoff when operating on closed throttle curve [DYNAMIC EXPRESSION]
		DFCO_torque_Nm class_REVS_dynamic_lookup = class_REVS_dynamic_lookup;
		DFCO_min_duration_secs = 2.1;								% DFCO defueled duration before triggering additional fueling
		DFCO_refuel_multiplier = [1.0, 1.3, 1.0];					% DFCO refuel multiplier profile
		DFCO_refuel_multiplier_time_secs = [0.0, 0.1, 1.1];			% DFCO refuel multiplier profile time [sec]
		DFCO_refuel_multiplier_limit_gps = inf;						% DFCO additional fuel limit [grams/sec]
		
		%transient_correction_mult = 1.40;							% Transient operation correction
		
		thermal;													% Engine thermal model
		
		mass_air_flow_map_type enum_engine_mass_air_flow_type;		% Location of mass airflow (intake vs exhaust)
		mass_air_flow_gps class_REVS_dynamic_lookup ;
		
		emissions;													% Final Emissions Map - for use with map based emissions model
		raw_emissions;												% Raw Emissions Maps - for aftertreatment simulation
		
		aftertreatment;												% Aftertreatment system class
		
		interp_signal_defaults;										% Default values to use for extra dynamic lookup axes stored as n by 2 cell array of signal names and scalar values
                                               
        
    end
	
    properties ( Hidden = true, Transient = true )
		interp_signal_override;                     % Override values to use (temporarily) in dynamic lookups 
	end
    
	properties ( Dependent = true, SetAccess = immutable, Transient = true)
				
        % Simplified 1D representation of max torque vs speed
		nominal_max_curve_torque_Nm;			% Simplified representation of maximum torque vs speed [Nm]
		nominal_max_curve_speed_radps;			% Simplified representation of maximum torque vs speed [radps]
		
		% Simplified 1D representation of min torque vs speed
		nominal_min_curve_torque_Nm;			% Simplified representation of minimum torque vs speed [Nm]
		nominal_min_curve_speed_radps;			% Simplified representation of minimum torque vs speed [radps]

        
%         nominal_fuel_map_speed_radps;			% Fuel consumption map speed breakpoints [rad/sec] - REVS3
% 		nominal_fuel_map_torque_Nm;				% Fuel consumption map torque breakpoints [Nm] - REVS3
% 		nominal_fuel_map_consumption_gps;		% Fuel consumption map [g/sec] - REVS3
% 
	end
	
	
	methods
				
		%% Constructor
		function obj = class_REVS_detailed_engine(varargin)
			
            %obj@class_REVS_engine_base();
                       
            obj.variant = 'detailed engine';
            
			% Populate dynamic lookup Bus Signal filters to avoid algeraic loops					
            obj.maximum_torque_Nm = class_REVS_dynamic_lookup;
            obj.maximum_torque_Nm.signal_filter = 'engine.torque*, engine.fuel*, engine.emissions*, engine.airflow*, engine.eng_load_Nm';
			
            obj.minimum_torque_Nm = class_REVS_dynamic_lookup;
            obj.minimum_torque_Nm.signal_filter = 'engine.torque*, engine.fuel*, engine.emissions*, engine.airflow*, engine.eng_load_Nm';			
			
            obj.fuel_consumption_gps = class_REVS_dynamic_lookup;
            obj.fuel_consumption_gps.signal_filter = 'engine.fuel*, engine.emissions*, engine.airflow*';
			
            % Idle speed target
            obj.idle_target_speed_radps = class_REVS_dynamic_lookup;           
            
            % Default to no thermal model
			obj.thermal = struct('variant','none');
            obj.thermal_management_norm = class_REVS_dynamic_lookup;
            obj.thermal_management_norm.signal_filter =  'engine.emissions*,engine.fuel*';
            obj.thermal_management_activation_normps = class_REVS_dynamic_lookup;
			obj.thermal_management_activation_normps.signal_filter =  'engine.emissions*,engine.fuel*';
			
            % Default mass air flow
            obj.mass_air_flow_map_type = enum_engine_mass_air_flow_type.intake;				% Location of mass airflow (intake vs exhaust)
            obj.mass_air_flow_gps  = class_REVS_dynamic_lookup;
            obj.mass_air_flow_gps.signal_filter =  'engine.fuel*, engine.airflow*, engine.emissions*';

			% Default to constant off cylinder deac		
			obj.deac_strategy = struct('variant','constant','enable_norm',0.6);
			
			% Default to emissions map & no aftertreatment
			obj.aftertreatment = struct('variant','none');
            
            
            % Default values for additional interpolation axes
            obj.interp_signal_defaults = {'veh_spd_mps', 0.0; 
                                          'eng_CDA_active_norm', 1.0}';
			
            if nargin >0 && isa(varargin{1},'class_REVS_engine')
                
                e = varargin{1};
                
                obj.maximum_torque_Nm.axis_1.signal = 'eng_spd_radps';
                obj.maximum_torque_Nm.axis_1.breakpoints = e.max_curve_speed_radps;
                obj.maximum_torque_Nm.axis_1.table = e.max_curve_torque_Nm;
                
                obj.minimum_torque_Nm.axis_1.signal = 'eng_spd_radps';
                obj.minimum_torque_Nm.axis_1.breakpoints = e.min_curve_speed_radps;
                obj.minimum_torque_Nm.axis_1.table = e.min_curve_torque_Nm;
                
                obj.fuel_consumption_gps.axis_1.signal = 'eng_spd_radps';
                obj.fuel_consumption_gps.axis_1.breakpoints = e.fuel_map_speed_radps;
                obj.fuel_consumption_gps.axis_2.signal = 'eng_fuel_trq_Nm';
                obj.fuel_consumption_gps.axis_2.breakpoints = e.fuel_map_speed_radps;                
                obj.fuel_consumption_gps.table = e.fuel_map_consumption_gps;  
                
                if ~all(isnan(e.deac_fuel_map_consumption_gps))
                    obj.fuel_consumption_gps.axis_3.signal = 'eng_CDA_active_norm';
                    obj.fuel_consumption_gps.axis_3.breakpoints = [0,1];    
                    obj.fuel_consumption_gps.table(:,:,2) = min(e.fuel_map_consumption_gps, e.deac_fuel_map_consumption_gps);  
                end
         
            end
            
		end
		
		%% Base Engine Characteristics
		
        function [idle_speed_radps] = get_idle_speed_radps(obj, varargin)
            idle_speed_radps = obj.idle_target_speed_radps.interp(obj.interp_signal_defaults{:}, varargin{:});          
        end
                
        function [speed_radps, torque_Nm] = get_max_torque_curve(obj, varargin)
            speed_radps = obj.maximum_torque_Nm.get_signal_breakpoints('eng_spd_radps');
            torque_Nm = max(0.0, obj.maximum_torque_Nm.interp(obj.interp_signal_defaults{:},varargin{:},'eng_spd_radps',speed_radps));
        end
        
        function [speed_radps, torque_Nm] = get_min_torque_curve(obj, varargin)      
            speed_radps = obj.minimum_torque_Nm.get_signal_breakpoints('eng_spd_radps');
            torque_Nm = obj.minimum_torque_Nm.interp(obj.interp_signal_defaults{:},varargin{:},'eng_spd_radps',speed_radps);
        end
            
        function [map_speed_radps, map_torque_Nm, map_consumption_gps] = get_fuel_map(obj, varargin)

               [min_curve_speed_radps, min_curve_torque_Nm] = get_min_torque_curve(obj, varargin{:});
            
                fuel_map_signals = obj.fuel_consumption_gps.get_signal_array();

                % Get Speed Axis
                map_speed_radps = obj.fuel_consumption_gps.get_signal_breakpoints( 'eng_spd_radps');
                    
                 
                % Get TOrque Axis
                if ismember( 'eng_fuel_trq_Nm', fuel_map_signals)
                    map_torque_Nm = obj.fuel_consumption_gps.get_signal_breakpoints( 'eng_fuel_trq_Nm');
                elseif ismember( 'eng_ind_fuel_trq_Nm', fuel_map_signals)
                    map_torque_Nm = obj.fuel_consumption_gps.get_signal_breakpoints( 'eng_ind_fuel_trq_Nm');
                elseif ismember( 'eng_trq_Nm', fuel_map_signals)
                    map_torque_Nm = obj.fuel_consumption_gps.get_signal_breakpoints( 'eng_fuel_trq_Nm');
                elseif ismember( 'eng_ind_trq_Nm', fuel_map_signals)
                    map_torque_Nm = obj.fuel_consumption_gps.get_signal_breakpoints( 'eng_ind_fuel_trq_Nm'); 
                else
                    error('Unable to determine torque axis signal for fuel map');
                end
                
               if  ismember( 'eng_ind_fuel_trq_Nm', fuel_map_signals) || ismember( 'eng_ind_trq_Nm', fuel_map_signals)
                    map_torque_extend_Nm = interp1( min_curve_speed_radps, min_curve_torque_Nm, [map_speed_radps(1), map_speed_radps(end)], linear');               
                    map_torque_Nm = map_torque_Nm(:) * [1,1] + map_torque_extend_Nm;                    
                    map_torque_Nm = unique(map_torque_Nm(:));   
               end
                  
                % Make mesh for interpolation
                [mesh_speed_radps, mesh_brk_torque_Nm] =  meshgrid( map_speed_radps, map_torque_Nm);
                mesh_ind_torque_Nm = mesh_brk_torque_Nm - interp1( min_curve_speed_radps, min_curve_torque_Nm, mesh_speed_radps, 'linear','extrap');
                
                % Tweak vararg signals
                vararg_sigs = varargin;                
                for i = 1:2:length( vararg_sigs)
                   switch  vararg_sigs{i}
                       
                       case 'deac_prop'
                           if isnumeric( vararg_sigs{i+1} )
                               %Nothing to do
                           elseif strcmpi(vararg_sigs{i+1},'enable') || strcmpi(vararg_sigs{i+1},'deac')  
                               vararg_sigs{i+1} = 1.0;
                           elseif strcmpi(vararg_sigs{i+1},'disable')
                               vararg_sigs{i+1} = 0.0;
                           end                               
                   end
                end
                           
                map_consumption_gps = obj.fuel_consumption_gps.interp( obj.interp_signal_defaults{:}, ... vararg_sigs{:}, 
                                    'eng_spd_radps', mesh_speed_radps, ...
                                    'eng_fuel_trq_Nm',  mesh_brk_torque_Nm, 'eng_trq_Nm',  mesh_brk_torque_Nm,...
                                    'eng_ind_fuel_trq_Nm', mesh_ind_torque_Nm, 'eng_ind_trq_Nm', mesh_ind_torque_Nm);    
            end

        
        % TODO: Add fuel map interp function - need to map signals before class_REVS_dynamic_lookup does heavy lifting
        
		
		%% Torque Limits
		
		function val = get.nominal_max_curve_speed_radps(obj)
            val = obj.maximum_torque_Nm.get_signal_breakpoints('eng_spd_radps');
		end
				
		function val = get.nominal_max_curve_torque_Nm(obj)
            val = max(0.0, obj.maximum_torque_Nm.interp('eng_spd_radps',obj.nominal_max_curve_speed_radps));			
		end
				
		function val = get.nominal_min_curve_speed_radps(obj)
            val =  obj.minimum_torque_Nm.get_signal_breakpoints('eng_spd_radps');
		end
				
		function val = get.nominal_min_curve_torque_Nm(obj)
            val = obj.minimum_torque_Nm.interp(obj.interp_signal_defaults,'eng_spd_radps',obj.nominal_min_curve_speed_radps);
		end

		
		%% Torque Response
		
		function obj = set.rising_torque_response_secs( obj, val)
			
			if isnumeric(val) && isscalar(val)
				obj.rising_torque_response_secs.axis_1.signal = 'eng_spd_radps';
				obj.rising_torque_response_secs.axis_1.breakpoints = [0,1000];
				obj.rising_torque_response_secs.table = [val,val];
			elseif isa( val, 'class_REVS_dynamic_lookup')
				obj.rising_torque_response_secs = val;
			else
				error('While setting property ''rising_torque_response_sec'' of class ''class_REVS_engine'':\nValue must be ''class_REVS_dynamic_lookup''.')
			end
		end
		
		function obj = set.falling_torque_response_secs( obj, val)
			
			if isnumeric(val) && isscalar(val)
				obj.falling_torque_response_secs.axis_1.signal = 'eng_spd_radps';
				obj.falling_torque_response_secs.axis_1.breakpoints = [0,1000];
				obj.falling_torque_response_secs.table = [val,val];
			elseif isa( val, 'class_REVS_dynamic_lookup')
				obj.falling_torque_response_secs = val;
			else
				error('While setting property ''falling_torque_response_sec'' of class ''class_REVS_engine'':\nValue must be ''class_REVS_dynamic_lookup''.')
			end
		end
		
		%% Getters
		
		function val = get.DFCO_torque_Nm(obj)
			if ~isempty( obj.DFCO_torque_Nm )
				val = obj.DFCO_torque_Nm;
			else
				val = obj.minimum_torque_Nm;
				val.table = val.table * 0.9;
			end
        end
		      
		
		function val = get.fast_torque_fuel_adjust_norm(obj)
			val = REVS_class_default( obj.fast_torque_fuel_adjust_norm, obj.combustion_type == enum_engine_combustion_type.compression_ignition, obj.disable_auto_calc  );
		end
		
		function val = get.idle_control_torque_reserve_Nm(obj)
			val = REVS_class_default( obj.idle_control_torque_reserve_Nm, -0.85 * interp1( obj.nominal_min_curve_speed_radps, obj.nominal_min_curve_torque_Nm, obj.nominal_idle_speed_radps ), obj.disable_auto_calc );		
        end
				
		function val = get.mass_air_flow_gps( obj)
			
			val = obj.mass_air_flow_gps;
			
			if isempty(val) && ~isempty( obj.displacement_L )
				val.axis_1.signal = 'eng_spd_radps';
				val.axis_1.breakpoints = [0,1000];
				val.table = [0,1000] * pi * 0.85 * obj.displacement_L;			
			end
						
		end
		
		function val = get.thermal_management_activation_normps( obj)
			
			val = obj.thermal_management_activation_normps;
			
			if isempty(val) 
				val.axis_1.signal = 'eng_spd_radps';
				val.axis_1.breakpoints = [0,1000];
				val.table = [0,0];			
			end
		end
		
		function val = get.thermal_management_norm( obj)
			
			val = obj.thermal_management_norm;
			
			if isempty(val) 
				val.axis_1.signal = 'eng_spd_radps';
				val.axis_1.breakpoints = [0,1000];
				val.table = [0,0];			
			end
		end
		
		        
        
		function val = get_power_speed( obj, pwr_thresh, pwr_mode, spd_mode)
			
			[trq, spd] = obj.hires_max_curve;
			power_W = spd .* trq ;
						
			if nargin < 3 || isempty( pwr_mode)
				pwr_thresh = pwr_thresh .* max(power_W);
			elseif strcmpi( pwr_mode, 'norm' )
				pwr_thresh = pwr_thresh .* max(power_W);	
			elseif strcmpi( pwr_mode, 'pct' )
				pwr_thresh = pwr_thresh / 100 .* max(power_W);		
			elseif strcmpi( pwr_mode, 'kW' )
				pwr_thresh = pwr_thresh  * 1000;		
			elseif ~strcmpi( pwr_mode, 'W' )
				error( 'Unknown power threshold mode %s, valid options are ''norm'', ''pct'', ''kW'', and ''W''', pwr_mode);		
			end
			
			idx_min =  find( power_W >= pwr_thresh ,1,'first');
			idx_max =  find( power_W >= pwr_thresh ,1,'last');
			
			if nargin < 4 || isempty( spd_mode) || strcmpi(  spd_mode, 'avg' )
				val = ( spd(idx_max) + spd(idx_min) )/2;
			elseif strcmpi(  spd_mode, 'min' )
				val = spd(idx_min);
			elseif strcmpi(  spd_mode, 'max' )
				val = spd(idx_max);
			elseif strcmpi(  spd_mode, 'range' )
				val = spd([idx_min,idx_max]);	
			else
				error( 'Unknown speed result mode %s, valid options are ''avg'', ''min'', ''max'', and ''range''', spd_mode);
			end
								
		end
				
        
		function [ engine_out ] = REVS_modify_engine( obj, varargin )
		%REVS_modify_engine takes in an engine structure and its displacement(L), apply engine modifiers,
		%   and returns the result in a new engine structure.
		%
		%   [ engine_new ] = REVS_modify_engine( engine, varargin )
		%
		%   Avaiable engine modifiers can be applied by varargin as follows:
		%   Scale engine: 'scale_factor', n
		%               where n = NewSize/OrigSize

		%% Parse varargs
		% Check if enough varargin was passed
		narginchk( 1, inf );


		% Engine Scaling Options
		% num_cylinders = parse_varargs( varargin, 'num_cylinders', [], 'numeric', {'integer', 'scalar'} );
		new_num_cylinders = parse_varargs( varargin, 'num_cylinders', [], 'numeric' );

		% Scale engine size: scale_factor = NewSize/OrigSize (scale_factor > 0)
		scale_factor = parse_varargs( varargin, 'scale_factor', [], 'numeric' );
		scale_displacement = parse_varargs( varargin, 'scale_displacement', [], 'numeric' );
		scale_max_power = parse_varargs( varargin, 'scale_max_power',[], 'numeric' );
		scale_avg_power = parse_varargs( varargin, 'scale_avg_power',[], 'numeric' );
		scale_max_torque = parse_varargs( varargin, 'scale_max_torque',[], 'numeric' );
		scale_avg_torque = parse_varargs( varargin, 'scale_avg_torque',[], 'numeric' );

		limit_bmep = parse_varargs( varargin, 'limit_bmep',nan, 'numeric','scalar' );
		limit_torque = parse_varargs( varargin, 'limit_torque',nan, 'numeric','scalar' );

		% window to use for averaging power or torque
		averaging_window = parse_varargs( varargin, 'avg_window',[1800,6000] * unit_convert.rpm2radps, 'numeric' );

		scaling_knock_adjust    = parse_varargs( varargin,'no_scaling_knock_adjust',   true,'toggle');
		scaling_friction_adjust = parse_varargs( varargin,'no_scaling_friction_adjust',true,'toggle');
		scaling_sv_ratio_adjust = parse_varargs( varargin,'no_scaling_sv_ratio_adjust',true,'toggle');

		% Fricion Adjustment
		input_friction_adjust_Nm = parse_varargs( varargin, 'friction_adjust_Nm', 0.0, 'numeric');
		input_friction_adjust_bar = parse_varargs( varargin, 'friction_adjust_bar', 0.0, 'numeric');

		knk_max_adjust = parse_varargs( varargin, 'knock_max_adjust', 1.1, 'numeric');

		% Add ro remove engine technologies
		rmv_deac = parse_varargs( varargin,'-deac',false,'toggle');
		%add_deacFC = parse_varargs( varargin,'+deacFC', false,'toggle');
		add_deacC = parse_varargs( varargin,'+deacC', 0, 'numeric');
		add_deacD = parse_varargs( varargin,'+deacD', 0, 'numeric');

% 		mod_GDI = parse_varargs( varargin,'GDI', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
% 		mod_DCP_DVVL = parse_varargs( varargin,'DCP_DVVL', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
% 		mod_CCP_DVVL = parse_varargs( varargin,'CCP_DVVL', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
% 		mod_DCP_CVVL = parse_varargs( varargin,'DCP_CVVL', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
% 		mod_DCP      = parse_varargs( varargin,'DCP', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
% 		mod_CCP      = parse_varargs( varargin,'CCP', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion

		new_fuel     = parse_varargs( varargin,'fuel', []); % pass in class_REVS_fuel object

		% deac_num_cylinders = parse_varargs(varargin,'deac_num_cylinders',[],'numeric'); % not fully vetted yet, at least bsed on GM 4.3L deac

		% Fill in if only window end points provided
		if numel(averaging_window) ==2
			averaging_window = linspace(averaging_window(1), averaging_window(end), 25);
		end

		%% Copy and create a new engine structure
		engine_out = obj;		
		
		if ~isempty(new_fuel)
			engine_out.fuel_consumption_gps = obj.fuel_consumption_gps * obj.fuel.energy_density_MJpkg / new_fuel.energy_density_MJpkg;
			engine_out.fuel = new_fuel;
		end

		%% Compute Deired Scaling

		engine_torque = interp1( obj.nominal_max_curve_speed_radps, obj.nominal_max_curve_torque_Nm, averaging_window,'linear','extrap' );
		engine_power = engine_torque .* averaging_window;

		engine_average_torque = sum( engine_torque)./ length(averaging_window);
		engine_average_power = sum( engine_power )./ length(averaging_window);


		if ~isempty( scale_displacement )
			scale_factor = scale_displacement / obj.displacement_L;
		elseif ~isempty( scale_max_power )
			scale_factor = scale_max_power / obj.max_power_W;
		elseif ~isempty( scale_max_torque )
			scale_factor = scale_max_torque / obj.max_torque_Nm;
		elseif ~isempty( scale_avg_power )
			scale_factor = scale_avg_power / engine_average_power;
		elseif ~isempty( scale_avg_torque )
			scale_factor = scale_avg_torque / engine_average_torque;
		elseif isempty(scale_factor) && ~isempty( new_num_cylinders )
			scale_factor = new_num_cylinders ./ obj.num_cylinders;
		elseif isempty(scale_factor)
			scale_factor = 1.0;
		end


		%% Adjust Physical Characteristics
		engine_out.displacement_L					= obj.displacement_L * (scale_factor);
		engine_out.inertia_kgm2						= obj.inertia_kgm2 .* (scale_factor);

		%% Set number of cylinders

		disp_limit_cyl =     [	3,	  4,	  6,   	  8,	16];
		disp_limit_max = 1.0*[1.5,	2.7,	4.5,	7.5,	15];
		%disp_limit_max = [1.5,	2.7,	3.55,	7.5,	15.0];
		disp_limit_min = 1.0*[0.6,	1.4,	2.5,	3.8,	8.0];
		%disp_limit_min = [0.6,	1.0,	2.5,	3.56,	8.0];


		if ~isempty( new_num_cylinders )
			% Use provided Cylinder Count
			engine_out.num_cylinders = new_num_cylinders;
		else
			engine_out.num_cylinders = obj.num_cylinders;
			% Add cylinders is displacement above limit
			while engine_out.displacement_L	> interp1( disp_limit_cyl, disp_limit_max, engine_out.num_cylinders)
				engine_out.num_cylinders = engine_out.num_cylinders + 1 + (engine_out.num_cylinders >= 4);
			end
			% Remove cylinders if displacement below limit
			while engine_out.displacement_L	< interp1( disp_limit_cyl, disp_limit_min, engine_out.num_cylinders)
				engine_out.num_cylinders = engine_out.num_cylinders - 1 - (engine_out.num_cylinders >= 6);
			end

		end


		%% Generate cylinder scale factors
		cyl_volume_scale_factor = scale_factor * obj.num_cylinders / engine_out.num_cylinders;
		cyl_linear_scale_factor = cyl_volume_scale_factor ^ (1/3);

		%% Adjust Cylinder Physical Characteristics
		engine_out.bore_mm	  = obj.bore_mm .* cyl_linear_scale_factor;
		engine_out.stroke_mm  = obj.stroke_mm .* cyl_linear_scale_factor;


		if ~isempty( new_num_cylinders) || engine_out.num_cylinders ~= obj.num_cylinders
			engine_out.name = sprintf( '%s converted to %0.2fL %d Cylinder', obj.name, engine_out.displacement_L, engine_out.num_cylinders);
		elseif scale_factor ~= 1
			engine_out.name = sprintf( '%s converted to %0.2fL', obj.name, engine_out.displacement_L);
		end

		% Make meshgrid like array of speed for later use
		fuel_map_speed_axis = obj.fuel_consumption_gps.get_signal_axis('eng_spd_radps');
		fuel_map_mesh_speed_radps = obj.fuel_consumption_gps.get_signal_grid( 'eng_spd_radps');
		

		% Find torque axis on which
		fuel_map_fuel_brk_torque_axis = obj.fuel_consumption_gps.get_signal_axis('eng_fuel_trq_Nm');
		fuel_map_fuel_ind_torque_axis = obj.fuel_consumption_gps.get_signal_axis('eng_fuel_ind_trq_Nm');
		fuel_map_brk_torque_axis = obj.fuel_consumption_gps.get_signal_axis('eng_trq_Nm');
		fuel_map_ind_torque_axis = obj.fuel_consumption_gps.get_signal_axis('eng_ind_trq_Nm');
		fuel_map_torque_axis = [fuel_map_fuel_brk_torque_axis, fuel_map_fuel_ind_torque_axis, fuel_map_brk_torque_axis,fuel_map_ind_torque_axis];
			
		
		if numel( fuel_map_torque_axis) > 1
			warning('Engine fuel map contains multiple axes referencing engine torque which may interfere with scaling functions.');
		end

		if ~isempty(fuel_map_fuel_brk_torque_axis)
			fuel_map_mesh_brk_torque_Nm = obj.fuel_consumption_gps.get_signal_grid('eng_fuel_trq_Nm');
			fuel_map_mesh_ind_torque_Nm = fuel_map_mesh_brk_torque_Nm - obj.minimum_torque_Nm.interp(obj.fuel_consumption_gps.get_full_signal_grid{:});
		elseif ~isempty( fuel_map_fuel_ind_torque_axis)
			fuel_map_mesh_ind_torque_Nm = obj.fuel_consumption_gps.get_signal_grid('eng_fuel_ind_trq_Nm');
			fuel_map_mesh_brk_torque_Nm = fuel_map_mesh_ind_torque_Nm + obj.minimum_torque_Nm.interp(obj.fuel_consumption_gps.get_full_signal_grid{:});
		elseif ~isempty( fuel_map_brk_torque_axis)
			fuel_map_mesh_brk_torque_Nm = obj.fuel_consumption_gps.get_signal_grid('eng_trq_Nm');
			fuel_map_mesh_ind_torque_Nm = fuel_map_mesh_brk_torque_Nm - obj.minimum_torque_Nm.interp(obj.fuel_consumption_gps.get_full_signal_grid{:});
		elseif ~isempty( fuel_map_ind_torque_axis)
			fuel_map_mesh_ind_torque_Nm = obj.fuel_consumption_gps.get_signal_grid('eng_ind_trq_Nm');
			fuel_map_mesh_brk_torque_Nm = fuel_map_mesh_ind_torque_Nm + obj.minimum_torque_Nm.interp(obj.fuel_consumption_gps.get_full_signal_grid{:});		
		end

		
		%% Friction Scaling

		% New A. Moskalik Method - Heywood Paper -- need table entries for 10
		% cylinders! Need to do something for 2 cylinder engines, too...
		FMEP_factors = [nan, nan, 41.60, 41.02, nan, 41.38, nan, 40.97;
						nan, nan,  2.82,  2.55, nan,  2.34, nan,  2.21];
		input_engine_FMEP_kPa =		[1,obj.num_cylinders/ obj.displacement_L]	* FMEP_factors( :, max(3, min(obj.num_cylinders, 8)));
		output_engine_FMEP_kPa =	[1,engine_out.num_cylinders/ engine_out.displacement_L] * FMEP_factors( :, max(3, min(engine_out.num_cylinders, 8)));

		scaling_friction_adjust_Nm =  scaling_friction_adjust * (output_engine_FMEP_kPa - input_engine_FMEP_kPa) * engine_out.displacement_L ./ (4 * pi );

		%% Other (user defined) Friction Adjustment

		total_friction_adjust_Nm = scaling_friction_adjust_Nm + input_friction_adjust_Nm + input_friction_adjust_bar * 100 * engine_out.displacement_L ./ (4 * pi );

		%% Calculate BSFC adjustment for heat loss due to surface to volume effect

		if scaling_sv_ratio_adjust

			% A. Moskalik first take from Novak & Blumberg SAE # 780943
			% bsfc_mult =  1 + scaling_sv_ratio_adjust *  -0.27* log(cyl_disp_scale_factor.^(1/3) ) ;

			% A Moskalik - makes it reversible
			heat_loss_adjust =  (cyl_volume_scale_factor ^-(1/11.5));

			% Add Speed Rolloff
			heat_loss_adjust =  1+ (heat_loss_adjust -1 ) .* ( 1250 .* unit_convert.rpm2radps ./ max(fuel_map_mesh_speed_radps,100) );

		else
			heat_loss_adjust = 1.0;

		end

		%% Adjust for increased knock when upscaling

		if scaling_knock_adjust && cyl_volume_scale_factor > 1

			knock_compare_scale = cyl_volume_scale_factor ^(1/3);

			equiv_ind_torque_mesh_Nm = fuel_map_mesh_ind_torque_Nm .* knock_compare_scale;
			equiv_brake_torque_mesh_Nm = equiv_ind_torque_mesh_Nm + obj.minimum_torque_Nm.interp( obj.fuel_consumption_gps.get_full_signal_grid);
			equiv_fuel_gps = interp2( obj.nominal_fuel_map_speed_radps, obj.nominal_fuel_map_torque_Nm, obj.nominal_fuel_map_consumption_gps, fuel_map_mesh_speed_radps, equiv_brake_torque_mesh_Nm, 'spline' );

			knk_adjust_orig = equiv_fuel_gps ./ max(eps, ( obj.nominal_fuel_map_consumption_gps .* knock_compare_scale));
			knk_adjust_orig(1,:) = 1.0;
			knk_adjust_orig = max( knk_adjust_orig, 1.0);

			min_bsfc = REVS_calc_engine_min_bsfc(obj);
			
            approx_mbt_trq = interp1( min_consumption_speed_radps, min_consumption_torque_Nm, fuel_map_mesh_speed_radps, 'linear','extrap');

			wot_trq_Nm = interp1( obj.nominal_max_curve_speed_radps, obj.nominal_max_curve_torque_Nm,  fuel_map_mesh_speed_radps,  'linear','extrap' );

			select_pts = knk_adjust_orig > 1.0 & equiv_brake_torque_mesh_Nm > approx_mbt_trq * 0.95 & equiv_brake_torque_mesh_Nm < wot_trq_Nm ;

			speed_mesh_radps_limited = max(fuel_map_mesh_speed_radps, 50);
			b = REVS_regress( knk_adjust_orig(select_pts), [ ones(sum(select_pts(:)),1),  equiv_ind_torque_mesh_Nm(select_pts), 1./fuel_map_mesh_speed_radps(select_pts), speed_mesh_radps_limited(select_pts)]); %, equiv_ind_torque_mesh_Nm(select_pts) ./ speed_mesh_radps(select_pts)]);

			knk_adjust = [ ones(size(fuel_map_mesh_speed_radps(:))),  equiv_ind_torque_mesh_Nm(:),   1./speed_mesh_radps_limited(:), fuel_map_mesh_speed_radps(:)]*b;%, equiv_ind_torque_mesh_Nm(:) ./ speed_mesh_radps_limited(:) ] * b;
			knk_adjust = max(1.0, reshape( knk_adjust, size(fuel_map_mesh_speed_radps)));


			% Find contour corresponding to knk_adjust_max
			cdata = contourc(obj.nominal_fuel_map_speed_radps, obj.nominal_fuel_map_torque_Nm, knk_adjust,[knk_max_adjust,knk_max_adjust]);

			if ~isempty(cdata)
				knk_limit_speed_radps = cdata(1,2:end);
				knk_limit_torque_Nm = cdata(2,2:end);

				%combine speeds
				combined_speeds_radps = union(knk_limit_speed_radps,obj.nominal_max_curve_speed_radps);

				knk_limit_torque_Nm = interp1( knk_limit_speed_radps, knk_limit_torque_Nm,combined_speeds_radps,'linear','extrap');
				orig_full_torque_Nm = interp1( obj.nominal_max_curve_speed_radps, obj.nominal_max_curve_torque_Nm,combined_speeds_radps,'linear','extrap');

				combined_torque_Nm = min(knk_limit_torque_Nm , orig_full_torque_Nm)/2 + orig_full_torque_Nm/2;
				% 	engine_out.nominal_max_curve_speed_radps = 	combined_speeds_radps;


				seg_slope = diff( combined_torque_Nm ) ./ diff(combined_speeds_radps);
				same = abs(diff(seg_slope)) < 1e-10* max(combined_torque_Nm) ;
				keep = [true;~same(:);true];

				engine_out.nominal_max_curve_speed_radps = combined_speeds_radps(keep);
				engine_out.nominal_max_curve_torque_Nm = combined_torque_Nm(keep);

			end

		else
			knk_adjust = 1.0;
		end


		%% Scale Torque and adjust for friction
		
% 		% Nominal Min & Max Torque Curves
% 		engine_out.nominal_max_curve_torque_Nm		= obj.nominal_max_curve_torque_Nm .* (scale_factor) - total_friction_adjust_Nm;
% 		engine_out.nominal_min_curve_torque_Nm		= obj.nominal_min_curve_torque_Nm .* (scale_factor) - total_friction_adjust_Nm; %for torque curve
	
		% Full Min & Max Torque Maps
		if any(~contains(obj.maximum_torque_Nm.get_signal_array, {'speed_radps', 'temp_degC', 'speed_mps'}))
			warning('Maximum torque map may contain signals that do not allow proper scaling, verify resulting map');
		end
		
		engine_out.maximum_torque_Nm			= obj.maximum_torque_Nm .* (scale_factor) - total_friction_adjust_Nm;

		if any(~contains(obj.minimum_torque_Nm.get_signal_array, {'eng_spd_radps', 'temp_degC', 'speed_mps'}))
			warning('Minimum torque contains lookup signals thay may result in improper scaling verify resulting map');
		end
		
		engine_out.minimum_torque_Nm			= obj.minimum_torque_Nm  .* (scale_factor) - total_friction_adjust_Nm; %for torque curve
		
		%% Limit maximum torque if requested
		limit_torque = min(limit_torque, limit_bmep * engine_out.displacement_L * 100 / (4*pi));		
		if ~isnan(limit_torque)
			engine_out.maximum_torque_Nm = min(engine_out.maximum_torque_Nm, limit_torque);			
		end

% 		% Change refrenced mesh to the scaled engine
% 		[fuel_map_mesh_speed_radps, brake_torque_mesh_Nm] = meshgrid( engine_out.nominal_fuel_map_speed_radps, engine_out.nominal_fuel_map_torque_Nm);
% 
% 		if add_deacC
% 			deacFC_max_reduction = interp1( [3,4,6,8], [0.332, 0.382, 0.482, 0.582], engine_out.num_cylinders, 'nearest');
% 		%    deacFC_bmep          =                [-2,	   -1,		0,		1,			2,			3,			4,			5,			6,	7,	8,	9,	10,	11];
% 		%    deacFC_reduction     = min(add_deacC, [1		1		1		0.61129		0.343841	0.170669	0.068004	0.015265	0	0	0	0	0	0]) * deacFC_max_reduction;
% 
% 			deacFC_bmep          =                [-2,	   -1,		0,		0.5,	1,		1.5,	2,      2.5,    3,      3.5,    4,      4.5,    5,      5.5,    6,      6.1];
% 			deacFC_reduction     = min(add_deacC, [0.490	0.490	0.490	0.358	0.258   0.183   0.130   0.093   0.068   0.052   0.042	0.035	0.030	0.025	0.019	0] / 0.49 * deacFC_max_reduction);
% 
% 			bmep_mesh_bar = brake_torque_mesh_Nm .*( 4 * pi / (engine_out.displacement_L /1000)*1e-5 );
% 
% 			engine_out.deac_nominal_fuel_map_consumption_gps = engine_out.nominal_fuel_map_consumption_gps .* (1- interp1( deacFC_bmep, deacFC_reduction, bmep_mesh_bar, 'linear','extrap'));
% 
% 			deac_pts = fuel_map_mesh_speed_radps >= 940 * unit_convert.rpm2radps & bmep_mesh_bar < 6;
% 			engine_out.deac_nominal_fuel_map_consumption_gps( ~deac_pts ) = nan;
% 
% 			%	engine_out.deac_num_cylinders = engine_out.num_cylinders * add_deacC;
% 
% 			% add deac strategy if missing or if wrong type
% 			if engine_out.deac_strategy.type ~= enum_engine_deac_select.dynamic_lookup        
% 				engine_out.deac_strategy.enable_norm = class_REVS_dynamic_lookup;
% 				engine_out.deac_strategy.enable_norm.axis_1.signal =  'gear_num';
% 				engine_out.deac_strategy.enable_norm.axis_1.breakpoints = [ 1, 2, 3 ];
% 				engine_out.deac_strategy.enable_norm.axis_2.signal =  'eng_spd_radps';
% 				engine_out.deac_strategy.enable_norm.axis_2.breakpoints = [ 98.4,	 99.5,	 313.6,	 314.7 ];
% 				engine_out.deac_strategy.enable_norm.axis_3.signal =  'cycle_pos_secs';
% 				engine_out.deac_strategy.enable_norm.axis_3.breakpoints = [ 120.0,	 121.0 ];
% 				engine_out.deac_strategy.enable_norm.table(:,:,1) = [
% 					0.0, 0.0, 0.0, 0.0;
% 					0.0, 0.0, 0.0, 0.0;
% 					0.0, 0.0, 0.0, 0.0];
% 				engine_out.deac_strategy.enable_norm.table(:,:,2) = [
% 					0.00, 0.00, 0.00, 0.00;
% 					0.00, 0.85, 0.85, 0.00;
% 					0.00, 1.00, 1.00, 0.00];
% 				engine_out.deac_strategy.output_digital_bool =  false;
% 				engine_out.deac_strategy.type = enum_engine_deac_select.dynamic_lookup;
% 
% 				engine_out.deac_strategy.enable_threshold  = 0;
% 				engine_out.deac_strategy.disable_threshold = 0;
% 			end
% 		end
% 
% 		if add_deacD
% 			deacFC_max_reduction = interp1( [3,4,6,8], [0.332, 0.382, 0.482, 0.582], engine_out.num_cylinders,'nearest');
% 			deacFC_bmep          = [-2,		-1,		0,		1,			2,			3,			4,			5,			6,	7,	8,	9,	10,	11];
% 			deacFC_reduction     = min(add_deacD/2, [1		1		1		0.61129		0.343841	0.170669	0.068004	0.015265	0	0	0	0	0	0]) .* deacFC_max_reduction; % deacD/2 since discrete deac is less effective than continuous
% 
% 			bmep_mesh_bar = brake_torque_mesh_Nm .*( 4 * pi / (engine_out.displacement_L /1000)*1e-5 );
% 
% 			engine_out.deac_nominal_fuel_map_consumption_gps = engine_out.nominal_fuel_map_consumption_gps .* (1- interp1( deacFC_bmep, deacFC_reduction, bmep_mesh_bar, 'linear','extrap'));
% 
% 			deac_pts = fuel_map_mesh_speed_radps <= 4000 * unit_convert.rpm2radps & fuel_map_mesh_speed_radps >= 900 * unit_convert.rpm2radps & bmep_mesh_bar < 6 * (1-add_deacD);
% 			engine_out.deac_nominal_fuel_map_consumption_gps( ~deac_pts ) = nan;
% 
% 			%	engine_out.deac_num_cylinders = engine_out.num_cylinders * add_deacD;
% 
% 		%     % add deac strategy if missing or if wrong type
% 		%     if engine_out.deac_strategy.type ~= enum_engine_deac_select.constant
% 		%         engine_out.deac_strategy.type = enum_engine_deac_select.constant;
% 		%         engine_out.deac_strategy.enable_norm = 0.6;
% 		%     end
% 
% 			% add deac strategy if missing or if wrong type
% 			if engine_out.deac_strategy.type ~= enum_engine_deac_select.dynamic_lookup        
% 				engine_out.deac_strategy.enable_norm = class_REVS_dynamic_lookup;
% 				engine_out.deac_strategy.enable_norm.axis_1.signal =  'gear_num';
% 				engine_out.deac_strategy.enable_norm.axis_1.breakpoints = [ 1, 2, 3 ];
% 				engine_out.deac_strategy.enable_norm.axis_2.signal =  'eng_spd_radps';
% 				engine_out.deac_strategy.enable_norm.axis_2.breakpoints = [ 98.4,	 99.5,	 313.6,	 314.7 ];
% 				engine_out.deac_strategy.enable_norm.axis_3.signal =  'cycle_pos_secs';
% 				engine_out.deac_strategy.enable_norm.axis_3.breakpoints = [ 120.0,	 121.0 ];
% 				engine_out.deac_strategy.enable_norm.table(:,:,1) = [
% 					0.0, 0.0, 0.0, 0.0;
% 					0.0, 0.0, 0.0, 0.0;
% 					0.0, 0.0, 0.0, 0.0];
% 				engine_out.deac_strategy.enable_norm.table(:,:,2) = [
% 					0.0, 0.0, 0.0, 0.0;
% 					0.0, 0.0, 0.0, 0.0;
% 					0.0, 1.0, 1.0, 0.0];
% 
% 				engine_out.deac_strategy.output_digital_bool =  false;
% 				engine_out.deac_strategy.type = enum_engine_deac_select.dynamic_lookup;
% 
% 				engine_out.deac_strategy.enable_threshold  = 0;
% 				engine_out.deac_strategy.disable_threshold = 0;
% 			end    
% 		end

		end

		
	end
	
	methods (Static = true)
				
        function obj = convert_from_standard_engine(obj3)
        
            common_fields = 9;
            obj = 0;
            
        end
    end
	
    
end

