classdef class_REVS_dynamic_lookup < matlab.mixin.CustomDisplay
	%class_REVS_dynamic_lookup
	%   Definition of class_REVS_dynamic_lookup class
	
	properties
		table double;								% n dimensional matrix for interpolation
		axis_1 = class_REVS_dynamic_lookup_axis;	
		axis_2 = class_REVS_dynamic_lookup_axis;
		axis_3 = class_REVS_dynamic_lookup_axis;
		axis_4 = class_REVS_dynamic_lookup_axis;
		axis_5 = class_REVS_dynamic_lookup_axis;
		axis_6 = class_REVS_dynamic_lookup_axis;
		axis_7 = class_REVS_dynamic_lookup_axis;
		axis_8 = class_REVS_dynamic_lookup_axis;
		axis_9 = class_REVS_dynamic_lookup_axis;
		axis_10 = class_REVS_dynamic_lookup_axis;
		interp_method = 'linear';
		code_gen_dimensions = 5;
		code_gen_num_breakpoints = 10;
		code_gen_signal_strlen = 50;
		interp_defaults = {}
		signal_filter = '';							% Signals to disallow from being selected (generally to avoid algebraic loops) 
	end
	
	properties (Dependent = true)
% 		mdl_vars;		
		num_dimensions;
 		table_size uint32;							% Calculated table size for simulink
		signal_list;
		breakpoints_max;
		breakpoints_min;
				
	end
	
	methods
		
		% Constructor
		function obj = class_REVS_dynamic_lookup( varargin )
			if nargin == 0
				% No Arguments - blank lookup
			elseif mod(nargin,2) ~= 1
				error('Incorrect number of input arguments, must be signal & breakpoint pairs followed by the interpolation table');
			else
				num_axes = (length(varargin)-1)/2;
				
				for a = 1:num_axes
					axis_str = sprintf('axis_%d',a);
					obj.(axis_str).signal = varargin{2*a-1};
					obj.(axis_str).breakpoints = varargin{2*a};
				end
				
				obj.table = varargin{end};
				
			end
		end
		
		
		function val = isempty(obj)
		% Replacement for isempty, class is empty when table and axes are empty
			val = isempty(obj.table) && isempty(obj.axis_1.signal) && isempty(obj.axis_1.breakpoints);
		end
		
		
		function val = get.table_size(obj)
			val = size(obj.table) - 1;
			val = uint32(val(val > 0));
		end
        
		function val = get.num_dimensions(obj)
			val = length(obj.table_size);
		end
		

		function val = get.signal_list(obj)	
		
			bus_signals = obj.get_signal_array();
			% combine signals into a single string
 			val = uint8([strjoin(bus_signals,','),0]);
			
		end

		function val = get.breakpoints_max(obj)
			
			val = zeros(1,obj.num_dimensions);
			
			for a = 1:obj.num_dimensions
				axis_str = ['axis_',int2str(a)];
				val(a) = obj.(axis_str).breakpoints_max;
			end
			
		end
		
		function val = get.breakpoints_min(obj)
			
			val = zeros(1,obj.num_dimensions);
			
			for a = 1:obj.num_dimensions
				axis_str = ['axis_',int2str(a)];
				val(a) = obj.(axis_str).breakpoints_min;
			end
			
		end
		
		
		function val = get_signal_array(obj)
			val = cell(obj.num_dimensions,1);
			for a = obj.num_dimensions:-1:1
				axis_str = ['axis_',int2str(a)];
				val{a} = obj.(axis_str).signal;
			end		
		end
		
		function val = get_signal_axis(obj, signal)
			signals = obj.get_signal_array;
			val = find( regexpcmp(signals,['^(.*\.)?', signal, '$']) );		
		end
		
		function val = get_signal_breakpoints( obj, signal)
			sig_axis = obj.get_signal_axis( signal);
			if isempty(sig_axis)
				val = [];
			else
				sig_axis_str = sprintf('axis_%d',sig_axis);
				val = obj.(sig_axis_str).breakpoints;
			end
		end
	
		function val = get_signal_grid(obj, signal)
			% Make meshgrid like array of breakpoints for the given signal
			sig_axis = obj.get_signal_axis(signal);
			if isempty( sig_axis )
				val = nan(obj.table_size);
				return;
			end
			
			sig_axis_str = sprintf('axis_%d',sig_axis);
			val = obj.(sig_axis_str).breakpoints;
			
			
            if obj.num_dimensions > 1
                % Expand and reshape to match table
                
                reshape_sz = ones(1, obj.num_dimensions);
                reshape_sz( sig_axis ) = numel( val );
           
            	repmat_sz = size(obj.table);
                repmat_sz( sig_axis ) = 1;
			
                val = repmat( reshape( val, reshape_sz ), repmat_sz);
            end

		end
		
		function val = get_full_signal_grid( obj )
			% Return cell array containing pairs of signal name and
			% signal mapping for interpolating witin other dynamic lookup
			% objects
			
			val = cell(obj.num_dimensions * 2,1);
			table_sz = obj.table_size+1;
            
            for a = obj.num_dimensions:-1:1
				axis_str = ['axis_',int2str(a)];
				val{2*a-1} = obj.(axis_str).signal;

				reshape_sz = ones(1, obj.num_dimensions);
				reshape_sz( a ) = table_sz(a);

				repmat_sz = table_sz;
				repmat_sz( a ) = 1;

				val{2*a} = repmat( reshape( obj.(axis_str).breakpoints , reshape_sz ), repmat_sz);
								
			end
			
		end
				
		function val = interp(obj, varargin)
			
			breakpoints = cell(obj.num_dimensions,1);
			lookup_pts = cell(obj.num_dimensions,1);
            
            if isscalar(varargin) && iscell(varargin{1})
                bus_signals = varargin{1};
            else
                bus_signals = varargin;
            end
                
            
            lookup_size = [];
			
			for a = 1:obj.num_dimensions
				axis_str = sprintf('axis_%d',a);
				signal_str =  obj.(axis_str).signal;
				breakpoints{a} = obj.(axis_str).breakpoints;
                
                idx = length( bus_signals )-1;
                while idx > 0 && ~strcmpi( bus_signals{idx}, signal_str)
                   idx = idx -2;                     
                end
                                
                if idx <= 0
                    error('Unable to interpolate, the signal %s for axis %d was not provided', signal_str, a)
                end
                
                
				lookup_pts{a} = bus_signals{idx+1};
                				
				if ~ obj.(axis_str).extrapolate_above
					lookup_pts{a} = min( lookup_pts{a}, obj.(axis_str).breakpoints(end) );
				end
				
				if ~ obj.(axis_str).extrapolate_below
					lookup_pts{a} = max( lookup_pts{a}, obj.(axis_str).breakpoints(1) );
                end
                
                if isscalar(lookup_pts{a})
                    % Scalar - All Good
                elseif isempty(lookup_size)
                    lookup_size = size( lookup_pts{a});               
                elseif size( lookup_pts{a}) ~= lookup_size
                    error('Unable to interpolate, inconsistent input sizes')
                end
                
                
            end
			
            % Expand any scalars           
            if ~isempty(lookup_size )
                expander = ones(lookup_size);
                for a = 1:length(lookup_pts)
                    if isscalar(lookup_pts{a})
                        lookup_pts{a} = lookup_pts{a} .* expander;
                    end
                end
            end
            
            
            val = interpn( breakpoints{:}, obj.table, lookup_pts{:} );
			
		end
		
		function val  = properties2export( obj, as_class)
			
			val = {};
			
			if ~isempty(obj)
				
				for a = 1:10
					axis_str = ['axis_',int2str(a)];
					if ~isempty(obj.(axis_str)) && (length(obj.table_size) >= a || ~isempty( obj.(axis_str).signal) )
						val{end+1} = [axis_str,'.signal'];
						val{end+1} = [axis_str,'.breakpoints'];
						
						if obj.(axis_str).extrapolate_above || obj.(axis_str).extrapolate_below
							val{end+1} = [axis_str,'.extrapolate_above'];
							val{end+1} = [axis_str,'.extrapolate_below'];
						end
					end
				end
				
				val{end+1} = 'table';
				
				if ~strcmpi( obj.interp_method ,'linear')
					val{end+1} = 'interp_method';
				end
				
				if nargin > 1 && ~as_class
					val{end+1} = 'table_size';
				end
				
			end
			
		end
		
		function val = mod_for_code_gen(obj)
		
			% convert object to a structure with similar format
			val = struct;
			
			signal_list_int8 = int8( zeros( 1, obj.code_gen_signal_strlen*obj.code_gen_dimensions));
			signal_list_int8( 1:length( obj.signal_list) ) =  obj.signal_list;
			val.signal_list = signal_list_int8;
			
			val.table = obj.table;
% 			val.code_gen_dimensions = max(obj.code_gen_dimensions, obj.num_dimensions);
 			val.num_dimensions = max(obj.code_gen_dimensions, obj.num_dimensions);
			val.table_size = obj.code_gen_num_breakpoints * ones( 1, val.num_dimensions );
			
			% Preallocate Breakpoint Limits (for controlling extrapolation)
			val.breakpoints_max = zeros(1,val.num_dimensions);
			val.breakpoints_min = zeros(1,val.num_dimensions);			
			
			% Convert existing table dimensions
			for a = 1:obj.num_dimensions
				axis_str = ['axis_',int2str(a)];
				
				% Copy and extend breakpoints to min number of pts
				val.table_size(a) = max(length(obj.(axis_str).breakpoints), obj.code_gen_num_breakpoints);
				in_breakpoints{a} = obj.(axis_str).breakpoints;	
				out_breakpoints{a} = interp1(1:length(in_breakpoints{a}), obj.(axis_str).breakpoints, 1:val.table_size(a),'linear','extrap');							
				
				% Store Extended Breakpoints
				val.(axis_str).breakpoints = out_breakpoints{a};
				
				% Store Limits based on extrapolation
				val.breakpoints_max(a) = obj.(axis_str).breakpoints_max;
				val.breakpoints_min(a) = obj.(axis_str).breakpoints_min;
				
			end
			
			%Add dimensions to reach limit
			for a = (obj.num_dimensions+1):val.num_dimensions
				axis_str = ['axis_',int2str(a)];
				val.(axis_str).breakpoints = 0:obj.code_gen_num_breakpoints-1;
			end
							
			% Construct grid from output breakpoints & extrapolate table
			[out_breakpoints{:}] = ndgrid(out_breakpoints{:});
			val.table = interpn(in_breakpoints{:}, obj.table, out_breakpoints{:}, 'makima' ); 
	
			% Add additional dimensions to table
			expand_rep = obj.code_gen_num_breakpoints * ones(1,val.num_dimensions);
			expand_rep( 1:obj.num_dimensions) = 1;
			val.table = repmat( val.table, expand_rep);
			
			% Write table size
			val.table_size = uint32(val.table_size - 1);

			
		end
		
	end
	
	methods ( Access = protected)
		
		% Select properties for display
		function val = getPropertyGroups(obj)
			val = matlab.mixin.util.PropertyGroup(obj.show_active_fields);
		end
		
		% Determine which fields are used for display &
		function val = show_active_fields(obj)
			
			val = {'axis_1'};
			
			for a = 2:10
				axis_str = ['axis_',int2str(a)];
				
				if ~isempty(obj.(axis_str)) && (length(obj.table_size) >= a || ~isempty( obj.(axis_str).signal)  )
					val{end+1,1} = axis_str;
				end
				
			end
			val{end+1,1} = 'table';
			val{end+1,1} = 'interp_method';
			val{end+1,1} = 'table_size';
			val{end+1,1} = 'code_gen_dimensions';
			val{end+1,1} = 'code_gen_num_breakpoints';
			val{end+1,1} = 'code_gen_signal_strlen';
			val{end+1,1} = 'signal_filter';
		end
		
	end
	
	methods( Hidden = true )
		
% 		% Override fieldnames only showing desired fields
% 		function val = fieldnames(obj)
% 			val = obj.show_active_fields;
% 		end
% 		
% 		% Override fieldnames only showing desired fields
% 		function val = properties(obj)
% 			val = obj.show_active_fields;
%         end
        
		function val = plus( lu1, lu2 )
			if isa( lu1, 'class_REVS_dynamic_lookup') && isa( lu2, 'class_REVS_dynamic_lookup')			
				[val, interp_args] = class_REVS_dynamic_lookup.operator_prep( lu1, lu2 );			
				val.table = lu1.interp( interp_args{:}) + lu2.interp(interp_args{:} );
			elseif isa( lu1, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu1;
				val.table = val.table + lu2;
			elseif isa( lu2, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu2;
				val.table = lu1 + val.table;				
			else
				error('Undefined operator ''+'' for input arguments of type %s and %s', class(lu1), class(lu2) );
			end
		end
        
		
		function val = minus( lu1, lu2 )
			if isa( lu1, 'class_REVS_dynamic_lookup') && isa( lu2, 'class_REVS_dynamic_lookup')			
				[val, interp_args] = class_REVS_dynamic_lookup.operator_prep( lu1, lu2 );			
				val.table = lu1.interp( interp_args{:}) - lu2.interp(interp_args{:} );
			elseif isa( lu1, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu1;
				val.table = val.table - lu2;
			elseif isa( lu2, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu2;
				val.table = lu1 - val.table;				
			else
				error('Undefined operator ''-'' for input arguments of type %s and %s', class(lu1), class(lu2) );
			end
		end
			
		function val = uminus( lu1 )
			val = lu1;
			val.table = -val.table;
		end
	
		function val = uplus( lu1 )
			val = lu1;
        end
		
        function val = abs( lul )            
           val = lul;
           val.table = abs(val.table);
        end
        
		function val = times( lu1, lu2 )
			if isa( lu1, 'class_REVS_dynamic_lookup') && isa( lu2, 'class_REVS_dynamic_lookup')			
				[val, interp_args] = class_REVS_dynamic_lookup.operator_prep( lu1, lu2 );			
				val.table = lu1.interp( interp_args{:}) .* lu2.interp(interp_args{:} );
			elseif isa( lu1, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu1;
				val.table = val.table .* lu2;
			elseif isa( lu2, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu2;
				val.table = lu1 .* val.table ;				
			else
				error('Undefined operator ''.*'' for input arguments of type %s and %s', class(lu1), class(lu2) );
			end
		end
		
		function val = mtimes( lu1, lu2 )
			if isa( lu1, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu1;
				val.table = val.table * lu2;
			elseif isa( lu2, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu2;
				val.table = lu1 * val.table ;				
			else
				error('Undefined operator ''*'' for input arguments of type %s and %s', class(lu1), class(lu2) );
			end
		end
		
		function val = rdivide( lu1, lu2 )
			if isa( lu1, 'class_REVS_dynamic_lookup') && isa( lu2, 'class_REVS_dynamic_lookup')			
				[val, interp_args] = class_REVS_dynamic_lookup.operator_prep( lu1, lu2 );			
				val.table = lu1.interp( interp_args{:}) ./ lu2.interp(interp_args{:} );
			elseif isa( lu1, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu1;
				val.table = val.table ./ lu2;
			elseif isa( lu2, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu2;
				val.table = lu1 ./ val.table;				
			else
				error('Undefined operator ''./'' for input arguments of type %s and %s', class(lu1), class(lu2) );
			end
		end
		
		function val = power( lu1, lu2 )
			if isa( lu1, 'class_REVS_dynamic_lookup') && isa( lu2, 'class_REVS_dynamic_lookup')			
				[val, interp_args] = class_REVS_dynamic_lookup.operator_prep( lu1, lu2 );			
				val.table = lu1.interp( interp_args{:}) .^ lu2.interp(interp_args{:} );
			elseif isa( lu1, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu1;
				val.table = val.table .^ lu2;
			elseif isa( lu2, 'class_REVS_dynamic_lookup') && isnumeric(lu2) && isscalar(lu2)
				val = lu2;
				val.table = lu1 .^ val.table ;				
			else
				error('Undefined operator ''.^'' for input arguments of type %s and %s', class(lu1), class(lu2) );
			end
		end
	end
			
	methods (Static = true , Hidden = true)
		
		
		function [val, interp_args] = operator_prep( lu1, lu2 )
			
			% Combine sommon signals and breakpoints
			
			signals = {};
			breakpoints = {};
			
			% Extract lookup 1 data
			for a = 1:lu1.num_dimensions
				axis_str = ['axis_',int2str(a)];
				signals{end+1} = lu1.(axis_str).signal;
				breakpoints{end+1} =  lu1.(axis_str).breakpoints;
			end
			
			% Add lookup 2 data
			for a = 1:lu2.num_dimensions				
				axis_str = ['axis_',int2str(a)];	
				[rep, sig_idx] = ismember( lu2.(axis_str).signal, signals );
				if rep					
					breakpoints{sig_idx} = unique( [breakpoints{sig_idx}, lu2.(axis_str).breakpoints ] );
				else % Add to the list
					signals{end+1} = lu2.(axis_str).signal;
					breakpoints{end+1} =  lu2.(axis_str).breakpoints;
				end	
			end
			
			% Make lookup from combined signals and breakpoints
			val = class_REVS_dynamic_lookup;
			
			for a = 1:numel(signals)
				axis_str = ['axis_',int2str(a)];			
				val.(axis_str).signal = signals{a};
				val.(axis_str).breakpoints = breakpoints{a};
			end
			
			% combine arguments for interpolation
			interp_args = [ signals ; breakpoints];
						
		end
					
	end
		
end

