classdef class_REVS_emachine
    %REVS_emachine
    %   REVS Electric Machine class definition
    
    properties
        
        name = '';
        source_filename = '';
        application_properties = [];
        
   		electrical_source = 'propulsion';   % electrical bus name prefix

        inertia_kgm2 = 0;                   % Inertia [kg m^2]
        
        max_speed_radps                     % Max speed [rad/s]
        max_torque_Nm                       % Max torque [Nm]
        max_motor_power_W                   % Max motor power [W]
        max_generator_power_W               % Max regen power [W]
%         max_speed_max_motor_power_W         % Max speed max motor power [W]    
%         max_speed_max_regen_power_W         % Max speed max regen power [W]
                
     	%max_torque_curve_speed_radps        % Max torque curve speed [rad/s]
        positive_torque_limit_Nm        class_REVS_dynamic_lookup = class_REVS_dynamic_lookup;   % Max motor torque curve [Nm]
        negative_torque_limit_Nm        class_REVS_dynamic_lookup = class_REVS_dynamic_lookup; 	% Max regen torque curve [Nm]

        electric_power_W                class_REVS_dynamic_lookup = class_REVS_dynamic_lookup;	% Power map [W]
        unpowered_torque_loss_Nm        class_REVS_dynamic_lookup = class_REVS_dynamic_lookup;	% Spin loss torque [Nm]
        
         
        interp_signal_defaults = {};             % Default values to use for extra dynamic lookup axes stored as n by 2 cell array of signal names and scalar values
        power_scaled_ratio;
        gen_raw_data_in;
    end
    
    
    
    properties ( Dependent )        

    end
    
    properties ( Hidden = true, Transient = true )
		disable_auto_calc = false;          % Used to supress automatic calculation of dependent properties [bool]
    end
    
    properties ( Hidden = true)
        has_pos_torque_limit = false;
        has_neg_torque_limit = false
	end
    
    methods


        function val = get.max_torque_Nm(obj)
           
            if obj.disable_auto_calc || ~isempty( obj.max_torque_Nm)
                val = obj.max_torque_Nm;
            elseif obj.has_pos_torque_limit || obj.has_neg_torque_limit             
                val = max([max(obj.negative_torque_limit_Nm.table), max(obj.positive_torque_limit_Nm.table)] );
            else
                val = obj.max_torque_Nm;
            end
            
        end
               
        function val  = get.max_motor_power_W(obj)
            
            if obj.disable_auto_calc || ~isempty( obj.max_motor_power_W)
                val = obj.max_motor_power_W;
            elseif obj.has_pos_torque_limit                  
                pwr = obj.positive_torque_limit_Nm.table .* obj.positive_torque_limit_Nm.get_signal_grid('emach_spd_radps');
                val = max( pwr(:) );
            else
                 val = obj.max_motor_power_W;
            end
            
        end

        function val  = get.max_generator_power_W(obj)
            
            if obj.disable_auto_calc || ~isempty( obj.max_generator_power_W)
                val = obj.max_generator_power_W;
            elseif obj.has_neg_torque_limit                
                pwr = obj.negative_torque_limit_Nm.table .* obj.negative_torque_limit_Nm.get_signal_grid('emach_spd_radps');
                val = max( -pwr(:) );
            else
                val = obj.max_generator_power_W;
            end
            
        end
        
        
        
        
        function val= get.positive_torque_limit_Nm(obj)
            
            if obj.disable_auto_calc || obj.has_pos_torque_limit
                val = obj.positive_torque_limit_Nm;
                               
            elseif  ~isempty( obj.max_torque_Nm) && ~isempty( obj.max_motor_power_W) && ~isempty( obj.max_generator_power_W)
                
                c = obj.make_power_curves();
                
                val = obj.positive_torque_limit_Nm;
                val.axis_1.signal = 'emach_spd_radps';
                val.axis_1.breakpoints = [-fliplr(c.gen_speeds_radps), 0, c.mot_speeds_radps];
                val.table = [fliplr(c.gen_max_torque_Nm), obj.max_torque_Nm, c.mot_max_torque_Nm];                                
            else
                val = obj.positive_torque_limit_Nm;           
            end
            
            
        end
        
        function obj = set.positive_torque_limit_Nm(obj, val)
            obj.positive_torque_limit_Nm = val;
            obj.has_pos_torque_limit = true;
        end
        
        function val= get.negative_torque_limit_Nm(obj)
            
            if obj.disable_auto_calc || obj.has_neg_torque_limit
                val = obj.negative_torque_limit_Nm;
                               
            elseif  ~isempty( obj.max_torque_Nm) && ~isempty( obj.max_motor_power_W) && ~isempty( obj.max_generator_power_W)                
                                c = obj.make_power_curves();
                
                val = obj.positive_torque_limit_Nm;
                val.axis_1.signal = 'emach_spd_radps';
                val.axis_1.breakpoints = [-fliplr(c.mot_speeds_radps), 0, c.gen_speeds_radps];
                val.table = -[fliplr(c.mot_max_torque_Nm), obj.max_torque_Nm, c.gen_max_torque_Nm];              
            else                
                val = obj.negative_torque_limit_Nm;
            end
            
        end
        
        
        function val= get.unpowered_torque_loss_Nm(obj)
            
            if ~isempty(obj.unpowered_torque_loss_Nm.table)
                val = obj.unpowered_torque_loss_Nm;
                               
            else
                
                val = obj.unpowered_torque_loss_Nm;
                val.axis_1.signal = 'emach_spd_radps';
                val.axis_1.breakpoints = [-15000, 15000];
                val.table = [0, 0];              

            end
            
        end
        
        function obj = set.negative_torque_limit_Nm(obj, val)
            obj.negative_torque_limit_Nm = val;
            obj.has_neg_torque_limit = true;
        end
        
        
        function obj = load_data(obj, data, varargin)
            
            scale_axes                   = parse_varargs(varargin,'scale_axes',(false),'toggle');
            smoothness_value             = parse_varargs(varargin,'smoothness',0.0005);
            
                
            idx_data = [];
            val_data = [];
            
            for d = 1:length(data) 
                
                if scale_axes            
                    data(d).speed_rpm =  data(d).speed_rpm./max(data(d).speed_rpm).*(obj.max_speed_radps.*unit_convert.radps2rpm);
                    data(d).torque_Nm = data(d).torque_Nm./max(data(d).torque_Nm).*obj.max_torque_Nm;              
                end
                
               [pts, val, loss_W] = process_input(data(d));                 
               idx_data = vertcat(idx_data, pts);
               val_data = vertcat( val_data, val);               
            end
            

            
           if size(idx_data,2) <=2
               signals = {'emach_trq_Nm', 'emach_spd_radps' };
            elseif max(idx_data(:,3)) / max( 1e-3, min(idx_data(:,3))) < 1.2
               disp('Note: Since, the loaded input voltage data for this dataset has a limited range there is no voltage dimension in the resulting power consumption table.');
               idx_data = idx_data(:,1:2);
               signals = {'emach_trq_Nm', 'emach_spd_radps'};
           else
               signals = {'emach_trq_Nm', 'emach_spd_radps', 'emach_voltage_V'};
           end
           

           
           % Normalized data for interpolation
           data_range = max(idx_data,[],1) - min(idx_data,[],1);
           idx_norm = idx_data ./ data_range;
           
           
           % Generate breakpoints for each axis
           breakpoints = cell(size(idx_data,2), 1);
           for i = 1:size(idx_data,2)
               
               % Find clusters in the data
               breakpoints{i} = find_clusters(idx_norm(:,i));  
               
               % Handle upper limit                            
                if i <= 2 && breakpoints{i}(end) <= -0.05
                    % Add breakpoint at 0
                    breakpoints{i} = [breakpoints{i}, 0]; 
               elseif i <= 2 && breakpoints{i}(end) <= 0.02
                    % Move max breakpoint to zero  
                    breakpoints{i}(end) = 0;  
                elseif max(idx_norm(:,i)) > breakpoints{i}(end) + 0.05
                    % Some points beyond last breakpoint
                   breakpoints{i} = [breakpoints{i},  max(idx_norm(:,i))];                    
                elseif max(idx_norm(:,i)) > breakpoints{i}(end)
                     % Some points beyond but near last breakpoint
                    breakpoints{i}(end) =   max(idx_norm(:,i));
                end                  
   
                    
                if i <= 2 && breakpoints{i}(1) >= 0.05
                    % Add breakpoint at 0
                    breakpoints{i} = [0, breakpoints{i}]; 
               elseif i <= 2 && breakpoints{i}(1) >= -0.02
                    % Move max breakpoint to zero  
                    breakpoints{i} = [0, breakpoints{i}(2:end)];  
               elseif min(idx_norm(:,i)) < breakpoints{i}(1) - 0.05
                    breakpoints{i} = [  min(idx_norm(:,i)), breakpoints{i}];
               elseif min(idx_norm(:,i)) < breakpoints{i}(1)
                    breakpoints{i}(1) = min(idx_norm(:,i));
               end        
           end
           
%% Estimate Electric power from Powerloss domain

%            loss_power_W = ndgridfit( idx_norm, loss_W, [], breakpoints, 'smoothness',smoothness_value);         
           
           
           elec_power_W = ndgridfit( idx_norm, val_data, [], breakpoints, 'smoothness',smoothness_value);
           elec_power_W = permute( elec_power_W, [2,1,3:ndims(elec_power_W)]);
           

           
           % Denormalize breakpoints
           for i = 1:size(idx_data,2)
               breakpoints{i} = breakpoints{i} * data_range(i);
           end
           
          
           if breakpoints{1}(1) == 0
               % Only Positive Torques - mirror over axis w/ adjustment for power losses
              	mech_power_W = breakpoints{1,1}'*breakpoints{2,1};
              	power_loss_W = (elec_power_W - mech_power_W) * data(1).mirrorfactor;
             	gen_elec_power = -mech_power_W + power_loss_W;
             	elec_power_W = [ gen_elec_power(end:-1:2,:,:) ; elec_power_W];
            	breakpoints{1} = [-breakpoints{1}(end:-1:2), breakpoints{1}];  
 
           elseif breakpoints{1}(end) == 0 
               % Only Negative Torques - mirror over axis w/ adjustment for power losses
              	mech_power_W = breakpoints{1,1}'*breakpoints{2,1};
              	power_loss_W = mech_power_W - elec_power_W;
             	mot_elec_power = -mech_power_W + power_loss_W;
             	elec_power_W = [ elec_power_W(end:-1:2,:,:) ; mot_elec_power];
            	breakpoints{1} = [-breakpoints{1}(end:-1:2), breakpoints{1}];     
           end
           

            if min(breakpoints{1}) > -max(breakpoints{1})
               	%Blending data to make generator limit torque = motoring limit
%               torque
                mech_power_W = breakpoints{1,1}'*breakpoints{2,1};
                dataset = [data.speed_rpm(data.torque_Nm < 0)*pi/30, -data.torque_Nm(data.torque_Nm < 0)]; % Test data set
                [gridset_N, gridset_T] = meshgrid(breakpoints{2,1},breakpoints{1,1}); % Create speed and torque grid
                gridset = [gridset_N(gridset_T > 0),gridset_T(gridset_T > 0)]; % Select the motoring side speed and torque
                setidx = ~ismembertol(gridset,dataset,0.1,'ByRows',true); % compare the grid data against the true test data and find the index for those points to be copied to gen side
                
                addlBN = gridset(setidx,1); 
                addlBT = -gridset(setidx,2);
                elecP = -elec_power_W(gridset_T > 0);
                lossP = abs(addlBN.*addlBT - elecP(setidx));                
                
                mirror_factor =  0.89; % for VW.ID.3 ; % 0.94 for Tesla Model 3
                addlEP = addlBN.*addlBT + lossP./mirror_factor;
                
                blend_idx = ((breakpoints{1}) > -min(breakpoints{1}) & breakpoints{1} > 0);
                
                breakpoints_1_blend = ((breakpoints{1,1}(blend_idx)));                                      
             
                breakpoints{1} = [-fliplr(breakpoints_1_blend), breakpoints{1}];
                [Ngrid, Tgrid] = meshgrid(breakpoints{2,1},breakpoints{1,1}); % Create speed and torque grid
                
                elec_power_W = griddata([gridset_N(:);addlBN],[gridset_T(:);addlBT],[elec_power_W(:);addlEP],...
                    Ngrid,Tgrid);                
            end
          
          
           if all( breakpoints{2} >= 0 )
               elec_power_W = [ rot90(elec_power_W(:,2:end,:),2), elec_power_W];
               breakpoints{2} = [-breakpoints{2}(end:-1:2), breakpoints{2}];
           end
           

           
        	index        = cell(1, ndims(obj.electric_power_W));
            index(:)     = {':'};
           for d = 1:length(signals)
               axis_str = sprintf('axis_%d', d);
               obj.electric_power_W.(axis_str).signal = signals{d};
               obj.electric_power_W.(axis_str).breakpoints = breakpoints{d};
               if any(breakpoints{d} == 0)
                   index{d} = find(breakpoints{d} == 0 );    %& elec_power_W<0 
               end
%                smoothness_value
           end
           
%            elec_power_W(index{:}) = 0;
             
           obj.electric_power_W.table = elec_power_W;                    
           
            function valid = is_valid_data(data, field)              
                valid = isfield(data, field) && ~isempty( data.(field)) && any(~isnan(data.(field)),'all');
            end
           
            function [axes_data, elec_W, loss_W] = process_input(d_in)
                                
                if is_valid_data(d_in, 'speed_radps') 
                    speed_radps = d_in.speed_radps;
                elseif is_valid_data( d_in, 'speed_rpm')
                    speed_radps = d_in.speed_rpm .* unit_convert.rpm2radps;
                else
                    error('Provided data must include a valid speed signal');
                end
                
                if is_valid_data(d_in, 'torque_Nm')
                    torque_Nm = d_in.torque_Nm;
                else
                    error('Provided data must include a valid torque signal');
                end
                
                if is_valid_data( d_in, 'voltage_V')
                    voltage_V = d_in.voltage_V;
                else
                    voltage_V = [];
                end
                
                if is_valid_data(d_in, 'elec_power_W')
                    elec_W = d_in.elec_power_W;
                elseif is_valid_data( d_in, 'voltage_V') && is_valid_data( d_in, 'current_A')
                    elec_W = d_in.voltage_V .* d_in.current_A;
                elseif is_valid_data(d_in, 'loss_power_W')
                    mech_W = torque_Nm .* speed_radps;
                    loss_W = d_in.loss_power_W;
                    elec_W = mech_W + loss_W;
                elseif is_valid_data( d_in, 'efficiency_norm')
                    eff_norm  = d_in.efficiency_norm ;
                    if numel(torque_Nm) == numel(eff_norm) && numel(speed_radps) == numel(eff_norm)
                         mech_W = torque_Nm .* speed_radps;
                         elec_W =  mech_W./ eff_norm;
                         elec_W(elec_W<mech_W) = mech_W(elec_W<mech_W).*eff_norm(elec_W<mech_W);
                    elseif numel( speed_radps) == size( eff_norm,1) && numel(torque_Nm) == size( eff_norm,2)
                        mech_W = torque_Nm(:) * speed_radps(:)';
                        elec_W =  mech_W./ eff_norm';
                        eff_norm_t = eff_norm';
                        elec_W(elec_W<mech_W) = mech_W(elec_W<mech_W).*eff_norm_t(elec_W<mech_W);
                    elseif numel( speed_radps) == size( eff_norm,2) && numel(torque_Nm) == size( eff_norm,1)
                        mech_W = torque_Nm(:) * speed_radps(:)';
                        elec_W =  mech_W./ eff_norm;
                        elec_W(elec_W<mech_W) = mech_W(elec_W<mech_W).*eff_norm(elec_W<mech_W);
                    else
                        error('Unable to compute electrical power consumption, potential data dimension mismatch');
                    end                   
                else
                    error('Provided data must include or be able to compute electrical power consumption');
                   
                end

                speed_radps = repmat( speed_radps, size( elec_W)./ size( speed_radps));
                torque_Nm = repmat( torque_Nm, size( elec_W)./ size( torque_Nm));
                voltage_V = repmat( voltage_V, size( elec_W)./ max(1,size( voltage_V)));
                
                axes_data = [ torque_Nm(:), speed_radps(:), voltage_V(:) ];              
                elec_W = elec_W(:);
                idx_q1_q2 = axes_data(:,1) >=0;
                loss_W = abs(elec_W) - abs(axes_data(:,1).*axes_data(:,2));
                loss_W(~idx_q1_q2) = abs(axes_data(~idx_q1_q2,1).*axes_data(~idx_q1_q2,2)) - abs(elec_W(~idx_q1_q2));
                
                
                rmv = any(isnan( axes_data),2) | isnan(elec_W);
                axes_data(rmv,:) = [];
                elec_W(rmv) = [];
                loss_W(rmv) = [];
                
                
            end
            
           
        end
        
        function [speed_radps, torque_Nm, elec_power_W] = get_electric_power_map(obj, varargin)
        % Generate simplified (2D) map of electrical consumption / generation based from dynamic lookup.
        % Use default or provided variables for additional axis signals      
        
             % Get Axis Breakpoints
             speed_radps = obj.electric_power_W.get_signal_breakpoints( 'emach_spd_radps');
             torque_Nm = obj.electric_power_W.get_signal_breakpoints( 'emach_trq_Nm');       
             voltage_V = obj.electric_power_W.get_signal_breakpoints( 'emach_voltage_V');  
                          
             % Make mesh for interpolation
             [mesh_speed_radps, mesh_torque_Nm] =  meshgrid( speed_radps, torque_Nm);
                                      
             elec_power_W = obj.electric_power_W.interp( 'emach_voltage_V', median(voltage_V),...
                 obj.interp_signal_defaults{:},varargin{:}, ...  
                 'emach_spd_radps', mesh_speed_radps, 'emach_trq_Nm',  mesh_torque_Nm);
            
        end
        
        
        function [speed_radps, torque_Nm, efficiency_norm] = get_efficiency_map(obj)
           
            [speed_radps, torque_Nm, elec_power_W] = obj.get_electric_power_map(obj, varargin);             
             mech_power_W = speed_radps .* torque_Nm;
             
             efficiency_norm = (1- abs( mech_power_W - elec_power_W ) ./ max(abs(elec_power_W), abs(mech_power_W)));
                         
        end
                      
        function [speed_radps, torque_Nm] = get_max_torque_curve(obj,  varargin)
            speed_radps = obj.positive_torque_limit_Nm.get_signal_breakpoints('emach_spd_radps');
            voltage_V = obj.electric_power_W.get_signal_breakpoints( 'emach_voltage_V');
            
            torque_Nm = max(0.0, obj.positive_torque_limit_Nm.interp('emach_voltage_V', median(voltage_V),...
                obj.interp_signal_defaults{:},varargin{:},...
                'emach_spd_radps',speed_radps));

                 
        end

        function [speed_radps, torque_Nm] = get_min_torque_curve(obj,  varargin)
            speed_radps = obj.negative_torque_limit_Nm.get_signal_breakpoints('emach_spd_radps');
            voltage_V = obj.electric_power_W.get_signal_breakpoints( 'emach_voltage_V');
            
            torque_Nm = min(0.0, obj.negative_torque_limit_Nm.interp('emach_voltage_V', median(voltage_V),...
                obj.interp_signal_defaults{:},varargin{:}, 'emach_spd_radps',speed_radps));
        end
        
        function write_mscript(obj, file, varargin)
%             varargs = {'show_file','scale_type','object_name'};
            show_file = parse_varargs( varargin,'show_file',false,'toggle');
            object_name_str = parse_varargs(varargin,'object_name','mg');
            
            fid = fopen( file ,'w+');

            % Disable auto calculation in getter functions
            obj.disable_auto_calc = true;


            %% Write to m-file
            fprintf( fid,'%% ALPHA ELECTRIC MOTOR DEFINITION\n'); 
            fprintf( fid,'%% Generated %s\n',datestr(now)); 

            fprintf( fid,'\n%% Constructor\n'); 
            fprintf( fid, '%s = %s();\n',object_name_str, class(obj) );
            fprintf( fid, gen_property_str(obj,object_name_str,'name') );
            fprintf( fid, '%s.source_filename = mfilename;\n',object_name_str);

            fprintf( fid,'\n%% Physical Description\n');
            fprintf(fid, gen_property_str(obj,object_name_str,'electrical_source'));
            fprintf(fid, gen_property_str(obj,object_name_str,'inertia_kgm2'));
            
            if isprop(obj, 'gear')
                fprintf(fid, gen_property_str(obj.gear,[object_name_str '.gear'],'ratio'));
                fprintf(fid, gen_property_str(obj.gear,[object_name_str '.gear'],'efficiency_norm'));
            end
            
            
            fprintf( fid,'\n%% Capacity Limits\n');
            fprintf(fid, gen_property_str(obj,object_name_str,'max_speed_radps'));
            fprintf(fid, gen_property_str(obj,object_name_str,'max_torque_Nm'));
            fprintf(fid, gen_property_str(obj,object_name_str,'max_motor_power_W'));
            fprintf(fid, gen_property_str(obj,object_name_str,'max_generator_power_W'));
            fprintf(fid, gen_property_str(obj,object_name_str,'positive_torque_limit_Nm'));
            fprintf(fid, gen_property_str(obj,object_name_str,'negative_torque_limit_Nm'));
            
            
            
            fprintf( fid,'\n%% Losses & Efficiency\n');
            fprintf(fid, gen_property_str(obj,object_name_str,'electric_power_W'));
            fprintf(fid, gen_property_str(obj,object_name_str,'unpowered_torque_loss_Nm'));
            
            % TODO: Add more properties
            
            fclose(fid);

            % Allow time for the file to close before publisher tries to use it
            pause(1);

            function str = gen_property_str( mg,object_name_str, prop)
                val = mg.(prop);
                if ~( isobject(val) || isstruct(val) ) && ( isempty( val) ||  all(isnan(val(:))))
                    str = '';
                else
                    str = export2mscript(val,[object_name_str,'.',prop],'exclude_empty','tab_separator');		
                end
            end

                              
        end
               
    end
  
    methods (Hidden)
       
        function val = make_power_curves(obj)
            
            %mot_max_speed_radps = min([obj.max_speed_radps, obj.max_motor_power_W ./ ( 0.01 * obj.max_torque_Nm)]);
            %mot_corner_speed_radps = obj.max_motor_power_W / obj.max_torque_Nm;
            mot_max_torque_Nm = linspace( obj.max_torque_Nm, 0.01 * obj.max_torque_Nm, 20);
            mot_speeds_radps =  obj.max_motor_power_W ./ mot_max_torque_Nm;
            
            %gen_max_speed_radps = min([obj.max_speed_radps, obj.max_generator_power_W ./ ( 0.01 * obj.max_torque_Nm)]);
            %gen_corner_speed_radps = obj.max_generator_power_W / obj.max_torque_Nm;
            gen_max_torque_Nm = linspace( obj.max_torque_Nm, 0.01 * obj.max_torque_Nm, 20);
            gen_speeds_radps =  obj.max_generator_power_W ./ gen_max_torque_Nm;
            
            
            if ~isempty( obj.max_speed_radps )
                mot_remove_points = mot_speeds_radps > obj.max_speed_radps;
                mot_speeds_radps( mot_remove_points ) = [];
                mot_max_torque_Nm( mot_remove_points ) = [];
                mot_speeds_radps(end+1) = obj.max_speed_radps;
                mot_max_torque_Nm(end+1) = obj.max_motor_power_W ./ obj.max_speed_radps;
                
                
                gen_remove_points = gen_speeds_radps > obj.max_speed_radps;
                gen_speeds_radps( gen_remove_points ) = [];
                gen_max_torque_Nm( gen_remove_points ) = [];
                gen_speeds_radps(end+1) = obj.max_speed_radps;
                gen_max_torque_Nm(end+1) = obj.max_generator_power_W ./ obj.max_speed_radps;
            end
            
            mot_speeds_radps(end+1) = mot_speeds_radps(end) * 1.01;
            mot_max_torque_Nm(end+1) = 0;

            gen_speeds_radps(end+1) = gen_speeds_radps(end) *1.01;
            gen_max_torque_Nm(end+1) = 0;
            
            val.mot_speeds_radps = mot_speeds_radps;
            val.mot_max_torque_Nm = mot_max_torque_Nm;
            val.gen_speeds_radps = gen_speeds_radps;
            val.gen_max_torque_Nm = gen_max_torque_Nm;
            
            
        end
        
    end
   
        
    methods (Static)
       
 
        
    end
 
end

