classdef class_REVS_engine < class_REVS_engine_base
	%class_REVS_engine
	%   Definition of class_REVS_engine class
    
    properties
        
        % Operating Range Parameters
        full_throttle_speed_radps;              % Maximum torque curve speed breakpoints [rad/sec]
        full_throttle_torque_Nm;                % Maximum torque curve [Nm]
                
        closed_throttle_speed_radps;              % Minimum (Motoring) torque curve speed breakpoints [rad/sec]
        closed_throttle_torque_Nm ;               % Minimum (Motoring) torque curve [Nm]              
        
        % Torque Response Parameters
        naturally_aspirated_speed_radps;						% Maximum torque curve without boost speed breakpoints [rad/sec]
        naturally_aspirated_torque_Nm;							% Maximum torque curve without boost [Nm]
        
        power_time_constant_secs = 0.2;                         % Power time constant [s]
        
        boost_time_constant_secs = 0.5;                         % boost time constant [s]
        boost_falling_time_constant_secs = 0.3;                 % boost falling time constant [s]
        
        % Run state
        run_state_activation_speed_radps = 1;				% Speed above which engine control activates [rad/sec]
        run_state_activation_delay_secs = 0.5;              % Time above activation speed before control begins [sec]
        run_state_deactivation_speed_radps = 0;				% Speed below which enginecontrol deactivates [rad/sec]
        
        % Idle Parameters
        idle_target_speed_radps class_REVS_dynamic_lookup = class_REVS_dynamic_lookup;	% Target idle speed Dynamic Lookup - can be set to scalar value [rad/sec]
        idle_control_Kp = 25;									% PID Proportional Term for speed control
        idle_control_Ki = 65;									% PID Integral Term for speed control
        idle_control_Kd = 1;									% PID Derivative Term for speed control
        idle_control_ramp_radps = 100 * unit_convert.rpm2radps; % Soft landing buffer for idle speed [rad/s]
        idle_control_ramp_time_secs = 1.5;                      % Soft landing decay duration for idle speed [sec]
        idle_control_torque_reserve_Nm;							% Speed Control Excess Slow Torue to ensure fast control authority [Nm]
        idle_control_slow_est_time_constant_sec;                % Estimate is slow control time constant to use for speed control [sec]
        
        fuel_map_speed_radps;			% Fuel consumption map speed breakpoints [rad/sec]
        fuel_map_torque_Nm				% Fuel consumption map torque breakpoints [Nm]
        
        fuel_map_gps;       % Fuel consumption map [g/sec]
        
        deac_fuel_map_gps;							% Fuel consumption map with cylinder deac active [g/sec]
        
        deac_strategy;											% Structure / Class defining type and calibration for cylinder deactivation logic
		deac_num_cylinders = 0;									% Number of cylinders that have deactivation capability
		
        deac_transition_on_duration_secs = 0.99;					% Cylinder deactivation turn on fuel map blending
        deac_transition_off_duration_secs = 0.11;					% Cylinder deactivation turn off fuel map blending
        deac_transition_off_fuel_multiplier = [1.0 1.0];			% Cylinder deactivation turn off penalty fuel multiplier profile
        deac_transition_off_fuel_multiplier_time_secs = [0.0 0.1];	% Cylinder deactivation turn off penalty fuel multiplier profile time [sec]
        deac_transition_off_fuel_multiplier_limit_gps = inf;		% Cylinder deactivation turn off penalty maximum additional fueling [g/sec]
        
        fast_torque_fuel_adjust_norm;							% Portion of fuel cut associated with fast torque reduction [0 - 1] generally 0 for gasoline 1 for diesel
        
        DFCO_enable_condition = '@veh_spd_mps>5';				% Conditions to allow decel fuel cutoff when operating on closed throttle curve [DYNAMIC EXPRESSION]
        DFCO_min_duration_secs = 2.1;							% DFCO defueled duration before triggering additional fueling
        DFCO_refuel_multiplier = [1.0, 1.3, 1.0];				% DFCO refuel multiplier profile
        DFCO_refuel_multiplier_time_secs = [0.0, 0.1, 1.1];		% DFCO refuel multiplier profile time [sec]
        DFCO_refuel_multiplier_limit_gps = inf;                 % DFCO additional fuel limit [grams/sec]
        
        transient_correction_mult = 1.40;						% Transient operation correction
                
    end
    
    properties (Hidden, Dependent)
        
        idle_speed_radps;
    end
    
            
     methods 
         
        %% Constructor
        function obj = class_REVS_engine          
            
            obj.variant = 'basic engine';
		
        end
        
        function speed_radps = get_idle_speed_radps(obj, varargin)            
            speed_radps = min(obj.idle_target_speed_radps.table(:));
        end

        
        function [speed_radps, torque_Nm] = get_max_torque_curve(obj, varargin)
           speed_radps = obj.full_throttle_speed_radps;
           torque_Nm = obj.full_throttle_torque_Nm;
        end
        
        function [speed_radps, torque_Nm] = get_min_torque_curve(obj, varargin)
           speed_radps = obj.closed_throttle_speed_radps;
           torque_Nm = obj.closed_throttle_torque_Nm;                       
        end
        
        
        function [speed_radps, torque_Nm, consumption_gps] = get_fuel_map(obj, varargin)
            
            deac_prop = parse_varargs(varargin, 'deac_prop', 1.0);
            
            speed_radps = obj.fuel_map_speed_radps; 
            torque_Nm = obj.fuel_map_torque_Nm;
            
            if ischar(deac_prop) && strcmpi(deac_prop,'deac')
				consumption_gps = obj.deac_fuel_map_gps;
            elseif ischar(deac_prop) && strcmpi(deac_prop,'enabled')
 				consumption_gps = min( obj.fuel_map_gps , obj.deac_fuel_map_gps );           
            elseif ischar(deac_prop) && strcmpi(deac_prop,'disabled')
 				consumption_gps =  obj.fuel_map_gps ;               
            else
				consumption_gps = min( obj.fuel_map_gps , (deac_prop * obj.deac_fuel_map_gps + (1-deac_prop) * obj.fuel_map_gps) );
            end                        
        end
        
        
        function [fuel_gps] = interp_fuel_map(obj, varargin )
           
            if any( strcmpi('speed_radps',varargin))
                speed_idx = find(strcmpi('speed_radps',varargin));
                speed_pt = varargin{speed_idx+1};
                varargin(speed_idx+[0,1])=[];
            elseif any( strcmpi('speed_rpm',varargin))
                speed_idx = find(strcmpi('speed_rpm',varargin));
                speed_pt = varargin{speed_idx+1} * unit_convert.rpm2radps;
                varargin(speed_idx+[0,1])=[];
            else
                error('Missing lookup speed, specifty either speed_rpm or speed_radps');
            end
            
            if any( strcmpi('torque_Nm',varargin))
                torque_idx = find(strcmpi('torque_Nm',varargin));
                torque_pt = varargin{speed_idx+1};
                varargin(torque_idx+[0,1])=[];
            else
                error('Missing lookup speed, specifty either speed_rpm or speed_radps');
            end
            
            [map_speed_radps, map_torque_Nm, map_consumption_gps] = obj.get_fuel_map( varargin);
            
            fuel_gps = interp2(map_speed_radps, map_torque_Nm, map_consumption_gps, speed_pt, torque_pt,'linear');
                        
        end
        
        %% Getters
        
		
        function val = get.naturally_aspirated_speed_radps( obj )
            val = REVS_class_default( obj.naturally_aspirated_speed_radps, obj.full_throttle_speed_radps, obj.disable_auto_calc );
		end
		
        function val = get.naturally_aspirated_torque_Nm( obj )
            val = REVS_class_default( obj.naturally_aspirated_torque_Nm, obj.full_throttle_torque_Nm, obj.disable_auto_calc  );
        end
                
        function val = get.fast_torque_fuel_adjust_norm(obj)
            val = REVS_class_default( obj.fast_torque_fuel_adjust_norm, obj.combustion_type == enum_engine_combustion_type.compression_ignition, obj.disable_auto_calc  );
        end
        
        function val = get.idle_control_torque_reserve_Nm(obj)
            val = REVS_class_default( obj.idle_control_torque_reserve_Nm, max( 10, obj.max_torque_Nm* 0.025 ), obj.disable_auto_calc );
        end
        
       function val = get.idle_control_slow_est_time_constant_sec(obj)
            val = REVS_class_default( obj.idle_control_slow_est_time_constant_sec, obj.power_time_constant_secs, obj.disable_auto_calc );
        end

        
        function val = get.deac_fuel_map_gps( obj)
            
            if obj.disable_auto_calc || ~isempty( obj.deac_fuel_map_gps)
				val = obj.deac_fuel_map_gps;
			else
                val = nan * ones( size( obj.fuel_map_gps));
            end
                       
		end
        
		function val = get.deac_strategy( obj)
			
			if obj.disable_auto_calc || ~isempty( obj.deac_strategy)
				val = obj.deac_strategy;
			else
				val.variant = 'constant'; % name of library cylinder deac logic select variant target block
				val.enable_norm = 0.6;
			end
			
        end
		
        function val = get.idle_speed_radps( obj)
            val = obj.idle_target_speed_radps;
        end
        
        function obj = set.idle_speed_radps(obj, val)
            if isempty(obj.idle_target_speed_radps)
                warning('Idle speed setpoint has moved from idle_speed_radps to idle_target_speed_radps and will be removed in a future release.')
            end
            obj.idle_target_speed_radps = val;
        end
        
                		
        %% Regular Methods
                
		function val = has_deac(obj)
			val = any(~isnan(obj.deac_fuel_map_gps(:)));	
		end
		        
		function val = merged_fuel_map_gps(obj,deac_prop)
			if nargin < 2
				deac_prop = 1.0;
			end
			
			if ischar(deac_prop) && strcmpi(deac_prop,'deac')
				val = obj.deac_fuel_map_gps;
			else
				val = min( obj.fuel_map_gps , (deac_prop * obj.deac_fuel_map_gps + (1-deac_prop) * obj.fuel_map_gps) );
			end	
        end	

	
        function write_mscript(obj, file, varargin)

            show_file = parse_varargs( varargin,'show_file',false,'toggle');
            % citation = parse_varargs( varargin,'no_citation',true,'toggle');

            fid = fopen( file ,'w+');

            % Disable auto calculation in getter functions
            obj.disable_auto_calc = true;


            %% Write to m-file
            fprintf( fid,'%% ALPHA ENGINE DEFINITION\n'); 
            fprintf( fid,'%% Generated %s\n',datestr(now)); 

            % if citation
            % 	fprintf( fid,'%% SUGGESTED CITATION: \n%% %s - Full Engine Map Package. Version %s. Ann Arbor MI: US EPA National Vehicle and Fuel Emissions Laboratory, National Center for Advanced Technology, %s\n',engine.name, datestr(now(),'yyyy-mm-dd'), datestr(now(),'yyyy')); 
            % end

            fprintf( fid,'\n%% Constructor\n'); 
            fprintf( fid,'engine = %s();\n', class(obj) );
            fprintf( fid, gen_property_str(obj,'name') );
            fprintf( fid, 'engine.source_filename = mfilename;');

            fprintf( fid, gen_property_str(obj,'matrix_vintage') );

            fprintf( fid,'\n%% Physical Description\n');
            fprintf(fid, gen_property_str(obj,'displacement_L'));
            fprintf(fid, gen_property_str(obj,'num_cylinders'));
            fprintf(fid, gen_property_str(obj,'combustion_type'));
            fprintf(fid, gen_property_str(obj,'compression_ratio'));
            fprintf(fid, gen_property_str(obj,'inertia_kgm2'));
            fprintf(fid, gen_property_str(obj,'bore_mm'));
            fprintf(fid, gen_property_str(obj,'stroke_mm'));

            fprintf( fid, '\n%% Maximum Torque Curve\n');
            fprintf(fid, gen_property_str(obj,'full_throttle_speed_radps')); 
            fprintf(fid, gen_property_str(obj,'full_throttle_torque_Nm'));

            if ~isequal(obj.full_throttle_speed_radps, obj.naturally_aspirated_speed_radps) || any( obj.full_throttle_torque_Nm(:) > obj.naturally_aspirated_torque_Nm(:) )
                fprintf(fid, gen_property_str(obj,'naturally_aspirated_speed_radps')); 
                fprintf(fid, gen_property_str(obj,'naturally_aspirated_torque_Nm'));
            end

            fprintf( fid, '\n%% Minimum Torque Curve\n');
            fprintf(fid, gen_property_str(obj,'closed_throttle_speed_radps'));
            fprintf(fid, gen_property_str(obj,'closed_throttle_torque_Nm'));

            fprintf( fid, '\n%% Fuel Map\n');
            fprintf(fid, gen_property_str(obj,'fuel_map_speed_radps'));
            fprintf(fid, gen_property_str(obj,'fuel_map_torque_Nm'));
            fprintf(fid, gen_property_str(obj,'fuel_map_gps'));

            fprintf(fid, gen_property_str(obj,'deac_fuel_map_gps'));

            fprintf( fid, '\n%% Fuel Properties\n');
            if class_REVS_fuel.check_id(obj.fuel.id)
                fprintf(fid, 'engine.fuel = class_REVS_fuel(''%s'');\n', obj.fuel.id);
            else
                % Not in table write individual fields
                fprintf(fid, gen_property_str(obj,'fuel'));
            end

            fprintf( fid, '\n%% Idle Speed\n');
            fprintf(fid, gen_property_str(obj,'idle_target_speed_radps'));

            fprintf( fid, '\n%% Calibration Adjustment Factors\n');

            % Print any additional non-default values
            completed_props = {'name','matrix_vintage',...
                'displacement_L','combustion_type','compression_ratio', 'inertia_kgm2','bore_mm','stroke_mm','num_cylinders',...
                'full_throttle_speed_radps','full_throttle_torque_Nm','naturally_aspirated_speed_radps','naturally_aspirated_torque_Nm',...
                'closed_throttle_speed_radps','closed_throttle_torque_Nm',...
                'fuel_map_speed_radps','fuel_map_torque_Nm','fuel_map_gps','deac_fuel_map_gps',...
                'fuel','idle_target_speed_radps',...
                'pedal_map_type','pedal_map_speed_radps','pedal_map_pedal_norm','pedal_map_Nm'};
            class_meta = metaclass(obj);
            prop_meta = class_meta.PropertyList;
            calibration_props = 0;

            for p = 1:length(prop_meta)

                prop = prop_meta(p).Name;

                if ismember( prop, completed_props) 
                    % Skip it already done
                elseif prop_meta(p).Hidden || prop_meta(p).Constant || prop_meta(p).Transient || prop_meta(p).Dependent
                    % Skip it
                elseif prop_meta(p).HasDefault && isequal(obj.(prop), prop_meta(p).DefaultValue)
                    % Skip It - same as default
                else	
                    prop_str = gen_property_str(obj,prop);

                    if ~isempty( prop_str)
                        fprintf(fid, prop_str);
                        calibration_props = calibration_props+1;
                    end
                end

            end

            if calibration_props == 0
                fprintf( fid, '%% -- None -- \n');
            end


            if show_file
                fprintf( 'Generated file: %s\n\n',file) ;
                fseek(fid,0,'bof');
                line = 1;
                while ~feof(fid)
                    fprintf('%i\t%s\n',line, fgetl(fid));
                    line = line+1;
                end

            end

            fclose(fid);

            % Allow time for the file to close before publisher tries to use it
            pause(1);

            function str = gen_property_str( engine, prop)
                val = engine.(prop);
                if ~( isobject(val) || isstruct(val) ) && ( isempty( val) ||  all(isnan(val(:))))
                    str = '';
                else
                    str = export2mscript(val,['engine.',prop],'exclude_empty','tab_separator');		
                end
            end

end






			
    end
        
end

