classdef (Abstract) class_REVS_engine_base
	%class_REVS_engine
	%   Definition of class_REVS_engine class
    
    properties
               
        name = '';                      % Description of engine
        source_filename = '';           % used to keep track of original filename (before REVS_modify_engine, for example)
        matrix_vintage = enum_matrix_vintage.present;
        variant = '';                   % name of library engine variant target block
                
        combustion_type;				% Combustion type specified via enum_engine_combustion_type
        
        displacement_L;					% Engine displacement [liters]
        bore_mm = [];                   % cylinder bore [mm]
        stroke_mm = [];                 % piston stroke [mm]
        num_cylinders = 4;              % number of engine cylinders
        compression_ratio = [];			% geometric compression ratio
        configuration = [];				% engine physical configuration ( I4, V6, etc.)
        
        inertia_kgm2;                   % Engine rotational inertia [kg*m^2]
        
        fuel class_REVS_fuel;			% Structure Containing map fuel properties
        base_fuel class_REVS_fuel;			% Structure Containing map fuel properties of the base engine
                                
    end
    
    properties ( Abstract ) % Properties that must be supplied by any subclass implementations
    
       
    end
        
    properties ( Dependent = true, Hidden = true, Transient = true )
           
        pedal_map_type;											% Generic or custom accelerator pedal to torque mapping DEPRECATED
        pedal_map_speed_radps;									% Accelerator pedal to torque map speed breakpoints [radians/sec]DEPRECATED
        pedal_map_pedal_norm;									% Accelerator pedal to torque map pedal breakpoints [0 - 1]DEPRECATED
        pedal_map_Nm;											% Accelerator pedal to torque map [Nm]DEPRECATED
        
%         full_throttle_speed_radps;                              % DEPRECATED
%         full_throttle_torque_Nm;                                % DEPRECATED
%         full_throttle_power_W;                                  % DEPRECATED
%         
%         closed_throttle_speed_radps;                            % DEPRECATED
%         closed_throttle_torque_Nm;                              % DEPRECATED
        
    end
    
    properties ( Dependent = true, SetAccess = immutable, Transient = true)
 
        
        nominal_idle_speed_radps;               % Nominal (scalar) idle speed [rad/sec] 
          		
        max_torque_Nm;                          % Max torque [Nm]
        max_torque_min_speed_radps;             % Max torque min [rad/s]
        max_torque_max_speed_radps;             % Max torque max [rad/s]
        max_torque_avg_speed_radps;             % Max torque avg [rad/s]
        
        min_torque_Nm;                          % Min torque [Nm]
        
        max_power_W;                            % Max power [W]      
        max_power_min_speed_radps;              % Max power min [rad/s]
        max_power_max_speed_radps;              % Max power max [rad/s]
        max_power_avg_speed_radps;              % Max power avg [rad/s]
        max_test_speed_radps;                   % Max test speed [rad/s]
        
	end
    	
	properties ( Hidden = true, Transient = true )
		disable_auto_calc = false;
	end
	
    methods
        
        
        %% Constructor
        function obj = class_REVS_engine_base
            
            % Nothing left here
		
		end
        
    end
    
    methods (Abstract)
        
        [speed_radps] = get_idle_speed_radps(obj, varargin);
        
        [speed_radps, torque_Nm] = get_max_torque_curve(obj, varargin);
        
        [speed_radps, torque_Nm] = get_min_torque_curve(obj, varargin);
        
        [speed_radps, torque_Nm, consumption_gps] = get_fuel_map(obj, varargin);
        
    end

    methods   
        
        function [trq, spd] = hires_max_curve( obj)
            [nom_speed_radps, nom_torque_Nm] = obj.get_max_torque_curve();            
            spd = 0:0.1:nom_speed_radps(end);
            trq = interp1( nom_speed_radps, nom_torque_Nm, spd );
        end 
        
        function [ spd ] = calc_power_speed( obj, power_norm, mode, varargin )
            
            if nargin < 3
                mode = 'range';
            end
            
            [max_curve.speed_radps, max_curve.torque_Nm] = obj.get_max_torque_curve();
            
            hires_speeds_radps = 0:0.1:max_curve.speed_radps(end);
            hires_torque_Nm = interp1(max_curve.speed_radps, max_curve.torque_Nm, hires_speeds_radps);
            hires_power_W = hires_speeds_radps .* hires_torque_Nm;
                       
            idx = find( hires_power_W >= max(hires_power_W) .* power_norm);
            
            if  strcmpi(mode , 'min' ) || strcmpi(mode , 'lower' )               
                idx = find( hires_power_W >= max(hires_power_W) .* power_norm, 1, 'first');    % Lowest speed that meets target engine power
            elseif strcmpi(mode , 'max' ) || strcmpi(mode , 'upper' )              
                idx = find( hires_power_W >= max(hires_power_W) .* power_norm, 1, 'last');     % Highest speed that meets target engine power
            else
                idx = find( hires_power_W >= max(hires_power_W) .* power_norm);
                idx = [idx(1), idx(end)];     % Highest and Lowest speeds that meet target engine power       
            end
            
            spd = hires_speeds_radps(idx);            
        end

        function [ spd ] = calc_torque_speed( obj, power_norm, mode, varargin )
            
            if nargin < 3
                mode = 'range';
            end
            
            [max_curve.speed_radps, max_curve.torque_Nm] = obj.get_max_torque_curve();
            
            hires_speeds_radps = 0:0.1:max_curve.speed_radps(end);
            hires_torque_Nm = interp1(max_curve.speed_radps, max_curve.torque_Nm, hires_speeds_radps);
                       
            if  strcmpi(mode , 'min' ) || strcmpi(mode , 'lower' )               
                idx = find( hires_torque_Nm >= max(hires_torque_Nm) .* power_norm, 1,'first');    % Lowest speed that meets target engine power
            elseif strcmpi(mode , 'max' ) || strcmpi(mode , 'upper' )              
                idx = find( hires_torque_Nm >= max(hires_torque_Nm) .* power_norm, 1, 'last');     % Highest speed that meets target engine power
            else
                idx = find( hires_torque_Nm >= max(hires_torque_Nm) .* power_norm);
                idx = [idx(1), idx(end)];     % Highest and Lowest speeds that meet target engine power       
            end
            
            spd = hires_speeds_radps(idx);            
        end
        
        
        
        %% Getters
        
		function val = get.inertia_kgm2( obj )
            val = REVS_class_default( obj.inertia_kgm2, interp1([1.0, 5.0], [0.07 , 0.27], obj.displacement_L, 'linear','extrap'), obj.disable_auto_calc );
        end
				
        function val = get.nominal_idle_speed_radps( obj )
            val = get_idle_speed_radps(obj);
        end
        
        function val = get.max_torque_Nm(obj)
            [~, trq_Nm] = obj.get_max_torque_curve();
            val = max( trq_Nm );
        end
        
        function val = get.max_torque_max_speed_radps(obj)
            [trq, spd] = obj.hires_max_curve();           
            idx =  find( trq > max(trq) * 0.98,1,'last');
            val = spd(idx);
        end
               
        function val = get.max_torque_min_speed_radps(obj)
            [trq, spd] = obj.hires_max_curve();
            idx =  find( trq > max(trq) * 0.98,1,'first');
            val = spd(idx);
        end
        
        function val = get.max_torque_avg_speed_radps(obj)
            [trq, spd] = obj.hires_max_curve();
            [max_trq, idxm] = max(trq);
            idx1 =  find( trq > max_trq * 0.98,1,'last');
            idx2 =  find( trq > max_trq * 0.98,1,'first');           
            val = mean(spd([idxm,idx1,idx2]));
        end        
               
        function val = 	get.min_torque_Nm(obj)
            [spd_radps, trq_Nm] = obj.get_min_torque_curve();
            hires_spd_radps = spd_radps(1):1:spd_radps(end);
            hires_trq_Nm = interp1( spd_radps, trq_Nm, hires_spd_radps);          
            val = min(hires_trq_Nm(hires_spd_radps <= obj.max_test_speed_radps)); 
        end
        
        
        function val = get.max_power_W(obj)
            [trq, spd] = obj.hires_max_curve();
            val = max( spd .* trq );
        end
        
        function val = get.max_power_max_speed_radps(obj)
            [trq, spd] = obj.hires_max_curve();           
            pow_W = spd .* trq ;
            idx =  find( pow_W > max(pow_W) * 0.98,1,'last');
            val = spd(idx);
        end
               
        function val = get.max_power_min_speed_radps(obj)
            [trq, spd] = obj.hires_max_curve();
            pow_W = spd .* trq ;
            idx =  find( pow_W > max(pow_W) * 0.98,1,'first');
            val = spd(idx);
        end
        
        function val = get.max_power_avg_speed_radps(obj)
            [trq, spd] = obj.hires_max_curve();
            pow_W = spd .* trq;
            [max_pow_W, idxm] = max(pow_W);
            idx1 =  find( pow_W > max_pow_W * 0.98,1,'last');
            idx2 =  find( pow_W > max_pow_W * 0.98,1,'first');           
            val = mean(spd([idxm,idx1,idx2]));
        end
        
        function val = get.max_test_speed_radps(obj)
            [trq_Nm, spd_radps] = obj.hires_max_curve();
            power_W = spd_radps .* trq_Nm ;
            
            % calculations following 1065.510 to find Maximum Test Speed (MTS)
            P_normi                    = power_W ./ max(power_W);
            f_normi                    = spd_radps ./ obj.max_power_avg_speed_radps;
            sum_squares                = f_normi.^2 + P_normi.^2;
            
            max_test_speed_min_radps   = spd_radps( find(sum_squares >= 0.98 * max(sum_squares), 1, 'first') );
            max_test_speed_max_radps   = spd_radps( find(sum_squares >= 0.98 * max(sum_squares), 1, 'last') );
                       
            val                        = mean( [max_test_speed_max_radps max_test_speed_min_radps] );
        end
        
        
        %% Regular Methods
        
        function val = cycle_percent_speed_reference_radps(obj, varargin)
			idle_speed_radps = obj.get_idle_speed_radps( varargin{:});
            max_test_speed_radps = obj.max_test_speed_radps( varargin{:});          
            val = idle_speed_radps + [0.0, 0.85] .* ( max_test_speed_radps - idle_speed_radps);
		end
						
		function val = cycle_abc_speed_reference_radps(obj, varargin)
			[trq, spd] = obj.hires_max_curve;
            idle_speed_radps = obj.get_idle_speed_radps( varargin{:});
			power_W = spd .* trq;
			
			fn_lo   = spd( find(power_W >= 0.5 * max(power_W), 1, 'first') );
			fn_hi   = spd( find(power_W >= 0.7 * max(power_W), 1, 'last') );
			
			val = [0.25, 0.5, 0.75, 1.0 ] .* (fn_hi - fn_lo) + fn_lo;
			val = [idle_speed_radps, val];
        end
        
        

        
        
        function val = max_power_max_rpm(obj)
            val = obj.max_power_max_radps .* unit_convert.radps2rpm;
        end
        
        function val = max_power_min_rpm(obj)
            val = obj.max_power_min_radps .* unit_convert.radps2rpm;
        end
        
        function val = max_power_avg_rpm(obj)
            val = obj.max_power_avg_radps .* unit_convert.radps2rpm;
        end
        
        function val = max_test_speed_rpm(obj)
            val = obj.max_test_speed_radps * unit_convert.radps2rpm;
        end
            
        function val = fuel_map_output_power_W(obj, varargin)
            [speed_radps, torque_Nm, ~] = obj.get_fuel_map( varargin{:});
            val = kron(speed_radps(:)', torque_Nm(:));
		end
        
        function val = fuel_map_fuel_power_W(obj, varargin)
            [~, ~, consumption_gps] = obj.get_fuel_map( varargin);
            val = consumption_gps * obj.fuel.energy_density_MJpkg * 1000;
		end
        		
        function val = fuel_map_bsfc_gpkWh(obj, varargin)
            [~, ~, consumption_gps] = obj.get_fuel_map( varargin);
            val = consumption_gps ./ (obj.fuel_map_output_power_W / 1000 / 3600);
            % remove spurious points, if necessary
            val(val <= 0 | val >= 2000) = NaN;
		end
		
        function val = fuel_map_efficiency_norm(obj, varargin)
        val = max(0, obj.fuel_map_output_power_W)./ obj.fuel_map_fuel_power_W(varargin{:});
            % remove spurious points, if necessary
            val( val > 1 | val < 0 ) = NaN;
		end
        		
        function val = fuel_map_flow_gpi(obj, varargin)
            [speed_radps, torque_Nm, consumption_gps] = obj.get_fuel_map( varargin);           
            val = consumption_gps ./ (repmat(speed_radps(:)' ,length(torque_Nm),1))*4*pi / obj.num_cylinders;
            % remove spurious points, if necessary
            val(isinf(val)) = NaN;
		end
		        
        function val = max_efficiency_norm(obj, varargin)
        efficiency_map_norm = obj.fuel_map_efficiency_norm(varargin{:});
            val = max(efficiency_map_norm(:));
        end
        
        function val = min_bsfc_gpkWh(obj,varargin)
            bsfc_map_gpkWh = obj.fuel_map_bsfc_gpkWh(varargin{:});
            val = min(bsfc_map_gpkWh(:));
        end

        function val = min_bsfc_curves(obj, varargin)
            
            % Establish speed points to check and corresponding maximum
            % torque
            
            [max_curve.speed_radps, max_curve.torque_Nm] = get_max_torque_curve(obj, varargin{:});
            [fuel_map.speed_radps, fuel_map.torque_Nm, fuel_map.consumption_gps] = get_fuel_map(obj, varargin{:});
        
            spd_radps = linspace(1, max(max_curve.speed_radps), 1000);
            max_trq_Nm = interp1(max_curve.speed_radps, max_curve.torque_Nm, spd_radps); % Preinterpolate for speed
	
            pwr_W = linspace(1, max(spd_radps .* max_trq_Nm), 100);
            min_consumption_flow_gps = zeros( size( pwr_W ) );
            min_consumption_speed_radps = zeros( size( pwr_W) );
            min_consumption_upper_speed_radps = zeros( size( pwr_W) );
            min_consumption_lower_speed_radps = zeros( size( pwr_W) );
	
            % Compute for max power
            torques_Nm = pwr_W(end)./spd_radps;
            fuel_rate_gps =  interp2(fuel_map.speed_radps, fuel_map.torque_Nm, fuel_map.consumption_gps, spd_radps, torques_Nm);
            fuel_rate_gps(torques_Nm > max_trq_Nm) = Inf;
            fuel_rate_gps(fuel_rate_gps<0) = Inf;
            
            if all(isinf(fuel_rate_gps)) %If the last point does not have a valide fuel consumption it leads to any subsequent point not having a valid speed value
                min_consumption_flow_gps(end) = max(max(fuel_map.consumption_gps));
                min_consumption_speed_radps(end) = max(spd_radps);
                min_consumption_upper_speed_radps(end) = max(spd_radps);
                min_consumption_lower_speed_radps(end) = max(spd_radps);
            else
                
                [min_consumption_flow_gps(end), min_idx] = min( fuel_rate_gps);                
                min_consumption_speed_radps(end) = spd_radps( min_idx);
                min_consumption_upper_speed_radps(end) = spd_radps( min_idx);
                min_consumption_lower_speed_radps(end) = spd_radps( min_idx);
            end
            
            

            for idx = (length(pwr_W)-1):-1:1
                torques_Nm = pwr_W(idx)./spd_radps;
                fuel_rate_gps =  interp2(fuel_map.speed_radps, fuel_map.torque_Nm, fuel_map.consumption_gps, spd_radps, torques_Nm);
                fuel_rate_gps(torques_Nm > max_trq_Nm) = Inf;
                fuel_rate_gps(fuel_rate_gps<0) = Inf;

                [min_consumption_flow_gps(idx), min_idx] = min( fuel_rate_gps );                
                
                min_consumption_pts = fuel_rate_gps < min_consumption_flow_gps(idx) * 1.015;                
                min_consumption_lo_idx = find(min_consumption_pts,1,'first');
                min_consumption_hi_idx = find(min_consumption_pts,1,'last'); 
                
                avg_speed_radps{idx} = mean(spd_radps( [min_consumption_lo_idx, min_idx, min_idx, min_consumption_hi_idx]));
                try
                min_consumption_speed_radps(idx) = min(avg_speed_radps{idx}, min_consumption_speed_radps(idx+1) );

                min_consumption_upper_speed_radps(idx) = min(spd_radps( min_consumption_lo_idx), min_consumption_upper_speed_radps(idx+1));
                min_consumption_lower_speed_radps(idx) = min(spd_radps( min_consumption_hi_idx), min_consumption_lower_speed_radps(idx+1));
                catch

                end

            end

            %Force Unique speeds - monotonic
            [ val.speed_radps, unique_pts] = unique(min_consumption_speed_radps,'last');
            val.flow_gps = min_consumption_flow_gps(unique_pts);            
            val.torque_Nm = pwr_W(unique_pts) ./ val.speed_radps;
            val.power_W = pwr_W(unique_pts);
            
            [ val.min_consumption_upper_speed_radps, unique_pts] = unique(min_consumption_upper_speed_radps,'last');                       
            val.min_consumption_upper_torque_Nm = pwr_W(unique_pts) ./ val.min_consumption_upper_speed_radps;
            
            [ val.min_consumption_lower_speed_radps, unique_pts] = unique(min_consumption_lower_speed_radps,'last');             
            val.min_consumption_lower_torque_Nm = pwr_W(unique_pts) ./ val.min_consumption_lower_speed_radps;

        end
        
        
        %% Setters & Getters for Deprecated / Renamed
        
        
        function obj = set.pedal_map_type(obj,val)
            warning('pedal_map_type has been relocated to the controls subsystem and class_REVS_controls and this property will be removed in a future release');           
        end
        
        function val = get.pedal_map_type(obj)
            warning('pedal_map_type has been relocated to the controls subsystem and class_REVS_controls and this property will be removed in a future release');
            val = [];            
        end
        
        function obj = set.pedal_map_Nm(obj,val)
            warning('pedal_map_Nm has been relocated to the controls subsystem and class_REVS_controls and this property will be removed in a future release');
        end
        
        function val = get.pedal_map_Nm(obj)
            warning('pedal_map_Nm has been relocated to the controls subsystem and class_REVS_controls and this property will be removed in a future release');
            val = [];            
        end
        
        function obj = set.pedal_map_pedal_norm(obj,val)
            warning('pedal_map_pedal_norm has been relocated to the controls subsystem and class_REVS_controls and this property will be removed in a future release');         
        end
        
        function val = get.pedal_map_pedal_norm(obj)
            warning('pedal_map_pedal_norm has been relocated to the controls subsystem and class_REVS_controls and this property will be removed in a future release');
            val = [];            
        end

        function obj = set.pedal_map_speed_radps(obj,val)
            warning('pedal_map_speed_radps has been relocated to the controls subsystem and class_REVS_controls and this property will be removed in a future release');         
        end        
        
        function val = get.pedal_map_speed_radps(obj)
            warning('pedal_map_speed_radps has been relocated to the controls subsystem and class_REVS_controls and this property will be removed in a future release');
            val = [];            
        end
        			
    end
        
end

