classdef class_REVS_logging_object < dynamicprops & matlab.mixin.CustomDisplay & matlab.mixin.Copyable
	%class_REVS_logging_object
	%A very struct like object that can be built into whatever form is
	%needed for storing the necessary data
	
	properties ( GetAccess = protected, SetAccess = immutable )
		root;		% Link to root object if available
	end
	
	methods
		function obj = class_REVS_logging_object(root)
			if nargin > 0
				obj.root = root;
			end
		end
	end
	
	
	methods( Hidden = true )
		
		% Hide handle & dynamicprops methods
		function mp = addprop(varargin)
			mp = addprop@dynamicprops(varargin{:});
		end
		function lh = addlistener(varargin)
			lh = addlistener@dynamicprops(varargin{:});
		end
		function obj = findobj(varargin)
			obj = findobj@dynamicprops(varargin{:});
		end
		function mp = findprop(varargin)
			mp = findprop@dynamicprops(varargin{:});
		end
		function notify(varargin)
			notify@dynamicprops(varargin{:});
		end
		function delete(varargin)
			delete@dynamicprops(varargin{:});
		end
		
		% Override fieldnames only showing desired fields
		function val = fieldnames(obj)
			val = obj.properties;
		end
		
		% Override fieldnames only showing desired fields
		function val = properties(obj)
			val = sort(builtin('properties',obj));
        end
        
        function result = isproperty( obj, subfield)
            
           field_list = strsplit(subfield,'.');           
           o = obj;
           
           for i = 1:numel(field_list)
               if isprop(o, field_list{i})
                   o = o.(field_list{i});
               else
                   result = false;
                   return
               end
           end
            
           result = true;
        end
        
		function add_calculated_properties( obj )
			
			props = properties(obj);
			
			% Recusively process tree
			for p = 1:length(props)
				try
					if isa( obj.(props{p}), 'class_REVS_logging_object')
						obj.(props{p}).add_calculated_properties;
					end
				end
			end
			
			% Add fields for current node
			
			% inertia compensated torque
			add_calc_iner_comp_torque(obj)
			
			% mass compensated force
			add_calc_mass_comp_force(obj)
			
			% total kJ (pos_kJ - neg_kJ)
			add_calc_total_energy( obj)
			
			% loss kJ (input_kJ - output_kJ)
			add_calc_loss_kJ(obj);
			
			% mechanical rotational power (torque_Nm * speed_radps)
			add_calc_rotational_power(obj);
			
			% mechanical linear power (force_N * speed_mps)
			add_calc_linear_power(obj);
			
			% electrical power (current_A * volts_V)
			add_calc_elec_power(obj);
			
			% total shift count
			add_calc_shift_count(obj);
			
			% Add calcs that are just unit changes
			add_calc_kWh(obj);          % Energy in kWhr (kJ -> kWhr)
            add_calc_speed_rpm(obj);    % Speed in RPM (rad/s -> RPM)
			
        end
		
        function add_calc_speed_rpm(obj)
                props = properties(obj);
            
           		match = regexp(props,'^.*(?=_spd_radps$)', 'match');
				match = vertcat( match{:}); 
        
                for p = 1:length(match)
					meta = obj.addprop([match{p},'_spd_rpm']);
					meta.Dependent = 1;
                    meta.NonCopyable = false;
					meta.GetMethod = eval(['@(obj) obj.',match{p},'_spd_radps * unit_convert.radps2rpm']);
				end
   
        end
        
		function add_calc_iner_comp_torque(obj)
			
			match_time = ~isempty( obj.root) && isprop( obj.root,'time');
			
			if match_time
				props = properties(obj);
				match_torque = regexp(props,'^.*(?=_trq_Nm$)', 'match');
				match_torque = vertcat( match_torque{:});
				
				match_speed = regexp(props,'^.*(?=_spd_radps$)', 'match');
				match_speed = vertcat( match_speed{:});
				
				match_iner = regexp(props,'^.*(?=_tot_iner_kgm2$)', 'match');
				match_iner = vertcat( match_iner{:});
				
				match = intersect(match_iner, intersect( match_speed, match_torque )) ;
				
				for p = 1:length(match)
					meta = obj.addprop([match{p},'_iner_comp_trq_Nm']);
					meta.Dependent = 1;
                    meta.NonCopyable = false;
					meta.GetMethod = eval(['@(obj) obj.',match{p},'_trq_Nm  -  obj.',match{p},'_tot_iner_kgm2 .* class_REVS_logging_object.calc_accel( obj.',match{p},'_spd_radps, obj.root.time)']);
				end
			end
			
		end
		
		function add_calc_mass_comp_force(obj)
			
			match_time = ~isempty( obj.root) && isprop( obj.root,'time');
			
			if match_time
				props = properties(obj);
				match_force = regexp(props,'^.*(?=_frc_N$)', 'match');
				match_force = vertcat( match_force{:});
				
				match_speed = regexp(props,'^.*(?=_spd_mps$)', 'match');
				match_speed = vertcat( match_speed{:});
				
				match_mass = regexp(props,'^.*(?=_tot_mass_kg$)', 'match');
				match_mass = vertcat( match_mass{:});
				
				match = intersect(match_mass, intersect( match_speed, match_force )) ;
				
				for p = 1:length(match)
					meta = obj.addprop([match{p},'_iner_comp_trq_Nm']);
					meta.Dependent = 1;
                    meta.NonCopyable = false;
					meta.GetMethod = eval(['@(obj) obj.',match{p},'_frc_N  -  obj.',match{p},'_tot_mass_kg .* class_REVS_logging_object.calc_accel( obj.',match{p},'_spd_mps, obj.root.time)']);
				end
			end
			
		end
		
		
		function add_calc_loss_kJ(obj)			
			props = properties(obj);
			
			match_input = ~all(cellfun(@isempty,regexp(props,'^input[0-9]*_pos_kJ$')));
			match_output = ~all(cellfun(@isempty,regexp(props,'^output[0-9]*_pos_kJ$')));
			
			if match_input && match_output
				meta = obj.addprop('loss_kJ');
				meta.Dependent = 1;
                meta.NonCopyable = false;
				meta.GetMethod = @class_REVS_logging_object.calc_loss_kJ;
			end
			
			
		end
		
		function add_calc_rotational_power( obj )			
			props = properties(obj);
			
			match_torque = regexp(props,'^.*(?=_trq_Nm$)', 'match');
			match_torque = vertcat( match_torque{:});
			
			match_speed = regexp(props,'^.*(?=_spd_radps$)', 'match');
			match_speed = vertcat( match_speed{:});
			
			match = intersect( match_speed, match_torque );
			
			for p = 1:length(match)
				meta = obj.addprop([match{p},'_power_kW']);
				meta.Dependent = 1;
                meta.NonCopyable = false;
				meta.GetMethod = eval(['@(obj) obj.',match{p},'_trq_Nm .* obj.',match{p},'_spd_radps / 1000']);
			end
		end
		
		function add_calc_linear_power( obj)
			
			props = properties(obj);
			
			match_force = regexp(props,'^.*(?=_frc_N$)', 'match');
			match_force = vertcat( match_force{:});
			
			match_speed = regexp(props,'^.*(?=_spd_mps$)', 'match');
			match_speed = vertcat( match_speed{:});
			
			match = intersect( match_speed, match_force );
			
			for p = 1:length(match)
				meta = obj.addprop([match{p},'_power_kW']);
				meta.Dependent = 1;
                meta.NonCopyable = false;
				meta.GetMethod = eval(['@(obj) obj.',match{p},'_frc_N .* obj.',match{p},'_spd_mps / 1000']);
			end
		end
		
		function add_calc_elec_power(obj)
			props = properties(obj);
			
			match_current = regexp(props,'^.*(?=_curr_A$)', 'match');
			match_current = vertcat( match_current{:});
			
			match_volts = regexp(props,'^.*(?=_volt_V$)', 'match');
			match_volts = vertcat( match_volts{:});
			
			match = intersect( match_volts, match_current );
			
			for p = 1:length(match)
				meta = obj.addprop([match{p},'_power_kW']);
				meta.Dependent = 1;
                meta.NonCopyable = false;
				meta.GetMethod = eval(['@(obj) obj.',match{p},'_curr_A .* obj.',match{p},'_volt_V / 1000']);
			end
			
		end
		
		function add_calc_total_energy( obj)
			props = properties(obj);
			
			match_pos = regexp(props,'^.*(?=_pos_kJ$)', 'match');
			match_pos = vertcat( match_pos{:});
			
			match_neg = regexp(props,'^.*(?=_neg_kJ$)', 'match');
			match_neg = vertcat( match_neg{:});
			
			match = intersect( match_neg, match_pos );
			
			for p = 1:length(match)
				meta = obj.addprop([match{p},'_tot_kJ']);
				meta.Dependent = 1;
                meta.NonCopyable = false;
				meta.GetMethod = eval(['@(obj) obj.',match{p},'_pos_kJ - obj.',match{p},'_neg_kJ']);
			end
		end
		
		function add_calc_kWh(obj)
			props = properties(obj);
			
			match = regexp(props,'^.*(?=_kJ$)', 'match');
			
			for p = 1:length(match)
				if ~isempty(match{p})
					meta = obj.addprop([match{p}{1},'_kWh']);
					meta.Dependent = 1;
                    meta.NonCopyable = false;
					meta.GetMethod = eval(['@(obj) obj.',match{p}{1},'_kJ / 3600']);
				end
			end
			
		end
		
		function add_calc_shift_count(obj)		
			props = properties(obj);
			
			if ismember('num_upshifts', props) && ismember('num_downshifts',props)
				meta = obj.addprop('num_shifts');
				meta.Dependent = 1;
                meta.NonCopyable = false;
				meta.GetMethod = @(obj) obj.num_upshifts + obj.num_downshifts;
			end
		end
		
	end
	
	
	methods ( Access = protected)
		
		% Select properties for display
		function val = getPropertyGroups(obj)		
			props = properties(obj);
			
			obj_props = {};
			in_props = {};
			out_props = {};
			other_props = {};
			
			for p = 1:length(props)
				if isobject(obj.(props{p}))
					obj_props{end+1} = props{p};
				elseif strncmp( props{p},'input_',6)
					in_props{end+1} = props{p};
				elseif strncmp( props{p},'output_',7)
					out_props{end+1} = props{p};
				else
					other_props{end+1} = props{p};
				end
			end
			
			obj_props = sort(obj_props);
			in_props = sort(in_props);
			out_props = sort(out_props);
			other_props = sort(other_props);
			
			props = {obj_props{:}, in_props{:}, out_props{:}, other_props{:}}';
			
			val = matlab.mixin.util.PropertyGroup(props);			
        end
		       
	end
	
	
	methods (Static, Hidden)
		
		function loss_kJ = calc_loss_kJ(obj)			
			input_energy_pos = sum_fields_regexp(obj, '^input[0-9]*_pos_kJ$');
			input_energy_neg = sum_fields_regexp(obj, '^input[0-9]*_neg_kJ$');
			output_energy_pos = sum_fields_regexp(obj, '^output[0-9]*_pos_kJ$');
			output_energy_neg = sum_fields_regexp(obj, '^output[0-9]*_neg_kJ$');
			
			loss_kJ = (input_energy_pos - output_energy_pos) + (output_energy_neg - input_energy_neg);			
		end
		
		function alpha_radps2 = calc_accel(speed, time)			
			alpha_radps2 = diff(speed)./diff(time);
			alpha_radps2 = ([alpha_radps2(1) ; alpha_radps2] + [alpha_radps2 ; alpha_radps2(end)])/2;			
        end
		
        function obj = from_struct(s)
           
            if ~isstruct(s)
                error('Unable to convert input, must be of type "struct"');
            end
            
            obj = class_REVS_logging_object();
            fields = fieldnames(s);
            
            for f = 1:length(fields)
                obj.addprop(fields{f});
                obj.(fields{f}) = s.(fields{f});
            end
            
        end
        
	end
	
end

