classdef class_REVS_result < dynamicprops
    %class_REVS_result
    %   Definition of class_REVS_result class
    
    properties
        
        phase;
        weighted;
        
    end
        
    methods
        
        function obj = class_REVS_result()
            
            powertrain_type = evalin('caller', 'vehicle.powertrain_type');
            
            drive_cycle = evalin('caller','drive_cycle');
            drive_cycle.name = char(drive_cycle.name);
            result_phase_names = drive_cycle.phase_name;
            
            
            if powertrain_type.is_conventional || powertrain_type.is_hybrid || powertrain_type.is_P0 || powertrain_type.is_P2
                meta = obj.addprop('map_fuel');
                obj.map_fuel = evalin('caller','engine.fuel');
                meta.SetMethod = @(o) error('Base simulation fuel is not an editable property');
                
                obj.addprop('output_fuel');
                
                try
                    obj.output_fuel = evalin('caller','vehicle.fuel');
                catch
                    obj.output_fuel = obj.map_fuel;
                end
                
                
                if isempty(obj.output_fuel)
                    obj.output_fuel = obj.map_fuel;
                end
                
            end

            
            if powertrain_type.is_hybrid || powertrain_type.is_bev                
                obj.addprop('charging_efficiency');
                obj.charging_efficiency = 0.90;                 
            end
                        

            % Make phase and total appropriate format for model type
            if powertrain_type.is_conventional
                obj.phase = class_REVS_CVM_result(obj, result_phase_names);
            elseif powertrain_type.is_hybrid || powertrain_type.is_P0 
                obj.phase = class_REVS_HVM_result(obj, result_phase_names);
            elseif powertrain_type.is_bev
                obj.phase = class_REVS_EVM_result(obj, result_phase_names);
            else
                obj.phase = class_REVS_VM_result(obj, result_phase_names);
                warning('Unknown powertrain type "%s" falling back to generic vehicle result',powertrain_type)
            end
            
                       
            
            % Get list of matching workspace variables
            ws_vars = sort(evalin('caller','who(''rsltp__*'');'));
            
            for v = 1:length(ws_vars)
                
                % root field to work off - all
                working_p = obj.phase;
                
                % Only load non-empty data
                if ~evalin('caller',['isempty(', ws_vars{v} ,');'] )
                    
                    var_fields = strsplit( ws_vars{v}, '__');
                    for f = 2:(length(var_fields)-1)
                        % Add nonexistent properties
                        if ~isprop( working_p,var_fields{f})
                            working_p.addprop(var_fields{f});
                            working_p.(var_fields{f}) = class_REVS_logging_object(obj.phase);
                        end
                        
                        % Move along tree
                        working_p = working_p.(var_fields{f});
                    end
                    
                    if ~isprop( working_p,var_fields{end})
                        working_p.addprop(var_fields{end});
                    end
                    
                    working_p.( var_fields{end} ) = evalin('caller',ws_vars{v});
                    
                end
                
                % cleanup workspace
                evalin('caller',['clear ', ws_vars{v}]);
            end
            
            %After data is loaded add calculated properties
            obj.phase.add_calculated_properties;
            
            % Add drive quality metrics
            calc_drive_quality = numel(evalin('caller','who(''datalog'',''vehicle'')')) == 2 ;
            if calc_drive_quality               
                datalog = evalin('caller','datalog');
                vehicle = evalin('caller','vehicle'); 
                
                if datalog.isproperty('drive_cycle.phase') && datalog.isproperty('drive_cycle.spd_mps') && datalog.isproperty('vehicle.output_spd_mps') && datalog.isproperty('time') 
                    obj.phase.add_SAE_J2951( vehicle, datalog.time, datalog.drive_cycle.phase, datalog.vehicle.output_spd_mps, datalog.drive_cycle.spd_mps);
                end
            end
                        
            % Generate weighed cycle results
            cycles = drive_cycle.subcycles;
         
            if isempty(cycles)
                cycles = struct('cycle',drive_cycle.name, 'phases', 1:max(drive_cycle.phase));
            end
            
            pd_props = properties(obj.phase);

            for c = 1:numel(cycles)
                
                if contains(cycles(c).cycle, 'FTP')
                    aggregator = class_REVS_EPA_FTP_cycle_aggregator(obj.phase, obj.phase, cycles(c).phases);
                    aggregator_func = @class_REVS_EPA_FTP_cycle_aggregator;
                else
                    aggregator = class_REVS_default_cycle_aggregator(obj.phase, obj.phase, cycles(c).phases);
                    aggregator_func = @class_REVS_default_cycle_aggregator;
                end
                                
                if powertrain_type.is_conventional
                    wr = class_REVS_CVM_result(obj, cycles(c).cycle, aggregator);
                elseif powertrain_type.is_hybrid || powertrain_type.is_P0 || powertrain_type.is_P2
                    wr = class_REVS_HVM_result(obj, cycles(c).cycle,  aggregator);
                elseif powertrain_type.is_bev
                    wr = class_REVS_EVM_result(obj, cycles(c).cycle,  aggregator);
                else
                    wr = class_REVS_VM_result(obj, cycles(c).cycle,  aggregator);
                end
                                
                for p=1:numel(pd_props)

                    prop = pd_props{p};
                    prop_meta = findprop(obj.phase,prop);
                    if isa(obj.phase.(prop),'class_REVS_logging_object')
                        if ~isprop(wr,prop)                    
                            meta = wr.addprop( prop);
                        end
                        wr.(prop) = class_REVS_cycle_result_object(obj.phase.(prop),  cycles(c).phases, aggregator_func, obj.phase);            
                    end
                end
                
                if isempty(obj.weighted)
                    obj.weighted = wr;
                else
                    obj.weighted(end+1) = wr;
                end
            end
    
            % Generate EPA combined results 55% FTP 45% HWFET
            cycle_results = fieldnames(obj.weighted);
            ftp_cycle_idx = find( startsWith(cycle_results, 'EPA_FTP'),1,'last');
            hwfet_cycle_idx= find( startsWith(cycle_results, 'EPA_HWFET'),1,'last');
            
            if ~isempty(ftp_cycle_idx) && ~isempty(hwfet_cycle_idx)
               
                aggregator = class_REVS_EPA_combined_aggregator(obj.weighted, obj.weighted, cycle_results([ftp_cycle_idx, hwfet_cycle_idx]));
                aggregator_func = @class_REVS_EPA_combined_aggregator;
                                
                if powertrain_type.is_conventional
                    wr = class_REVS_CVM_result(obj, 'EPA_combined', aggregator);
                elseif powertrain_type.is_hybrid || powertrain_type.is_P0 || powertrain_type.is_P2
                    wr = class_REVS_HVM_result(obj, 'EPA_combined', aggregator);
                elseif powertrain_type.is_bev
                    wr = class_REVS_EVM_result(obj, 'EPA_combined', aggregator);
                else
                    wr = class_REVS_VM_result(obj, 'EPA_combined', aggregator);
                end

                for p=1:numel(pd_props)

                    prop = pd_props{p};
                    prop_meta = findprop(obj.phase,prop);
                    if isa(obj.phase.(prop),'class_REVS_logging_object')
                        if ~isprop(wr,prop)                    
                            meta = wr.addprop( prop);
                        end
                        wr.(prop) = class_REVS_cycle_result_object(obj.phase.(prop), cycle_results([ftp_cycle_idx, hwfet_cycle_idx]), aggregator_func, obj.phase );            
                    end
                end
                
                obj.weighted(end+1) = wr;
                
            end
            
            % Add performance metrics (if available)
            perf_idx = find( startsWith({cycles.cycle}, 'REVS_Performance', 'IgnoreCase', true), 1);
            if ~isempty(perf_idx)
               
                try 
                    perf_stats = REVS_calc_performance_stats( datalog.time, datalog, 'start_phase', cycles(perf_idx).phases(1));
                catch
                    perf_stats = [];
                end
                
                if ~isempty(perf_stats)
                    obj.addprop('performance');
                    obj.performance = class_REVS_logging_object.from_struct(perf_stats);
                end
                
            end
            
            
            % Dirty way to make a deep copy - equivalent to saving to a MAT
            % file and loading. Would be good to add proper copy
            % functionality to class_REVS_logging_object
            copyStream = getByteStreamFromArray(obj);            
            obj.addprop('unadjusted');
            obj.unadjusted = getArrayFromByteStream(copyStream);
                        
        end
        
       
        function print( obj, fid )
            
            if nargin < 2 || isempty(fid)
                fid = 1;
            end
            
            obj.print_phase(fid);
            obj.print_weighted(fid);   
            obj.print_performance(fid);
        end
        
        function print_phase( obj, fid )
            
            if nargin < 2 || isempty(fid)
                fid = 1;
            end
            
            fprintf(fid, '       ---------- Phase Results ----------\n');
            obj.phase.print(fid);            
        end
        
        function print_weighted( obj, fid )
            
            if nargin < 2 || isempty(fid)
                fid = 1;
            end
            
            fprintf(fid, '       --------- Weighted Results --------\n');
            cycles = fieldnames(obj.weighted);
            for c = 1:numel(obj.weighted)
                obj.weighted(c).print(fid);
            end
            
        end
        
                
        function print_performance(obj, fid)
            
            if nargin < 2 || isempty(fid)
                fid = 1;
            end
            
            if isprop(obj,'performance')
                fprintf(fid, '       -------- Vehicle Performance --------\n');
                fprintf(fid,'         0 to 30 time: %7.4f sec \n', obj.performance.zero_to_thirty_time_secs);
                fprintf(fid,'         0 to 60 time: %7.4f sec \n', obj.performance.zero_to_sixty_time_secs);
                fprintf(fid,'        30 to 50 time: %7.4f sec \n', obj.performance.thirty_to_fifty_time_secs);
                fprintf(fid,'        50 to 70 time: %7.4f sec \n', obj.performance.fifty_to_seventy_time_secs);
                fprintf(fid,'        1/4 mile time: %7.4f sec @ %6.2f mph\n', obj.performance.quarter_mile_time_secs, obj.performance.quarter_mile_speed_mph );
                fprintf(fid,'        1/2 mile time: %7.4f sec @ %6.2f mph\n', obj.performance.half_mile_time_secs, obj.performance.half_mile_speed_mph );
                fprintf(fid,'        Top Speed    : %6.2f mph\n', obj.performance.top_speed_mph);
                fprintf(fid,'\n');
            end
        end
        
    end
end


