classdef class_REVS_sim_batch < handle & matlab.mixin.Copyable
    %class_REVS_sim_batch supports batch runs of sim cases, as defined
    %   by a set of configuration strings in a config set
    
    properties
        cpu_count               = [];
        matlabname              = '';
        timestamp               = [];
        descriptor              = 'sim'
        
        verbose                 = 1; % goes to REVS.verbose
        
        model                   = 'REVS_VM';    % use full path if model is not in REVS_common or its subfolders or have model open while running
        logging_config      class_REVS_logging_config;
        
        output_verbose          = 3;        % 0 = minimum output, 1 = min + detailed configs, 2 = 1 + phase results, 3 = all outputs
        output_restrict_phases  = [];       % list of drive cycle phases to NOT output to results file
        
        publish_params          = false;    % Publish input workspaces to output folder as .txt and .mat files
        retain_output_workspace = false;    % Retain post-simulation workspace in sim_case
        save_input_workspace    = false;    % Save pre-simulation workspace to a .mat file
        save_output_workspace   = false;    % Save post-simulation workspace to a .mat file
        
        save_output_console     = true;     % Save batch console log output to file 
        show_output_console     = true;     % Display batch console outputs
        save_case_console       = true;     % Save case console log output to file
        show_case_console       = false;    % Display case console output 
               
        output_file;            % structure of output file vars
        result_files            = {}; % cell array of result files from distributed run
        
        output_path             = 'output/';       
        param_path              = 'param_files/';
        script_path             = 'scripts/';
        sim_path                = '';   % folder where simulation executables (slprj) will be placed
        
        config_keys class_REVS_sim_config_key % =  class_REVS_sim_config_key('aggregation_keys', 'default', {});
        config_set;                 % set of configuration strings that can be used to define sim_case objects generated from config key packages

        case_preprocess_scripts class_REVS_sim_config_script
        case_postprocess_scripts class_REVS_sim_config_script
        batch_postprocess_scripts class_REVS_sim_config_script
        
        setup_data_columns          = '';       
        
        sim_case class_REVS_sim_case;   % holds class_REVS_sim_case object(s)
        sims_completed              = 0;    % count of sims successfully completed in this batch
        sims_skipped                = 0;    % count of sims skipped in this batch
        sims_failed                 = 0;    % count of sims which failed to run in this batch
        
        disable_sims            = false; % set true to test process without running sims
                
        network_hostname        = '';   % name of computer on LAN                
        parallel_batch_path     = '';   % path of currently executing parallel batch
        parallel_batch_cleanup  = false;
        parallel_base_path      = '';   % path under which to store parallel batch jobs
        parallel_nodes          = {};   % cell array of nodes to use for networked parallel execution
        parallel_scheduler      = ''    % scheduler node to manage a shared job cluster 
        parallel_min_chunk_size = nan;  % minimum number of simulations to distribute to a node
        parallel_path_folders   = {};   % folders to copy to batch path for use on other nodes
        parallel_dispy_debug    = false;
        parallel_dispy_opts     = {};    % Additional arguments to pass to revs_dispy.py

        globals;                % generic property to hold values across sim_case simulations, can be used as a struct or any other Matlab var
    end
    
    properties ( Dependent, Transient )
        
        sim_configs struct;              % simulation configuration - the expanded set of simulation cases
        size;                            % number of simulations in the batch

    end
    
    
    properties ( Hidden,  Access = protected)
                
        cache_sim_configs;              % stored expanded simulation configurations
          
        start_path              = '';   % path from which batch simulation is run
        log_fid;                        % file ID for batch log file
                
    end
    
    
    methods
        %% -----------------------------------------------------------------------
        
        function val = get.size( obj )
            val = length(obj.sim_configs);
        end
        
        function [] = getenv(obj)
            
            [~, obj.network_hostname] = system('hostname');
            obj.network_hostname = strtrim(obj.network_hostname);
            obj.matlabname       = matlab.engine.engineName;
            if isempty(obj.matlabname)
                obj.matlabname = 'M';
            else
                obj.matlabname = strrep(obj.matlabname,'MATLAB','ML');
            end
            obj.cpu_count      = getenv('NUMBER_OF_PROCESSORS');
            
        end
        
        function obj = class_REVS_sim_batch(default_log_list)
            %addpath([pwd '\dev\'])
            

            obj.logging_config = class_REVS_logging_config;
            
            obj.getenv();   % get environment variables
            
            obj.config_keys = class_REVS_sim_config_key('aggregation_keys', 'default', {});
                        
            if nargin > 0
                warning('Providing the default log list in the constructor is a feature that will be removed in the future, use the add_log method to add simulation logging');
                obj.logging_config.add_log( default_log_list );
            end
            
        end
            
        
        
        %% Configuration of Simulation Loggging
        function add_log(obj,  log_list)
            obj.logging_config.add_log( log_list );
        end
        
        %% 
        function add_config_opt(obj, opts, varargin)
            
            if ~isa(opts, 'class_REVS_sim_config_options')
                error('Expecting a class_REVS_sim_config_options object as input');
            end
            
            for o = 1:numel(opts)
                
                opt = opts(o);
                duplicate = false(size(opt.keys));
                existing_keys = {obj.config_keys.key};
                existing_tags = {obj.config_keys.tag};

                for k = 1:length(opt.keys)                
                    if ismember(opt.keys(k).key , existing_keys ) 
                        if ~isequaln( opt.keys(k) , obj.get_key_info(opt.keys(k).key))                   
                            error('Configuration option includes a key "%s" that is already in use', opt.keys(k).key)
                        else
                            duplicate(k) = true;
                        end              
                    elseif ~isempty(opt.keys(k).tag) && ismember(opt.keys(k).tag , existing_tags )               
                        error('Configuration option includes a key tag "%s" that is already in use', opt.keys(k).tag)               
                    end                
                end


                obj.config_keys = [ obj.config_keys ; opt.keys(~duplicate) ];

                if ismember('keys_only', varargin)
                    % Don't add scripts
                    continue
                end
                
%                 obj.case_load_scripts = [ obj.case_load_scripts; opt.load_scripts ];
                obj.case_preprocess_scripts = [ obj.case_preprocess_scripts; opt.case_preprocess_scripts ];
                obj.case_postprocess_scripts = [ obj.case_postprocess_scripts; opt.case_postprocess_scripts ];
                obj.batch_postprocess_scripts = [ obj.batch_postprocess_scripts; opt.batch_postprocess_scripts ];
            
            end
            
        end
        
        function add_key(obj, key, varargin)      
            k = class_REVS_sim_config_key(key, varargin{:}); 
            
            if any( isequaln(k, obj.config_keys) )
               warning('Key for "%s" already exists', key);
            elseif ~isempty(k.tag) && ismember(k.tag, {obj.config_keys.tag} )
                error('Key includes a key tag "%s" that is already in use', k.tag)   
            else
                 obj.config_keys = [ obj.config_keys ; k ];  
            end
                                 
        end
        
        function val = get_key_info(obj, key)         
            keys = {obj.config_keys.key};           
            idx = find(strcmp(key, keys)); 
            
            if isempty(idx)
                error('No key matching "%s" found.', key);
            elseif numel(idx) > 1
                error('Multiple keys matching "%s" found.', key);
            else
                val = obj.config_keys(idx);
            end
                
        end
        
        function show_keys(obj)
            fprintf('\t   Key                                    |     Tag                |     Default Value                  |     Provided by                |     Description\n')
            fprintf('\t-------------------------------------------------------------------------------------------------------------------------------------------------------\n')
            
            for k = 1:numel(obj.config_keys)
                
               fprintf('\t%s  |', limit_and_pad( obj.config_keys(k).key, 40));
               fprintf('  %s  |', limit_and_pad( obj.config_keys(k).tag, 20));
               
               default = obj.config_keys(k).default;
               if isempty( default)
                    default_str = '';
               elseif isnumeric(default) || islogical(default)
                     default_str = mat2str(default);
               elseif ischar(default)
                     default_str = default;
               elseif iscell(default) && isempty(default)
                   default_str = '{ }';               
               elseif iscellstr(default) || isstring(default)
                    default_str = ['{', strjoin(default, ', '),'}'];
               else
                    default_str = '???';
               end                   
               fprintf('  %s  |', limit_and_pad( default_str, 32));
               
               fprintf('  %s  |', limit_and_pad( obj.config_keys(k).provided_by, 28));

               fprintf('  %s\n', obj.config_keys(k).description);

            end
            
            
            function out = limit_and_pad( str, len)                
                if length(str) > len               
                    str = str(1:len);
                    str(end-2:end) = '...';                   
                end
                out = pad( str,len);    
            end
            
        end
                        
        function add_case_preprocess_script(obj, script, sequence)
            obj.case_preprocess_scripts(end+1) = class_REVS_sim_config_script(script, sequence);
        end
        
        function set.case_preprocess_scripts(obj, val)
            [~,idx] =sort( [val.sequence]);
            obj.case_preprocess_scripts = val(idx);
        end
        
        function add_case_postprocess_script(obj, script, sequence)
            obj.case_postprocess_scripts(end+1) = class_REVS_sim_config_script(script, sequence);
        end
        
        function set.case_postprocess_scripts(obj, val)
            [~,idx] =sort( [val.sequence]);
            obj.case_postprocess_scripts = val(idx);
        end
        
        function add_batch_postprocess_script(obj, script, sequence)
            obj.batch_postprocess_scripts(end+1) = class_REVS_sim_config_script(script, sequence);
        end
        
        function set.batch_postprocess_scripts(obj, val)
            [~,idx] =sort( [val.sequence]);
            obj.batch_postprocess_scripts = val(idx);
        end
        
        
        function load_config_strings(obj, config_strings)
        
            if ischar( config_strings )
                config_strings = {config_strings};
            end
            
            obj.config_set = cell(numel(config_strings), 1);
            
            for c = 1:numel(config_strings)               
                 aggregation_loc = strfind( config_strings{c}, '||');
                 obj.config_set{c}.aggregation_keys = {};
                 
                 found = false(size(config_strings{c}));
                 
                 for k = 1:numel(obj.config_keys)
                     
                     key_info = obj.config_keys(k);
                     [tag_tokens, tag_start, tag_end] = regexp(config_strings{c},['(?<=(^|\s))(' key_info.tag ':)(.*?)(?=((\s+\+)|(\s+\|\|)|$))'],'tokens', 'start', 'end');
                                         
                     if numel(tag_tokens) > 1
                         warning('Multiple entries found for config tag "%s", using "%s"', key_info.tag,  tag_tokens{1}{2});                         
                     elseif isempty(tag_tokens)
                         continue;
                     end
                     
                     value = tag_tokens{1}{2};
                     
                     found(tag_start:tag_end) = true;
                     
                     
                    if key_info.eval
                        obj.config_set{c}.(key_info.key) = eval(value);
                    elseif regexpcmp(value, '\{.*\}')
                        obj.config_set{c}.(key_info.key) = {eval(value)};
                    else    
                        obj.config_set{c}.(key_info.key) = {value};
                    end
                    
                    if tag_start > aggregation_loc
                        obj.config_set{c}.aggregation_keys{end+1} = key_info.key;
                    end
                                                                                      
                 end
                 
                 unused = config_strings{c}(~found);                
                 unused_tags = regexp(unused,['(?<=(^|\s))(\w*:.*?)(?=((\s+\+)|(\s+\|\|)|$))'],'match');
                                         
                 if ~isempty(unused_tags)
                    warning('Found %d unused tags on configuration string %d : \n\t\t%s', numel(unused_tags), c, strjoin(unused_tags, sprintf('\n\t\t')));
                 end
                 
            end
                        
        end

        function set.config_set(obj, val)         
            obj.config_set = val;
            obj.cache_sim_configs = [];            
        end
        
        function [val] = get.sim_configs(obj)
        
            if ~isempty(obj.cache_sim_configs)
                val = obj.cache_sim_configs;
                return
            end
                                   
            for k = 1:numel(obj.config_keys)
                key_info = obj.config_keys(k);             
                default_config.(key_info.key) = key_info.default;
            end
            
            key_list = {obj.config_keys.key};
            
            
            default_config.base_hash = '';
            default_config.aggregation_hash = '';
            
            if isstruct(obj.config_set)
                config_set = num2cell(obj.config_set);
            elseif iscell(obj.config_set)
                config_set = obj.config_set;
            else
                error('Config set is is an unknown type, expecting struct or cell array of structs');
            end
            
            
            
            val = rmfield(default_config, 'aggregation_keys');            
            val.base_hash = '';
            val.aggregation_hash = '';
            val.simulation_hash = '';
            val = val([]);
            
            for c = 1:numel(config_set)
                
                filled_config = default_config;
                keys = fieldnames(config_set{c});
                for k = 1:numel(keys)
                    
                    if ischar( config_set{c}.(keys{k}))
                        % Convert chars to cell array of chars (don't iterate over string characters)
                        filled_config.(keys{k}) = { config_set{c}.(keys{k}) };
                    else
                        filled_config.(keys{k}) = config_set{c}.(keys{k}); 
                    end
                    
                    if ~ismember(keys{k}, key_list)
                       warning('Config set includes reference to key "%s" that has not been defined',  keys{k})
                       key_list{end+1} = keys{k};
                    end                    
                end
                                
                aggregation_keys = filled_config.aggregation_keys;
                filled_config = rmfield( filled_config, 'aggregation_keys');
                                               
                keys = fieldnames(filled_config);
                options = ones(1,numel(keys));
                
                for k = 1:numel(keys)
                    options(k) = max(1,numel(filled_config.(keys{k})));
                end
                
                % Full Factorial of options & make initial filled cases
                if prod( options ) > 1e6
                    warning('Configuration set would requireat least %d simulations.', prod( options ));
                    cont = input('Continue with simulations? Y/N [N]: ','s');
                    
                    if ~ismember(upper(cont), {'Y','YES'})
                        error('Simulation aborted...');
                    end
                        
                end
                    
                ff = REVS_fullfact(options);
                sim_configs = repmat(filled_config, size(ff,1), 1);
                
                % Process keys not being aggregated over
                nonaggregation_keys = setdiff(keys, aggregation_keys);
                
                
                for k = 1:numel(nonaggregation_keys)
                    key = nonaggregation_keys{k};
                    [~,ff_ki] = ismember(key, keys);
                    if isempty(  filled_config.(key) )
                        % Empty = skip
                    elseif iscell( filled_config.(key))
                        for o = size(ff,1):-1:1
                            sim_configs(o).(key) = filled_config.(key){ff(o,ff_ki)};
                        end
                    else 
                       for o = size(ff,1):-1:1
                            sim_configs(o).(key) = filled_config.(key)(ff(o,ff_ki));
                       end
                    end
                end
                
                % Compute aggreagtion hash (aggregation keys not expanded
                % yet)
               aggregation_hash = arrayfun(@hash, sim_configs, 'UniformOutput', false);
                
                % Expand keys being aggregated over               
                 for k = 1:numel(aggregation_keys)
                    key = aggregation_keys{k};
                    [~,ff_ki] = ismember(key, keys);
                    if isempty(  filled_config.(key) )
                        % Empty = skip
                    elseif iscell( filled_config.(key))
                        for o = size(ff,1):-1:1
                            sim_configs(o).(key) = filled_config.(key){ff(o,ff_ki)};
                        end
                    else 
                       for o = size(ff,1):-1:1
                            sim_configs(o).(key) = filled_config.(key)(ff(o,ff_ki));
                       end
                    end
                end
                
                %[sim_configs(:).base_hash = 
                
                sim_hash = arrayfun(@hash, sim_configs, 'UniformOutput', false);
                                
                [sim_configs.base_hash] = deal(hash( filled_config));
                [sim_configs.aggregation_hash] = deal(aggregation_hash{:});
                [sim_configs.simulation_hash] = deal(sim_hash{:});
                
                
                val = [val; sim_configs];
                
            end
            
            %obj.config_set_updated = false;
            obj.cache_sim_configs = val;           

            
        end
        

        %% -----------------------------------------------------------------------
        function [sim_config] = get_sim_config(obj, index)
              sim_config = obj.sim_configs(index);
        end
        
        function [out] = collect_case_values(obj, expr)
                              
            out = arrayfun( @(c) eval(['c.',expr]),   obj.sim_case', 'UniformOutput',false) ;
            
            dims = cellfun( @ndims, out);
            height = cellfun( @(x) size(x,1), out);
            width = cellfun( @(x) size(x,2), out);
            
            if all( dims == 2) && all(height == 1) && all(width == width(1))              
                out = vertcat(out{:});               
            end
           
        end
        
        %% -----------------------------------------------------------------------
        function [result_file] = run_sim_cases(obj, config_select, parallel_slave)
                                 
            if nargin < 3
                parallel_slave = false;
            end
            
            warning('off','MATLAB:MKDIR:DirectoryExists');
            
            % update environment variables
            obj.getenv(); 
            obj.start_path = pwd;
            obj.timestamp = now;
            
            % Set cleanup to reset working directory on crash
            onCleanup( @() cd(obj.start_path));
            
            warning('off','backtrace')

            if ~isfolder( obj.output_path)
                mkdir( obj.output_path);
            end

            obj.prepare_output_file(parallel_slave);           
            obj.prepare_log_file(parallel_slave);
            
            % Add the script folder to the path (temporarily)
            
            if isfolder(obj.script_path)
                cd(obj.script_path);
                script_dir = pwd;
                addpath(script_dir);
            else
                 script_dir = '';
            end

            cd(obj.start_path);
            
            if parallel_slave
                % Just use Matlabname (shorter path to stay under windows limit)
                obj.sim_path = [obj.output_path obj.matlabname '_sim/'];
            else
                obj.sim_path = [obj.output_path obj.descriptor '_sim/'];
            end
                        
            mkdir(obj.sim_path);
            
            if nargin < 2
                config_select = 1:length(obj.sim_configs);
            else
                config_select = unique(config_select);                        
            end
                        
            batch_size = numel(config_select);
                        
            if parallel_slave   
                
               obj.ts_log( 'Batch Descriptor: %s', obj.descriptor);
               obj.ts_log( 'Simulation Path: %s', obj.sim_path);
               obj.ts_log( 'Simulation Batch Size: %d', batch_size);
               
               if nargin >= 2
                   if numel(config_select) < 10
                        obj.ts_log( 'Running Selected Cases: %s ', int2str(config_select));
                   else
                       obj.ts_log( 'Running Selected Cases: %s ... %s', int2str(config_select(1:5)), int2str(config_select(end-5:end)));
                   end
               end
                
            else
                                
                obj.log( 'Batch Descriptor: %s', obj.descriptor);
                obj.log( 'Simulation Path: %s', obj.sim_path);
                obj.log( 'Simulation Batch Size: %d', batch_size);
                
                if nargin >= 2
                    obj.log( 'Running Selected Cases: %s ', int2str(config_select) );                            
                end
                
                obj.log('\nConfiguration Keys: ');
                obj.log('%s\n', evalc('obj.show_keys'));

                obj.log('\nSim Case Preprocess Scripts: ');
                obj.log('%s\n', obj.case_preprocess_scripts.script );

                obj.log('\nSim Case Postprocess Scripts: ');
                obj.log('%s\n', obj.case_postprocess_scripts.script );

                obj.log('\nSim Batch Postprocess Scripts: ');
                obj.log('%s\n', obj.batch_postprocess_scripts.script ); 

                 h = waitbar(0,sprintf('\n\n\n\n\n\n'));
                 waitbar(0,h,sprintf('Running Simulations\nElapsed Time 0 minutes\nRemaining Time: ? minutes\n\n'));
            end

            start_time = clock;
            sim_time   = start_time;
                                               
            idx = 0;
            obj.sims_completed = 0;
            obj.sims_skipped = 0;
            obj.sims_failed = 0;
            obj.output_file.prev_length_data_columns = 0;
                                
            if ~parallel_slave
               fprintf('\n*****************************************\n Running %d Simulations \n*****************************************\n\n', length(config_select));
            end
                                
            for i = config_select
                
                cd(obj.start_path)
                
                idx = idx + 1;
                obj.ts_log( 'Preparing Sim Case #%d - ( %d / %d )', i, idx, batch_size );

                sim_case_num = i;
                if ~obj.retain_output_workspace % Override storage to use index 1 and save memory
                    sim_case_num = 1;
                end

                obj.sim_case(sim_case_num) = obj.setup_sim_case(i);     
                
                if obj.sim_case(sim_case_num).prepare_workspace() 
                    obj.ts_log( 'Sim Case #%d Failed - see %s for details', i, obj.sim_case(sim_case_num).console_log_file);
                    obj.sims_failed = obj.sims_failed + 1;
                    continue                   
                end

                if obj.disable_sims 
                    continue
                end

                obj.ts_log( 'Running Sim Case #%d - ( %d / %d )', i, idx, batch_size);

                status = obj.sim_case(sim_case_num).sim;

                if status == 1
                    obj.ts_log( 'Sim Case #%d Failed - see %s for details', i, obj.sim_case(sim_case_num).console_log_file);
                    obj.sims_failed = obj.sims_failed + 1;
                    continue
                elseif status == -1
                    obj.ts_log( 'Sim Case #%d Skipped - see %s for details', i, obj.sim_case(sim_case_num).console_log_file);
                    obj.sims_skipped = obj.sims_skipped + 1;  
                    continue
                end

                obj.postprocess_sim_case(i);

                if ~obj.retain_output_workspace
                    obj.sim_case(sim_case_num).workspace = [];
                    obj.sim_case(sim_case_num) = [];
                end

                obj.sims_completed = obj.sims_completed + 1;
                
                % Estimate remaiaing time
                sim_time = clock;                   
                elapsed_time = etime(sim_time, start_time);
                time_per_sim = elapsed_time / idx;

                obj.ts_log( 'Completed Sim Case #%d - ( %d / %d ) - %.1f sec/sim', i, idx, batch_size, time_per_sim);

                if ~parallel_slave

                    remaining_time = (batch_size - idx) * time_per_sim;
                    est_completion_time = datetime('now','Format','eee h:mm a');
                    est_completion_time.Second = est_completion_time.Second + remaining_time;

                    try                       
                        progress_str = sprintf('Running Sim [%d] %d / %d\nElapsed Time %.1f Minutes (%.1f secs/sim)\nRemaining Time %.1f Minutes\nEstimated Completion %s\n', i, idx, batch_size, elapsed_time/60, time_per_sim, remaining_time/60, est_completion_time);
                        waitbar((idx) / batch_size, h, progress_str);
                    end
                end


            end

            sim_time = clock;
            elapsed_time = etime(sim_time, start_time);              
            if ~parallel_slave
                try
                    progress_str = sprintf('Completed %d Simulations %s\nElapsed Time %.1f Minutes\n%.1f Seconds per Sim\n\n', batch_size, datetime('now','Format','eee h:mm a'), elapsed_time/60, elapsed_time/batch_size);
                    waitbar(1, h, progress_str);
                end
            end

            pause(0.1);

            obj.ts_log( 'Completed %d Simulations in %.1f Minutes (%.1f sec/sim)', batch_size, elapsed_time/60, elapsed_time/batch_size);

            [rmdir_status, rmdir_message] = rmdir(obj.sim_path,'s');

            if ~rmdir_status
                fprintf('Working directory cleanup failed: %s\n',rmdir_message);
            end

            if ~parallel_slave
                for s = 1:numel(obj.batch_postprocess_scripts)
                    run(obj.batch_postprocess_scripts{s});
                end
            end

            if ~isempty(script_dir)
                rmpath(script_dir)
            end

            if obj.save_output_console 
                fclose(obj.log_fid);
            end
                
                       
            if nargout > 0
                result_file = obj.output_file.name;
            end
            
        end
        
        %% -----------------------------------------------------------------------
        function sim_case = setup_sim_case(obj, i)

            sim_case                           = class_REVS_sim_case;
            sim_case.param_path                 = obj.param_path;
            sim_case.script_path                = obj.script_path;
            sim_case.output_path                = obj.output_path;
            sim_case.timestamp                  = datestr(obj.timestamp,'yyyy_mm_dd_HH_MM_SS');
            sim_case.descriptor                 = obj.descriptor;
            sim_case.sim_path                   = obj.sim_path;
            
            sim_case.name                       = num2str(i);
            sim_case.model                      = obj.model;
            sim_case.verbose                    = obj.verbose;
            sim_case.logging_config             = copy(obj.logging_config);
            
            sim_case.save_console               = obj.save_case_console;
            sim_case.show_console               = obj.show_case_console;
            sim_case.save_input_workspace       = obj.save_input_workspace;
            sim_case.save_output_workspace      = obj.save_output_workspace;
            
            sim_case.sim_config                 = obj.sim_configs(i);
            sim_case.preprocess_scripts         = {obj.case_preprocess_scripts.script};
            sim_case.postprocess_scripts        = {obj.case_postprocess_scripts.script};
            
            sim_case.batch_sim_ptr = obj;
        end       
        
        %% -----------------------------------------------------------------------
        function [] = postprocess_sim_case(obj, sim_case)
            if obj.retain_output_workspace
                sim_case_num = sim_case;
            else
                sim_case_num = 1;
            end
                        
            [fid, MESSAGE] = fopen(fullfile(obj.start_path, obj.output_file.name),'A');
            if fid < 0
                error([obj.output_file.name ' : ' MESSAGE]);
            end
            
            output_verbose = obj.output_verbose; %#ok<PROPLC,NASGU>
            
            % load workspace
            obj.sim_case(sim_case_num).extract_workspace; % pull simulation workspace into current workspace so data_columns aren't completely awkward
            result = obj.sim_case(sim_case_num).workspace.result; %#ok<NASGU> % rename for easier access / portability
            
            
            
            if REVS.logging_config.has_package('REVS_log_fuel_economy') || REVS.logging_config.has_package('REVS_log_all') || ~isempty(obj.setup_data_columns)
                
                if vehicle.powertrain_type == enum_powertrain_type.conventional
                    REVS_setup_data_columns_CVM;
                elseif vehicle.powertrain_type == enum_powertrain_type.battery_electric
                    % EVM
                    REVS_setup_data_columns_EVM;
                else
                    % HVM
                    REVS_setup_data_columns_HVM;
                end
                
                run(obj.setup_data_columns);
                
                % Write output file data
                if length(data_columns) ~= obj.output_file.prev_length_data_columns
                    write_column_row(fid, data_columns, 'header', 'verbose', obj.output_verbose, 'insert_blank_row', ftell(fid) > 0);
                end
                write_column_row(fid, data_columns, 'data', 'verbose', obj.output_verbose);
                
                obj.output_file.prev_length_data_columns = length(data_columns);
            end
            
            fclose(fid);
            
        end
        
        %% -----------------------------------------------------------------------
        function [] = run_sim_cases_parallel(obj, configs)
            
            obj.start_path = pwd;
            obj.timestamp = now;
            
            if ~isempty(obj.parallel_base_path)
                % have a place to store things
            elseif isempty( obj.parallel_nodes )
                obj.parallel_base_path = tempdir;
                fprintf('Storing batch shared files in: %s\n', obj.parallel_base_path)
            else
                error('A networked distributed parallel batch requires a storage location, set the parallel_base_path property and try again');
            end
            
            if ~isfolder(obj.output_path)
                disp('Creating output folder...');
                mkdir(fullfile(obj.start_path, obj.output_path));
            end

            fprintf('Generating Distributable Sim Batch...\n')
            obj.prepare_parallel_sim_batch(); % export sim_batch and required files to network for shared access
            

            if isempty( obj.parallel_nodes )
                % Local Only
                dispy_batch_options = '';
            elseif iscell( obj.parallel_nodes)  
                dispy_batch_options = ['--nodes ', strjoin(obj.parallel_nodes)];
            elseif ischar( obj.parallel_nodes)
                dispy_batch_options = ['--nodes ', strjoin(strsplit(obj.parallel_nodes,', '))];
            end
            
            if nargin == 1
                configs = 1:obj.size;
            end
            
            if nargin >= 2
                dispy_batch_options = [dispy_batch_options, ' --cases ', int2str(configs)];
            end
                                    
            if ~isempty(obj.parallel_min_chunk_size) && ~isnan(obj.parallel_min_chunk_size)
                dispy_batch_options = [dispy_batch_options, ' --min_job_size', int2str(obj.parallel_min_chunk_size)];
            end

            if obj.parallel_dispy_debug 
                dispy_batch_options = [dispy_batch_options, sprintf(' --debug %d',  obj.parallel_dispy_debug)  ];
            end
            
            dispy_batch_options = [dispy_batch_options, strjoin(obj.parallel_dispy_opts, ' ')];


            if obj.disable_sims == false
                batch_size = numel(configs);
                fprintf('Sending %d Job Batch to Cluster...\n', batch_size)
                
                pe = pyenv; % get Matlab python environment info
                
                cmd = sprintf('"%s" "%s" --run_batch "%s" %s\n', pe.Executable, which('revs_dispy.py'), obj.parallel_batch_path, dispy_batch_options);
                fprintf('%s\n', cmd)
                batch_fail = system(cmd,'-echo');
               
                if batch_fail
                    error('*** Batch failure, unable to process result files ***');
                end
                    
                pause(2); % wait to make sure result file closes before continuing
                
                fprintf('Collating Result Files...\n');
                obj.process_parallel_result_files();
                
                for s = 1:numel(obj.batch_postprocess_scripts)
                   run(obj.batch_postprocess_scripts{s});
                end

            end
            
            if obj.parallel_batch_cleanup              
                REVS_rmdir(obj.parallel_base_path )
            end
            
        end
                
        %% -----------------------------------------------------------------------
        function [] = salvage_parallel_result_files(obj)
            % in the event of a crashed batch that still has some result
            % files, this method can be used to collate those files to the
            % local output folder
            obj.process_parallel_result_files(strcat(obj.parallel_batch_path, filesep, obj.output_path));
        end
        
        function [] = process_parallel_result_files(obj, result_files)
            % collate results files from each Matlab instance into a single output file
            
            if nargin < 2               
            
                obj.result_files = readcell(fullfile(obj.parallel_batch_path,'batch_result_files.txt'),'Delimiter','','Whitespace','');
                obj.result_files = strcat( {obj.parallel_batch_path},{filesep}, obj.result_files);
            
            elseif iscell(result_files)
                
                obj.result_files = result_files;
            
            else
                % see salvage_parallel_result_files(obj) above:
                % use this option to recover a partially failed run,
                % by passing a folder to look for matching files.
                % Results will be collated but not in numerical order so
                % will require some manual post-processing to determine
                % failed cases and sort results
                
                dir_info = dir(fullfile(result_files,'/*.csv'));
                obj.result_files = {dir_info.name};
                obj.result_files = strcat(result_files, obj.result_files);
                
            end
            
            if sum(strcmp(obj.result_files,'None')) > 0
                warning('Some Result Files Were Lost');
                disp(find(strcmp(obj.result_files,'None')));
            end
            
            % result files include the network path...     
             mkdir(obj.output_path);
             obj.output_file.name = [obj.gen_output_filename(), '_results.csv'];            
                          
             collate_files(obj.result_files, obj.output_file.name);
                         
            %for f = 1:length(obj.result_files)
            %    delete(obj.result_files{f});    % cleanup, but only after all files have been processed...
            %end
                        
        end
                
        %% -----------------------------------------------------------------------
        function [] = prepare_parallel_sim_batch(obj)

            obj.parallel_batch_path = fullfile(obj.parallel_base_path, [obj.network_hostname '_' datestr(now,'yyyy_mm_dd_HH_MM_SS') '_' obj.descriptor]);
            
            REVS_rmdir(obj.parallel_batch_path); % start fresh?            
            pause(1); % wait for delete to finish
                        
            % Make a copy we can tweak
            disp('Copying sim_batch object...');
            sim_batch = copy(obj);   
                        
            sim_batch.start_path = sim_batch.parallel_batch_path;
            sim_batch.descriptor = [sim_batch.network_hostname '_' sim_batch.matlabname '_' sim_batch.descriptor];
            
            disp('Uploading param files...');
            sim_batch.param_path = 'param_files';
            param_destination = fullfile(obj.parallel_batch_path, sim_batch.param_path);
            mkdir(param_destination);
            copyfile(fullfile(obj.param_path,'*'), param_destination) 
            
            if isfolder(obj.script_path)
                disp('Uploading scripts...');
                sim_batch.script_path = 'scripts';
                script_destination = fullfile(obj.parallel_batch_path, sim_batch.script_path);
                mkdir(script_destination);
                copyfile(fullfile(obj.script_path,'*'), script_destination)
            end
             
            disp('Creating Shared Output folder...');
            REVS_mkdir(fullfile(obj.parallel_batch_path, obj.output_path));

            disp('Uploading Matlab Tools...');
            matlab_tools_path = fullfile(fileparts(which('unit_convert')),'..');          
            items = {'datatypes', 'extern', 'functions','utilities'};            
            for i = 1:numel(items)
                copyfile(fullfile(matlab_tools_path,items{i}), fullfile(obj.parallel_batch_path, 'Matlab_Tools', items{i}))
            end
                        
            disp('Uploading REVS_Common...');
            revs_common_path = fileparts(which('REVS_VM'));
            %Copy selected parts because param_files folder is big
            copyfile(fullfile(revs_common_path,'*'), fullfile(obj.parallel_batch_path, 'REVS_Common'))
            
%             revs_common_items = strcat(revs_common_path,{'/datatypes','/functions','/log_packages','/config_packages','/helper_scripts','/libraries','/REVS_VM.mdl'});
%             for c = 1:length(revs_common_items)
%                 copyfile(revs_common_items{c},fullfile(obj.parallel_batch_path, 'REVS_Common/'));
%             end          
                        
            % Create script to configure matlab instances
            fp = fopen(fullfile(obj.parallel_batch_path,'batch_node_prep_script.m'),'w');
            fprintf(fp,"addpath(genpath('Matlab_Tools'));\n");
            fprintf(fp,"addpath(genpath('REVS_Common'));\n");
            fprintf(fp,"warning('off','MATLAB:class:errorParsingClass')\n"); % disable bogus fixed point warning messages (fxptui/getexplorer)
            fprintf(fp,"disp('Node Configuration Complete...')\n");
            fclose(fp);
            
            disp('Expanding config set...');
            length( sim_batch.sim_configs);
            
            disp('Saving sim_batch...');
            save(fullfile(sim_batch.parallel_batch_path, 'sim_batch.mat'),'sim_batch');
                       
        end
        
        function [fname_base] = gen_output_filename(obj, parallel_slave)
                                    
            output_descriptor = obj.descriptor;
            
            if  nargin > 1 && parallel_slave 
                output_descriptor = [obj.network_hostname '_' obj.matlabname '_' output_descriptor];
            end
                       
            output_descriptor = regexprep(output_descriptor,'[ _]*','_');
            
            fname_base = [obj.output_path datestr(obj.timestamp,'yyyy_mm_dd_HH_MM_SS') '_' output_descriptor];
        end
        
        function prepare_output_file(obj, parallel_slave)
                        
            obj.output_file.name = [obj.gen_output_filename(parallel_slave), '_results.csv'];            
            fid = fopen(obj.output_file.name, 'A');
            
            if obj.log_fid < 0
                error('Unable to create batch output file %s',log_filename);
            end

            % Write Something?
            fclose(fid);
                       
        end
        
        
        function prepare_log_file(obj, parallel_slave)
            
            if ~obj.save_output_console
                obj.log_fid = -1;
                return
            end
            
            log_filename = [obj.gen_output_filename(parallel_slave), '.log'];
            obj.log_fid = fopen(log_filename,'W');
            
                
            if obj.log_fid < 0 
                error('Unable to create batch log file %s',log_filename);                
            end
                

        end
        
        function log(obj, msg, varargin)
                        
            log_msg = sprintf(msg, varargin{:});
            
            if obj.show_output_console
                fprintf('%s\n', log_msg );
            end
            
            if obj.log_fid > 0 
                fprintf( obj.log_fid, '%s\n', log_msg );
            end
                
        end
        
        
        function ts_log(obj, msg, varargin)

            log_msg = sprintf(msg, varargin{:});
            
            if length( obj.matlabname ) > 1
                log_msg = sprintf('%s - %s - %s', datetime('now',"Format","uuuu-MM-dd HH:mm:ss"), obj.matlabname, log_msg);
            else
                log_msg = sprintf('%s - %s', datetime('now',"Format","uuuu-MM-dd HH:mm:ss"), log_msg);
            end
            
            if obj.show_output_console
                fprintf('%s\n', log_msg );
            end
            
            if obj.log_fid > 0 
                fprintf( obj.log_fid, '%s\n', log_msg );
            end
                
        end
                    
    end
    
    methods(Static)
        
    end
    
end

